;;; -*- Mode: LISP; Package: YY-GEO; Syntax: Common-Lisp; Base: 10 -*-
;;;
(in-package :yy-geo)

(defclass viewer
	  ()
    ((sequence-number :initarg :id#)
     (window :initarg :window :initform nil)
     (superior :initarg :superior)
     (view-matrix :initform (make-view-matrix))
     (view-point :initform (list 80.0 100.0  50.0))
     (aim-point :initform (list .0 .0 .0))
     (vanishig :initform (list .0 .0 .0))
     (view-factor :initform 1.0) 		;;need check no good value
     (draw-hidden-face :initform t)
     (draw-sequence-number :initform nil)
     (draw-axis :initform t)
     (preset-axis)
     (objects :initform nil)
     (fill-color :initform  yy::*transparent*
		 ;;(yy::make-color :red 60000 :green 60000 :blue 60000)
		 )
     (line-color :initform  yy::*SkyBlue-color*)
              ;;(yy::make-color :red 0 :green 0 :blue 60000)
     (face-color :initform yy::*transparent*)
     (line-width :initform 3)
     (point-radius :initform 3)
     ))

(defmethod print-object ((viewer viewer) stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (SEQUENCE-NUMBER) viewer
    (format stream "<VIEWER ~d>" SEQUENCE-NUMBER)))


(defmethod initialize-instance :after ((view viewer) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (sequence-number superior preset-axis window
			       view-matrix view-point aim-point vanishig) view
    (change-view-matrix view-matrix view-point aim-point vanishig)
    (with-slots (views bodys) superior
      (setf sequence-number (get-next-element views))
      (make-viewer-window view)
      (push view views)
      (let ((obj))
	(multiple-value-setq (obj preset-axis)
	  (yy::with-output-as-presentation
	    (view window)))
	obj)
      (display superior view))
    (redisplay view)))

(defun make-viewer-window (view)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (yy::make-window-instance 
    'viewer-window
    :horizontal-scroll-visible t
    :vertical-scroll-visible t
    :window-region
    (yy::make-region :left 100 :width 700 :bottom 100 :height 500)
    :translate-coordinate :left-top ;;:left-bottom
    :title-bar 'yy::switch-title-bar
    :window-frame 'viewer-frame
;;;    :transform-matrix (yy::make-transform-matrix)
    :drawing-region (yy::make-region)
    :title-bar-string (with-slots (sequence-number) view
			(format nil "View ~d" sequence-number))
    :switches *viewer-switches*
    :superior view))
(defgeneric change-view-point (viewer new-view-point &optional redraw-edge-only))
(defmethod change-view-point ((stream viewer) new-view-point &optional (redraw-edge-only nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (view-matrix view-point aim-point vanishig) stream
    (unless (equal view-point new-view-point)
      ;; need change view-matrix
      (setf view-point new-view-point)
      (change-view-matrix view-matrix view-point aim-point vanishig)
      (redisplay stream t t redraw-edge-only))))
(defgeneric change-aim-point (viewer new-aim-point &optional redraw-edge-only))
(defmethod change-aim-point ((stream viewer) new-aim-point  &optional redraw-edge-only)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (view-matrix view-point aim-point vanishig) stream
    (unless (equal aim-point new-aim-point)
      ;; need change view-matrix
      (setf aim-point new-aim-point)
      (change-view-matrix view-matrix view-point new-aim-point vanishig)
      (redisplay stream t t redraw-edge-only))))

(defmethod change-vanishig ((stream viewer) new-vanishig)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (view-matrix view-point aim-point view-factor vanishig) stream
    (unless (equal vanishig new-vanishig)
      ;; need change view-matrix
      (setf vanishig new-vanishig)
      (change-view-matrix view-matrix view-point aim-point new-vanishig)
      (redisplay stream))))
(defgeneric change-view-factor (viewer new-view-factor &optional redraw-edge-only))
(defmethod change-view-factor ((stream viewer) new-view-factor
			       &optional (redraw-edge-only nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (view-factor objects) stream
    (unless (= view-factor new-view-factor)
      (setf view-factor new-view-factor)
      (redisplay stream t nil redraw-edge-only))))

(defmethod draw-axis ((stream viewer))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (draw-axis preset-axis window) stream
    (when window
      (yy::with-shape-presentation-alone
	(stream window 'viewer preset-axis)
	(when draw-axis
	  (draw-arrow stream '(0 0 0) '(100 0 0) "X")
	  (draw-arrow stream '(0 0 0) '(0 100 0) "Y")
	  (draw-arrow stream '(0 0 0) '(0 0 100) "Z"))))))

(defgeneric draw-arrow (stream position vector &optional string))
(defmethod draw-arrow ((stream viewer) position vector &optional (string ""))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (view-matrix view-factor) stream
	      (multiple-value-bind (x1 y1)
		  (calculate-view-position view-matrix position view-factor)
		(multiple-value-bind (x2 y2)
		    (calculate-view-position view-matrix vector view-factor)
		  (with-slots (window) stream
			      (with-slots (center-x center-y) window
					  (setf (yy::graphic-color window) *r-color*)
					  (yy::draw-line-xy window
							    (+ center-x x1)
							    (+ center-y y1)
							    (+ center-x x2)
							    (+ center-y y2))
					  (yy::draw-string-xy window string (+ center-x x2) (+ center-y y2))))
		  ;;(draw-line-xy stream x1 y1 x2 y2 string)
		  ))))

(defgeneric redisplay (stream &optional recompute-position set-visibility
			      redraw-edge-only))
(defmethod redisplay ((stream viewer)
		      &optional
		      (recompute-position t)
		      (set-visibility t)
		      (redraw-edge-only nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (objects) stream
    (dolist (object objects)
	(redraw-view-solid object recompute-position set-visibility redraw-edge-only))
    (draw-axis stream)))

(defmethod make-view-object ((stream viewer) (object object))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (make-instance 'view-solid
		 :view stream
		 :object object))

(defmethod make-view-object ((stream viewer) (vertex vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (polyhedron) vertex
    (let ((view-solid (get-view-object stream polyhedron)))
      (make-instance 'view-vertex
		     :view stream
		     :solid view-solid
		     :object vertex))))

(defmethod make-view-object ((stream viewer) (edge edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (polyhedron head tail) edge
    (let ((view-solid (get-view-object stream polyhedron)))
      (make-instance 'view-edge
		     :solid view-solid
		     :view stream
		     :object edge))))

(defmethod make-view-object ((stream viewer) (face  face))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (polyhedron vertices) face
    (let ((view-solid (get-view-object stream polyhedron)))
      (make-instance 'view-face
		     :view stream
		     :solid view-solid
		     :object face))))

(defmethod delete-view-object ((stream viewer) (object basic-object))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((view-object (get-view-object stream object)))
    (when view-object
      (with-slots (presentation-object) view-object
	(with-slots (window) stream
	  (yy::delete-presentation-object
	    view-object window
	    (class-name (class-of view-object))
	    presentation-object)))
      (with-slots (solid) view-object
	(remove-depend-view-object solid view-object))
      (with-slots (view-objects) object
	(setf view-objects (delete view-object view-objects))))))


;;(SCL:FUNDEFINE '(METHOD DELETE-VIEW-OBJECT (VIEWER OBJECT)))
(defmethod delete-view-object ((stream viewer) (object object))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((view-object (get-view-object stream object)))
    (with-slots (objects) stream
      (setf objects (delete view-object objects)))
    (with-slots (body view-objects) object
      (setf view-objects (delete view-object view-objects))
      (delete-view-object stream body))))

(defmethod delete-view-object :before ((stream viewer) (object object))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (objects) stream
    (setf objects (delete (get-view-object stream object) objects))))

(defmethod delete-view-object ((stream viewer) (object polyhedron))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((view-object (get-view-object stream object)))
    (when view-object
      (with-slots (presentation-object) view-object
	(with-slots (window) stream
	  (YY::DELETE-FORM-ROOT-PRESENTATION-OBJECT window presentation-object))))
    (with-slots (view-objects) object
      (setf view-objects (delete view-object view-objects)))))

(defmethod delete-view-object :before ((stream viewer) (body polyhedron))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  ;; invisible object is faster delete presentation object
  (let ((solid (get-view-object stream body)))
    (set-view-visible-in-solid solid nil)
    (reset-already-redraw solid)
    (redraw-with-depends solid))
  (with-slots (faces edges vertices) body
    (dolist (face faces)
      (delete-view-object stream face))
    (dolist (edge edges)
      (delete-view-object stream edge))
    (dolist (vertex vertices)
      (delete-view-object stream vertex))))

(defmethod get-view-object ((viewer viewer) (object basic-object))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (view-objects) object 
    (dolist (view-object view-objects)
      (with-slots (view) view-object
	(if (eq view viewer)
	    (return view-object))))))

;;;
(defmethod accept-object-from-view ((stream viewer) view-type)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (window) stream
    (let ((view-object (yy::accept view-type window)))
      (with-slots (object) view-object
	(values object view-object)))))

;;;
(defmethod change-object ((stream viewer) (object object))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (body matrix) object
    (with-slots (window objects) stream
      (dolist (obj objects)
	(with-slots ((geo-obj object)) obj
	(when (equal object geo-obj)
	  (redraw-viewed-object object)))))))

(defmethod change-object ((stream viewer) (objects list))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (dolist (object objects)
    (change-object stream object)))

;;;
(defmethod moving-terminate ((window viewer-window))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (selection) window
    (null selection)))

(defmethod moving-finish ((window viewer-window) state)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (previous-x previous-y selection (view superior)) window
    (yy::draw-prompt "Now Finishing")
    (when selection
      (when (not (zerop (logand (yy::mouse-state-button-state state)
				yy::*mouse-left-1*)))
	(setf selection nil
	      previous-x nil
	      previous-y nil)))))

