;;; -*- Mode: LISP; Package: YY-GEO; Syntax: Common-Lisp; Base: 10 -*-
(in-package :yy-geo)

(defclass basic-view-object ()
    ((object :initarg :object)
     (presentation-object :initform nil)
     (view :initarg :view :accessor view)
     (already-redraw :initform nil :accessor already-redraw)
     (visiblep :initform t :accessor visiblep)))

(defmethod initialize-instance :after ((view-object basic-view-object) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (object) view-object
    (with-slots (view-objects) object
      (push view-object view-objects))))

(defclass view-object-in-solid (basic-view-object)
    ((solid :initform nil :initarg :solid)))

(defmethod initialize-instance :after ((view-object view-object-in-solid) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (solid) view-object
    (with-slots ((solid-presentation presentation-object) view) solid
      (with-slots (window) view
	(with-slots (presentation-object) view-object
	  (yy::with-shape-presentation-alone
	    (solid window 'view-solid solid-presentation :redraw nil)
	    (let ((obj))
	      (multiple-value-setq (obj presentation-object)
		(yy::with-inside-output-as-presentation
		 (view-object window)))
	      obj)))))
    (add-depend-view-object solid view-object)))

(defclass view-vertex (view-object-in-solid yy::position)
    ((faces :initform nil)
     (edges :initform nil)))

(defmethod initialize-instance :after ((vertex view-vertex) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (compute-view-vertex vertex))

(defclass view-edge (view-object-in-solid)
    ((head :initarg :head)
     (tail :initarg :tail)))

(defmethod initialize-instance :after ((edge view-edge) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (head tail object view) edge
    (with-slots ((3d-head head) (3d-tail tail)) object
      (setf head (get-view-object view 3d-head)
	    tail (get-view-object view 3d-tail)))
    (add-depend-view-object head edge)
    (add-depend-view-object tail edge)))

(defclass view-face (view-object-in-solid)
    ((vertices :initform nil :initarg :vertices)
     (edges :initform nil :initarg :edges)
     (front-face :initform nil)))

(defmethod initialize-instance :after ((face view-face) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (view vertices edges object) face
    (with-slots ((face-vertices vertices)(face-edges edges)) object
      (setf vertices (mapcar #'(lambda (x)
				 (get-view-object view x))
			     face-vertices)
	    edges (mapcar #'(lambda (x)
			      (get-view-object view x))
			  face-edges)))
    (dolist (vertex vertices)
      (add-depend-view-object vertex face))))
	

(defclass view-solid (basic-view-object)
    ((vertices :initform nil :initarg :vertices)
     (edges :initform nil)
     (faces :initform nil)
     ))

(defmethod initialize-instance :after ((solid view-solid) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (object presentation-object view) solid
    (with-slots (objects window) view
      ;; Set in Viewer Object
      (push solid objects)
      ;; Make Presentation-Object
      (let ((obj))
	(multiple-value-setq (obj presentation-object)
	  (yy::with-output-as-presentation
	    (solid window)))
	obj))
    ;; Making inside objects
    (with-slots (body) object
      (with-slots (vertices faces edges view-objects) body
	;; Set in Polyhedron Object
	(push solid view-objects)
	(dolist (vertex vertices)
 	  (make-instance 'view-vertex :view view :solid solid :object vertex))
	(dolist (edge edges)
	  (make-instance 'view-edge   :view view :solid solid :object edge))
	(dolist (face faces)
	  (make-instance 'view-face   :view view :solid solid :object face)))))
  (redraw-view-solid solid))

(defmethod  get-view-object ((solid view-solid) (vertex vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (vertices) solid
    (dolist (view-vertex vertices)
      (with-slots (object) view-vertex
	(if (equal object vertex)
	    (return view-vertex))))))

#+debug
(defmethod add-depend-view-object :after ((object1 basic-view-object)
					  (object2 basic-view-object))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots ((view1 view)) object1
    (with-slots ((view2 view)) object2
      (unless (eq view1 view2)
	(format t "~% ~A ~A ~A ~A "view1 object1 view2 object2)))))
    
(defmethod add-depend-view-object ((solid view-solid) (vertex view-vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (vertices) solid
    (push vertex vertices)))
(defmethod add-depend-view-object ((solid view-solid) (edge view-edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (edges) solid
    (push edge edges)))
(defmethod add-depend-view-object ((solid view-solid) (face view-face))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (faces) solid
    (push face faces)))

(defmethod add-depend-view-object ((vertex view-vertex) (edge view-edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (edges) vertex
    (push edge edges)))
(defmethod add-depend-view-object ((vertex view-vertex) (face view-face))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (faces) vertex
    (push face faces)))
(defmethod add-depend-view-object ((vertex view-vertex) (solid view-solid))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots ((solid-in-vertex solid)) vertex
    (setf solid-in-vertex solid)))

(defmethod remove-depend-view-object ((solid view-solid) (vertex view-vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (vertices) solid
    (setf vertices (delete vertex vertices))))
(defmethod remove-depend-view-object ((solid view-solid) (edge view-edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (edges) solid
    (setf edges (delete edge edges))))
(defmethod remove-depend-view-object ((solid view-solid) (face view-face))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (faces) solid
    (setf faces (delete face faces))))

(defmethod remove-depend-view-object ((vertex view-vertex) (edge view-edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (edges) vertex
    (setf edges (delete edge edges))))
(defmethod remove-depend-view-object ((vertex view-vertex) (face view-face))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (faces) vertex
    (setf faces (delete face faces))))

(defmethod compute-view-vertex ((view-vertex view-vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots ((vertex object) view yy::x yy::y) view-vertex
    (with-slots (view-matrix view-factor window) view
      (with-slots (center-x center-y) window
	(with-slots (polyhedron) vertex
	  (with-slots ((3d-object object)) polyhedron
	    (with-slots (matrix) 3d-object
	      (let ((pos (calculate-matrix-vertex matrix vertex)))
		(multiple-value-bind (xpos ypos) 
		    (calculate-view-position view-matrix pos view-factor)
		  (setf yy::x (+ xpos center-x)
			yy::y (+ ypos center-y)))))))))))


(defmethod redraw-viewed-object ((view-vertex view-vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots ((vertex object) view already-redraw visiblep
	       presentation-object) view-vertex
    (unless already-redraw
      (with-slots (window point-radius fill-color line-color
			  draw-sequence-number) view
	(yy::with-inside-shape-presentation
	  (view-vertex window 'view-vertex presentation-object)
	  (when visiblep
	    (setf (yy::filled-type window) yy::*FillSolid*
		  (yy::graphic-color window) fill-color)
	    (yy::draw-circle window view-vertex point-radius)
	    (when draw-sequence-number 
	      (with-slots (sequence-number) vertex
		(setf (yy::graphic-color window) *m-color*)
		(yy::draw-string window (format nil "~d" sequence-number) view-vertex))))))
      (setf already-redraw t))))

(defmethod redraw-viewed-object ((edge view-edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (head tail already-redraw view solid visiblep
		    object presentation-object) edge
    (unless already-redraw
      (with-slots (window line-color fill-color line-width select-color
			  draw-sequence-number) view
	(yy::with-inside-shape-presentation
	  (edge window 'view-edge presentation-object)
	  (when visiblep
	    (setf (yy::graphic-color window) line-color
		  (yy::line-width window) line-width)
	    (yy::draw-line window head tail)
	    (setf (yy::line-width window) 1)
	    (when draw-sequence-number 
	      (with-slots (sequence-number) object
		(let ((x (round (+ 10 (yy::position-x head) (yy::position-x tail)) 2))
		      (y (round (+ (yy::position-y head) (yy::position-y tail)) 2)))
		  (setf (yy::graphic-color window) *g-color*)
		  (yy::draw-string-xy window (format nil "~d" sequence-number) x y)
		  (setf (yy::graphic-color window) line-color))))
	    ))
	(setf already-redraw t)))))

(defmethod redraw-viewed-object ((face view-face))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (vertices already-redraw view solid visiblep presentation-object) face
    (unless already-redraw
      (with-slots ((stream window) face-color) view
	(yy::with-inside-shape-presentation
	  (face stream 'view-face presentation-object)
	  (when visiblep
	    (setf (yy::graphic-color stream) face-color
		  (yy::filled-type stream) yy::*FillSolid*)
	    (apply #'yy::draw-polygon stream vertices)
	    ;;(apply #'yy::draw-polygon-inside stream 2 2 vertices)
	    ))
	(setf already-redraw t)))))

(defmethod redraw-viewed-object ((solid view-solid))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (faces already-redraw view visiblep presentation-object) solid
    (unless already-redraw
      (with-slots ((stream window) fill-color) view
	(when visiblep
	  (setf (yy::filled-type stream) yy::*FillSolid*
		(yy::graphic-color stream) fill-color)
	  (dolist (face faces)
	    (with-slots (vertices front-face) face
	      (when front-face
		(apply #'yy::draw-polygon stream vertices))))))
      (setf already-redraw t))))


(defgeneric redraw-view-solid (view-solid &optional
					  recompute-position
					  set-visibility
					  redraw-edge-only))
(defmethod redraw-view-solid ((solid view-solid)
			      &optional
			      (recompute-position t)
			      (set-visibility t)
			      (redraw-edge-only nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots ((view-vertices vertices)
	       (view-faces faces)
	       (view-edges edges)
	       object view) solid
    (with-slots (visible-p) object
      (when visible-p
	;; calculate positions
	(when recompute-position
	  (dolist (item view-vertices)
	    (compute-view-vertex item))))
      ;; set visibility
      (when set-visibility
	(set-view-visible-in-solid solid nil)
	(when visible-p
	  (with-slots (draw-hidden-face) view
	    (if draw-hidden-face
		(set-view-visible-in-solid solid t)
		(set-front-visible solid))))))
    ;; now redraw start
    (reset-already-redraw solid)
    (redraw-with-depends solid redraw-edge-only)))

(defmethod set-view-visible-in-solid ((solid view-solid) t-or-nil)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots ((view-vertices vertices)
	       (view-faces faces)
	       (view-edges edges)
	       object) solid
    (dolist (item view-edges)
      (setf (visiblep item) t-or-nil))
    (dolist (item view-faces)
      (setf (visiblep item) t-or-nil))
    (dolist (item view-vertices)
      (setf (visiblep item) t-or-nil))
    (setf (visiblep solid) t-or-nil)))

(defmethod set-front-visible ((solid view-solid))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  ;; Set Front Face
  (set-front-face-in-solid solid)
  (with-slots (faces visiblep view) solid
    (setf visiblep t)
    (dolist (face faces)
      (with-slots (edges vertices front-face object) face
	(when front-face
	  (unless edges
	    (with-slots (object) face
	      (with-slots ((object-edges edges)) object
		(setf edges (mapcar #'(lambda (x)
					(get-view-object view x))
				    object-edges)))))
	  (dolist (edge edges)
	      (setf (visiblep edge) t))
	  (dolist (vertex vertices)
	    (setf (visiblep vertex) t))
	  (setf (visiblep face) t)))
      )))

(defmethod set-front-face-in-solid ((solid view-solid))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (object faces view) solid
    (with-slots (view-matrix) view
      (with-slots (matrix) object
	(dolist (face faces)
	  (set-front-face face matrix view-matrix)
	  )))))

(defmethod set-front-face ((face view-face) matrix view-matrix)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (object front-face) face
    (with-slots (normal-vector) object
      (multiple-value-bind (x y z)
	  (calculate-view-position
	    view-matrix
	    (multiple-value-list
	      (calculate-matrix
		matrix
		normal-vector)) 1.0)
	x y z
	(setf front-face (plusp z))))))

;;;
(defmethod reset-already-redraw ((solid view-solid))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots ((view-vertices vertices)
	       (view-faces faces)
	       (view-edges edges)
	       object) solid
    (dolist (item view-edges)
      (setf (already-redraw item) nil))
    (dolist (item view-faces)
      (setf (already-redraw item) nil))
    (dolist (item view-vertices)
      (setf (already-redraw item) nil))
    (setf (already-redraw solid) nil)))

(defmethod reset-already-redraw ((vertex view-vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  #+with-shape-presentation-alone-all
  (with-slots (solid) vertex
    (reset-already-redraw solid))
  #-with-shape-presentation-alone-all
  (with-slots (solid edges faces) vertex
    (setf (already-redraw vertex) nil)
    (setf (already-redraw solid) nil)
    (dolist (edge edges)
      (setf (already-redraw edge) nil))
    (dolist (face faces)
      (setf (already-redraw face) nil)))
  )

(defmethod reset-already-redraw ((edge view-edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  #+with-shape-presentation-alone-all
  (with-slots (solid) edge
    (reset-already-redraw solid))
  #-with-shape-presentation-alone-all
  (with-slots (head tail solid) edge
    (reset-already-redraw head)
    (reset-already-redraw tail))
  )

(defmethod reset-already-redraw ((face view-face))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  #+with-shape-presentation-alone-all
  (with-slots (solid) face
    (reset-already-redraw solid))
  #-with-shape-presentation-alone-all
  (with-slots (vertices) face
    (dolist (vertex vertices)
      (reset-already-redraw vertex)))
  )

(defmethod reset-already-redraw ((view-objects list))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (dolist (view-object view-objects)
    (reset-already-redraw view-object)))

(defmethod redraw-vertex-with-depends ((vertex view-vertex) &optional (edge-only nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (edges faces solid) vertex
    (unless edge-only
      (redraw-viewed-object solid)
      (redraw-viewed-object vertex) 
      (dolist (face faces)
	(redraw-viewed-object face)))
    (dolist (edge edges)
      (redraw-viewed-object edge)))
  )

(defmethod redraw-with-depends ((vertex view-vertex) &optional (edge-only nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  #+with-shape-presentation-alone-all
  (with-slots (solid) vertex
    (redraw-with-depends solid edge-only))
  #-with-shape-presentation-alone-all
  (with-slots (edges faces solid view) vertex
    (with-slots (window) view
      (with-slots (presentation-object) solid
	(yy::with-shape-presentation-alone
	  (solid window 'view-solid presentation-object)
	  (redraw-vertex-with-depends vertex edge-only)))))
  )

(defmethod redraw-with-depends ((edge view-edge) &optional (edge-only nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  #+with-shape-presentation-alone-all
  (with-slots (solid) edge
    (redraw-with-depends solid edge-only))
  #-with-shape-presentation-alone-all
  (with-slots (head tail solid view) edge
    (with-slots (presentation-object) solid
      (with-slots (window) view
	(yy::with-shape-presentation-alone
	  (solid window 'view-solid presentation-object)     
	  (redraw-vertex-with-depends head edge-only)
	  (redraw-vertex-with-depends tail edge-only)))))
  )

(defmethod redraw-with-depends ((face view-face) &optional (edge-only nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (solid) face
    #+with-shape-presentation-alone-all
    (redraw-with-depends solid edge-only))
  #-with-shape-presentation-alone-all
  (with-slots (edges vertices solid view) face
    (with-slots (presentation-object) solid
      (with-slots (window) view
	(yy::with-shape-presentation-alone
	  (solid window 'view-solid presentation-object)
	  (dolist (vertex vertices)
	    (redraw-vertex-with-depends vertex edge-only))))))
  )

(defmethod redraw-with-depends ((solid view-solid) &optional (edge-only nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (vertices faces edges view presentation-object) solid
    (with-slots ((stream window)) view
      (yy::with-shape-presentation-alone-all
	(solid stream 'view-solid presentation-object)
	(unless edge-only
	  (redraw-viewed-object solid)
	  (dolist (item vertices)
	    (redraw-viewed-object item)))
	(dolist (item edges)
	  (redraw-viewed-object item))
	(unless edge-only
	  (dolist (item faces)
	    (redraw-viewed-object item)))
	))))
;;;
(defmethod yy::moved-presented-object ((vertex view-vertex) stream x y)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore stream))
  (with-slots (edges faces solid yy::x yy::y) vertex
    (incf yy::x x)
    (incf yy::y y)
;    (redraw-viewed-object solid)
;    (dolist (face faces)
;      (redraw-viewed-object face))
    (dolist (edge edges)
      (redraw-viewed-object edge))))

(defmethod yy::moved-presented-object ((edge view-edge) stream x y)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (head tail) edge
    (yy::move-presented-object head stream 'view-vertex x y)
    (yy::move-presented-object tail stream 'view-vertex x y)
    ))

(defmethod yy::moved-presented-object ((face view-face) stream x y)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (vertices) face
    (dolist (vertex vertices)
      (yy::move-presented-object vertex stream 'view-vertex x y))))

(defmethod yy::moved-presented-object ((solid view-solid) stream x y)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (vertices) solid
    (dolist (vertex vertices)
      (yy::move-presented-object vertex stream 'view-vertex x y))
    ))

