;;; -*- Mode: LISP; Package: YY-GEO; Syntax: Common-Lisp; Base: 10 -*-
(in-package :yy-geo)

(defclass basic-object ()
    ((sequence-number :initarg :id#)
     (changed :initform nil)
     (property-list :initform nil)
     (attributes :initform nil)
     (view-objects :initform nil)))

(defmethod print-object ((self basic-object) stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (SEQUENCE-NUMBER) self
    (format stream "<~S ~d>" (class-name (class-of self)) SEQUENCE-NUMBER)))

(defclass object (basic-object)
    ((name :initarg :name)	;string
     (superior :initarg :superior)
     (body :initarg :body)	;polyhedron
     (visible-p :initform t)
     (matrix :initform (make-4x4-matrix))	;for geometry
     (global-matrix :initform (make-4x4-matrix))	;for dyna
     (local-matrix :initform (make-4x4-matrix))))	;for dyna

(defmethod print-object ((object object) stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (sequence-number name) object
    (format stream "<~S ~a ~d>" (class-name (class-of object)) name sequence-number)))

(defmethod initialize-instance :after ((object object) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (body) object
    (with-slots ((parent object)) body
      (setf parent object))
    (with-slots (superior) object
      (with-slots (bodys views) superior
	(push object bodys)
	(display superior object)
	(dolist (view views)
	  (make-view-object view object))))))

(defclass polyhedron (basic-object)
    ((object :initarg :object)
     (vertices :initform nil)			;vertex
     (edges :initform nil)			;edges
     (faces :initform nil)			;faces
     ))

(defclass object-in-polyhedron (basic-object)
    ((polyhedron :initarg :polyhedron)))

(defmethod initialize-instance :after ((self object-in-polyhedron) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (polyhedron) self
    (with-slots (view-objects) polyhedron
      (dolist (view-object view-objects)
	(with-slots (view) view-object
	  (make-view-object view self))))))

(defclass vertex (object-in-polyhedron)
    ((x :initform 0.0 :initarg :x :accessor vertex-x)
     (y :initform 0.0 :initarg :y :accessor vertex-y)
     (z :initform 0.0 :initarg :z :accessor vertex-z)))

(defmethod print-object ((vertex vertex) stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (sequence-number x y z) vertex
    (format stream "<VERTEX ~d [~d ~d ~d]>" sequence-number x y z)))

(defmethod initialize-instance :after ((vertex vertex) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (polyhedron sequence-number) vertex
    (with-slots (vertices) polyhedron
      (setf sequence-number (get-next-element vertices))
      (push vertex vertices))))

(defclass edge (object-in-polyhedron)
    ((head :initarg :head)
     (tail :initarg :tail)
     (for-wing)
     (back-wing)
     (ccw-for-wing)
     (ccw-back-wing)
     (right-face)
     (left-face)))

(defmethod initialize-instance :after ((edge edge) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (polyhedron sequence-number) edge
    (with-slots (edges) polyhedron
      (setf sequence-number (get-next-element edges))
      (push edge edges))))

(defmethod print-object ((edge edge) stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (sequence-number polyhedron) edge
    (with-slots ((poly-ID sequence-number)) polyhedron 
      (format stream "<EDGE ~d in POLYHEDRON ~d>" sequence-number  poly-ID))))

(defclass face (object-in-polyhedron)
    ((edges :initform nil)
     (vertices :initform nil :initarg :vertices)
     (normal-vector :initform nil :initarg :normal-vector)))

(defmethod print-object ((face face) stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (sequence-number polyhedron) face
    (with-slots ((poly-ID sequence-number)) polyhedron 
    (format stream "<FACE ~d in POLYHEDRON ~d>" sequence-number poly-ID))))

(defmethod initialize-instance :after ((face face) &rest initargs)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
	   (ignore initargs))
  (with-slots (polyhedron sequence-number) face
    (with-slots (faces) polyhedron
      (setf sequence-number (get-next-element faces))
      (push face faces)))
  (set-normal-vector face))

(defmethod set-normal-vector ((face face))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (vertices normal-vector) face
    (let ((length (length vertices))
	  ans)
      (dotimes (pos length)
	(let ((v1 (nth-with-ring pos vertices length))
	      (v2 (nth-with-ring (+ pos 1) vertices length))
	      (v3 (nth-with-ring (+ pos 2) vertices length)))
	  (setf ans (normal-vector-by-vertex v1 v2 v3))
	  (when ans
	    (return))))
      (setf normal-vector ans))
    (unless normal-vector
      (setf normal-vector '(0 0 0)))
    normal-vector))

(defmethod replace-edge ((target edge) (old edge) (new edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (for-wing back-wing ccw-for-wing ccw-back-wing) target
    (when (eq for-wing old)
      (setf for-wing new))
    (when (eq back-wing old)
      (setf back-wing new))
    (when (eq ccw-for-wing old)
      (setf ccw-for-wing new))
    (when (eq ccw-back-wing old)
      (setf ccw-back-wing new))))

(defmethod remove-object ((object basic-object))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (view-objects) object
    (dolist (view-object view-objects)
      (with-slots (view) view-object
	(delete-view-object view object)))))

(defmethod remove-object :after ((vertex vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (polyhedron) vertex
    (with-slots (vertices) polyhedron
      (setf vertices (delete vertex vertices)))))
(defmethod remove-object :after ((edge edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (polyhedron) edge
    (with-slots (edges) polyhedron
      (setf edges (delete edge edges)))))
(defmethod remove-object :after ((face face))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (polyhedron) face
    (with-slots (faces) polyhedron
      (setf faces (delete face faces)))))

(defmethod remove-object ((objects list))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (dolist (object objects)
    (remove-object object)))

(defmethod make-vertex-from-previous-vector ((vertex vertex) (vector list))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (x y z polyhedron) vertex
    (let ((new-x (+ x (get-vector-x vector)))
	  (new-y (+ y (get-vector-y vector)))
	  (new-z (+ z (get-vector-z vector))))
      (make-vertex polyhedron new-x new-y new-z))))


(defmethod make-vertex ((polyhedron polyhedron) x y z)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (make-instance 'vertex
		 :x (float x)
		 :y (float y)
		 :z (float z)
		 :polyhedron polyhedron))

(defmethod make-edge ((polyhedron polyhedron) (head vertex)(tail vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (make-instance 'edge
		 :head head
		 :tail tail
		 :polyhedron polyhedron))


(defmethod segment-between ((polyhedron polyhedron) (vertex1 vertex) (vertex2 vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (edges) polyhedron
    (dolist (edge edges)
      (with-slots (head tail) edge
	(if (or (and (eql vertex1 head) (eql vertex2 tail))
		(and (eql vertex2 head) (eql vertex1 tail)))
	    (return edge))))))

(defmethod changed-object ((face face))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (vertices edges view-objects) face
    (dolist (view-object view-objects)
      (with-slots (view (view-vertices vertices) (view-edges edges)) view-object
	(dolist (vertex view-vertices)
	  (remove-depend-view-object vertex view-object))
	(setf view-vertices (mapcar #'(lambda (x)
					(get-view-object view x))
				    vertices)
	      view-edges (mapcar #'(lambda (x)
				     (get-view-object view x))
				 edges))
	(set-normal-vector face)
	(dolist (vertex view-vertices)
	  (add-depend-view-object vertex view-object))
	(reset-already-redraw view-object)
	(redraw-with-depends view-object)))))

(defmethod changed-object ((edge edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (head tail view-objects) edge
    (dolist (view-object view-objects)
      (with-slots (view (view-head head) (view-tail tail)) view-object
	(remove-depend-view-object view-head view-object)
	(remove-depend-view-object view-tail view-object)
	(setf view-head (get-view-object view head)
	      view-tail (get-view-object view tail))
	(add-depend-view-object view-head view-object)
	(add-depend-view-object view-tail view-object)
	(reset-already-redraw view-object)
	(redraw-with-depends view-object)))))

(defmethod changed-object ((vertex vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (view-objects) vertex
    (dolist (view-object view-objects)
      (compute-view-vertex view-object)
      (reset-already-redraw view-object)
      (redraw-with-depends view-object))))