;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: lucid-defsys.lisp
;;;  Author: Simoncelli/Heeger
;;;  Description: System definition file for OBVIUS in Lucid Common Lisp.
;;;  Creation Date:  Spring, 1988
;;;  Modified:  Fall, 1989
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Make sure this is Mac Common Lisp
#-MCL
(eval-when (load compile eval)
  (error "This file is meant to be run in Mac Common Lisp only"))

;; Copy and paste are functions exported from CCL package.
;; CL-USER uses CCL package, therefore gets these symbols from CCL.
;; In order to avoid conflicts, we will unexport these symbols from CCL,
;; and unintern them from CL-USER.
(unexport 'copy 'ccl)
(unintern 'copy 'cl-user)
(unexport 'paste 'ccl)
(unintern 'paste 'cl-user)
(unexport '@ 'ccl)
(unintern '@ 'cl-user)
(unexport '@@ 'ccl)
(unintern '@@ 'cl-user)
(unexport '@@@ 'ccl)
(unintern '@@@ 'cl-user)

;;; Loading code.
(setq *autoload-lisp-package* nil)
(ccl::require :lisp-package)
(ccl::require :loop)
(ccl::require :FF)
(ccl::require :RESOURCES)
(ccl::require :quickdraw)

;;; Make clos be a nickname for ccl
(let* ((package (find-package 'ccl))
       (name (package-name package))
       (nicknames (package-nicknames package)))
  (rename-package package name (cons "CLOS" nicknames)))

;;; Set up Obvius package
(defpackage "OBVIUS"  (:nicknames "OBV") (:use "COMMON-LISP"))
(in-package :obvius)

;;; Import only those functions that are in Steele. 
;;; *** we can't find these functions, which need to be imported
;;  (import '(defsubst handler-case condition type-reduce) 'OBVIUS)

;;; Export a few symbols that are defined in this file:
(export '(obv-compile-load obv-source-load 
          obv-require *obvius-version*))

(ccl::provide 'OBVIUS)		   ;Add OBVIUS to the *modules* list
(pushnew :OBVIUS *features*)	   ;Add OBVIUS to the *features* list

(use-package 'OBVIUS 'CL-USER)	   ;CL-USER package "uses" the OBVIUS package

(defconstant *obvius-version* 3.0 "OBVIUS version number.")
	     
;;; This value should be set appropriately for the current source
;;; code.  That is, it should hold a value one greater than the last
;;; patch file that has been incorporated into the source tree.  When
;;; the patches are loaded, this variable will be incremented to one
;;; greater than the number of the last patch file.  Note that this
;;; heppens when you make a new lisp world AND at runtime.  This
;;; allows available patches to be compiled into the world, and newer
;;; patches to be loaded at runtime.
(defvar *starting-patch-file* 1
  "This variable holds the current patch level (or, equivalently, the
number of the next patch file to load) of the OBVIUS environment in
which it is evaluated.  Note that this value is incremented by
load-obvius-patches.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Stuff for loading and compiling OBVIUS files

;;; Load pathnames for this site:
(load (merge-pathnames "mcl-site-paths.lisp" 
                       "Macintosh HD:Obvius:"))

(defvar *obvius-features* nil
  "List of modules loaded using obv-require")

;;; Require a module for obvius.  Works like the Common Lisp require
;;; function, except it looks for pathnames in *obvius-module-plist*
;;; (defined below) and fills in missing directories in filenames with
;;; obvius source-path.  *** This is a little broken -- If user loads
;;; the file(s) manually, they will not go on the *obvius-features*
;;; list.
(defun obv-require (module-name
		    &key
		    (pathnames (or (getf *obvius-module-plist* module-name)
				   (string-downcase module-name)))
		    (initialize t))
  (setq pathnames (mapcar #'(lambda (path) (merge-pathnames path *lisp-source-path*))
                          (if (listp pathnames) pathnames (list pathnames))))
  (when (not (member module-name *obvius-features*))
    (dolist (file pathnames)
      (obv-compile-load file))
    (setq *obvius-features* (pushnew module-name *obvius-features*))
    (when initialize (run-initialization-functions))
    t))					;return t => files loaded

;;; A function to compile-load obvius source code that knows which
;;; directory to look in for source code, and which directory to
;;; compile into.  Also sets optimizations appropriately.  Also, bypasses
;;; the global *source-file-suffix*, which may not be correct for the
;;; OBVIUS source code.
(defun obv-compile-load (basename)
  (obv-compile basename)
  (obv-binary-load (file-namestring basename))
  )

;;; Call compile-if-necessary on OBVIUS source file, compiling into
;;; correct directory.  Also sets optimizations appropriately.
(defun obv-compile (basename)
  (compile-if-necessary
   (merge-pathnames (merge-pathnames basename *lisp-source-path*) ".lisp")
   :output-file (merge-pathnames *binary-file-suffix*
                                 (merge-pathnames *binary-path* basename))
   ))

(defun obv-source-load (basename)
  (load (merge-pathnames
         (merge-pathnames basename *lisp-source-path*) ".lisp")))

(defun obv-binary-load (basename)
  (format t ";;; Loading file ~a~%" basename)
  (load (merge-pathnames *binary-file-suffix*
			 (merge-pathnames basename *binary-path*))))




;;; This compiles source files if the binary (compiled version) is not
;;; up-to-date or if it does not exist.  Takes any kwywords that the
;;; compile-file function takes.  Additional keywords: 1) :umask
;;; keyword allows you to set the read/write/execute protection mask
;;; on the binary file, 2) :optimizations keyword that allow you to
;;; pass options that you would pass to (proclaim (optimize ...)), 3)
;;; :brief-optimize-message keyword that overrides the standard
;;; :optimize-message keyword and prints a one line message.  The
;;; function returns binary file pathname, and a non-nil second value
;;; if the file had to be compiled.  NOTE: We can't use Lucid:load
;;; since it doesn't allow separate binary pathname specification and
;;; wouldn't allow us to modify optimizations or umask.
(defun compile-if-necessary (source-pathname &rest compiler-options
					     &key
					     (output-file source-pathname)
					     &allow-other-keys
					     &aux action)

  ;; Add suffix to source-pathname if not already there.
  (setq source-pathname (merge-pathnames source-pathname *source-file-suffix*))
  ;; Force output-file suffix to be *binary-file-suffix*, fill in any
  ;; holes from source-pathname.
  (setq output-file
	(merge-pathnames *binary-file-suffix*
			 (merge-pathnames output-file source-pathname)))
  (when (not (probe-file source-pathname))
    (error "Source file ~S does not exist" source-pathname))
  (when (or (not (probe-file output-file))
	    (> (file-write-date source-pathname)
	       (file-write-date output-file)))
    (setq action :compiled)
    (format t "~%;;; Compiling file ~S:~%" source-pathname)
    (apply #'compile-file source-pathname :output-file output-file
           compiler-options)
    (format t ";;; Done compiling file ~S~%~%" source-pathname))
  (values output-file action))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Definition of the OBVIUS system and subsystems.

;;; Compilation dependencies:
;;; - Once viewables module is loaded, should be
;;;   to compile images module or operations module in either order.
;;; - Once pictures module is loaded, should be able to
;;;   compile drawings or bltables in either order.

;;; Run-time dependencies (over and above compilation dependencies):
;;; - operations depends on images
;;; - overlay and hardcopy depends on drawings and bltables

;;; For standard configuration of modules, see mcl-site-init.lisp

(defvar *obvius-system-files* 
  (list "fred-hacks"
        "mcl-clos-extensions" "mcl-hacks" "misc"
        "mcl-ffi-macros" "memory" "generic-fns" 
        "array-ops" "arrayio-c" "matrix" "list-ops"))

(defvar *obvius-viewable-files*
  (list "viewable" "viewable-classes" 
        "viewable-matrix" "viewable-sequence" "discrete-function" ))

(defvar *obvius-image-files*
  (list "image" "bit-image" "image-matrix" 
        "image-sequence" "image-pair" "synth")) 
        
(defvar *obvius-operations-files*
  (list "generic-ops" "user-macros"  
        "mcl-image-loops"  "imops" "warp" "fft" "fileio" "filter"))

(defvar *obvius-picture-files*
  (list "picture" "pane" "picture-classes"
        "overlay" "hardcopy"))

(defvar *obvius-drawing-files*
  (list "coord-xforms" "drawing" "graph" "surface-plot"))

(defvar *obvius-blting-files*
  (list "gray" "flipbook" "pasteup"))

(defvar *mcl-window-files*  
  (list
   "mcl-screen" "mcl-window" 
   "mcl-draw"
   ;;"mcl-blt" 
   ;;"mcl-mouse-utilities" "mcl-mouse"
   ;;"mcl-dialogs" "mcl-menus"
   ))

;;; P-list of module symbols and filenames.  Note that filename can be
;;; a single file or a list of files.  Directory defaults to
;;; *lisp-source-path*.

(defvar *obvius-module-plist*
  (list :viewables *obvius-viewable-files*
        :images *obvius-image-files*
        :operations *obvius-operations-files*
        :pictures *obvius-picture-files*
        :drawings *obvius-drawing-files*
        :bltables *obvius-blting-files*
        :mcl-windows *mcl-window-files*
        :color '("color-image" "color-picture")
        :contour-plot "contour-plot"
        :matrix '("svd" "qr" "regress" "row-ops")
        :statistics '("gaussian-noise" "numerical-recipes" "statistics" )
        :stepit "stepit"
        :simplex "simplex"
        :pyramid '("gaussian-pyramid" "pyramid")
        :steer '("steer" "steer-pyramid")
	:psychophysics '("gamma" "psychophysics" "psychophysics-analysis")
        ;;***   :kalman
        ;;***   :Canny
        ;;***   :HIPS  ;Mike Landy's image processing package
        ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Foreign (C) libraries.

;;; Runs a "make" on the foreign code.  Returns a value which indicates 
;;; whether the libraries were remade and need to be loaded again.
;;; this results in new libraries and executables which are placed at 
;;; their respective places.

(defun update-C-libraries ()
  (warn "Update-C-libraries not implemented."))

;;; This is put on the *initialization-functions* list to be executed at startup.
;; Load a few obj files at a time, else we appear to run out of memory.
;; Notes on libraries:
;;   matrix.c requires CSANELib.o (fabs)
;;   matrix.c requires Runtime.o (ULMULT)
;;   imops.c requires Interface.o (GETPTRSIZE)
;;; *** libs might want to live in a different place from source.
;;; Currently, we assume the libraries are in the c-source dir.
(defun load-C-libraries ()
  (let ((libs '("StdCLib.o" "Math.o" "CSANELib.o" "Runtime.o" "Interface.o"))
        (file-lists '(("3d-ops.c.o" "byteswap.c.o" "chartohex.c.o" "convolve.c.o"
                       "color.c.o" "dither.c.o" "edges.c.o" "fft.c.o"
                       "hex-convolve.c.o"  "p-convolve.c.o" "mpw-hacks.c.o")
                      ("imops.c.o"  "matrix.c.o" "mpw-hacks.c.o")
                      ("surface-plot.c.o"  "svd.c.o" "mpw-hacks.c.o"
                       ;; "tiff-access.c.o" ;; undefined TiffPrintDirectory
                       )
                      ("warper.c.o"  "wrap.c.o" "zoom.c.o" "mpw-hacks.c.o"))) 
        (lib-funcs '("memcpy" "memset" "fopen" "fclose" "fread" "fwrite" "fprintf")))
    (setq libs (loop for lib in libs collect
                     (merge-pathnames *binary-path* lib)))
    ;; Load C object files and associated library functions
    (loop for files in file-lists
          for num from 0
          for paths = (loop for file in files collect
                            (merge-pathnames *binary-path* file))
          for ffenv-name = (intern (format nil "LIB-~A" num))
          do (ccl::ff-load paths :replace t :ffenv-name ffenv-name :libraries libs))
    ;; Load the functions we want from the C library
    (ccl::ff-load nil :entry-names lib-funcs :ffenv-name 'libs :replace t :libraries libs))
  (obv-source-load "mcl-ffi"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Loading and Running OBVIUS starting from an empty Lisp World.

;;; Starting from an empty lisp world, this loads all of the obvius code,
;;; compiling when necessary, and then initializes the system.
(defun run-obvius ()
  (load-obvius)				;load code, compiling if necessary
  (initialize-obvius))			;initialize obvius

;;; This function loads all of the obvius lisp code, compiling when
;;; necessary.  It also compiles the C code into a library which will
;;; be loaded by initialize-obvius at run-time.  It does not
;;; initialize OBVIUS!
(defun load-obvius ()
  (format t "~%;;; Loading OBVIUS system files ...~%")
  (unless (probe-file *binary-path*)
    (ccl:create-directory *binary-path*))
  (mapc 'obv-compile-load *obvius-system-files*)
  (let ((site-init (merge-pathnames "mcl-site-init.lisp" *obvius-directory-path*)))
    (if (probe-file site-init)
	(load site-init)
	(warn "Cannot find file ~S" site-init)))
  (format t ";;; Done loading OBVIUS system files.~%~%")
  (format t "~%;;; Loading C Libraries ...~%")
  (load-C-libraries)
  (format t "~%;;; Done Loading C Libraries ...~%")
  )

;;; Load files from the patches directory with names of the form
;;; patch-<num>.lisp in numerical order (according to <num>), starting
;;; with the one designated by the variable *starting-patch-file*.
;;; Increments this variable to be one more than the last patch number
;;; loaded.  *** wasteful re-parsing here....
(defun load-obvius-patches ()
  (declare (special *obvius-directory-path* *source-file-suffix*
		    *starting-patch-file*))
  (let* ((dir (pathname (concatenate 'string *obvius-directory-path* "patches/")))
	 (prefix "patch-")
	 (len (length prefix))
	 (patch-files (directory (merge-pathnames dir *source-file-suffix*)))
	 (*redefinition-action* nil))	;get rid of redefinition-warnings
    (declare (special *redefinition-action*))
    ;; delete filenames that are not of the form "patch-<num>.lisp", where
    ;; <num> is greater than *starting-patch-file*.
    (setq patch-files
	  (delete-if-not
	   #'(lambda (name &aux num)
	       (and (> (length name) len)
		    (string= name prefix :end1 len)
		    (setq num (parse-integer name :start len :junk-allowed t))
		    (>= num *starting-patch-file*)))
	   patch-files
	   :key #'pathname-name))
    ;; sort files according to <num>:
    (setq patch-files
	  (sort patch-files #'<
		:key #'(lambda (p)
			 (parse-integer (pathname-name p) :start len :junk-allowed t))))
    ;; Compile and load the patch files:
    (dolist (file patch-files)
      (obv-compile-load file)
      (setq *starting-patch-file*
	    (1+ (parse-integer (pathname-name file) :start len :junk-allowed t))))
    *starting-patch-file*))

(defun compile-obvius-modules ()
  (loop for plist = *obvius-module-plist* then (cddr plist)
	until (null plist)
	for files = (cadr plist)
	do
	(if (consp files)
	    (mapc #'(lambda (f) (obv-compile f)) files)
	    (obv-compile files))))

(defvar site-patch-filename "site-patches.lisp")

;;; This function must be called to initialize obvius.  It loads patch
;;; files that are not already loaded, loads the foreign (C)
;;; libraries, initializes the window system, loads the user's startup
;;; file and starts the OBVIUS read-eval-print loop (repl).  It is
;;; called by run-obvius (if running from an empty world) and
;;; startup-internal (if running from a world containing OBVIUS).
;;; NOTE: patches and the window-init files are loaded BEFORE
;;; initializing the window system.  We load the C libraries at run
;;; time because otherwise it is hard to patch changes to the C code.

;;; *** no repl, no push-onto-eval-queue
(defun initialize-obvius ()
  (format t "~%;;; >>>---  Initializing OBVIUS version ~A  ---<<<~%~%" *obvius-version*)
  (load-obvius-patches)
  (let ((site-patches (merge-pathnames site-patch-filename *obvius-directory-path*)))
    (if (probe-file site-patches)
	(obv-compile-load site-patches)
	(warn "Cannot find file ~S" site-patches)))
  (load (merge-pathnames
         (merge-pathnames "mcl-window-init" *obvius-directory-path*) ".lisp"))
  (run-initialization-functions)	;window system initializations
  (load (merge-pathnames
         (merge-pathnames "mcl-obvius-init" *obvius-directory-path*) ".lisp"))
  (run-initialization-functions))

#|
(defun initialize-obvius ()
  (format t "~%;;; >>>---  Initializing OBVIUS version ~A  ---<<<~%~%" *obvius-version*)
  (load-C-libraries)   ;*** should this be done as an initialization from lucid-ffi.lisp?
  (load-obvius-patches)
  (let ((site-patches (merge-pathnames site-patch-filename *obvius-directory-path*)))
    (if (probe-file site-patches)
	(obv-compile-load site-patches)
	(warn "Cannot find file ~S" site-patches)))
  (cond ((probe-file "~/.obvius-windows") (load "~/.obvius-windows"))
	((probe-file "~/obvius-window-init.lisp") (load "~/obvius-window-init.lisp"))
	(t  (obv-source-load "lucid-window-init")))
  (run-initialization-functions)	;window system initializations
  (push-onto-eval-queue
   (cond ((probe-file "~/.obvius") '(load "~/.obvius"))
	 ((probe-file "~/obvius-init.lisp") '(load "~/obvius-init.lisp"))
	 (t '(obv-source-load "lucid-obvius-init"))))
  (push-onto-eval-queue '(run-initialization-functions)) ;Other initializations
  (repl))
|#

;;; Code in various files puts functions (or symbols) on this list.
;;; They will be eval'ed by initialize-obvius at startup.
;;; *** should probably do this with a lexical closure (hide the list).
(defvar *initialization-functions* nil)

;;; Run the functions on the *initialization-functions* list in
;;; reverse order (assume that they were added to the list using
;;; push or pushnew).
(defun run-initialization-functions ()
  (let (list-o-functions)
    ;; Do this to make sure multiple processes don't bash...
    (psetq list-o-functions *initialization-functions*
	   *initialization-functions* nil)
    (mapc #'(lambda (func) (funcall func)) (nreverse list-o-functions))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Building and saving a Lisp World containing OBVIUS.

;;; Save a lisp-world containing obvius.  Note that existing patch
;;; files are loaded before saving the world.  Also compiles all
;;; modules, so that they can be quickly loaded at runtime.
(defun make-obvius (&optional filename)
  (cerror "load obvius without saving a world"
          "make-obvius not implemented")
  (load-obvius)
  (load-obvius-patches)
  (save-lisp-world filename)
  (compile-obvius-modules))

;;; This function saves the current lisp world into the given pathname.
(defun save-lisp-world (pathname)
  (ccl::save-application pathname))
	
;;;; Define Lisp Startup hook to initialize obvius:
;;; *** don't know what needs to be done here for MCL.
