;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: mcl-screen.lisp
;;;  Author: Heeger
;;;  Description: 
;;;  Creation Date: 12/93 modified from lv-screen.lisp
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '())

;;; Make sure this is Mac Common Lisp
#-MCL
(eval-when (load compile eval)
  (error "This file is meant to be run in Mac Common Lisp only"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(eval-when (eval load)
  (pushnew 'initialize-mcl-window-system *initialization-functions*))

(defun initialize-mcl-window-system ()
  (make-screen))

;;; Called by initialize-mcl-window-system.  Also called by (current-screen)
;;; if there is no *current-pane* or *screen-list*.
(defun make-screen (&rest initargs)
  (apply 'make-instance 'mcl-screen initargs))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; mcl-screen object

;;; The values in the gray-lut correspond to sequential indexes into
;;; the gray colors of the screen's actual colormap.  The values in
;;; color-lut are also a meta cmap for color dithering color pictures.

;;; *** gray-shades is redundant: it is the size of gray-lut (but it
;;; is used by the dialog box that is automatically generated from the
;;; slots).  -DH
(def-simple-class mcl-screen (screen)
  ((selected-color :initform :red)
   (foreground :type t
	       :documentation "Foreground color of screen.  This is used as a \
foreground color for all panes on this screen.  Value can be a color name \
or a list of length three containing RGB values between 0.0 and 1.0")
   (background :type t
	       :documentation "Background color of screen.  This is used as a \
background color for all panes on this screen.  Value can be a color name \
or a list of length three containing RGB values between 0.0 and 1.0")
   gray-lut
   color-lut
   pseudo-color-lut
   registered-colors			;array of all registered colors
   (gray-shades :type (integer 2 255)
		:documentation
		"Determines the number of gray levels allocated in the colormap.")
   (gray-gamma :type (float 0.1 10.0)
	       :documentation
	       "Determines the gamma correction factor of the colormap.")
   (gray-dither :initform 64 :type (or (member nil t) number)
		:documentation
		"Determines whether pictures are dithered into the gray shades in \
the colormap.  If t, dither.  If nil, just quantize.  If a number, the pictures \
will be dithered if the value of gray-shades is less than this number.")
   (rgb-bits :type cons
	     :documentation
	     "List of three integers that determine the number of bits for \
dithering color pictures.")
   (pseudo-colors :type (integer 0 255)
		  :documentation
		  "Determines the number of pseudo colors allocated in the colormap.")
   (pseudo-color-function :documentation
			  "Function used to allocate pseudo-colors, argument is an integer, returns a list of rgb values.")
   )
  (:default-initargs
     :gray-shades 128
     :gray-gamma 0.9
     :rgb-bits '(2 2 2)
     :pseudo-colors 7
     :pseudo-color-function #'pseudo-color-ramp
     :background :white
     :foreground :black))

;;; Add to screen list, make array for registered-colors, make color map.
(defmethod initialize-instance ((screen mcl-screen) &rest initargs)
  (setq *screen-list* (nconc *screen-list* (list screen))) 
  (setf (getf initargs :registered-colors)
	(or (getf initargs :registered-colors)
	    (make-array 256)))
  ;;; *** make a color map here ***
  (apply #'call-next-method screen initargs)
  screen)

;;; This is called by initialize-instance and reinitialize-instance.
(defmethod shared-initialize ((screen mcl-screen) slot-names &rest initargs)
  (declare (ignore slot-names))
  ;; Fill slots according to initargs (some reset below).
  (call-next-method)			
  ;; Register the background color (weird to use the
  ;; setf methods to reset the slot, but it is correct).
  (when (getf initargs :background)
    (setf (background screen) (background screen)))
  ;; Allocate grays and register them.  Weird to use the
  ;; setf method to reset the slot, but it is correct (invokes set-gray-lut).
  (when (or (getf initargs :gray-gamma) (getf initargs :gray-shades))
    (setf (gray-shades screen) (gray-shades screen)))
  ;; Allocate colors and register them.
  (when (getf initargs :rgb-bits)
    (set-color-lut screen))
  (when (getf initargs :pseudo-colors)
    (allocate-pseudo-colors screen))
  screen)

(defmethod settable-parameters ((class-name (eql 'mcl-screen)))
  (append '(background
            gray-shades gray-gamma gray-dither
	    rgb-bits pseudo-colors pseudo-color-function)
	  (call-next-method)))

(defmethod (setf gray-shades) (num (screen mcl-screen))
  (setq num (clip num 2 (- 256
                           (* (expt 2 (first (rgb-bits screen)))
                              (expt 2 (second (rgb-bits screen)))
                              (expt 2 (third (rgb-bits screen)))))))
  (setf (slot-value screen 'gray-shades) num)
  (set-gray-lut screen :old-lut (gray-lut screen)))

(defmethod (setf gray-gamma) (num (screen mcl-screen))
  (setf (slot-value screen 'gray-gamma) num)
  (set-gray-lut screen :old-lut (gray-lut screen)))

(defmethod (setf rgb-bits) (list (screen mcl-screen))
  (unless (and (plusp (first list)) (plusp (second list)) (plusp (third list)))
    (error "Must be at least 1 bit for each r, g, and b."))
  (when (> (apply '+ list) 7)
    (error "Can not allocate more than 7 bits for color."))
  (setf (slot-value screen 'rgb-bits) list)
  (set-color-lut screen))

(defmethod (setf pseudo-colors) (num (screen mcl-screen))
  (setf (slot-value screen 'pseudo-colors) num)
  (allocate-pseudo-colors screen :old-lut (pseudo-color-lut screen)))

(defmethod (setf pseudo-color-function) (function (screen mcl-screen))
  (unless (fboundp function)
    (error "~a must be a symbol with a function binding" function))
  (setf function (symbol-function function))
  (setf (slot-value screen 'pseudo-color-function) function)
  (allocate-pseudo-colors screen :old-lut (pseudo-color-lut screen)))

;;; convert from rgb list to encoded color
(defmethod convert ((color-list list) (type (eql :encoded-color)))
  (cond ((every #'ccl::fixnump color-list)
         (apply #'ccl::make-color color-list))
        ((every #'floatp color-list)
         (apply #'ccl::make-color 
                (mapcar #'(lambda (x) (clip (floor (* x 65535)) 0 65535))
                        color-list)))
        (t (error "color ~a must be a list of fixnums or a list of floats"
                  color-list))))

(defmethod convert ((color fixnum) (type (eql :encoded-color)))
  color)

(defvar *obv-colors*)
(setq *obv-colors* (list :white ccl::*white-color*
                         :black ccl::*black-color*
                         :red ccl::*red-color*
                         :pink ccl::*pink-color*
                         :orange ccl::*orange-color*
                         :yellow ccl::*yellow-color*
                         :green ccl::*green-color*
                         :dark-green ccl::*dark-green-color*
                         :light-blue ccl::*light-blue-color*
                         :blue ccl::*blue-color*
                         :purple ccl::*purple-color*
                         :brown ccl::*brown-color*
                         :tan ccl::*tan-color*
                         :gray ccl::*gray-color*
                         :light-gray ccl::*light-gray-color*
                         :dark-gray ccl::*dark-gray-color*))

(defmethod convert ((color symbol) (type (eql :encoded-color)))
  (let ((encoded-color (getf *obv-colors* color)))
    (unless encoded-color
      (error "Color ~a is not available.  Options are ~a"
             color *obv-colors*))
    encoded-color))

;;; convert from encoded color to RGB triplet
(defmethod convert ((color fixnum) (type (eql :rgb-color)))
  (multiple-value-bind (r g b) (ccl::color-values color)
    (mapcar #'(lambda (x) (/ x 65535.0)) (list r g b))))

(defmethod convert ((color-list list) (type (eql :rgb-color)))
  (cond ((every #'floatp color-list)
         color-list)
        ((every #'ccl::fixnump color-list)
         (mapcar #'(lambda (x) (/ x 65535.0)) color-list))))

(defmethod (setf background) (color (screen mcl-screen))
  (dolist (p (pane-list screen))
    (setf (background p) color)) ;must use setf here to achieve proper side-effects.
  (setf (slot-value screen 'background) color)
  color)

(defmethod depth ((screen mcl-screen))
  ccl::*color-available*)

;;; *** This really only needs to be done on color-pictures, grays, and
;;; flipbooks of grays.
(defmethod set-not-current ((screen mcl-screen))
  (loop for pane in (pane-list screen) do
	(loop for pic in (picture-stack pane) do
	      (set-not-current pic))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Allocates grays in the colormap.  Makes internal lookup table
;;; to find allocated grays.  Destroys old colors.
(defmethod set-gray-lut ((screen mcl-screen)
			 &key
			 (gray-size (gray-shades screen))
			 (gray-gamma (gray-gamma screen))
			 (old-lut (gray-lut screen)))
  screen)

;;; Allocates colors in the X colormap for dithering color pictures.  Makes
;;; internal lookup table to find allocated colors.  Destroys old colors.
(defmethod set-color-lut ((screen mcl-screen))
  screen)

;;; Allocates colors in the X colormap for pseudo-color pictures.
(defmethod allocate-pseudo-colors
    ((screen mcl-screen)
     &key old-lut)
  screen)

;;; Example pseudo-color-function:
(defun pseudo-color-ramp (i size)
  (let* ((rval (/ i (- size 1)))
	 (gval (- 1 rval))
	 (bval (* 1/2 (- 1 (abs (- (* 2 rval) 1)))))
	 )
    ;;(print-db rval gval bval)
    (list rval gval bval)))



;;; Local Variables:
;;; buffer-read-only: t 
;;; fill-column: 79
;;; End:
