;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File:  x-window.lisp
;;;  Author: Simoncelli/Heeger
;;;  Description: x-windows using LispView
;;;  Creation Date: summer '90
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)

;;; These are defined in optional file x-control-panel.  Put them here
;;; to avoid export errors.
(export '(make-control-panel destroy-control-panel))

;;; This code requires the LispView interface to X windows (written by
;;; Rmori@Eng.Sun.Com and hmuller@Eng.Sun.Com).
#-:LispView
(error "You must load the LispVIew interface to X windows to run ~
        this version of OBVIUS-~a.  LispView is available from Sun ~
        Microsystems, or via anonymous ftp from MIT." obv::*obvius-version*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; X-pane object

;;; This method must be provided for each type of system-dependent screen.
(defmethod make-pane ((screen X-screen) &rest keys &key left bottom right top width height
		      border-width &allow-other-keys)
  (declare (ignore left bottom right top width height border-width))
  (apply 'make-instance 'x-pane :screen-of screen keys))

;;; The underlying window object, which inherits from the generic
;;; obvius:pane object and the lispview:base-window object.  Initargs for
;;; lispview:base-window are (:top :left :width :height :right :bottom
;;; :depth :foreground :badckground :backing-store :border-width) ***
;;; Add a :confirm-quit function to ask the user if they're sure!
(def-simple-class X-pane (pane lispview:base-window)
  ()
  (:default-initargs
    :width 300
    :height 256
    :border-width 1
    :interests *default-pane-interests*
    :mapped t
    :label "obvius pane"
    :keyboard-focus-mode :passive
    ;; :confirm-quit #'null		;prevent them from being destroyed
    :show-resize-corners t
    :icon (make-standard-pane-icon)
    ))

;;; NOTE: cannot re-use the icon itself - this really screws up the
;;; window system!
(let ((image nil))
  (defun make-standard-pane-icon ()
    (unless image
      (setq image (make-instance
		   'lispview:image
		   :filename
		   (merge-pathnames "obv.icon" *obvius-directory-path*))))
    (make-instance 'lispview:icon :label (list "" image))))

(defmethod initialize-instance :around ((pane X-pane) &rest initargs)
  (let ((screen (getf initargs :screen-of)))
    (unless (and screen (typep screen 'X-screen))
      (error "Must provide an X-screen as a :screen-of argument"))
    ;;enforce these items for consistency!
    (setf (getf initargs :display) (X-display screen))
    (setf (getf initargs :background) (background screen))
    (setf (getf initargs :foreground) (foreground screen))
    (apply #'call-next-method pane initargs)))

#|
;;; Destructively modify the icon.  Default version copies the
;;; standard icon into the image.  This can be specialized on picture
;;; types to provide an icon.
(defmethod compute-icon (pic image)
  )

;;; *** Modify this to dynamically set the icon to have correct string
;;; and icon which resembles the picture on top of the stack.
(defmethod (setf lv::closed) (val (pane x-pane))
  (with-slots (lispview:icon) pane
    (setf (icon-string lispview:icon)
	  (lispview:label pane))
    (compute-icon (car (picture-stack pane)) (icon-image lispview:icon)))
  (call-next-method))
|#

;;; Modify to set up picture-specific mouse bindings of pane.
(defmethod draw-pane ((pane X-pane) &rest keys)
  (unless (eq (lv::status pane) :destroyed)
    (prog1 
	(call-next-method)
      (set-picture-specific-mouse-bindings pane))))

;;; Modify to destroy picture-parameter dialogs corresponding to the
;;; picture.
(defmethod destroy :around ((pic picture) &key &allow-other-keys)
   (declare (special *slot-value-dialogs*))
   (let* ((screen (screen-of (pane-of pic)))
	  dlg)
     (when (typep screen 'X-screen)
       (setq dlg (gethash pic *slot-value-dialogs*))
       (when dlg (setf (lispview:status dlg) :destroyed)))
     (call-next-method)))

;;; *** Could be smarter about this: only need to draw a portion of
;;; the window!
(defmethod lispview:receive-event
    ((pane X-pane) (interest lispview:damage-interest) event)
  (declare (ignore event))
  (draw-pane pane :clear nil))

;;; Resize events: not used.
;(defmethod lispview:receive-event
;    ((pane X-pane) (interest (eql :BOUNDING-REGION-NOTIFICATION)) event))

;;; Background and foreground methods ust call the corresponding LispView methods.
(defmethod background ((pane X-pane))
  (lispview:background pane))

(defmethod (setf background) (color (pane X-pane))
  (setf (lispview:background pane) color))

(defmethod foreground ((pane X-pane))
  (lispview:foreground pane))

(defmethod (setf foreground) (color (pane X-pane))
  (setf (lispview:foreground pane) color))

(defmethod x-dim ((pane X-pane))
  (lispview:region-width (lispview:bounding-region pane)))

(defmethod y-dim ((pane X-pane))
  (lispview:region-height (lispview:bounding-region pane)))

(defmethod dimensions ((pane X-pane))
  (let ((rgn (lispview:bounding-region pane)))
    (list (lispview:region-height rgn)
	  (lispview:region-width rgn))))

(defmethod clear ((pane X-pane)
		  &key (y0 0) (x0 0) 
		  (y1 (y-dim pane))
		  (x1 (x-dim pane))
		  (color (background pane)))
  (draw-rect pane y0 x0 y1 x1 :foreground color))

;; *** Should change border color or something to indicate which is
;; selected.
(defmethod set-selected-pane ((pane X-pane))
  (when (not (eq pane *current-pane*))
    (call-next-method)))

;;; *** Kludged to trim ugly Common Lisp right decimal point
(defmethod set-pane-title-bar ((pane X-pane) title)
  (setf (lispview:label pane) (string-right-trim "." title)))

;;; Destroy the picture stack and clean up.
(defmethod (setf lispview:status) ((val (eql :destroyed)) (pane X-pane))
  (destroy pane))

(defmethod destroy ((pane X-pane) &key &allow-other-keys)
  (call-next-method)
  ;; *** Kludge: change class to avoid calls with above.
  (setq pane (change-class pane 'lispview::base-window))
  (setf (lispview:status pane) :destroyed))

(defmethod depth ((pane X-pane))
  (lispview:depth pane))

(defmethod X-display ((pane X-pane))
  (lispview:display pane))


;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
