;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File:  x-draw.lisp
;;;  Author: Heeger/Simoncelli
;;;  Description: x-windows using LISPVIEW (lispview)
;;;  Creation Date: summer '90
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '())

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; system dependent stuff to draw to lispview panes

(defmethod font ((pane X-pane))
  (lispview:font (lispview:graphics-context (X-display pane))))

(defmethod font-height ((font lv:font))
  (+ (lispview:font-ascent font) (lispview:font-descent font)))

(defmethod string-width ((font lv:font) string)
  (lv:string-width font string))

(defmethod draw-text ((pane X-pane) y x string &rest keys)
  (apply #'lispview:draw-string pane x (- y 2) string keys))

(defmethod draw-line ((pane X-pane) from-y from-x to-y to-x &rest keys &key foreground)
  (when foreground
    (setf (getf keys :foreground) (convert foreground 'lv:color)))
  (apply #'lispview:draw-line pane from-x from-y to-x to-y keys))

;;; *** fill-p should be a keyword
(defmethod draw-rect ((pane X-pane) y0 x0 y1 x1 &rest keys &key foreground)
  (when foreground
    (setf (getf keys :foreground) (convert foreground 'lv:color)))
  (apply #'lispview:draw-rectangle pane x0 y0 (- x1 x0) (- y1 y0)
	 :fill-p t keys))

;;; *** fill-p should be a keyword
(defmethod draw-circle ((pane X-pane) y-center x-center radius &rest keys &key foreground)
  (when foreground
    (setf (getf keys :foreground) (convert foreground 'lv:color)))
  (apply #'lispview:draw-arc pane
	 (- x-center radius) (- y-center radius)
	 (* 2 radius) (* 2 radius) 0 360
	 :fill-p t keys))

;;;; NOTE: this is actually faster and more cons-efficient than
;;;; calling the current version of lispview:draw-lines, since that
;;;; requires consing lists of vectors.  Call it with a LispVIew object!
(defmethod draw-lines ((X-thing lispview::drawable) y0 x0 y1 x1
		       &key
		       foreground line-width line-style
		       x-offset y-offset)
  (declare (type (array fixnum (*)) x0 x1 y0 y1)
	   (fixnum x-offset y-offset))
  (when foreground
    (setq foreground (convert foreground 'lv:color)))
  (let* ((X-display (lispview:display X-thing))
	 (gc (lispview:graphics-context X-display)))
    (lispview:with-graphics-context (gc :foreground foreground
					:line-style line-style
					:line-width line-width)
      (lispview:with-output-buffering X-display
	(cond ((or x-offset y-offset)
	       (setq x-offset (or x-offset 0))
	       (setq y-offset (or y-offset 0))
	       (dotimes (i (length y0))
		 (lispview:draw-line X-thing
				     (+ (aref x0 i) x-offset)
				     (+ (aref y0 i) y-offset)
				     (+ (aref x1 i) x-offset)
				     (+ (aref y1 i) y-offset)
				     :gc gc)))
	      (t (dotimes (i (length y0))
		   (lispview:draw-line X-thing
				       (aref x0 i) (aref y0 i)
				       (aref x1 i) (aref y1 i)
				       :gc gc))))))))

(defmethod draw-circles ((X-thing lispview::drawable) yarr xarr
			 &key foreground line-width line-style
			 fill-p radius)
  (declare (type (array fixnum (*)) xarr yarr))
  (when foreground
    (setq foreground (convert foreground 'lv:color)))
  (let* ((X-display (lispview:display X-thing))
	 (gc (lispview:graphics-context X-display)))
    (lispview:with-graphics-context (gc :foreground foreground
					:line-style line-style
					:line-width line-width)
      (lispview:with-output-buffering X-display
	(dotimes (i (length yarr))
	  (lispview:draw-arc X-thing
			     (- (aref xarr i) radius) (- (aref yarr i) radius)
			     (* 2 radius) (* 2 radius)
			     0 360
			     :fill-p fill-p :gc gc))))))

(defmethod draw-squares ((X-thing lispview::drawable) yarr xarr
			 &key foreground line-style line-width
			 fill-p size)
  (declare (type (array fixnum (*)) xarr yarr))
  (when foreground
    (setq foreground (convert foreground 'lv:color)))
  (let* ((X-display (lispview:display X-thing))
	 (gc (lispview:graphics-context X-display)))
    (lispview:with-graphics-context (gc :foreground foreground
					:line-style line-style
					:line-width line-width)
      (lispview:with-output-buffering X-display
	(dotimes (i (length yarr))
	  (lispview:draw-rectangle X-thing
				   (- (aref xarr i) size) (- (aref yarr i) size)
				   (* 2 size) (* 2 size)
				   :fill-p fill-p :gc gc))))))

(defmethod draw-rects ((X-thing lispview::drawable) y0 x0 y1 x1
		       &key
		       fill-p foreground line-width line-style)
  (declare (type (array fixnum (*)) x0 x1 y0 y1))
  (when foreground
    (setq foreground (convert foreground 'lv:color)))
  (let* ((X-display (lispview:display X-thing))
	 (gc (lispview:graphics-context X-display))
	 xo xs yo ys)
    (lispview:with-graphics-context (gc :foreground foreground
				        :line-style line-style
					:line-width line-width)
      (lispview:with-output-buffering X-display
	(dotimes (i (length y0))
	  (setq xs (abs (- (aref x1 i) (aref x0 i)))
		xo (min (aref x0 i) (aref x1 i))
		ys (abs (- (aref y1 i) (aref y0 i)))
		yo (min (aref y0 i) (aref y1 i)))
	  (lispview:draw-rectangle X-thing xo yo xs ys
				   :fill-p fill-p
				   :gc gc))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; draw-graph for X-drawables

(defmethod draw-graph ((drawable X-pane)
		       data graph-type
		       graph->frob-y graph->frob-x data->frob-y data->frob-x
		       y-range x-range y-axis x-axis
		       y-tick-step x-tick-step y-tick-length x-tick-length
		       y-tick-gap x-tick-gap y-tick-format-string x-tick-format-string
		       y-label x-label
		       &key
		       (font (font drawable))
		       color axis-color line-width
		       plot-symbol fill-symbol-p symbol-size
		       x-offset y-offset)
  (declare (ignore y-offset x-offset y-label x-label))
  (let* ((X-display (lispview:display drawable))
	 (gc (lispview:graphics-context X-display)))
    (when color
      (setq color (convert color 'lv:color)))
    (when axis-color
      (setq axis-color (convert axis-color 'lv:color)))
    (lispview:with-graphics-context (gc :foreground axis-color)
      (lispview::with-output-buffering X-display
	(when x-axis
	  (plot-x-axis drawable graph->frob-x (transform-point graph->frob-y x-axis)
		       x-range x-tick-step x-tick-length x-tick-gap
		       x-tick-format-string font))
	(when y-axis
	  (plot-y-axis drawable graph->frob-y (transform-point graph->frob-x y-axis)
		       y-range y-tick-step y-tick-length y-tick-gap
		       y-tick-format-string font))))
    (lispview:with-graphics-context
	(gc :line-width line-width
	    :foreground color
	    :clip-mask (lispview:make-region
			:top (transform-point graph->frob-y (cadr y-range))
			:left (transform-point graph->frob-x (car x-range))
			:bottom (transform-point graph->frob-y (car y-range))
			:right (1+ (transform-point graph->frob-x (cadr x-range)))))
      (let ((x-axis-pos
	     (transform-point graph->frob-y (or x-axis (apply 'clip 0 y-range)))))
	(plot-data graph-type data drawable
		   data->frob-y data->frob-x x-axis-pos
		   plot-symbol fill-symbol-p symbol-size)))))


;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
