;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: lv-color-patch.lisp
;;;  Author: Simoncelli, Heeger
;;;  Description: keep track of colors registered on obvius screens
;;;  Creation Date: 1991
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '())


;;; convert from symbol or list to lv::color
(defmethod convert ((sym symbol) (type (eql 'lispview:color)))
  (find-true-color :name sym :if-not-found :warn))

(defmethod convert ((ls list) (type (eql 'lispview:color)))
  (when (equal (car ls) 'quote)
    (setq ls (cadr ls)))
  (if (every #'numberp ls)
      (find-true-color :red (clip (float (car ls)) 0.0 1.0)
		       :green (clip (float (cadr ls)) 0.0 1.0)
		       :blue (clip (float (caddr ls)) 0.0 1.0)
		       :if-not-found :warn)
      (progn
	(warn "Couldn't allocate color specified ~a, substituting :white" ls)
	(find-true-color :name :white :if-not-found :warn))))

(defmethod convert ((col lv:color) (type (eql 'lispview:color)))
  col)


;;; Calls lispview:find-color with :if-not-found :realize.  If color
;;; returned is an approximation, we destroy it, giving a warning or
;;; error according to :if-not-found, and we return nil.
(defun find-true-color (&rest keys &key if-not-found &allow-other-keys)
  (remf keys :if-not-found)
  (let ((color (apply 'lispview:find-color :if-not-found :realize keys)))
    (when (not (eq color (lispview::proxy-color color)))
      (setf (lispview:status color) :destroyed)
      (setq color nil)
      (case if-not-found
	(:warn (warn "Color specified by ~A could not be allocated." keys))
	(:error (error "Color specified by ~A could not be allocated." keys))))
    color))

;;; Make-color is like lispview:find-color, but ALWAYS makes a NEW
;;; lispview:color object.  Registers the color in the registered-colors table.
;;; Also takes a keyword :if-not-exact specifying what to do if LispView
;;; returns a color which is an approximation to the requested color.  This is
;;; to fix the LispView bug: If I destroy a lv:color which corresponds to a
;;; reserved (system) colormap cell (pixel), LispView will consider that cell
;;; to be available (i.e.  a subsequent call to find-color will return an
;;; lv:color indexed to that cell)!  But the actual color of that cell will not
;;; be changed.  When a lv:color object is destroyed, the corresponding
;;; colormap cell is considered available free ONLY if there are no other
;;; lv:color objects using that cell.
(defun make-color (&rest find-color-args &key (if-not-exact :warn)
			 (display lispview:*default-display*)
			 &allow-other-keys)
  (remf find-color-args :if-not-exact)
  (let ((screen (find display *screen-list* :key 'X-display)))
    ;;if not an 8bit-X-screen, just do find-color
    (if (and screen (typep screen '8bit-X-screen))
	(multiple-value-bind (col status)
	    (apply 'lv:find-color find-color-args)
	  ;; if color is not exactly what was requested:
	  (when col
	    (unless (eq col (lispview::proxy-color col))
	      (case if-not-exact
		(:warn (warn "Color ~A is an approximation to the spec ~%~S"
			     col find-color-args))
		(:error
		 (error "No more colormap entries: cannot allocate color for spec ~%~S"
			find-color-args))
		(:cerror (cerror "Use the approximation~2*" ;ignore format args
				 "Color ~A is an approximation to the spec ~%~S"
				 col find-color-args))
		((nil) (setq col nil))))
	    ;;(format t "registering color ~a~%" col)
	    (when col
	      (if (and (eq status :already-created)
		       (aref (registered-colors screen) (lispview:pixel col)))
		  (incf (cdr (aref (registered-colors screen) (lispview:pixel col))))
		  (setf (aref (registered-colors screen) (lispview:pixel col))
			(cons col 1))))
	    (values col status)))
	(apply 'lv:find-color find-color-args))))
     
;;; Only call the lispview function (which de-alloctes the underlying
;;; window system resources) if there are no other lispview:colors
;;; relying on the same colormap cell (pixel)!
(defmethod (setf lv:status) :around ((state (eql :destroyed)) (col lv:color))
  ;;(format t "destroying color ~A~%" col)
  (let ((screen (find (lv:display col) *screen-list* :key 'X-display)))
    ;;if not an 8bit-X-screen, just destroy it
    (if (and screen (typep screen '8bit-X-screen))
	(let ((registered-colors (registered-colors screen)))
	  (if (aref registered-colors (lispview:pixel col))
	      (when (<= (decf (cdr (aref registered-colors (lispview:pixel col)))) 0)
		;;(format t "removing color ~A from registered-colors table~%" col)
		(setf (aref registered-colors (lispview:pixel col)) nil)
		(call-next-method))	;destroy the thing
	      (call-next-method)))
	(call-next-method))))

#|
;;; look at table of registered colors
(loop for i from 0 below 256 
      for registered-color = (aref (registered-colors (current-screen)) i)
      with count = 0
      do
      (when registered-color
	(incf count)
	(format t "~a:  ~a~%" i registered-color))
      finally (return count))
|#


;;; Local Variables:
;;; buffer-read-only: t 
;;; fill-column: 79
;;; End:
