;;; Draws anti-aliased lines into images.

(in-package 'obvius)
(export 'draw-line) 

; the method for drawing a line (does everything that draw-slave does + width)
(defmethod draw-line ((im image) ystart xstart yend xend
		      &key (val 1.0) (alias t) (width 1) ->)
  (with-result ((result ->) im 'draw-line im ystart xstart yend xend 
		:val val :alias alias :width width)
    (unless (equal im result) (copy im :-> result))
    (let ((data (data result))
	  (m (/-0 (- ystart yend) (- xstart xend) 0.0)))
      (draw-slave data ystart xstart yend xend :val val :alias alias)
      (if (> width 1)
	  (cond ((or (> (abs m) 1) (= xstart xend))
		 (draw-slave data ystart (+ xstart width) yend (+ xend width) 
			     :val val :alias alias)
		 (loop for i from 1 to (- width 1) do
		       (draw-slave data ystart (+ xstart i) yend (+ xend i) 
				   :val val :alias t)))
		(t
		 (draw-slave data (+ ystart width) xstart (+ yend width) xend 
			     :val val :alias alias)
		 (loop for i from 1 to (- width 1) do
		       (draw-slave data (+ ystart i) xstart (+ yend i) xend 
				   :val val :alias t))))))
    result))	

(defun x-chop (y-x-list operator m b &optional (x-limit 1))
  (let ((y (car y-x-list))
	(x (cdr y-x-list)))
    (if (apply operator (list x x-limit))
	(cons (+ (* m x-limit) b) x-limit)
	(cons y x))))

(defun y-chop (y-x-list operator m b &optional (y-limit 1))
  (let ((y (car y-x-list))
	(x (cdr y-x-list)))
    (if (apply operator (list y y-limit))
	(cons y-limit (/ (- y-limit b) m))
	(cons y x))))

(defun n-crossing (val1 val2 crossing)
  (or
   (and (<= val1 crossing) (>= val2 crossing))
   (and (<= val2 crossing) (>= val1 crossing))))

(defun within (val1 l-bound u-bound)
  (and (>= val1 l-bound) (<= val1 u-bound)))

(defun canclip-p (ystart xstart yend xend ydim xdim)
  (if 
   (and
    (>= (min ystart yend) 1)
    (>= (min xstart xend) 1)
    (<= (max ystart yend) ydim)
    (<= (max xstart xend) xdim))
   t
   (cond ((or				; y crossing detection...
	   (n-crossing ystart yend 1)	; does the line intersect
	   (n-crossing ystart yend ydim)) ; the y boundaries?
	  (or				; if so,
	   (or				; does it also
	    (n-crossing xstart xend 1)	; intersect the 
	    (n-crossing xstart xend xdim)) ; x boundaries as well?
	   (within xstart 1 xdim)	; are either of the x coordinates
	   (within xend 1 xdim)))	; within the x boundaries?
	 ((or				; x crossing detection...
	   (n-crossing xstart xend 1)	; does the line intersect
	   (n-crossing xstart xend xdim)) ; the x boundaries?
	  (or				; if so,
	   (within ystart 1 ydim)	; are either of the y coordinates
	   (within yend 1 ydim)))	; within the y boundaries
	 (t				; if the crossings don't happen,
	  nil))))			; return a nil result
				

;;; this function takes the starting and ending coordinates and clips appropriately
(defun clipit (ystart xstart yend xend ydim xdim m b)
  (cond ((canclip-p ystart xstart yend xend ydim xdim)
	 (let ((y-x-start
		(x-chop
		 (x-chop
		  (y-chop
		   (y-chop
		    (cons ystart xstart) '< m b)
		   '> m b ydim)
		  '< m b)
		 '> m b xdim))
	       (y-x-end
		(x-chop
		 (x-chop
		  (y-chop
		   (y-chop
		    (cons yend xend) '< m b)
		   '> m b ydim)
		  '< m b)
		 '> m b xdim)))
	   (setq
	    ystart (round (car y-x-start))
	    xstart (round (cdr y-x-start))
	    yend (round (car y-x-end))
	    xend (round (cdr y-x-end)))
	   (list ystart xstart yend xend)))
	(t
	 nil)))
	  
(defun clipit2 (ystart xstart yend xend ydim xdim)
  (let* ((m (/ (- ystart yend) (- xstart xend)))
	 (b (- ystart (* m xstart))))
    (cond ((canclip-p ystart xstart yend xend ydim xdim)
	   (let ((y-x-start
		  (x-chop
		   (x-chop
		    (y-chop
		     (y-chop
		      (cons ystart xstart) '< m b)
		     '> m b ydim)
		    '< m b)
		   '> m b xdim))
		 (y-x-end
		  (x-chop
		   (x-chop
		    (y-chop
		     (y-chop
		      (cons yend xend) '< m b)
		     '> m b ydim)
		    '< m b)
		   '> m b xdim)))
	     (setq
	      ystart (round (car y-x-start))
	      xstart (round (cdr y-x-start))
	      yend (round (car y-x-end))
	      xend (round (cdr y-x-end)))
	     (list ystart xstart yend xend)))
	  (t
	   nil))))

; line drawing function for lines of slope <= abs 1
(defun linex (y x xfinish d m i1 i2 val yinc data alias)
  (linexj y x xfinish d i1 i2 val yinc data) ; plot a jagged line
  (if (not alias)			; if anti-aliased
      (cond ((> yinc 0)			; if slope > 0
	     (setq m (- 1 m))	; set m to 1 - m
	     (linexa1 (- y 1) x xfinish 0.5 m val yinc data) ; plot top anti-aliaser
	     (linexa2 (+ y 1) x xfinish 0.5 m val yinc data)) ; plot bottom anti-aliaser
	    (t				; if slope < 0
	     (setq m (+ 1 m))	; set m to 1 + m
	     (linexa2 (- y 1) x xfinish 0.5 m val yinc data) ; plot top anti-aliaser
	     (linexa1 (+ y 1) x xfinish 0.5 m val yinc data))))) ; plot bottom anti-alias

			; function for drawing aliased lines of slope <= abs 1
(defun linexj (y x xfinish d i1 i2 val yinc data) ; j for "jaggy" line
  (setf (aref data y x) val)		; plot onto array
  (if (< x xfinish)			; check to see if line draw is done
      (if (< d 0)			; increment x if necessary
	  (linexj y (+ x 1) xfinish (+ d i1) i1 i2 val yinc data)
	  (linexj (+ y yinc) (+ x 1) xfinish (+ d i2) i1 i2 val yinc data))))

; function for anti-aliasing the top of a line slope > 0, bottom of a line slope < 0
(defun linexa1 (y x xfinish d a val yinc data) ; anti-alias type 1 for lines (abs slope) < 1
  (unless (= x xfinish) 
    (let ((point (aref data y x)))
      (setf (aref data y x) (+ point (* (- 1 d) (- val point))))) ; plot onto array
    (if (< d a)				; adjust intensity
	(linexa1 y (+ x 1) xfinish (+ d (- 1 a)) a val yinc data) 
	(linexa1 (+ y yinc) (+ x 1) xfinish (- d a) a val yinc data))))

; function for anti-aliasing the bottom of a line slope > 0, top of a line slope < 0
(defun linexa2 (y x xfinish d a val yinc data) ; anti-alias type 2 for lines (abs slope) < 1
  (unless (= x xfinish)
    (let ((point (aref data y x)))
      (setf (aref data y x) (+ point (* d (- val point))))) ; plot onto array
    (if (< d a)				; adjust intensity
	(linexa2 y (+ x 1) xfinish (+ d (- 1 a)) a val yinc data)
	(linexa2 (+ y yinc) (+ x 1) xfinish (- d a) a val yinc data))))

; line drawing function for lines of slope >= abs 1
(defun liney (y x yfinish d m i1 i2 val xinc data alias)
  (lineyj y x yfinish d i1 i2 val xinc data) ; plot a jagged line
  (if (not alias) ; if anti-aliased
      (cond ((> xinc 0) ; if xinc positive
	     (setq m (- 1 (/ 1 m))) ; set m to  1 - 1/m
	     (lineya1 y (- x 1) yfinish 0.5 m val xinc data) ; plot left anti-aliaser
	     (lineya2 y (+ x 1) yfinish 0.5 m val xinc data)) ; plot right anti-aliaser
	    (t ; if xinc = -1
	     (setq m (+ 1 (/ 1 m))) ; set m to 1 + 1/m
	     (lineya2 y (- x 1) yfinish 0.5 m val xinc data) ; plot left anti-aliaser
	     (lineya1 y (+ x 1) yfinish 0.5 m val xinc data))))) ; plot right anti-aliaser

; function for writing lines of slope > abs 1
(defun lineyj (y x yfinish d i1 i2 val xinc data) ; xinc=1:slope>0, xinc=-1:slope<0
  (setf (aref data y x) val)		; plot onto array
  (if (< y yfinish)			; check to see if line draw is done
      (if (< d 0)			; increment x if necessary
	  (lineyj (+ y 1) x yfinish (+ d i1) i1 i2 val xinc data)
	  (lineyj (+ y 1) (+ x xinc) yfinish (+ d i2) i1 i2 val xinc data))))

; function for anti-aliasing the left side of a line slope > 1,right side of a line slope < 1
(defun lineya1 (y x yfinish d a val xinc data) ; anti-alias type 1 for lines (abs slope) > 1
  (unless (= y yfinish)
    (let ((point (aref data y x)))
      (setf (aref data y x) (+ point (* (- 1 d) (- val point))))) ; plot onto array
    (if (< d a)				; adjust intensity
	(lineya1 (+ y 1) x yfinish (+ d (- 1 a)) a val xinc data)
	(lineya1 (+ y 1) (+ x xinc) yfinish (- d a) a val xinc data))))

; function for anti-aliasing the right side of a line slope > 1,left side of a line slope < 1
(defun lineya2 (y x yfinish d a val xinc data) ; anti-alias type 2 for lines (abs slope) > 1
  (unless (= y yfinish)
    (let ((point (aref data y x)))
	    (setf (aref data y x) (+ point (* d (- val point))))) ; plot onto array
    (if (< d a)				; adjust intensity
	(lineya2 (+ y 1) x yfinish (+ d (- 1 a)) a val xinc data)
	(lineya2 (+ y 1) (+ x xinc) yfinish (- d a) a val xinc data))))

; Bresenham line drawing routine
(defun draw-bresenham-line (ystart yend xstart xend val dy dx m data alias)
  (setq dx (abs dx) dy (abs dy))
  (if (> dx dy)				; check if slope is less than one
      (let* ((d (- (* 2 dy) dx))	; set up to call linex
	     (i1 (* 2 dy))
	     (i2 (* 2 (- dy dx))))	
	(if (> xend xstart)	
	    (if (> yend ystart)
		(linex ystart xstart xend d m i1 i2 val 1 data alias)
		(linex ystart xstart xend d m i1 i2 val -1 data alias))
	    (if (> yend ystart)
		(linex yend xend xstart d m i1 i2 val -1 data alias)
		(linex yend xend xstart d m i1 i2 val 1 data alias))))
      (let ((d (- (* 2 dx) dy))		; set up to call writelny
	    (i1 (* 2 dx))
	    (i2 (* 2 (- dx dy))))
	(if (> yend ystart)
	    (if (> xend xstart)
		(liney ystart xstart yend d m i1 i2 val 1 data alias)
		(liney ystart xstart yend d m i1 i2 val -1 data alias))
	    (if (> xend xstart)
		(liney yend xend ystart d m i1 i2 val -1 data alias)
		(liney yend xend ystart d m i1 i2 val 1 data alias))))))

; function for doing vertical lines
(defun vliney (y x yfinish val data)
  (setf (aref data y x) val)
  (if (< y yfinish)
      (vliney (+ y 1) x yfinish val data)))

(defun draw-slave (data ystart xstart yend xend 
			&key (val 1.0) (alias t))
  (let* ((xdim (- (x-dim data) 2))	; -1 for fact array starts at 0
	 (ydim (- (y-dim data) 2))	; another -1 to accomodate anti-aliased lines
	 (y-x-start-end nil))           ; for later use when clipping
    (if (= xend xstart)			; is it a vertical line?
	(if (and (> xend 0) (< xstart xdim)) ; if so, is the x coordinate in the array?
	    (if (> ystart yend)		; plot the vertical line as follows
		(vliney (max yend 0) xend (min ystart ydim) val data) ; clip ordinates
		(vliney (max ystart 0) xend (min yend ydim) val data))) ; clip ordinates
	(let* ((dy (- yend ystart))	; if it isn't a vertical line, get delta y,
	       (dx (- xend xstart))	; delta x,
	       (m (/ dy dx))		; the slope of the line
	       (b (- ystart (* m xstart)))) ; y intercept (as in mx + b)
	  (setq y-x-start-end (clipit ystart xstart yend xend ydim xdim m b))
	  (unless (null y-x-start-end)
	    (setq
	     ystart (first y-x-start-end)
	     xstart (second y-x-start-end)
	     yend (third y-x-start-end)
	     xend (fourth y-x-start-end))
	    (draw-bresenham-line ystart yend xstart xend val dy dx m data alias))))))
	    






