;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: lucid-hacks.lisp
;;;  Author: Heeger
;;;  Description: Lucid dependent extensions to common-lisp
;;;  Creation Date: 9/92
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '())

;;; Functions are listed by the lucid-independent source files that use them.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Setf iref defined in image.lisp and bit-image.lisp is
;;; lucid-dependent.  These methods must be defined in those files,
;;; after the image and bit-image classes are defined.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Used by obvius-init and obvius-window-init

(defun environment-variable (thing)
  (lcl:environment-variable thing))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun quit ()
  (lcl:quit))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Used to be defined in misc.lisp.

;;; ***  This is supposed to be defined in CLOS, but is currently not there!
;;; Macro which replaces symbols in body by s-expressions.  
#|
(defmacro symbol-macrolet (sym-list . body)
  (cons 'progn
	(mapcar #'(lambda (body-sexpr) 
		    (mapcar-tree #'(lambda (x) (or (cadr (assoc x sym-list))
						   x))
				 (list body-sexpr)))
		body)))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Used by control-panel (and probably elsewhere)

(defun arglist (func)
  (lcl::arglist func))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Used by fileio.lisp

(defun create-directory (path)
  (let ((path-string (namestring path)))
    (LCL:run-program "mkdir" :arguments (list path-string))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Used by hardcopy.lisp

;;; Both paths are strings (e.g., using namestring or format nil)
(defun copy-file (path1 path2)
  (LCL:run-program "cp" :arguments (list path1 path2)))

(defun ship-to-printer (path printer)  
  (with-status-message "Printing postscript file"
    (LCL:run-program "sh" :wait nil :arguments 
		     (list "-c" (format nil *print-command-string* path printer)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Used by memory.lisp

;;; Careful not to do extra consing in here!
(defun make-static-array (size type)
  (LCL:with-static-area
      (make-array size :element-type type)))

(defun static-array-p (arr)
  (and (arrayp arr) (LCL:stationary-object-p arr)))

(defun displaced-array-p (arr)
  (lcl:displaced-array-p arr))

(defmacro with-scheduling-inhibited (&body body)
  `(LCL:with-scheduling-inhibited ,@body))
    
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Multiprocessing stuff

;;; Only allow one process at a time to access the pane.  Nested calls
;;; to with-locked-pane to this by the same process are ok.  Very very
;;; ity bitty chance of a collision bug here between the time the lock
;;; is read and when it is set.  **** BUG: this may cause the
;;; mouse process to go into a wait state.  We should probably start
;;; another process to execute immediate mouse events.
(defmacro with-locked-pane (pane . body)
  (let ((the-pane (gensym))
	(original-lock (gensym))
	(original-process (gensym)))
    `(let* ((,the-pane ,pane)
	    (,original-process LCL:*current-process*)
	    (,original-lock (locked ,the-pane)))
      (cond ((not ,original-lock) (setf (locked ,the-pane) ,original-process))
	    ((not (eq ,original-lock ,original-process))
	     ;; Wait until pane becomes unlocked by the process that locked it.
	     (status-message "Waiting for locked pane ...")
	     (LCL:process-wait 
	      "Waiting for locked pane"
	      #'(lambda () (and (not (locked ,the-pane))
				(setf (locked ,the-pane) ,original-process))))))
      (unwind-protect
	   (progn ,@body)
	;;CURRENT lock created by this process so undo it!
	(when (not (eq ,original-lock ,original-process)) 
	  (setf (locked ,the-pane) nil)))))) 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Used by conversion.lisp (could use obvius functions obv::array-read and
;;; obv::array-write instead).

(defun write-array (&rest args)
  (apply 'lcl:write-array args))

(defun read-array (&rest args)
  (apply 'lcl:read-array args))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; stepit.lisp.

;;; Need to acquire a function pointer to the foreign callable.
(defun foreign-function-pointer (symbol)
  (let* ((name (string-downcase (symbol-name symbol)))
	 (address (LCL:foreign-variable-address name)))
    (LCL:make-foreign-pointer :address address)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Local Variables:
;;; buffer-read-only: t 
;;; fill-column: 79
;;; End:
