/* 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: color.c
;;;  Author: EJ Chichilnisky
;;;  Description: Color specific operations that must be fast.
;;;  Creation Date: Fall 1992
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
*/

#include <math.h>


#define CIE_CUTOFF 0.008856

/* Brill (1992) Col. Res. Appl. p 402 */
int internal_cmc(xyz,w,result)
float *xyz,*w,*result;
{
  float Sl,Sc,Sh,c,F,T,h,L,a,b;

  internal_cielab(xyz,w,result);
  L = result[0];
  a = result[1];
  b = result[2];
  if (L<16.0) {
    Sl = 0.511;
  }
  else {
    /*** is this right? notation in Brill is weird ***/
    Sl = 0.0409*75.0*L/(1.0 + 0.017*65.0*L);
  }

  c = sqrt(a*a + b*b);
  Sc = 0.0638*c/(1.0+0.0131*c) + 0.638;
  
  c = pow(c,4.0);
  F = sqrt(c/(c + 1900.0));

  /* Brill requires atan to return angle in range (0,2pi).
     atan2 returns values in the range (-pi,pi).*/
  h = atan2(b/a);
  if (h<0.0) h = h + 2.0*M_PI;

  if ( (h>2.8623) && (h<6.0214) ) {
    T = 0.56 + 0.2*abs( cos(2.9322+h) );
  }
  else {
    T = 0.36 + 0.4*abs( cos(0.6109+h) );
  }
  Sh = (F*T + 1.0 - F)*Sc;

  result[0] = Sl;
  result[1] = Sc;
  result[2] = Sh;

  return(0);
}


/* Wyszecki & Stiles, p. 828 */
int internal_cielab(xyz,w,result)
float *xyz,*w,*result;
{
  float L,a,b,Xn,Yn,Zn;

  Xn = xyz[0]/w[0];
  Yn = xyz[1]/w[1];
  Zn = xyz[2]/w[2];

  L = Yn;
  L = (L > CIE_CUTOFF) ? 116.0*pow(L,0.33333333333) - 16.0 : L * 903.3;

  Yn = (Yn > CIE_CUTOFF) ? pow(Yn,0.33333333) : 7.787 * Yn + 0.13793;
  Xn = (Xn > CIE_CUTOFF) ? pow(Xn,0.33333333) : 7.787 * Xn + 0.13793;
  Zn = (Zn > CIE_CUTOFF) ? pow(Zn,0.33333333) : 7.787 * Zn + 0.13793;

  result[0] = L;
  result[1] = 500.0 * (Xn - Yn);
  result[2] = 200.0 * (Yn - Zn);
  
  return(0);
}

/* Wyszecki & Stiles, p. 828 */
int internal_cieluv(xyz,w,result)
float *xyz,*w,*result;
{
  float L,u,v,X,Y,Z,Xn,Yn,Zn,un,vn,d,dn;

  X = xyz[0];
  Y = xyz[1];
  Z = xyz[2];

  Xn = w[0];
  Yn = w[1];
  Zn = w[2];

  L = Y/Yn;
  L =  (L > CIE_CUTOFF) ? 116.0*pow(L,0.33333333333) - 16.0 : L * 903.3 ;

  d = X + 15.0*Y + 3.0*Z;
  u = d ? 4.0*X / d : 0.0;
  v = d ? 9.0*Y / d : 0.0;
  dn = Xn + 15.0*Yn + 3.0*Zn;
  un = dn ? 4.0*Xn / dn : 0.0;
  vn = dn ? 9.0*Yn / dn : 0.0;

  result[0] = L;
  result[1] = 13.0 * L * (u - un);
  result[2] = 13.0 * L * (v - vn);
  
  return(0);
}


int internal_cmc_rows(xyz,w,result,rows)
float *xyz,*w,*result;
int rows;
{
  register int i;

  for (i=0;i<rows;i++) {
    internal_cmc(xyz,w,result);
    xyz += 3;
    result += 3;
  }
  return(0);
}


int internal_cieluv_rows(xyz,w,result,rows)
float *xyz,*w,*result;
int rows;
{
  register int i;

  for (i=0;i<rows;i++) {
    internal_cieluv(xyz,w,result);
    xyz += 3;
    result += 3;
  }
  return(0);
}

int internal_cielab_rows(xyz,w,result,rows)
float *xyz,*w,*result;
int rows;
{
  register int i;

  for (i=0;i<rows;i++) {
    internal_cielab(xyz,w,result);
    xyz += 3;
    result += 3;
  }
  return(0);
}

/*
int internal_luv_rows(xyz,white,result,rows)
float *xyz,*white,*result;
int rows;
{
  register int i;

  for (i=0;i<rows;i++) {
    internal_luv(xyz,white,result);
    xyz += 3;
    result += 3;
  }
  return(0);
}


int internal_luv(xyz,white,result)
float *xyz,*white,*result;
{
  float white_0, white_1, white_2, xyz_0, xyz_1, xyz_2, L;

  white_0 = white[0]; 
  white_1 = white[1]; 
  white_2 = white[0] + 15.0*white[1] + 3.0*white[2];

  xyz_0 = xyz[0]; 
  xyz_1 = xyz[1]; 
  xyz_2 = xyz[0] + 15.0*xyz[1] + 3.0*xyz[2];
  
  L = xyz_1 / white_1;

  if (L > 0.008856) 
    L =  116.0*pow(L,0.33333333333) - 16.0;
  else 
    L *= 903.3;

  result[0] = L;
  result[1] = 52.0*L*( (xyz_0/xyz_2) - (white_0/white_2));
  result[2] = 117.0*L*( (xyz_1/xyz_2) - (white_1/white_2));
  
  return(0);
}
 */
