;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: steer-pyramid.lisp
;;;  Author: David Heeger and Bill Freeman
;;;  Description: pyramid-steerable-filter image-transforms.
;;;            Program which came up with the filter taps:  
;;;                      ~freeman/lisp/steer/pyramid-alias.lisp
;;;  Creation Date: summer '89
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; History:
;;; NOTE:  Sept. 22, 1989  the non-separable pyramid tiling steering filters
;;;  steer backwards from the way they should.  ie, for them, positive angle = clockwise.
;;;  For all the other filters, including the separable versions of the 
;;;  pyramid tiling steering filters, positive angle = counterclockwise.  Eventually,
;;;  I'll fix the non-separable ones.  wtf.   See wtf labbook p. 52.
;;; April 19, 1990  freeman changed edge handling from  nil (wrap around) to "reflect1".
;;; June 20, 1990  Freeman modified to make an end-filter to stop the recursion.
;;;  So, I added an  end-filter  to the definition of steerable-pyramid,
;;;  and changed even-collapse, odd-collapse, and added *default-end-steerable-pyramid-filter*.
;;;  So the old default filters, stored as   bpeven0 ...  are still there on disk, but the
;;;  new filters are read in from   bpevena0 ...    
;;; july 3, 1991 modified this to read-in the newest version filters,  bpevenb0...
;;;  See /u/freeman/lisp/steer/pyramid-alias.lisp for general comments about the steerable
;;;  pyramid;  search for "july 3".  The x-y separable filters in this file are out of date.
;;; July 9, 1991  changed the edge-handler from reflect1 back to nil.  That significantly
;;;  reduced the reconstruction errors;  see ~/lisp/steer/pyramid-alias.lisp.
;;;  OLDEST FILTERS, designed summer, 1989:  bpeven0, bpeven1, ...
;;;  NEWER FILTERS, designed summer, 1990:  bpevena0, bpevena1, ...
;;;  NEWEST FILTERS, designed summer, 1991:  bpevenb0, bpevenb1, ...
;;;  Still newest filters, oriented filters:  July 1991:  bpevenc0, ... (oriented filters only.
;;;   the two low-pass filters are the same as version b).
;;; July 20, 1991.  Use even filters which have NO dc response, even though
;;; they give slightly higher maximum freq error.  Therefore, use bpevenb0,1,2,3,4  with
;;; bpoddc0,1,2,3.

;;; Sep. 17, 1992    Simoncelli ported to obvius-2.2

;;; Mar 17, 1993 The two knuckleheads totally rewrote this code (sorry
;;; Bill) and ported to obvius-3.0

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '(steerable-pyramid make-steerable-pyramid
	  hi-filter
	  quadrature-steerable-pyramid quadrature-make-steerable-pyramid
	  even-pyramid odd-pyramid))

(obv-require :steer)
(obv-require :pyramid)

(warn "this code has not yet been ported to OBVIUS version 3.0")

#|
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; STEERABLE PYRAMID CLASS

(def-simple-class steerable-pyramid (pyramid)
  ((terminal-low-filter :initform nil)
   (terminal-low-band :initform nil)
   (hi-filter :initform nil)
   (hi-band :initform nil))
  (:default-initargs :display-type nil))

(defmethod inferiors-of ((pyr steerable-pyramid))
  (append (low-images pyr) (remove nil (steerables pyr))))

(defmacro steerable-pyramid-p (obj)
  `(typep ,obj 'steerable-pyramid))

;;; note: no display type for these guys for now, look at the
;;; sub-objects to look at them.  if :level is non-nil, initially
;;; builds pyramid to that level
(defun make-steerable-pyramid
    (image &rest initargs &key (level nil)
	   low-filter hi-filter filters
	   display-type name ->)
  (when -> (setf (getf initargs :name) ->))
  (remf initargs :level)
  (remf initargs :low-filter)
  (remf initargs :filters)
  (with-result ((result nil)
		`(:class steerable-pyramid
		  :low-band ,image
		  :original ,image
		  :forward-filters ,filters :inverse-filters ,filters
		  :forward-low-filter ,low-filter :inverse-low-filter ,low-filter
		  :hi-filter hi-filter
		  :hi-band ,(apply-filter hi-filter image)
		  ,@initargs)
		'apply 'make-steerable-pyramid image initargs)
    (when level (build result level))
    result))

(defmethod set-result ((name t) (model steerable-pyramid))
  (check-type name viewable-name)
  (make-instance (class-of model)
		 :name name
		 :display-type (display-type model)
		 :forward-filters (forward-filters model)
		 :forward-low-filter (forward-low-filter model)
		 :inverse-filters (inverse-filters model)
		 :inverse-low-filter (inverse-low-filter model)
		 :original (original model)
		 :low-band (similar (low-band model))
		 :hi-band (similar (hi-band model))
		 :levels (mapcar #'similar (levels model))))

(defmethod initialize-instance :after
  ((pyr steerable-pyramid) &key &allow-other-keys)
  (call-next-method pyr)
  (when (hi-band pyr) (pushnew pyr (superiors-of (hi-band pyr)))))

(defmethod terminal-low-band ((pyr steerable-pyramid))
  (or (slot-value pyr 'terminal-low-band pyr)
      (setf (slot-value pyr 'terminal-low-band pyr)
	    (apply-filter (terminal-low-filter pyr) (low-band pyr)))))

(defmethod collapsible-low-band ((pyr steerable-pyramid))
  (expand-filter (terminal-low-filter pyr) (terminal-low-band pyr)))

(defmethod collapse :after ((pyr steerable-pyramid)
			    &rest initargs &key (to-level 0))
  (let ((res (call-next-method)))
    (when (zerop to-level)
      (expand-filter (hi-filter pyr) (hi-band pyr) :zero nil :-> res))
    res))

(defmethod build :before ((pyr steerable-pyramid) level)
   (when (and (<= (length (levels pyr)) level) (slot-value pyr 'terminal-low-band pyr))
     (destroy (slot-value pyr 'terminal-low-band pyr)))
   (unless (hi-band pyr)
     (setf (hi-band pyr)
	   (apply-filter (hi-filter pyr) (original pyr))))
   (call-next-method))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; DEFAULT STEERABLE PYRAMID FILTERS

;;; Weighted least squares (in the frequency domain) real filter design.
;;; desired-fft should be an even- or odd-symmetric image, with
;;; power-of-two dimensions.  Weight can be an image or a number which
;;; is used as an exponent for an exponentially decaying weight
;;; function.
(defun design-filter (desired-fft dims &key weight)
  (when (some #'evenp dims) (error "not written for even size kernels"))
  (with-local-viewables
      ((inverted (let ((inv (circular-shift desired-fft :offset
					    (mapcar #'(lambda (x) (floor x 2))
						    (dimensions desired-fft)))))
		   (flip-y (flip-x inv :-> inv) :-> inv)))
       (symmetric (let ((pwr (mean (variance desired-fft))))
		    (cond ((< (/ (mean-square-error inv desired-fft) pwr) 0.01) ;*** magic
			   t)
			  ((< (/ (mean-square-error (-. inv :-> inv) desired-fft) pwr) 0.01)
			   nil)
			  (t (error "desired-fft must be even- or odd- symmetric.")))))
       (fourier (make-fourier-basis (dimensions desired-fft)
				    :symmetric symmetric
				    :freq-range (mapcar #'(lambda (x) (ceiling x 2)) dims)))
       (wt (when (numberp weight)
	     (make-1-over-r (dimensions desired-fft) :exponent weight))))
    (when wt (circular-shift wt :offset (mapcar #'(lambda (x) (/ (1- x) 2)) (dimensions wt))
			     :-> wt))
    (multiple-value-bind (p-vect err)
	(linear-least-squares fourier dfft :return-mse t :weight (or wt weight))
      (unless silent
	(pdb err)
	(pvect p-vect :digits 8))
      (values (params-to-filter p-vect dims symmetric) err))))

;;; Returns vector of the values a_i minimizing [w (y - \sum_i a_i
;;; x_i)]^2 where x_i are the basis images, y is the desired model
;;; image, and w is a weighting image (can be nil).
(defun linear-least-squares (x-seq y &key (rank (length. x-seq))
				   weight return-mse)
  (let* ((dim (length. x-seq))
	 (mat (make-array (list dim dim) :element-type 'single-float))
	 (vect (make-array dim  :element-type 'single-float))
	 inv est)
    (with-local-viewables ((temp (similar y))
			   (frame-j (similar temp))
			   (wt2 (when weight (square weight))))
      (loop for j from 0 below dim
	    do
	    (status-message "Computing row: ~A of ~A" j dim)
	    (if wt2
		(mul (frame j x-seq) wt2 :-> frame-j)
		(copy (frame j x-seq) :-> frame-j))
	    (mul y frame-j :-> temp)
	    (setf (aref vect j) (mean temp))
	    (loop for i from j below dim
		  do
		  (mul (frame i x-seq) frame-j  :-> temp)
		  (setf (aref mat j i) (setf (aref mat i j) (mean temp))))))
    (setq inv (matrix-inverse mat :dimension-limit rank))
    (setq est (matrix-mul vect inv))
    (if return-mse
	(with-local-viewables ((err (dot-product x-seq est)))
	  (sub err y :-> err)
	  (square err :-> err)
	  (values est (mean err)))
	est)))

;;; make sine or cosine fourier basis functions of given size (dims),
;;; covering given frequency range.
(defun make-fourier-basis (dims &key symmetric freq-range)
  (make-image-sequence
   (loop with norm = (sqrt (apply #'* dims))
	 for wy from (- 1 (car freq-range)) below (car freq-range) ;wy over neg freqs too
	 nconc
	 (loop for wx from (if (and (zerop wy) (not symmetric)) 1 0)
	       below (cadr freq-range)
	       collect
	       (make-synthetic-image dims
				     #'(lambda (y x)
					 (/ (cos (+ (* y wy) (x wx))) norm))
				     :x-range (list 0 2-pi))))))

(defun params-to-filter (p-vect fdims symmetric)
  (let* ((params/2 (mapcar #'(lambda (x) (/ x 2)) (listify p-vect)))
	 (x-ctr (floor (cadr fdims) 2)))
    (loop with kernel  = (make-array fdims :element-type 'single-float)
	  for y from 0 below (car fdims)
	  do
	  (loop for x from 0 to x-ctr
		do
		(setf (aref kernel y (+ x x-ctr))
		      (car params/2))
		(setf (aref kernel y (+ x x-ctr))
		      (car params/2))
		(setq params/2 (cdr params/2))
		(when (and (zerop x) (zerop y)) (setf)
		

    (if symmetric
	(make-filter (append (reverse (cdr params/2))
			     (list (car params))
			     (cdr params/2)))




	(make-filter (append (mapcar #'- (reverse params/2))
			     (list 0.0)
			     params/2)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; QUADRATURE STEERABLE PYRAMID (has not been ported from Bill's 1.2
;;; code )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
|#

#|
;;; out of date test code

(setq imp64 (make-impulse '(64 64)))
(setq imp128 (make-impulse '(128 128)))
(setq tst-filter (car *default-odd-steerable-pyramid-filters*))
(setq lo-filter *default-low-steerable-pyramid-filter*)
(setq bnd1 (expand-filter lo-filter (expand-filter tst-filter imp64)))
(setq bnd0 (expand-filter tst-filter imp128))
(setq resp0 (circular-shift (square-magnitude (fft bnd0)) 64 64))
(setq resp1 (circular-shift (square-magnitude (fft bnd1)) 64 64))
(setq checktile (add resp0 resp1))

(setq rand (make-uniform-random-image '(64 64)))
(setq zone (make-cos-zone-plate '(128 128)))

(setq pyr (make-steerable-pyramid zone :level 2))
(setq sm0-pi/4 (square-magnitude pyr 0 (* pi (/ 1.0 4.0))))
(setq sm0-pi/2 (square-magnitude pyr 0 (* pi (/ 1.0 2.0))))
(setq sm0-3pi/4 (square-magnitude pyr 0 (* pi (/ 3.0 4.0))))
(setq sm0-pi (square-magnitude pyr 0 (* pi (/ 1.0 1.0))))
(setq sm0 (+. (+. sm0-pi/4 sm0-pi/2) (+. sm0-3pi/4 sm0-pi)))
(setq sm1-pi/4 (square-magnitude pyr 1 (* pi (/ 1.0 4.0))))
(setq sm1-pi/2 (square-magnitude pyr 1 (* pi (/ 1.0 2.0))))
(setq sm1-3pi/4 (square-magnitude pyr 1 (* pi (/ 3.0 4.0))))
(setq sm1-pi (square-magnitude pyr 1 (* pi (/ 1.0 1.0))))
(setq sm1 (+. (+. sm1-pi/4 sm1-pi/2) (+. sm1-3pi/4 sm1-pi)))
(setq sm1-exp (expand-filter sm1 (mul (low-filter pyr) 4.0)))
(setq sm2-pi/4 (square-magnitude pyr 2 (* pi (/ 1.0 4.0))))
(setq sm2-pi/2 (square-magnitude pyr 2 (* pi (/ 1.0 2.0))))
(setq sm2-3pi/4 (square-magnitude pyr 2 (* pi (/ 3.0 4.0))))
(setq sm2-pi (square-magnitude pyr 2 (* pi (/ 1.0 1.0))))
(setq sm2 (+. (+. sm2-pi/4 sm2-pi/2) (+. sm2-3pi/4 sm2-pi)))

(setq pyr1 (make-steerable-pyramid 
	    zone
	    :low-filter (make-separable-filter 
			 '(0.125 0.5 0.75 0.5 0.125)
			 '(0.125 0.5 0.75 0.5 0.125)
			 :step-vector '(2 2)
			 :edge-handler "reflect1")
	    :even-filters *default-even-steerable-filters*
	    :odd-filters *default-odd-steerable-filters*))
|#
  
