;;; A demonstration package for  steerable filters.

;;; See Vision and Modeling Technical Report 126 for more details on
;;; steerable filters.

;;; June 26, 1989 and March 8, 1990 Freeman wrote the demo package.
;;; Feb. 4, 1992  Freeman added the steerable pyramid demo package.
;;; Sep. 17, 1992 Simoncelli ported to obvius-2.2

(obv-require :steerable)

(set-default 'vector-field :arrow-heads nil)

(make-zone-plate '(64 64) :-> 'zone)

;;; make separable steerable basis set for default steerable filter
;;; (2nd deriv of Gaussian, "G2")
(make-steerable-basis zone :-> 'zone-sb)

;;; steer the filter to 45 degrees.  (the angle specification can be either a 
;;;   number or an image of angles).  ie, from the basis filters, synthesize
;;;   what applying G2 oriented at 45 degrees would give you.
(steer zone-sb (/ pi 4) :-> 'zone-45)

;;; make the steerable basis set for a quadrature pair of filters. 
;;;  The default filters are 2nd deriv of Gaussian (G2),
;;;  and it's Hilbert transform (H2).  This doesn't display, for now.
(make-quadrature-steerable-basis zone :-> 'zoneqsb)

;;; to display the G2 basis set
(even-steerable-basis zoneqsb)   

;;; to display the H2 basis set
(odd-steerable-basis zoneqsb)    

;;; steer the H2 filter to 30 degrees
(steer (odd-steerable-basis zoneqsb) (/ pi 6))

;;; find  square-root[ G2^2 + H2^2 ], if both the G2 and H2 filters are steered
;;;  to pi/6 radians.
(directional-magnitude zoneqsb (/ pi 6))

;;; Find the dominant orientation direction at each point in an image.  
;;; Apply a filter, adaptively, along that dominant orientation direction.

;;; (note:  orientation-map gives you the sin (2 theta) and cos(2 theta) coefficients
;;;        of the Fourier series for oriented energy as a function of angle.
;;;        The magnitude of these two components tells you the strength of the
;;;        orientation.  The relative phase gives you 2 theta, which you use
;;;        to get the orientation angle, -theta.)

(orientation-map zoneqsb :-> 'zonepair)
(complex-phase zonepair :-> 'zfangle)
(div zfangle 2.0 :-> 'zangle)
(magnitude zonepair :-> 'zmag)      ;;; strength of orientation at every point.
                                    ;;; Ripples are due to the approximation caused
                                    ;;; by fitting the Hilbert transform of G2 with a
                                    ;;; 3rd order polynomial.

;;; plot the dominant orientation at every point.  Scale the length of each
;;; vector by the square root of the strength of the orientedness.
(square-root zmag :-> 'sqrt-zmag)

(make-image-pair 
 (list (mul (cos. zangle) sqrt-zmag)
       (mul (sin. zangle) sqrt-zmag))
 :display-type 'vector-field :-> 'zpic)

;;; adaptively steer 2nd deriv of Gaussian along and against the orientation direction
(steer (even-steerable-basis zoneqsb) zangle :-> 'zalong)
;;; note the small over-all magnitude of this image, steered against the dominant orientation:
(steer (even-steerable-basis zoneqsb) (add zangle (/ pi 2)) :-> 'zagainst)

;;; Make floret plots (polar plots) of oriented energy at each position.  THIS TAKES WHILE.
(obv-compile-load "image-lines")    ;;; For plot-orientation
(plot-orientation zoneqsb :n-angles 16 :lengthen 0.7 :skip 8)


;;; other steerable filters, for higher orientational tuning:
(make-quadrature-steerable-basis zone 
				 :even-filters (make-g4-steerable-filters)
				 :odd-filters (make-h4-steerable-filters) :-> 'g4h4)
;;; display the basis functions
(even-steerable-basis g4h4)
(odd-steerable-basis g4h4)

;;; steer the odd-phase filters along 45 degrees.
(steer (odd-steerable-basis g4h4) (/ pi 4.0) :-> 'odd-diag)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; similarly with einstein:
;;; Find direction of dominant orientation, make plot, filter along direction
;;; of dominant orientation.
(load-image "/v/images/einstein")
(gauss-out einstein :-> 'ein)
(make-quadrature-steerable-basis ein :-> 'qsb)
(even-steerable-basis qsb)
(odd-steerable-basis qsb)

;;; make orientation map of einstein
(orientation-map qsb :-> 'einpair)
(div (complex-phase einpair) 2.0 :-> 'angle)
(magnitude einpair :-> 'mag)   ;;; strength of orientation at every point

(square-root mag :-> 'sqrt-mag)
(clip sqrt-mag 0 68 :-> 'cl-sqrt-mag)
(make-image-pair 
 (list (mul (cos. angle) cl-sqrt-mag)
       (mul (sin. angle) cl-sqrt-mag))
 :display-type 'vector-field :-> 'pic)

;;; do spatially adaptive angular steering, in the direction of orientation
(steer (even-steerable-basis qsb) angle :-> 'along)
(mul along -1.0 :-> 'alongplus)
(steer (even-steerable-basis qsb) (add angle (/ pi 2)) :-> 'against)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Feb. 4, 1992  Steerable pyramid demonstration.

(load-image "/v/images/einstein")
(gauss-out einstein :-> 'ein)

;;; Make the steerable pyramid, and build it up to level 2 (ie, build levels 0, 1, 2).
;;; THIS COMMAND TAKES A LONG TIME for the non-separable filters
;;; (separable versions are in process):
;;; This doesn't display anything.
(make-steerable-pyramid ein :level 2 :-> 'ein-pyr)

;;; show the low-pass filtered versions at the various pyramid levels
(access-low ein-pyr 0)
(access-low ein-pyr 1)
(access-low ein-pyr 2)

;;; show the oriented basis filters at various pyramid levels
(odd-steerable-basis (access-qsb ein-pyr 0))
(odd-steerable-basis (access-qsb ein-pyr 1))
(odd-steerable-basis (access-qsb ein-pyr 2))


;;; Make an orientation map for the second pyramid level.
;;; This doesn't display anything.
(access-qsb ein-pyr 1 :-> 'qsb1)

;;; find the dominant orientation at level 2 in the pyramid
(orientation-map qsb1 :-> 'pair)
(div (complex-phase pair) 2.0 :-> 'dominant-angle)

;;; steer the even phase filters along the dominant angle at level 2:
(steer (even-steerable-basis qsb1) dominant-angle :-> 'dom-steered)

