;*****************************************************************
;  OBJECT-METHODS.LISP

(defclass sim-object  ()                         ; no superclasses
  ((id  :accessor id     :initarg :id)
   (kind :accessor kind  :initarg :kind     :initform '())
   (environment :accessor environment       :initform '())
   (displayer :accessor displayer :initarg :displayer  :initform '())))

(defmethod unique-id ((self sim-object))
  (id self))

(defmethod kind-id ((self sim-object))
  (cond
   ((symbolp (kind self)) (kind self))
   ((listp (kind self)) (car (kind self)))
   (t (class-name (class-of self)))))

(defmethod print-object ((self sim-object) stream)
  (format stream "#{~a}" (id self)))

(defmethod (setf environment) :before (new-holder (self sim-object))
  (check-class new-holder 'environment))

;*****************************************************************
;  Implementing class?  
;  There's a lot of confusion as to what class? and check-class
;  are supposed to do.  The original intent was to generate a class
;  hierarchy that the T object system didn't automatically provide. 
;  The T code tried to mimic this hierarchy, 
;    e.g. map-node -> container -> sim-object or something like that, 
;   in the "kind" slot.  When the code was translated to CL, an attempt
;  was apparently made to let the CLOS type hierarchy do this work.
;  The problem is that it's very hard to get at a class's superclasses.
;  My read right now is that all the information about the classes is
;  still stored in the "kind" slot, thus scanning the CLOS hierarchy is 
;  unnecessary.  Ideally I think that CLASS? should go away entirely, 
;  and if we need to know whether something is a parcel, for example, 
;  we would ask that question directly --- parcels would then answer true
;  to a IS-PARCEL predicate.

(defun class? (self class-name)
  (cond
	((null self)
	  nil)
	(t (or (memq class-name (kind self))
		   (name-in-class-hierarchy? self class-name)
		   (memq class-name (extra-classes self))))))

(defun name-in-class-hierarchy? (self class-name)
  (and (class-name? class-name)
	   (typep self class-name)))

(defun class-name? (sym)
  (find-class sym nil))

