;
; path planning functions
;
(in-package 'xtile)

(defvar c-grid (make-string 4000))
(defvar occupied (make-string 256 :initial-element (int-char 0)))

;(defvar c-grid (with-static-area (make-string 4000)))
;(defvar occupied (with-static-area (make-string 256 
;					:initial-element (int-char 0))))

(setf (aref occupied (char-int #\#)) #\#)
(setf (aref occupied (char-int #\T)) #\#)
(setf (aref occupied (char-int #\t)) #\#)
(setf (aref occupied (char-int #\O)) #\#)
(setf (aref occupied (char-int #\o)) #\#)

; convert plan from string to list
(defun tw-convert-plan (str)
    (let (result)
	(dotimes (i (length str))
	    (setf result 
		  (cons 
		    (case (aref str i)
			(#\u 'u)
			(#\d 'd)
			(#\l 'l)
			(#\r 'r)
			(#\x 'x))
		    result)))
	(subseq (reverse result) 0 (position nil (reverse result)))))

; plan move to a square
(defun tw-plan-move (world agent row col)
    (dotimes (i (tw-rows world))
	(dotimes (j (tw-cols world))
	    (setf (aref c-grid (+ (* i (tw-cols world)) j))
		  (aref (tw-grid world) i j))))
    (setf (aref c-grid (+ (* row (tw-cols world)) col)) #\D)
    (tw-convert-plan (moveplan (tw-rows world) (tw-cols world) c-grid occupied 
			       (obj-row agent) (obj-col agent))))

; plan move to any empty square adjacent to an obj
(defun tw-plan-move-near (world agent obj)
    (dotimes (i (tw-rows world))
	(dotimes (j (tw-cols world))
	    (setf (aref c-grid (+ (* i (tw-cols world)) j))
		  (aref (tw-grid world) i j))))
    (dolist (cell (obj-cells obj))
	(let ((row (cell-row cell))
	      (col (cell-col cell)))
	    (mapc #'(lambda(i j) 
			(if (member (aref (tw-grid world) (+ row i) (+ col j)) 
				(cons nothing-char agent-chars))
			    (setf (aref c-grid (+ (* (+ row i) 
					 (tw-cols world)) col j)) #\D)))
		  '(-1  0  1 -1  1 -1  0  1)
		  '(-1 -1 -1  0  0  1  1  1))))
    (tw-convert-plan (moveplan (tw-rows world) (tw-cols world) c-grid occupied 
			       (obj-row agent) (obj-col agent))))

(defun tw-plan-push (world agent tile holecells)
  (let (dir)
    (dotimes (i (tw-rows world))
	(dotimes (j (tw-cols world))
	    (setf (aref c-grid (+ (* i (tw-cols world)) j))
		  (aref (tw-grid world) i j))))
    (setf (aref c-grid (+ (* (obj-row agent) (tw-cols world)) 
			  (obj-col agent)))
	  nothing-char)
    (setf (aref c-grid (+ (* (obj-row tile) (tw-cols world)) (obj-col tile))) 
	  nothing-char)
    (dolist (cell holecells)
	(setf (aref c-grid (+ (* (cell-row cell) (tw-cols world))
			      (cell-col cell))) #\D))
    (setf dir (find-agent-dir tile agent))
    (if (>= dir 0)
	(tw-convert-plan (pushplan (tw-rows world) (tw-cols world) 
				   c-grid occupied 
				   (obj-row tile) (obj-col tile) dir))
      'agent-not-by-the-tile)))

(defun find-agent-dir (tile agent)
    (let* ((drow (- (obj-row agent) (obj-row tile)))
	   (dcol (- (obj-col agent) (obj-col tile)))
	   dir)
	(if (= drow -1)
	    (if (= dcol -1)
		(setf dir 7)
		(if (= dcol 0)
		    (setf dir 0)
		    (if (= dcol 1)
			(setf dir 1)
			(setf dir -1))))
	    (if (= drow 0)
		(if (= dcol -1)
		    (setf dir 6)
		    (if (= dcol 1)
			(setf dir 2)
			(setf dir -1)))
		(if (= drow 1)
		    (if (= dcol -1)
			(setf dir 5)
			(if (= dcol 0)
			    (setf dir 4)
			    (if (= dcol 1)
				(setf dir 3)
				(setf dir -1))))
		    (setf dir -1))))
	dir))

;
; Find the closest tile to 'obj' in 'world'.  Returns the tile and the
; parameters for future calls to next-closest-tile, as seven values.
; 
(defun tw-closest-tile (obj world)
    (let ((grid-mark (make-array (list (tw-rows world) (tw-cols world))
				 :initial-element nil))
	  frontier)
	(dolist (c (obj-cells obj))
	    (setf frontier (cons (list (cell-row c) (cell-col c)) frontier))
	    (setf (aref grid-mark (cell-row c) (cell-col c)) t))
	(next-closest-tile frontier nil nil grid-mark (tw-grid world)
			   (tw-grid-which world))))

;
; Same as tw-closest-tile, but doesn't call next-closest-tile.  Returns
; six values.
; 
(defun next-closest-tile-setup (obj world)
    (let ((grid-mark (make-array (list (tw-rows world) (tw-cols world))
				 :initial-element nil))
	  frontier)
      (dolist (c (obj-cells obj))
        (setf frontier (cons (list (cell-row c) (cell-col c)) frontier))
        (setf (aref grid-mark (cell-row c) (cell-col c)) t))
      (values frontier nil nil grid-mark (tw-grid world)
              (tw-grid-which world))))
     
; 
; Find the next closest tile, in number of moves, assuming tiles and agents
; are not in the way.  'unexpanded' are cells which have been checked but
; not expanded.  'limbo' are cells which have been generated but not checked
; for tiles.  'expanded' are cells which have been checked.
(defun next-closest-tile (unexpanded limbo expanded marked twgrid twgridwhich)
  (let (this)
     (loop
	(when (null limbo)
	    (when (null unexpanded)
		(when (null expanded)
		    (return-from next-closest-tile
			(values nil nil nil nil nil nil nil)))
		(setf unexpanded expanded)
		(setf expanded nil))
	    (setf this (car unexpanded))
	    (setf unexpanded (cdr unexpanded))
	    (mapc #'(lambda(drow dcol)
                      (let ((row (+ (car this) drow))
                            (col (+ (cadr this) dcol)))
                        (when (and (not (aref marked row col))
                                   (not (member (aref twgrid row col)
                                                obstacle-or-hole-chars)))
                          (setf limbo (cons (list row col) limbo)))))
                  '( 0  0 -1  1)
                  '(-1  1  0  0)))
	(when limbo
	    (setf this (car limbo))
	    (setf expanded (cons this expanded))
	    (setf limbo (cdr limbo))
	    (setf (aref marked (car this) (cadr this)) t)
	    (when (member (aref twgrid (car this) (cadr this)) tile-chars)
	       (return-from next-closest-tile 
		  (values (aref twgridwhich (car this) (cadr this)) unexpanded
			limbo expanded marked twgrid twgridwhich)))))))


; Simulate a plan.  This should be fast, but isn't really right now
; because it's just lisp code.  That's life.
; If the planner's right, this should always work.  If it isn't, this
; isn't always sufficient to notice, so more checking should be done.

(defun tw-simulate (world agent plan)
  (let ((olddisp *display-on*))
    (setf *display-on* nil)
    (dolist (move plan)
	    (when (not (tw-step world agent move 0))
		  (setf *display-on* olddisp)
		  (return-from tw-simulate nil)))
    (setf *display-on* olddisp)
    t
    )
  )

(defun find-same-obj (obj objlist)
    (dolist (x objlist)
	(when (eq (obj-name obj) (obj-name x))
	    (return-from find-same-obj x)))
    nil)
		
(defun find-obj-with-name (name objlist)
    (dolist (x objlist)
	(when (eq name (obj-name x))
	    (return-from find-obj-with-name x)))
    nil)

; 
; Fill a cell of 'hole' asking for 'next-closest-tile' with parameters p1..p6
; and trying to make a plan for it.  If that fails, try next-closest-tile
; again.
; 
(defun fill-next-closest (world agent hole p1 p2 p3 p4 p5 p6)
    (let (next-tile
	  (ok-world world)
	  new-world newer-world
	  (ok-hole hole)
	  new-hole newer-hole
	  (ok-agent agent)
	  new-agent newer-agent
	  plan-to plan-from)

	(loop
            (multiple-value-setq (next-tile p1 p2 p3 p4 p5 p6)
              (next-closest-tile p1 p2 p3 p4 p5 p6))
	    (when (not next-tile)
		(return-from fill-next-closest 
		    (values nil '(x) nil nil nil p1 p2 p3 p4 p5 p6)))
	    (setf new-world (tw-copy-world ok-world))
	    (setf new-agent (find-same-obj ok-agent 
				    (tw-agent-list new-world)))
  	    (setf new-hole
		      (find-same-obj ok-hole (tw-hole-list new-world)))
	    (setf plan-to (tw-plan-move-near ok-world ok-agent next-tile))
	    (when (not (equal plan-to '(x)))
		(tw-simulate new-world new-agent plan-to)
		(setf newer-world (tw-copy-world new-world))
		(setf newer-agent (find-same-obj new-agent 
				    (tw-agent-list newer-world)))
  		(setf newer-hole
		      (find-same-obj new-hole (tw-hole-list newer-world)))
		(setf plan-from (tw-plan-push new-world new-agent next-tile 
				    (obj-cells new-hole)))
		(when (not (equal plan-from '(x)))
		    (tw-simulate newer-world newer-agent plan-from)
		    (return-from fill-next-closest
			    (values next-tile plan-to new-world
				    plan-from newer-world
				    p1 p2 p3 p4 p5 p6)))))))
; 
; Expand a 'fill' intention in Martha's intention format.  If the act
; has null subacts, 'world' should be the starting world, and the first
; move/push/rest subacts are created.  Otherwise, the starting world is
; taken from the starting-world parameter of the 'rest' subact.
; 
(defun expand-fill (act world path-planner)
    (let* (hole-obj agent-obj tile-obj this-world
	   p1 p2 p3 p4 p5 p6 
	   plan1 world1
	   plan2 world2
	   move-act push-act rest-act
	   info new-part)
	(if (null (act-subacts act))
	    (progn
		(setf this-world world)
		(setf hole-obj 
		      (find-obj-with-name (binding 'hole (act-parms act))
					  (tw-hole-list this-world)))
                (if (equal path-planner 'direct)
                    (multiple-value-setq (p1 p2 p3 p4 p5 p6)
                      (next-closest-tile-setup hole-obj world))))
                
	    (progn
		(setf info (binding 'planner-state-info
			        (act-parms (car (last (act-subacts act))))))
		(setf p1 (nth 0 info))
		(setf p2 (nth 1 info))
		(setf p3 (nth 2 info))
		(setf p4 (nth 3 info))
		(setf p5 (nth 4 info))
		(setf p6 (nth 5 info))
		(setf this-world 
		      (binding 'starting-world 
			       (act-parms (car (last (act-subacts act))))))))
	(setf hole-obj (find-obj-with-name (binding 'hole (act-parms act))
					   (tw-hole-list this-world)))
	(setf agent-obj (first (tw-agent-list this-world)))
        (cond ((equal path-planner 'direct)
               (multiple-value-setq (tile-obj plan1 world1 
                                              plan2 world2
                                              p1 p2 p3 p4 p5 p6)
                 (fill-next-closest this-world agent-obj hole-obj 
                                    p1 p2 p3 p4 p5 p6)))
              ((equal path-planner 'search)
               (multiple-value-setq (tile-obj plan1 world1 plan2 world2 p1)
                 (spp-fill-next-closest this-world agent-obj hole-obj p1)))
              (t (format t "Error: unkown path-planner ~s" path-planner)))
	(when (or (equal plan1 '(x))
		  (equal plan2 '(x)))
	    (setf (act-status2 act) 'to-be-discarded)
	    (return-from expand-fill nil))
        (when (not (null plan1))
	  (setf move-act 
	      (make-act 
		  :id (make-name 'act)
		  :type 'move-near
		  :status1 'intention
		  :status2 'to-be-executed
		  :parms (list (make-parm :name 'tile
					  :binding (obj-name tile-obj))
			       (make-parm :name 'starting-world
					  :binding this-world))))
	  (setf (act-subacts move-act)
	      (mapcar #'(lambda(move)
			  (let ((actname (make-name 'act)))
			    (make-act :id actname
				      :type move
				      :atomic t
				      :status1 'intention
				      :status2 'to-be-executed
				      :parms (list (make-parm :name 'id
						      :binding actname)))))
		      plan1)))
	(setf push-act
	      (make-act 
		  :id (make-name 'act)
		  :type 'push
		  :status1 'intention
		  :status2 'to-be-executed
		  :parms (list (make-parm :name 'tile
					  :binding (obj-name tile-obj))
			       (make-parm :name 'hole
					  :binding (obj-name hole-obj))
			       (make-parm :name 'starting-world
					  :binding world1))))
	(setf (act-subacts push-act)
	      (mapcar #'(lambda(move)
			  (let ((actname (make-name 'act)))
			    (make-act :id actname
				      :type move
				      :atomic t
				      :status1 'intention
				      :status2 'to-be-executed
				      :parms (list (make-parm :name 'id
						      :binding actname)))))
		      plan2))
	(when (null (act-subacts act))
	    (setf (act-subacts act)
	       (list (make-act
		  :id (make-name 'act)
		  :type 'rest
		  :status1 'intention
		  :status2 'to-be-planned
		  :parms (list (make-parm :name 'hole 
				          :binding (obj-name hole-obj)))))))
	(setf rest-act (car (last (act-subacts act))))
	(setf (act-parms rest-act) (list (make-parm :name 'hole
						 :binding (obj-name hole-obj))
					 (make-parm :name 'starting-world
						    :binding world2)
					 (make-parm :name 'planner-state-info
						    :binding (list p1 p2 p3 
								p4 p5 p6))))
	(if (find-obj-with-name (binding 'hole (act-parms act))
				(tw-hole-list world2))
	    (setf new-part (list move-act push-act rest-act))
	    (progn
		(setf (act-status2 act) 'to-be-executed)
		(setf new-part (list move-act push-act))))
	(if (null move-act)
	    (setf new-part (cdr new-part)))
	(setf (act-subacts act)
	      (append (butlast (act-subacts act))
		      new-part))
	world2))
