;;;************************************************************************
;;; x-graphics.lisp		D. Musliner
;;;
;;; - note special provisions needed b/c Xwindows inverts the Y axis from the PS
;;;  	standard, which our device-indep. uses.
;;;************************************************************************

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.


;;; NOTE NOTE hostname parsing from DISPLAY env var is not right if host name
;;; really ends in a digit (e.g., mach2)

(require 'clx)

(require 'di-graphics)

(when (not (member "X-GRAPHICS" *modules* :test #'equal))
  	(format t "; Note uninterning ALIST symbol before using XLIB package~%")
  	(unintern 'alist)	;; makes load smoother: XLIB exports this too
  	(use-package 'xlib))

(provide 'x-graphics)

(proclaim '(optimize (speed 3) (safety 1)))

;;; ---------------------------------------------------------------------------
;;; X graphics global constants.

(defvar *hostname* nil)			;; holds string name of Xserver host
					;; - defaults to local machine on init
(defvar *display* nil)
(defvar *screen* nil)
(defvar *root-window* nil) 
(defvar *window* nil)
;(defvar *gcontext* nil)

(defvar *black-pixel* nil)
(defvar *white-pixel* nil)

(defvar *pm* nil)		;; buffer pixmap
(defvar *pmgc* nil)		;; buffer pixmap graphics context

(defvar *width* nil)
(defvar *height* nil)

(defvar *x-x-mag* 1)
(defvar *x-y-mag* 1)

(defvar *x-x-offset* 0)
(defvar *x-y-offset* 0)

;;; ---------------------------------------------------------------------------
;;; pass in width and height in inches, and coords to fit w/in that screen.
;;; we use the xy-mag to scale the dimensions, and the xy-offset to offset for
;;; nonzero xmin ymin.

(defun di-X-initialize-graphics (width height xmin ymin xmax ymax)
  (let (x-ppi y-ppi)

	;; if a previous display is running, kill it
  (if *display* (close-display *display*))

	;; if no other Xserver host defined, use local display
  (if (not *hostname*) (setf *hostname* (system:getenv "DISPLAY")))
  	;; remove :0 from *hostname* if necessary (stupid CLX!).
  (setf *host* (string-right-trim "0123456789:." *hostname*))
  
  (setf *display* (open-display *host*))
  (setf (display-after-function *display*) #'display-finish-output)
  (setf *screen* (first (display-roots *display*)))

	;; get screen's pix-per-inch characteristics.
  (setf x-ppi (* 25.4 (/ (screen-width *screen*) 
			 (screen-width-in-millimeters *screen*))))
  (setf y-ppi (* 25.4 (/ (screen-height *screen*) 
			 (screen-height-in-millimeters *screen*))))
	;; remap window dimensions from inches to pixels.
  (setf *width* (round (* x-ppi width)))
  (setf *height* (round (* y-ppi height)))

	;; setup xy-mag and xy-offset to fit coord range w/in window.
  (setf *x-x-offset* (- xmin))
  (setf *x-y-offset* (- ymin))
  (setf *x-x-mag* (/ *width* (- xmax xmin)))
  (setf *x-y-mag* (/ *height* (- ymax ymin)))

  (setf *root-window* (screen-root *screen*))
  (setf *black-pixel* (screen-black-pixel *screen*))
  (setf *white-pixel* (screen-white-pixel *screen*))
  (setf *window* (create-window 
  			:parent *root-window*
		      	:x 0
		      	:y 0
		      	:width *width*
		      	:height *height*
		      	:backing-store :always 
		      	:border-width 1
		      	:border *white-pixel*
		      	:background *black-pixel*
		      	:bit-gravity :center
			:event-mask 
			    '(:exposure :button-press :enter-window)))

  (setf (wm-name *window*) "Lisp X-graphics")
  (setf (wm-icon-name *window*) "Lisp X-graphics")
  (setf (wm-normal-hints *window*)
	(make-wm-size-hints :x 0
			  :y 0
			  :width *width*
			  :height *height* ))
  (map-window *window*)
;  (setf *gcontext* (create-gcontext 
;				:foreground *white-pixel*
;			       	:background *black-pixel*
;			       	:drawable *window*))

	;;; now open the buffer pixmap
	;;; - we draw in the buffer pixmap and then blast it to the window
	;;;   with a copy-area call
  (setf *pm* (create-pixmap 
			:width *width*
			:height *height*
			:depth (drawable-depth *window*)
  			:drawable *window*))

  (setf *pmgc* (create-gcontext 
	       		:background *black-pixel*
  			;;  :function boole-eqv
	       		:drawable *pm*))

	;; by default, open fixed width font
  (setf (gcontext-font *pmgc*) (open-font *display* "fixed"))
))

;;; ---------------------------------------------------------------------------
(defun di-X-draw-in-background ( &aux temp)
  (setf (gcontext-background *pmgc*) *black-pixel*)
  (setf (gcontext-foreground *pmgc*) *white-pixel*))

;;; ---------------------------------------------------------------------------
(defun di-X-draw-in-foreground ( &aux temp)
  (setf (gcontext-background *pmgc*) *white-pixel*)
  (setf (gcontext-foreground *pmgc*) *black-pixel*))

;;; ---------------------------------------------------------------------------
(defun di-X-label-drawing (label)
  (setf (wm-icon-name *window*) label)
  (setf (wm-name *window*) label))

;;; ---------------------------------------------------------------------------
(defun di-x-deinitialize-graphics ()
  (if *display* (close-display *display*))
  (setf *display* nil))

;;; ---------------------------------------------------------------------------
;;; Function copy-buffer-to-window
;;; - copies pixmap buffer to window, thus displaying new world state

(defun copy-buffer-to-window ()
  ;; (setf (gcontext-function *pmgc*) boole-1)
  (copy-area *pm* *pmgc* 0 0 	
	     (drawable-width *pm*) (drawable-height *pm*)
	     *window* 0 0) 
  ;;(setf (gcontext-function *pmgc*) boole-eqv)
  (display-finish-output *display*))

;;; ---------------------------------------------------------------------------
;;; Function clear-buffer
;;; - clears entire buffer pixmap to background color

(defun clear-buffer ()
       	(setf (gcontext-foreground *pmgc*) *black-pixel*)
  	;;(setf (gcontext-function *pmgc*) boole-1)
  	(draw-rectangle *pm* *pmgc* 
		  	0 0 
		  	(drawable-width *pm*) (drawable-height *pm*)
		  	T)
  	;;(setf (gcontext-function *pmgc*) boole-eqv)
       	(setf (gcontext-foreground *pmgc*) *white-pixel*))

;;; ---------------------------------------------------------------------------
;;; Function clear-window
;;; - clears buffer pixmap to background color, copies to display window

(defun clear-window ()
  (clear-buffer)
  (copy-buffer-to-window))

;;; ---------------------------------------------------------------------------
;;; To invert the Y axis to conform to my PS-like standard,
;;; 	new miny = total-height - miny - rect-height
;;; note should make filled and label and other options be &key options
;;; and include the ignore-other-keys flag, so that difff graphics packages
;;; can choose to support these or not, depending, but all calls need the
;; basic rectangle and will give no errors on any syntax beyond that.

(defun di-x-draw-rectangle (xmin ymin xmax ymax 
			    &key (filled nil) (label nil)
			    &allow-other-keys &aux width height)

  (setf xmin (round (* (+ xmin *x-x-offset*) *x-x-mag*)))
  (setf xmax (round (* (+ xmax *x-x-offset*) *x-x-mag*)))
  (setf width (- xmax xmin))
  (cond ((< *x-y-mag* 0) ;; when negative Y mag, dont invert Y axis (for MICE)
  	 (setf ymin (round (- (* (+ ymin *x-y-offset*) *x-y-mag*))))
  	 (setf ymax (round (- (* (+ ymax *x-y-offset*) *x-y-mag*))))
	 (setf height (- ymax ymin))
	 )
	(T
  	 (setf ymin (round (* (+ ymin *x-y-offset*) *x-y-mag*)))
  	 (setf ymax (round (* (+ ymax *x-y-offset*) *x-y-mag*)))
	 (setf height (- ymax ymin))
	 (setf ymin (- *height* height ymin))))

  ;; (format t "drawing rect ~A ~A ~A ~A~%" xmin ymin width height)
  (draw-rectangle *pm* *pmgc* xmin ymin width height filled)
  (if label
	(draw-glyphs *pm* *pmgc* 
			(round (+ xmin 
				  (/ width 2)
				  (- (/ (text-width *pmgc* label) 2))))
			(+ ymin (round (/ height 2)) 4) 
			label)))

;;; ---------------------------------------------------------------------------
;;; - fits circle w/in rectangle coords given.

(defun di-x-draw-circle (xmin ymin xmax ymax
			    &key (filled nil) (label nil)
			    &allow-other-keys &aux width height)
  (setf xmin (round (* (+ xmin *x-x-offset*) *x-x-mag*)))
  (setf xmax (round (* (+ xmax *x-x-offset*) *x-x-mag*)))
  (setf width (- xmax xmin))
  (cond ((< *x-y-mag* 0) ;; when negative Y mag, dont invert Y axis (for MICE)
  	 (setf ymin (round (- (* (+ ymin *x-y-offset*) *x-y-mag*))))
  	 (setf ymax (round (- (* (+ ymax *x-y-offset*) *x-y-mag*))))
	 (setf height (- ymax ymin))
	 )
	(T
  	 (setf ymin (round (* (+ ymin *x-y-offset*) *x-y-mag*)))
  	 (setf ymax (round (* (+ ymax *x-y-offset*) *x-y-mag*)))
	 (setf height (- ymax ymin))
	 (setf ymin (- *height* height ymin))))

  (draw-arc *pm* *pmgc* xmin ymin width height -.1 6.283 filled)
  (if label
	(draw-glyphs *pm* *pmgc* 
			(round (+ xmin 
				  (/ width 2)
				  (- (/ (text-width *pmgc* label) 2))))
			(+ ymin (round (/ height 2)) 4) 
			label))
)
	
;;; ---------------------------------------------------------------------------

(defun x-convert-x (x) (round (* (+ x *x-x-offset*) *x-x-mag*)))
(defun x-convert-y (y) 
  (cond ((< *x-y-mag* 0)
	 (round (- (* (+ y *x-y-offset*) *x-y-mag*))))
	(T
	 (- *height* (round (* (+ y *x-y-offset*) *x-y-mag*))))))

;;; ---------------------------------------------------------------------------

(defun di-x-draw-segments (&rest seglist)
 (setf seglist (map-even #'x-convert-x seglist))
 (setf seglist (map-odd #'x-convert-y seglist))
 (draw-segments *pm* *pmgc* seglist))

;;; ---------------------------------------------------------------------------

(defun di-x-draw-triangle (x1 y1 x2 y2 x3 y3 &key (filled nil)
			    &allow-other-keys &aux point-list)
  (setf point-list (list x1 y1 x2 y2 x3 y3 x1 y1))
  (setf point-list (map-even #'x-convert-x point-list))
  (setf point-list (map-odd #'x-convert-y point-list))
  (draw-lines *pm* *pmgc* point-list :fill-p filled))

;;; ---------------------------------------------------------------------------

(defun di-x-draw-lines (point-list &key (filled nil) (label nil)
			    &allow-other-keys)
  (setf point-list (map-even #'x-convert-x point-list))
  (setf point-list (map-odd #'x-convert-y point-list))
  (draw-lines *pm* *pmgc* point-list :fill-p filled))

;;; ---------------------------------------------------------------------------

(defun di-x-center-text (x y text)
  (setf x (x-convert-x x))
  (setf y (x-convert-y y))
  (draw-glyphs *pm* *pmgc* (- x (/ (text-width *pmgc* text) 2)) y text))

;;; ---------------------------------------------------------------------------
