;;;; **********************************************************************
;;;; tools.lisp		- D. Musliner		
;;;; - general purpose enhancements to lisp environment
;;;;
;;;; *********************************************************************

;;; needed for Allegro 4.0.1
#+ALLEGRO-V4.0
(setf *CLTL1-IN-PACKAGE-COMPATIBILITY-P* T)
#+ALLEGRO-V4.0
(setf comp:*cltl1-compile-file-toplevel-compatibility-p* T)

(provide 'tools)

;;; **********************************************************************
;;; setup Allegro-dependent search paths according to what machine we are on.
;;; - try to load fasl or lisp version from a subdirectory named after the type 
;;; of CPU we are running on, thus allowing us to keep multiple, device 
;;; dependent .fasl files around in subdirectories.

(defvar *cpu* nil)

(cond ((member :SPARC *features*)
    	(setf *cpu* "sparc"))
      ((member :RIOS *features*)
    	(setf *cpu* "rios"))
      ((member :SGI4D *features*)
    	(setf *cpu* "sgi4d"))
      ((member :MIPS *features*)
    	(setf *cpu* "mips"))
      ((member :M68K *features*)
    	(setf *cpu* "m68k"))
      ((member :TI *features*)
    	(setf *cpu* "ti"))
      ((member :APPLE *features*)
    	(setf *cpu* "apple"))
      (T
	(format t "WARNING: DJM tools.lisp could not determine CPU type for enhanced load")))

#+ALLEGRO
(setf system:*load-search-list*
      (list 
	(list 'newest 
		(make-pathname :directory *cpu* :type "fasl")
 		(make-pathname :directory "." :type "fasl")
		(make-pathname :directory "." :type "cl")
		(make-pathname :directory "." :type "lisp"))
	;;(list 'newest 
	;;      (make-pathname :directory "~djm/src/lisp" :type "fasl")
	;;  	(make-pathname :directory "~djm/src/lisp" :type "lisp"))
	 excl::*library-code-fasl-pathname*
	 excl::*library-code-cl-pathname*))

#+ALLEGRO
(setf system:*require-search-list* system:*load-search-list*)

;;; **********************************************************************
(defvar *program-name* nil)

(defvar *debug* 0)

;;; **********************************************************************
(defmacro debug (level format-string &rest allargs)
  `(when (>= *debug* ,level)
	(format t ";; ~A: " *program-name*)
	(format t ,format-string ,@allargs)
	(format t "~%")))

;;; **********************************************************************
(defmacro perror (format-string &rest allargs)
  `(let () 
	(format t ";; ~A: ERROR -- " *program-name*)
  	(format t ,format-string ,@allargs)
  	(format t "~%")))

;;; **********************************************************************
;;; Function ls
;;; - lists directory (default current).

(defun ls ( &optional (pathname "."))
  (setf pathname (make-pathname :directory pathname))
  (dolist (filename (directory pathname))
	(format t "~A~%" filename)))

;;; **********************************************************************
;;; Macro pwd
;;; - prints current (working) directory

(defmacro pwd () (chdir "."))    

;;; **********************************************************************
;;; Macro while
;;; - execs body as long as pred returns non-nil value. 
;;; - returns nil

;(defmacro while (pred &BODY body)
;  `(loop (unless ,pred (return nil)) ,@body))    

;;;***********************************************************************
;;; Macro for
;;; (for (variable start-value end-value increment) S-expression*)
;;; A 'for' loop like a normal language would have.

(defmacro for (arg-list &body body)
"(for (variable start-value end-value increment) S-expression*)
	- A 'for' loop like a normal language would have (DJM tools)."

  (let  ((var (first arg-list))
	 (start (second arg-list))
	 (end (third arg-list))
	 (inc (fourth arg-list)))
  	(append (list 'do (list (list var start (list '+ var inc)))
			(list (list '= var end)))
		body)))
	
;;;***********************************************************************
;;; Function rank-and-choose

(defun rank-and-choose (rank-function choose-function arglist &rest rest-args)
  "(rank-and-choose rank-function choose-function arglist &rest rest-args)
        - Applies choose-function to select a single member of arglist
          based on rankings given by rank-function (with rest-args) (DJM tools).
        - example: (rank-and-choose #'first #'max '((5 foo) (6 boo) (1 roo)))."

  (let  ((ranked-list (apply #'my-mapcar 
				(list* rank-function arglist rest-args))))
        (nth (position (apply choose-function ranked-list) ranked-list)
             arglist)))

;;;***********************************************************************
;;; Function random-choice
;;; - returns randomly selected element of arg list

(defun random-choice (arg)
  (cond ((null arg) nil)
        (T (nth (random (length arg)) arg))))

;;;***********************************************************************
;;; Macro setvar
;;; - declares global var and sets to the value.

(defmacro setvar (var value)
  `(defvar ,var)
  `(setf ,var ,value))

;;;***********************************************************************
;;; Macro getassoc
;;; - returns element of alist associated w/ key.

(defmacro getassoc (key alist) `(second (assoc ,key ,alist)))

;;;***********************************************************************
;;; Macro setassoc
;;; - deletes any element associated w/ key in alist, then
;;; adds an entry associating key w/ new-element.
 
(defmacro setassoc (key new-element alist)
  `(let* ((old-assoc (assoc ,key ,alist)))

  	(if old-assoc (setf ,alist (delete old-assoc ,alist)))
 	(push (list ,key ,new-element) ,alist)))

;;;----------------------------------------------------------------------------
;;; - deletes any element associated w/ key in alist, then
;;; adds an entry associating key w/ new-element.
;;; new-element is treated as an atom, and new associated element will be list.

(defmacro setlassoc (key new-element alist)
  `(let* ((old-assoc (assoc ,key ,alist)))

        (if old-assoc (setf ,alist (delete old-assoc ,alist)))
        (push (list ,key (list ,new-element)) ,alist)))

;;;----------------------------------------------------------------------------
;;; adds an element to the value associated in alist with key.  Note
;;; new-element is treated as an atom, and new associated element will be list.

(defmacro addlassoc (key new-element alist)
  `(let* ((old-assoc (assoc ,key ,alist)) old-value)

        (cond (old-assoc
                (setf old-value (second old-assoc))
                (setf ,alist (remove old-assoc ,alist))
                (push (list ,key (append old-value (list ,new-element))) ,alist)
                )
              (T (push (list ,key (list ,new-element)) ,alist)))))

;;;***********************************************************************
;;; Macro *=
;;; - simple macro to ease multiplication.

(defmacro *= (location increment)
  `(setf ,location (* ,location ,increment)))

;;;***********************************************************************
;;; Macro +=
;;; - simple macro to ease incrementing 

(defmacro += (location increment)
  `(setf ,location (+ ,location ,increment)))

;;;***********************************************************************
;;; Macro -=
;;; - simple macro to ease incrementing 

(defmacro -= (location increment)
  `(setf ,location (- ,location ,increment)))

;;;***********************************************************************
;;; Macro 1+=
;;; - simple macro to ease incrementing by one

(defmacro 1+= (location)
  `(setf ,location (1+ ,location)))

;;; **********************************************************************
;;; Macro ! 
;;; - reloads the last file loaded

(defmacro ! () (load *last-file-loaded*))

;;; **********************************************************************
;;; Macro !!
;;; - repeats the last command	(specific to Allegro)

(defmacro !! () (eval +))

;;; **********************************************************************
;;; Macro ++
;;; - increments argument by one

(defmacro ++ (foo) `(setf ,foo (1+ ,foo)))

;;; **********************************************************************
;;; Macro --
;;; - decrements argument by one

(defmacro -- (foo) `(setf ,foo (- ,foo 1)))

;;; **********************************************************************
(defun my-remove-if (function arglist &rest rest-args)
  "(my-remove-if predicate arglist &rest rest-args)

	Returns list of elements of arglist for which predicate returns nil
	when applied to a list of the element plus rest-args (DJM tools)."

  (let  ((return-val nil))
	(dolist (element arglist)
		(if (not (apply function (list* element rest-args)))
		    (setf return-val (append return-val (list element)))))
	return-val))


;;; **********************************************************************
(defun my-remove-if-not (function arglist &rest rest-args)
  "(my-remove-if-not predicate arglist &rest rest-args)

	Returns list of elements of arglist for which predicate returns non-nil
	when applied to a list of the element plus rest-args (DJM tools)."

  (let  ((return-val nil))
	(dolist (element arglist)
		(if (apply function (list* element rest-args))
		    (setf return-val (append return-val (list element)))))
	return-val))

;;; **********************************************************************
(defun any (function arglist &rest rest-args)
  "(any predicate arglist &rest rest-args)
	
	Predicate is applied to a list made of element 0 of arglist 
	and rest-args, then element 1 and rest-args, etc.  Returns T when
	predicate returns first non-nil value, or nil if end of arglist is
	reached (DJM tools)."

  (let  ((return-val nil))
  	(dolist (element arglist)
	  	(when (apply function (list* element rest-args)) 
			(setf return-val T)
			(return)))
	return-val))

;;; **********************************************************************
(defun my-mapcar (function arglist &rest rest-args)
  "(my-mapcar function arglist &rest rest-args)
	
	Returns list made up of return from function applied to a list made of 
	element 0 of arglist and rest-args, then element 1 and rest-args, etc.
	(DJM tools)."

  (let  ((return-val nil))
  	(dolist (element arglist)
		(setf return-val (append return-val (list (apply function (list* element rest-args))))))
	return-val))

;;; **********************************************************************

(defun map-stride (function arglist offset stride &rest rest-args)
"(map-stride function list offset stride &rest rest-args)
	
	Applies function to the elements of arglist which are stride-apart,
starting with element # offset, returning list made up of cons'd results of
the function, or the original element if function was not run (see also
map-odd and map-even) (DJM tools)."

  (-- stride)
  (let ((index offset) (return-val nil))
  	(if (quote rest-args)
	    (dolist (element arglist)
		 (cond ((= index 0) 
		 	(setf return-val (append return-val (list 
				   (apply function (list* element rest-args)))))
			(setf index stride))
		       (T 
		 	(setf return-val (append return-val (list element)))
			(-- index))))
	    (dolist (element arglist)
		 (cond ((= index 0) 
		 	(setf return-val (append return-val (list 
				   (apply function element))))
			(setf index stride))
		       (T 
		 	(setf return-val (append return-val (list element)))
			(-- index))))
		)
  	return-val))
		
(defmacro map-even (function arglist &rest rest-args)
"(map-even function list offset stride &rest rest-args)
	See map-stride"

  `(map-stride ,function ,arglist 0 2 ,@rest-args))

(defmacro map-odd (function arglist &rest rest-args)
"(map-odd function list offset stride &rest rest-args)
	See map-stride"

  `(map-stride ,function ,arglist 1 2 ,@rest-args))

;;; **********************************************************************
(defun docfun (function) (documentation function 'function))
	
;;; ***************************************************************************
(defmacro my-compile (file)
  (let ((output (make-pathname :name file :directory *cpu* :type "fasl"))
	(probe (probe-file (make-pathname :directory *cpu*))))
    (cond (probe
	   `(compile-file-if-needed ,file 
				    :output-file (make-pathname :name ,file 
								:directory *cpu*
								:type "fasl")))
	  (t
	   (break "~%*** Create '~a' directory at the current directory, and compile again~%" *cpu*)
	   nil))))

#+TI
(defmacro my-compile (file)
  `(compile-file ,file :output-file 
		(make-pathname :name ,file :directory *cpu* :type "fasl")))

;;; **********************************************************************
;;; Function load 	***** shadows usual function lisp:load *****
;;; - examines filename argument and the existing files and determines which
;;;   file to load, and whether the .lisp must be recompiled before a .fasl
;;;   is loaded.
;;; - essentially, this performs part of the 'make' function
;;; - it will respect explicit file types in the argument, but will print out
;;;   a warning if the requested file type is older than the other (ie, if
;;;   the .fasl is older than the .lisp, and you specify '(load *.lisp) 
;;;   it will load the .fasl but will warn you).
;;; - to get it to load the most recent, do not specify a type for the filename
;;;   (ie, just use '(load "foo")'
;;;   - in that case, if a fasl exists it will always load the fasl 
;;; 	(after compiling again if necessary)
;;; - for some reason, this function does not like to be compiled
;;; **********************************************************************

;;; this version does not match the comments above: it tries to load the
;;; **********************************************************************
;;; Function package-call
;;; - allows us to make a function call given the package in a variable, so
;;; 	we dont depend on fixed previous locations of functions we shadow:
;;; - store their old packages in vbles, then use package-call to get to the
;;; 	original versions.
;
;(defun package-call (package function &rest args)
;  (eval (read-from-string (format nil 
;		"(apply #'~A:~A '~S)" package function args))))
;
;;;; **********************************************************************
;
;(defvar *old-load-package* (package-name (symbol-package 'load)))
;
;(defvar *last-file-loaded* nil)
;
;(shadow 'load)
;
;;;; **********************************************************************
;;;; Function load
;
;(defun load (filename) 
;  (setf *last-file-loaded* filename)
;  (let* ((faslname (make-pathname
;                               :directory *cpu*
;                               :name filename
;                               :type "fasl"))
;        (lispname (make-pathname
;                               :directory *cpu*
;                               :name filename
;                               :type "lisp")))
;
;	(cond ((probe-file faslname) 
;		(package-call *old-load-package* 'load faslname))
;	      ((probe-file lispname) 
;		(package-call *old-load-package* 'load lispname))
;	      (T 
;		(package-call *old-load-package* 'load filename)))))
;
;
;;;; **********************************************************************
;(defvar *old-require-package* (package-name (symbol-package 'require)))
;
;;(shadow 'require)
;  
;;;; **********************************************************************
;;;; Function require
;
;(defun my-require (package &optional (filename nil))
;  (if (not filename) (setf filename (string-downcase (symbol-name package))))
;  (let* ((faslname (make-pathname
;                               :directory *cpu*
;                               :name filename
;                               :type "fasl"))
;        (lispname (make-pathname
;                               :directory *cpu*
;                               :name filename
;                               :type "lisp")))
; 
;	(cond ((probe-file faslname)
;		(format t "calling require with cpu/*.fasl~%")
;		(package-call *old-require-package* 'require package faslname))
;	      ((probe-file lispname)
;		(format t "calling require with cpu/*.lisp~%")
;		(package-call *old-require-package* 'require package lispname))
;	      (T
;		(format t "calling require with orig filename~%")
;		(package-call *old-require-package* 'require package filename)))))
;

;;(defun load (filename) 
;  (let* ((origname (merge-pathnames filename))
;	 (faslname (make-pathname 
;			    	:directory (pathname-directory origname)
;		    	    	:name (pathname-name origname)
;		    	 	:type "fasl"))
;	 (lispname (make-pathname 
;			    	:directory (pathname-directory origname)
;		    	    	:name (pathname-name origname)
;		    	 	:type "lisp"))
;	 (exist-fasl (probe-file faslname))
;	 (exist-lisp (probe-file lispname))
;	 (newestname (if (file-older-p faslname lispname) lispname faslname))
;	 (orignametype (pathname-type origname)))
;
;  (setf *last-file-loaded* filename)
;		;; if both lisp and fasl exist and command did not specify which
;		;; to load, compile lisp again if necessary and then load fasl
;
;  (cond	((and exist-fasl exist-lisp (not orignametype))
;	 (when (equal newestname lispname) 
;	        (format t ";; Note: compiling lisp before loading fasl~%")
;		(compile-file lispname))
;	 (lisp:load faslname))
;
;		;; if both lisp and fasl exist and fasl is older, load it only
;		;; if command explicitly said to load fasl
;
; 	((and exist-fasl exist-lisp 
;	      (equal newestname lispname))
;	 (if (equal orignametype "fasl")
;	     (progn (format t ";; *** Warning, loading fasl despite more recent lisp source~%")
;		    (lisp:load faslname))
;	     (lisp:load lispname)))
;
;		;; if both lisp and fasl exist and lisp is older, load it only
;		;; if command explicitly said to load lisp
;
;  	((and exist-fasl exist-lisp 
;	      (equal newestname faslname))
;	 (if (equal orignametype "lisp")
;	     (progn (format t ";; *** Warning, loading lisp despite more recent fasl~%")
;		    (lisp:load lispname))
;	     (lisp:load faslname)))
;
;		;; if only lisp exists and command said lisp or nothing, load it
;	((and exist-lisp 
;	      (or (equal orignametype "lisp")
;	     	  (null orignametype)))
;	 (lisp:load lispname))
;
;		;; if only lisp exists and command said fasl, compile and load
;	((and exist-lisp (equal orignametype "fasl"))
;	 (format t ";; Note: compiling lisp before loading fasl~%")
;	 (compile-file lispname)
;	 (lisp:load faslname))
;
;		;; if only fasl exists and command said fasl or nothing, load it
;	((and exist-fasl 
;	      (or (equal orignametype "fasl")
;	     	  (null orignametype)))
;	 (lisp:load faslname))
;
;	(T (lisp:load origname)))
;))
;
