;;;; ti-extensions.lisp
;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *             TI Explorer Implementation Dependent Extensions
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher/Philip Johnson/Daniel Corkill/Kelly Murray
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  This file contains extensions to Common Lisp needed by GBB.  The functions
;;;  in this file depend on the details of the TI Explorer implementation.  The
;;;  file Umass-Extended-Lisp.Lisp contains extensions to Common Lisp that are
;;;  written in Common Lisp.
;;;
;;;  03-25-87 File Created.  (Gallagher)
;;;  08-23-87 Updated to Release 3.  (Gallagher)
;;;  09-09-88 Updated to Release 4.  (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(ecase (si:get-system-version)
  (2 (pushnew :explorer-release-2 *features*))
  (3 (pushnew :explorer-release-3 *features*))
  (4 (pushnew :explorer-release-4 *features*))
  (6 (pushnew :explorer-release-6 *features*)))

(in-package 'umass-extended-lisp
	    :use '(lisp))

#+(or EXPLORER-RELEASE-3 EXPLORER-RELEASE-4 EXPLORER-RELEASE-6)
(import '(ticl::string-append))

(export '(defstruct-p
	  defstruct-conc-name
	  structure-slot-names
	  structure-slot-p
	  get-structure-slot
	  deftransform
	  delete-transform
	  %pass%
	  #+(or EXPLORER-RELEASE-3 EXPLORER-RELEASE-4 EXPLORER-RELEASE-6)
            %pointer
	  ))

(use-package '(lisp))

(proclaim '(optimize (speed 3) (safety 1)))

;;; The functions that must be defined in this file are:
;;;
;;;     defstruct-p
;;;     get-defstruct-description
;;;     defstruct-conc-name
;;;     structure-slot-names
;;;     structure-slot-p
;;;     get-structure-slot  (and corresponding setf method)
;;;     deftransform
;;;     delete-transform
;;;     %pointer
;;;
;;; See the documentation with each function for arguments and purpose.



;;;; --------------------------------------------------------------------------
;;;;   Generic Defstruct Functions
;;;; --------------------------------------------------------------------------

;;; It is a hole in Common Lisp that you can't access a slot in a structure
;;; via the slot name.  For example, if I have an instance of the following
;;; defstruct,
;;;
;;;    (defstruct (foo (:conc-name FOO->))
;;;      (a nil)
;;;      (b nil))
;;;
;;; I want to be able to get the value of slot A with a call like:
;;;
;;;    (get-structure-slot object 'A).
;;;
;;; This is because I may not know what the accessor function is for that
;;; slot.  Even if I did know the name of the accessor function, I wouldn't
;;; be able to use it to alter the value of the slot because Setf needs to
;;; know the function name at compile time.  This means that code such as
;;; the following won't work.
;;;
;;;    (defun set-structure-slot (object slot-accessor new-value)
;;;      (setf (funcall slot-accessor object) new-value))
;;;
;;; Obviously, setf shouldn't need to handle that as a place form.  This
;;; just points out the need for a general slot accessor function which
;;; will work as a place form for setf.

;;; GBB uses the functions Get-Structure-Slot and Defstruct-Conc-Name
;;; which are defined below.


(defun get-defstruct-description (name)

  "GET-DESFSTRUCT-DESCRIPTION name

   If `Name' has been defined as a structure then return its
   description.  Otherwise signal an error."

  (si::get-defstruct-description name))


(defun defstruct-p (name)

  "DEFSTRUCT-P name

   This predicate returns a non-nil value if `Name' is the name of a
   defstruct type; nil otherwise.  `Name' must be a symbol."
 
  (si::getdecl name 'si::defstruct-description))


(defun defstruct-conc-name (structure)

  "DEFSTRUCT-CONC-NAME structure

   Return a string which is the prefix for building defstruct slot
   accessors for this structure.  STRUCTURE must be a symbol which
   names a structure type."

  (let* ((description (get-defstruct-description structure)))
    (if (si::defstruct-description-conc-name description)
        (string (si::defstruct-description-conc-name description))
        "")))


(defun structure-slot-names (type)

  "STRUCTURE-SLOT-NAMES type

   This function returns a list of structure slot names
   given the type (symbol) of the structure."

  (let* ((description (get-defstruct-description type))
	 (slot-data (si::defstruct-description-slot-alist description)))
    (mapcar #'first slot-data)))


(defun structure-slot-p (type slot)

  "STRUCTURE-SLOT-P type slot

   This function returns true if `Slot' is the name of a slot in the
   defstruct type named by `Type.'"

  (and (structure-slot-index-1 type slot) t))


;;; Structure Slot Index Computation Function:
;;;
;;; This is common to both the access and the setting functions.  It
;;; computes the index into the internal data structure (usually a
;;; vector) where the slot value is stored.


(defun structure-slot-index (type slot)

  ;; Returns the index of the slot or signals an error
  ;; if the slot doesn't exist.
  
  (or (structure-slot-index-1 type slot)
      (error "~S is not a slot in ~S." slot type)))


(defun structure-slot-index-1 (type slot)

  ;; Returns the index of the slot or nil if the slot doesn't
  ;; exist.  Used by STRUCTURE-SLOT-P and STRUCTURE-SLOT-INDEX.

  (let* ((description (get-defstruct-description type))
	 (slot-data (si::defstruct-description-slot-alist
		      description))
	 (index (second (assoc slot slot-data
			       ;; We use string= to avoid package
			       ;; problems.
			       :TEST #'string=))))
    (when index
      (case (second description)
	 (:NAMED-ARRAY (+ 1 index))
	 (otherwise index)))))


;;;
;;; Structure Slot Access Functions :::
;;;

(defun get-structure-slot (object slot)

  "GET-STRUCTURE-SLOT object slot

Returns the value of a slot in a structure
given the name of the slot.

This function may be used as a place form for SETF."

  (let ((type (type-of object)))
    (aref object (structure-slot-index type slot))))


;;;
;;; Structure Slot Setting Functions :::
;;;

(defun set-structure-slot (object slot value)

  ;; SET-STRUCTURE-SLOT object slot value
  ;;
  ;; Sets the value of a slot in a structure
  ;; given the name of the slot."

  (let ((type (type-of object)))
    (si::set-aref object (structure-slot-index type slot) value)))


(defsetf get-structure-slot set-structure-slot)


;;;; --------------------------------------------------------------------------
;;;;   Compiler Transforms.
;;;; --------------------------------------------------------------------------

;;; The body of the transform should return the symbol %PASS% if it
;;; doesn't apply.  If it does apply then it should return the
;;; transformed form.  For example, a common optimization is to do
;;; keyword analysis at compile time.
;;;
;;;     (defun foo (x &key a b)
;;;       ...)
;;;
;;;     (deftransform (foo keyword) (x &rest keyword-args)
;;;       (if (literal-keywords-in-arglist keyword-args)
;;;           (let ((a (getf keyword-args :a))
;;;                 (b (getf keyword-args :b)))
;;;             `(foo-internal ,x ,a ,b))
;;;           '%pass%))
;;;
;;;     (defun foo-internal (x a b)
;;;       ...)


(defmacro DEFTRANSFORM ((fn-name transform-name) arglist &body body)
  
  "DEFTRANSFORM (fn-name transform-name) arglist {form}*

   Defines a source level compiler transform for FN-NAME labeled
   TRANSFORM-NAME.  If FN-NAME already has a transform labeled
   TRANSFORM-NAME then the old one will be replaced."

  (let ((transoptfun (intern (string-append fn-name "-" transform-name "-OPT")))
	(transfun (intern (string-append fn-name "-" transform-name "-TRANSFORM"))))
    `(progn
       (defun ,transfun ,arglist . ,body)
       (defun ,transoptfun (wholeform)
	 (if (not (eq (car wholeform) ',fn-name))
	     wholeform
	     (let ((newform (apply ',transfun (cdr wholeform))))
	       (if (and (symbolp newform)
			(string= newform '%PASS%))
		   wholeform
		   newform))))
       (compiler::add-optimizer ,fn-name ,transoptfun))))


(defun delete-transform (fn-name transform-name)

  "DELETE-TRANSFORM fn-name transform-name

   This function will remove the compiler transform for FN-NAME
   designated by TRANSFORM-NAME."

  (flet ((same-symbol-name (x y)
	   (and (symbolp y) (string= x y))))
    (let* ((optimizer-name (string-append fn-name "-" transform-name "-OPT"))
	   (optimizers (get fn-name 'compiler::optimizers))
	   (n (etypecase optimizers
                (null    0)
                (symbol  (if (same-symbol-name optimizer-name optimizers)
                             1 0))
                (list    (count optimizer-name optimizers
                                :test #'same-symbol-name)))))
    ;; Because I don't know what package the deftransform was run in, I
    ;; don't know what package the optimizer function was interned in.
    ;; To get around this problem, I look for symbols with the same name
    ;; and hope that there is only one.
    (case n
       ;; There is no optimizer at all.
       (0 nil)
       ;; Only one.  This must be it.
       (1 (setf (get fn-name 'compiler::optimizers)
		(if (listp optimizers)
                    (delete optimizer-name optimizers :test #'same-symbol-name)
                    nil)))
       ;; More than one?
       (otherwise
	(error "Can't determine which compiler transform to delete.~@
                For function: ~s, transform name: ~s."
	       fn-name transform-name))))))


;;;; --------------------------------------------------------------------------
;;;;   Miscellaneous Non Common Lisp Functions
;;;; --------------------------------------------------------------------------

;;; The LISP package on the Explorers (release 2) is not pure common
;;; lisp so %pointer is already there.  In release 3 get %pointer from
;;; system.

#+(or EXPLORER-RELEASE-3 EXPLORER-RELEASE-4 EXPLORER-RELEASE-6)
(defun %pointer (object)
  (system::%pointer object))


;;;; --------------------------------------------------------------------------
;;;;   Patches
;;;; --------------------------------------------------------------------------

;;; Patch to TI's DEFINE-MODIFY-MACRO:

;;; The original code was severely brain-damaged.  It forgot to reverse the
;;; additional-arglist-names list it constructed, and did not handle &REST
;;; arguments properly.  This version is patterned after the Spice Lisp
;;; version and appears to work.

;; From file SETF.LISP#> SYS2; SYS:

SYSTEM-INTERNALS#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "SYSTEM-INTERNALS")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: SYS2; SETF.#"

#+EXPLORER-RELEASE-2

(defmacro DEFINE-MODIFY-MACRO (name additional-arglist action-function
                               &OPTIONAL doc-string)
  "Define a construct which, like INCF, modifies the value of its first argumenty.
NAME is defined so that (NAME place additional-args) expands into
  (SETF place (action-function place additional-args))
except that subforms of place are evaluated only once."
  (let ((additional-arg-names nil)
        (rest-arg nil))
    ;; Parse out the variable names and rest arg from the lambda list.
    (do ((xl additional-arglist (rest xl))
	 (arg nil))
	((null xl))
      (setq arg (first xl))
      (cond ((eq arg '&OPTIONAL))
	    ((eq arg '&REST)
	     (if (symbolp (second xl))
		 (setq rest-arg (second xl))
		 (error "Non-symbol &REST argument in definition of ~S." name))
	     (if (null (cddr xl))
		 (return nil)
		 (error "Additional arguments following &REST arg in DEFINE-MODIFY-MACRO.")))
	    ((memq arg '(&KEY &ALLOW-OTHER-KEYS &AUX))
	     (error "~S not allowed in DEFINE-MODIFY-MACRO lambda list." arg))
	    ((symbolp arg)
	     (push arg additional-arg-names))
	    ((and (listp arg) (symbolp (first arg)))    ; Optional arg syntax.
	     (push (first arg) additional-arg-names))))
    (setq additional-arg-names (nreverse additional-arg-names))      
    `(defmacro ,name (place . ,additional-arglist)
       ,doc-string
       (if (symbolp place)

	   ;; Special case for simple SETQs to speed up the expansion process and
           ;; generate better code ::

           `(setq ,place
                  ,,(if rest-arg
                       ``(,',action-function ,place ,,@additional-arg-names ,@,rest-arg)
                       ``(,',action-function ,place ,,@additional-arg-names)))

           ;; General case ::

           (multiple-value-bind (tempvars tempargs storevars storeform refform)
               (get-setf-method place)
             ;; Build a let* form using tempvars and tempargs.  Note that the original
             ;; code also included additional-arg-names in this safety binding.  They
             ;; are NOT included in this version.
             (do ((tv tempvars (rest tv))
                  (ta tempargs (rest ta))
                  (let-list nil (cons (list (first tv) (first ta)) let-list)))
                 ((null tv)
                  (push
                    (list (first storevars)
                          (sublis-eval-once
                            (pairlis tempvars tempargs)
                            ,(if rest-arg
                                 `(list* ',action-function refform ,@additional-arg-names ,rest-arg)
                                 `(list ',action-function refform ,@additional-arg-names))))
                    let-list)
                  `(let* ,(nreverse let-list)
                     ,storeform))))))))

)) ;; End of compiler-let.


;;; ---------------------------------------------------------------------------
;;;				  End of File
;;; ---------------------------------------------------------------------------
