;;;************************************************************************
;;; ps-graphics.lisp		D. Musliner
;;;
;;;************************************************************************

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.


(provide 'ps-graphics)

(require 'di-graphics)

;(in-package 'di-graphics)

;;; ---------------------------------------------------------------------------
;;; PostScript output system global variables.
;;; - these are assumed to remain constant while generating an output file 
;;;	(eg, we assume mags & offsets are const, so dont have to add each 
;;;	time, can just do translate/scale at start).

(defvar *ps-filename* "ps-output")	;; default output file name.

(defvar *ps-fontsize* 16)		;; default font size.

(defvar *ps-fontskip* nil) 	;; used to vertically center labels in objects.

(defvar *ps-width* nil)		;; picture size user coords.
(defvar *ps-height* nil)

(defvar *ps-width-points* nil)		;; picture size in points.
(defvar *ps-height-points* nil)

(defvar *ps-x-mag* 1)		;; axis magnification factors, multiplied into
(defvar *ps-y-mag* 1)		;; all user's values.

(defvar *ps-x-offset* 0)	;; axis offsets, get added to user's values.
(defvar *ps-y-offset* 0)

(defvar *ps-image-x-offset* 20)	       ;; how far image is moved on page
(defvar *ps-image-y-offset* 20)	       ;; to get away from unprintable corners

	;; normally, ps output is surrounded by a single-pixel-wide black
	;; frame.  To shut off, set ps-border to nil and border widths to 0

(defvar *ps-border* T)			;; flag: whether to print black border.
(defvar *ps-border-width* 20)		;; width of whitespace border inside
					;; black border box [20] (in points)
(defvar *ps-whiteborder-width* 36)	;; width of whitespace border outside 
					;; black border box [36] (in points)

(defvar *ps-invert-y-axis* nil)		;; flag: is Y axis inverted?

;;; ---------------------------------------------------------------------------
;;; - opens output file, sets up the size of the image (args) and away we go..
;;; - initialize offsets to push the image around on page as needed.

;;; - as of now 7/16, we these dimensions specify size of the window or
;;; 	framed image we draw, but the coords w/in that window are still 
;;;	starting at 0,0.
;;; 	- that is, the xmin ymin specify where on the page we put the origin,
;;;	  and, eg, the viable X coords are then 0..(xmax-xmin).

;;; width and height given in inches.
;;; xyminmax specify user coords to map into this window.

(defun di-ps-initialize-graphics (width height xmin ymin xmax ymax)
  (setf *ps-width-points* (* 72 width))
  (setf *ps-height-points* (* 72 height))
  (setf *ps-invert-y-axis* nil)

	;; set up xy-mags and xy-offsets to convert to points.
  (setf *ps-x-offset* (- xmin))
  (setf *ps-y-offset* (- ymin))
  (setf *ps-width* (- xmax xmin))
  (setf *ps-height* (- ymax ymin))
  (setf *ps-x-mag* (/ *ps-width-points* *ps-width*))
  (setf *ps-y-mag* (/ *ps-height-points* *ps-height*))
  (setf *ps-x-offset* (+ xmin *ps-whiteborder-width* *ps-border-width*))
  (setf *ps-y-offset* (+ ymin *ps-whiteborder-width* *ps-border-width*))
)

;;; ---------------------------------------------------------------------------
(defun di-ps-start-drawing ()
	;; the .lps suffix stands for "Lisp PostScript"
  (setf *ps-filename* (merge-pathnames *ps-filename* "*.lps"))
  (when (open *ps-filename* :direction :probe)
      (format t "~%Moving old file ~A to ~A.bak~%" *ps-filename* *ps-filename*)
      (rename-file *ps-filename* 
		   (make-pathname :name (pathname-name *ps-filename*) :type "lps.bak")))
  (setf *ps-output* (open *ps-filename* :direction :output)) 
  (format *ps-output* "%!~%")
  (format *ps-output* "%%DocumentFonts: Times-Roman~%")
  (format *ps-output* "%%Creator: Lisp Graphics~%")
  (format *ps-output* "%%Title: ~A~%" *ps-filename*)
  (format *ps-output* "%%Pages: 1~%")
  (format *ps-output* "%%BoundingBox: ~A ~A ~A ~A~%"
	(- *ps-image-x-offset* *ps-whiteborder-width*)
	(- *ps-image-y-offset* *ps-whiteborder-width*)
  	(+ *ps-image-x-offset* *ps-width-points* (* 2 *ps-whiteborder-width*))
  	(+ *ps-image-y-offset* *ps-height-points* (* 2 *ps-whiteborder-width*)))
  (format *ps-output* "%%EndComments~%")
  (format *ps-output* "%%EndProlog~%")
  (format *ps-output* "%%Page 1 1~%~%")

	;; set up font ; note assumes xmag==ymag for best scalefont
  (format *ps-output* "/Times-Roman findfont ~A ~A div scalefont setfont~%~%" 
		*ps-fontsize* (float *ps-x-mag*))

		;; scaled, one half of approx height of font chars.
		;; used to vertically center labels in objects
  (setf *ps-fontskip* (/ (/ *ps-fontsize* 3) *ps-y-mag*))

	;; set up line width (also dependent on xmag==ymag for best results)
  (format *ps-output* "1 ~A div setlinewidth~%~%" (float *ps-x-mag*))

	;; add whitespace border and translation away from corner of page.
  (format *ps-output* "~A ~A translate~%~%" 
	  (+ *ps-image-x-offset* *ps-x-offset*)
	  (+ *ps-image-y-offset* *ps-y-offset*))

	;; set up scaling to user coords
  (format *ps-output* "~A ~A scale~%~%" (float *ps-x-mag*) (float *ps-y-mag*))

	;; is Y axis inverted?
  (cond (*ps-invert-y-axis* 
	 (format *ps-output* "/inverted-Y 1 def~%")
  	 (format *ps-output* "0 ~A translate 1 -1 scale~%" (float *ps-height*)))
  	(T (format *ps-output* "/inverted-Y 0 def~%")))

	;; write out definitions for various functions.
  (init-draw-rectangle)
  (init-draw-circle)
  (init-draw-triangle)
  (init-center-text)
  (init-center-text-in-rect)

	;; draw line border around picture
  (if *ps-border*
  	(di-ps-draw-rectangle (float (- (/ *ps-border-width* *ps-x-mag*)))
		(float (- (/ *ps-border-width* *ps-y-mag*)))
		(float (+ *ps-width* (/ *ps-border-width* *ps-x-mag*)))
		(float (+ *ps-height* (/ *ps-border-width* *ps-y-mag*)))))

)

;;; ---------------------------------------------------------------------------

(defun di-ps-deinitialize-graphics ())

;;; ---------------------------------------------------------------------------
;;; - puts label centered above the line border, inside whiteborder.

(defun di-ps-label-drawing (label)
  (format *ps-output* "~A ~A (~A) center-text~%"
	(float (/ *ps-width* 2))
  	(float (+ *ps-height* (/ *ps-border-width* *ps-y-mag*) 
		  (/ *ps-whiteborder-width* *ps-y-mag* 2)))
	label))

;;; ---------------------------------------------------------------------------
;;; closes PostScript file in proper form

(defun di-ps-finish-drawing ()
  (format *ps-output* "~%showpage~%~%%%Trailer~%")
  (close *ps-output*))

;;; ---------------------------------------------------------------------------

(defun di-ps-draw-rectangle (xmin ymin xmax ymax
                             &key (filled nil) (label nil)
                             &allow-other-keys)
  (format *ps-output* "~A ~A ~A ~A " xmin ymin xmax ymax)
  (if filled (format *ps-output* "1 ")
  	     (format *ps-output* "0 "))
  (if label (format *ps-output* "(~A) " label)
  	    (format *ps-output* "(!@#) "))	;; note reserved label
  (format *ps-output* "draw-rectangle~%")
)

;;; ---------------------------------------------------------------------------

(defun di-ps-draw-circle (xmin ymin xmax ymax
                             &key (filled nil) (label nil)
                             &allow-other-keys)
  (format *ps-output* "~A ~A ~A ~A " xmin ymin xmax ymax)
  (if filled (format *ps-output* "1 ")
  	     (format *ps-output* "0 "))
  (if label (format *ps-output* "(~A) " label)
  	    (format *ps-output* "(!@#) "))	;; note reserved label
  (format *ps-output* "draw-circle~%")
)

;;; ---------------------------------------------------------------------------

(defun di-ps-draw-triangle (x1 y1 x2 y2 x3 y3
                             &key (filled nil)
                             &allow-other-keys)
  (format *ps-output* "~A ~A ~A ~A ~A ~A " x1 y1 x2 y2 x3 y3)
  (if filled (format *ps-output* "1 ")
  	     (format *ps-output* "0 "))
  (format *ps-output* "draw-triangle~%")
)

;;; ---------------------------------------------------------------------------

(defun di-ps-draw-segments (&rest seglist)
  (cond ((null (first seglist)) (format *ps-output* "stroke~%"))
        (T (format *ps-output* "~A ~A moveto~%" 
		   (first seglist) (second seglist))
	   (format *ps-output* "~A ~A lineto~%" 
		   (third seglist) (fourth seglist))
	   (apply #'di-ps-draw-segments (rest (rest (rest (rest seglist))))))))
	  

;;; ---------------------------------------------------------------------------
(defun di-ps-draw-lines (&rest seglist)
  (format t "PS DRAW LINES NOT DONE YET~%"))

;;; ---------------------------------------------------------------------------
(defun di-ps-center-text (x y text)
  (format *ps-output* "~A ~A (~A) center-text~%" x y text))

;;; ---------------------------------------------------------------------------
(defun di-ps-invert-Y-axis ()
  (setf *ps-invert-y-axis* T))

;;; ---------------------------------------------------------------------------
;;; outputs definition for PS function to draw [labeled][filled] rectangle.

(defun init-draw-rectangle ()
  (format *ps-output* "/draw-rectangle {            % rectangle function~%")
  (format *ps-output* "	/label exch def~%")
  (format *ps-output* "	/fillflag exch def~%")
  (format *ps-output* "	/ymax exch def~%")
  (format *ps-output* "	/xmax exch def~%")
  (format *ps-output* "	/ymin exch def~%")
  (format *ps-output* "	/xmin exch def~%")
  (format *ps-output* "	newpath~%")
  (format *ps-output* "	xmin ymin moveto~%")
  (format *ps-output* "	xmin ymax lineto~%")
  (format *ps-output* "	xmax ymax lineto~%")
  (format *ps-output* "	xmax ymin lineto~%")
  (format *ps-output* "	closepath fillflag 1 eq {fill} {stroke} ifelse~%")
  (format *ps-output* "	label (!@#) ne~%") 
  (format *ps-output* "	 {xmin ymin xmax ymax label center-text-in-rect} if~%") 
  (format *ps-output* "} def~%"))

;;; ---------------------------------------------------------------------------

(defun init-draw-circle ()
  (format *ps-output* "/draw-circle {            % circle function~%")
  (format *ps-output* "	/label exch def~%")
  (format *ps-output* "	/fillflag exch def~%")
  (format *ps-output* "	/ymax exch def~%")
  (format *ps-output* "	/xmax exch def~%")
  (format *ps-output* "	/ymin exch def~%")
  (format *ps-output* "	/xmin exch def~%")
  (format *ps-output* "	newpath~%")
  (format *ps-output* "	xmax ymin ymax ymin sub 2 div add moveto~%")
  (format *ps-output* "	xmin xmax xmin sub 2 div add~%")
  (format *ps-output* "	ymin ymax ymin sub 2 div add~%")
  (format *ps-output* "	xmax xmin sub 2 div 0 360 arc~%")
  (format *ps-output* "	closepath fillflag 1 eq {fill} {stroke} ifelse~%")
  (format *ps-output* "	label (!@#) ne~%") 
  (format *ps-output* "	 {xmin ymin xmax ymax label center-text-in-rect} if~%") 
  (format *ps-output* "} def~%"))

;;; ---------------------------------------------------------------------------
;;; - triangle does not take label arg like circle & square.

(defun init-draw-triangle ()
  (format *ps-output* "/draw-triangle {            % triangle function~%")
  (format *ps-output* "	/fillflag exch def~%")
  (format *ps-output* "	/y3 exch def~%")
  (format *ps-output* "	/x3 exch def~%")
  (format *ps-output* "	/y2 exch def~%")
  (format *ps-output* "	/x2 exch def~%")
  (format *ps-output* "	/y1 exch def~%")
  (format *ps-output* "	/x1 exch def~%")
  (format *ps-output* "	newpath~%")
  (format *ps-output* "	x1 y1 moveto x2 y2 lineto x3 y3 lineto x1 y1 lineto~%")
  (format *ps-output* "	closepath fillflag 1 eq {fill} {stroke} ifelse~%")
  (format *ps-output* "} def~%"))

;;; ---------------------------------------------------------------------------
;;;; frames are rectangles w/ arc-smoothed corners
;;;; defines PS function expecting xorig, yorig, width, height on stack
;;;; - note the method used here only works b/c of right angled corners...

(defun init-draw-frame ()
  (format *ps-output* "/draw-frame {            % frame function~%")
  (format *ps-output* " /height exch def~%")
  (format *ps-output* " /width exch def~%")
  (format *ps-output* " /yorig exch def~%")
  (format *ps-output* " /xorig exch def~%")
  (format *ps-output* " /top yorig height add def~%")
  (format *ps-output* " /right xorig width add def~%")
  (format *ps-output* " newpath ~%")
  (format *ps-output* " xorig yorig arc-radius add moveto~%")
  (format *ps-output*
          "     xorig top xorig arc-radius add top arc-radius arcto~%")
  (format *ps-output* " 4 {pop} repeat~%")
  (format *ps-output*
          "     right top right top arc-radius sub arc-radius arcto~%")
  (format *ps-output* " 4 {pop} repeat~%")
  (format *ps-output*
          "     right yorig right arc-radius sub yorig arc-radius arcto~%")
  (format *ps-output* " 4 {pop} repeat~%")
  (format *ps-output*
          "     xorig yorig xorig yorig arc-radius add arc-radius arcto~%")
  (format *ps-output* " 4 {pop} repeat~%")
  (format *ps-output* " closepath stroke~%")
  (format *ps-output* "} def~%")
)

;;;;************************************************************************
;;;; expects xorig yorig string
;;; NOTE to use this will need to set global ps var text-space

(defun init-draw-framed-text ()
  (format *ps-output* "/draw-framed-text {           % framed-text function~%")
  (format *ps-output* " /ft-string exch def~%")
  (format *ps-output* " /ft-yorig exch def~%")
  (format *ps-output* " /ft-xorig exch def~%")
  (format *ps-output* " ft-xorig text-space sub~%")
  (format *ps-output* " ft-yorig text-space sub~%")
  (format *ps-output* " ft-string stringwidth pop~%")
  (format *ps-output* "   twice-text-space add~%")
  (format *ps-output* "   12 twice-text-space add~%")
  (format *ps-output* " draw-frame~%")
  (format *ps-output* " ft-xorig ft-yorig moveto ft-string show~%")
  (format *ps-output* "} def~%")
)

;;;;************************************************************************
(defun init-center-text ()
  (format *ps-output* "/center-text {~%")
  (format *ps-output* " /label exch def~%")
  (format *ps-output* " /yorig exch def~%")
  (format *ps-output* " /xorig exch def~%")
  (format *ps-output* "	xorig label stringwidth pop 2 div sub yorig moveto~%")
  (format *ps-output* "	inverted-Y 1 eq~%")
  (format *ps-output* "	  { gsave 0 ~A rmoveto 1 -1 scale label show grestore }~%" (float *ps-fontskip*))
  (format *ps-output* "	  { 0 -~A rmoveto label show }~%" (float *ps-fontskip*))
  (format *ps-output* "	ifelse~%")
  (format *ps-output* "} def~%")
)

;;;;************************************************************************
(defun init-center-text-in-rect ()
  (format *ps-output* "/center-text-in-rect {~%")
  (format *ps-output* "	/label exch def~%")
  (format *ps-output* "	/ymax exch def~%")
  (format *ps-output* "	/xmax exch def~%")
  (format *ps-output* "	/ymin exch def~%")
  (format *ps-output* "	/xmin exch def~%")
  (format *ps-output* "	xmin xmax xmin sub 2 div add~%") 
  (format *ps-output* " ymin ymax ymin sub 2 div add~%")
  (format *ps-output* "	label center-text~%")
  (format *ps-output* "} def~%"))

;;;;************************************************************************
(defun framed-text (xorig yorig string)
  (format *ps-output* "~A ~A (~A) framed-text~%" xorig yorig string))
