;; mice.lisp
   
;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.

;;;            This work has been sponsored in part by:
;;;               the NSF (IRI-9010645, IRI-9015423)
;;;         the University of Michigan Rackham Graduate School
;;;


;(in-package 'MICE)

;(export '(mice))

(proclaim '(optimize (speed 3) (safety 1)))

;;; ***************************************************************************

(defun sort-agent-predicate (a1 a2)
  (or (< (agent$current-time a1) (agent$current-time a2))
      (and (= (agent$current-time a1) (agent$current-time a2))
           (> (compute-agent-authority a1) (compute-agent-authority a2))
           )))


;;; ---------------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   THESE DEFSTRUCTS SHOULD BE MOVED TO THE STRUCTURES FILE WHEN A
;;;   DEFAULT VARIABLE SETTING FILE IS CREATED
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *graphics?* t)

(defvar *sort-agent-predicate* #'sort-agent-predicate)

(defvar *default-sort-agent-predicate* #'sort-agent-predicate)

;;; ---------------------------------------------------------------------------

(defun mice (environment-file &KEY (time-limit 200))

  "MICE environment-file
Run Michigan's Intelligent Coordination Experiment"

  (setf environment-file (merge-pathnames environment-file "*.env"))
  (initialize-mice environment-file)
  (setf *time-limit* time-limit)
  (execute-mice)
  (finalize-mice))

;;; ---------------------------------------------------------------------------

(defun make-random-location (&KEY (x 1) (y 1))
  (let* ((new-location (make-location :X (random x) :Y (random y)))
         (ge (get-grid-element new-location))
         (live-agents (live-agents-at-loc-in-schedule-queue new-location)))
    (if (or live-agents   ; technically, should see whether they block it
            (and ge
                 (grid-element$features ge)))
        (if (and (= x 1) (= y 1))
            (error "Cannot find random location within ~a ~a" x y)
            (make-random-location :X x :Y y))
        new-location)))

;;; ---------------------------------------------------------------------------

(defun live-agents-at-loc-in-schedule-queue (loc)
  (let ((agts nil))
    (dolist (agt *agent-schedule-queue*)
      (when (and (equalp loc (find-agent-location agt))
                 (not (agent$removal-time agt)))
        (setf agts (cons agt agts))))
    agts))

;;; ---------------------------------------------------------------------------

(defun make-acons-list (alist)
  (cond ((null alist) nil)
        (t (acons (first alist) (second alist) (make-acons-list (rest (rest alist)))))))

;;; ---------------------------------------------------------------------------

(defun initialize-mice (env-file)
  (setf *simulation-data* (make-simulation-data))
  (setf *current-time* 0)
  (setf *move-back-count* nil)
  (setf *move-count* nil)
  (setf *attempted-move-count* nil)
  (setf *agent-schedule-queue* nil)
  (setf *mice-channels* nil)		; Jaeho's
  (setf *all-agents* nil)
  (setf *links* nil)
  (setf *attempted-links* nil)
  (setf *remote-hosts* nil)
  (setf *time-limit* 200)
;  (setf *verbose?* nil)
;  (setf *sort-agent-predicate* *default-sort-agent-predicate*)
  (setf (simulation-data$grid *simulation-data*)
        (make-array (list 100 100)
;                          (1+ (- (region$x-max (simulation-data$overall-region *simulation-data*))
;                                 (region$x-min (simulation-data$overall-region *simulation-data*))))
;                          (1+ (- (region$y-max (simulation-data$overall-region *simulation-data*))
;                                 (region$y-min (simulation-data$overall-region *simulation-data*)))))
                    :element-type 'grid-element))
  ;(let ((*package* (find-package 'mice)))
  ;  (load env-file))
    (load env-file)
  (when *remote-hosts*
    (dolist (remote-host *remote-hosts*)
      (initialize-slave-mice remote-host)))
  (if *graphics?* (initialize-mice-graphics-window))
;  (setf *all-agents* (copy-list *agent-schedule-queue*))
;  (when *mice-default-random-state?* (setf *random-state* (read (open "random-state.lisp"))))
  (setf *longest-move-time* (find-longest-move-time *agent-schedule-queue*)))

;;; ---------------------------------------------------------------------------

(defun find-longest-move-time (agent-list)
  (apply #'max
         (mapcar #'(lambda (agent)
                     (max (move-data$north (agent$move-data agent))
                          (move-data$south (agent$move-data agent))
                          (move-data$east  (agent$move-data agent))
                          (move-data$west  (agent$move-data agent))))
                 agent-list)))

;;; ---------------------------------------------------------------------------

(defun finalize-mice ()
  (if *graphics?*
    (draw-board-configuration (simulation-data$overall-time *simulation-data*)))
;  (setf *agent-schedule-queue* *all-agents*)
  nil)

;;; ---------------------------------------------------------------------------

(defun execute-mice ()
  (execution-hook)
  (do ((current-agent (find-current-agent) (when (mice-continue-p) (find-current-agent))))
      ((not current-agent))
    (setf *current-time* (agent$current-time current-agent))
    (execute-agent current-agent)
    (setf (agent$current-time current-agent) *current-time*)
    (execution-hook)))

;;; ---------------------------------------------------------------------------

(defun execution-hook ()
  "Apply domain constraints"
  (control-board))

;;; ---------------------------------------------------------------------------

(defun find-current-agent ()
  (first (setf *agent-schedule-queue* (sort *agent-schedule-queue* *sort-agent-predicate*))))

;;; ---------------------------------------------------------------------------

(defun execute-agent (agent)
  (setf *current-agent* agent)
  (when *debug?*
    (format t "~&~30,30,,'-<~> Executing agent ~A at time ~A ~30,30,,'-<~>"
            (agent$name agent)
            *current-time*))
  (let ((agent-commands nil)
        (initial-time *current-time*))
    (loop
      (when (> *current-time* initial-time) (return))
      (unless agent-commands (setf agent-commands (get-agent-commands agent)))
      ; :STOP and :QUIESCENT are special commands because MICE itself can return them
      (cond ((eq agent-commands :STOP)
             (setf *agent-schedule-queue* (remove agent *agent-schedule-queue*))
             (format t "Removing agent ~A from schedule queue due to :STOP~%" agent)
             (setf *current-time* most-positive-fixnum))
            ((eq agent-commands :QUIESCENT)
             (setf *current-time* (+ *current-time* (interpret-mice-command :QUIESCENT agent nil)))
             (setf agent-commands (setf (agent$command-buffer agent) nil)))
            ((not (listp agent-commands)) (error "Illegal mice command ~A" agent-commands))
            (t (multiple-value-bind (elapsed-time revised-commands)
                   (interpret-mice-command (first agent-commands) agent (rest agent-commands))
                 (setf *current-time* (+ *current-time* elapsed-time))
                 (setf agent-commands (setf (agent$command-buffer agent) revised-commands))))))))

(defun get-agent-commands (agent)
  (cond ((and (agent$command-buffer agent) (not *interruptable-command-sequences?*))
         (agent$command-buffer agent))
        ((agent$invocation-function agent)
         (get-commands-from-agent agent))
        (t :QUIESCENT)))

(defun get-commands-from-agent (agent)
  (if *real-time-knob*
      (compute-real-time-commands-for-agent agent)
      (funcall (agent$invocation-function agent) agent)))

;;; ---------------------------------------------------------------------------

(defun interpret-mice-command (command agent subsequent-commands)
  (when *verbose?* (format t "Agent ~a is executing command ~a~%" (agent$name agent) command))
  (let ((command-keyword (if (listp command) (first command) command)))
    (case command-keyword
      (:INTERRUPT (values 0 nil))
      (:GOTO (let* ((target-location (make-location :X (second command) :Y (third command)))
                    (current-location (find-agent-location agent))
                    (break-predicate   (fourth command))
                    (direction-list
                      (compute-shortest-track-between-locations current-location target-location))
                    (move-and-time-list
                      (compute-direction-and-cost-list current-location target-location agent direction-list)))
               (values 0 (cons (list :GOTO-INTERNAL move-and-time-list break-predicate) subsequent-commands))))
      (:GOTO-INTERNAL
       (cond ((null (second command)) ; done moving
              (values 0 subsequent-commands))
             ((eval (third command))  ; evaluate break predicate
              (values 0 (append (get-commands-from-agent agent) (cons command subsequent-commands))))
             (t (let ((direction (first (first (second command))))
                      (elapsed-time (second (first (second command)))))
                  (move-agent agent direction elapsed-time)
                  (setf (second command) (rest (second command)))
                  (values elapsed-time (cons command subsequent-commands))))))
      (:MOVE (let* ((direction (second command))
                    (elapsed-time (find-movement-time agent direction)))
               (cond (elapsed-time (move-agent agent direction)
                                   (values elapsed-time subsequent-commands))
                     (t (cerror "Continuing takes 1 time unit" "Attempt to move in forbidden direction ~A" direction)
                        (values 1 subsequent-commands)))))
      (:SCAN
       (cond ((sensor-data-p (second command))
              (setf (agent$scan-data-buffer agent) (funcall #'scan-with-sensor agent (second command)))
              (values (eval (sensor-data$time (second command))) subsequent-commands))
             ((eq (second command) :ALL)
              (let ((max-time 0))
		(unless (listp (agent$scan-data-buffer agent)) (setf (agent$scan-data-buffer agent) nil))
                (dolist (sensor (agent$sensors agent))
                  (setf (agent$scan-data-buffer agent)
                        (append (agent$scan-data-buffer agent)
                                (funcall #'scan-with-sensor agent sensor)))
                  (setf max-time (max max-time (eval (sensor-data$time sensor)))))
                (values max-time subsequent-commands)))
             ((integerp (second command)) (values (second command) subsequent-commands))
             (t (error "Illegal argument to the sensor command: ~A" (second command)))))
      (:RECV				; Added 1/8/92 Jaeho Lee
       (let ((channel (channel-structure (second command)))
	     ;; The agent$receive-message-buffer might have some non-list state value such as :empty.
	     ;; In this case remove this value and fill it with received messages (possibly nil).
	     (old (if (listp (agent$receive-message-buffer agent)) (agent$receive-message-buffer agent) nil)))
	 (cond (channel
		(setf (agent$receive-message-buffer agent) 
		  (append old (apply #'recv-messages (cons agent (cdr command)))))
		(values (eval (channel$time-to-receive channel)) subsequent-commands))
	       (t (error "Illegal argument to the :RECV command: ~A" (second command))))))
      (:SEND				; Added 1/8/92 Jaeho Lee
       (let ((channel (channel-structure (second command))))
	 (cond (channel
		(apply #'send-message (cons agent (cdr command)))
		(values (eval (channel$time-to-send channel)) subsequent-commands))
	       (t (error "Illegal argument to the :SEND command: ~A" (second command))))))
      (:REASONING (values (second command) subsequent-commands))        ; where the agent thinks for a time but does not act
      (:QUIESCENT (values 1 subsequent-commands))   ; idle takes 1 time unit
      (:NULL-ACTION (values (if (second command) (second command) 1) subsequent-commands))
      (:STOP
       (setf *agent-schedule-queue* (remove agent *agent-schedule-queue*))
       (format t "Removing agent ~A from schedule queue due to :STOP~%" agent)
       (setf *current-time* most-positive-fixnum))
      (:LINK (create-link (agent$name agent) (second command) (third command))
             (values (compute-link-cost agent (agent-structure (second command))) subsequent-commands))
      (:UNLINK (remove-link (agent$name agent) (second command))
               (values (compute-unlink-cost agent (agent-structure (second command))) subsequent-commands))
      ; rotate function returns the sim time cost
      (:ROTATE (let ((direction (second command))
                     (number-of-quadrants (third command)))
                (cond ((member direction '(:NORTH :SOUTH :EAST :WEST))
                       (multiple-value-bind (new-direction number-quadrants)
                                            (find-rotation-for-absolute-direction agent direction)
                                 (values 0 (cons (list :ROTATE new-direction number-quadrants) subsequent-commands))))
                      ((null direction) (values (move-data$rotate-one-quadrant (agent$move-data agent)) subsequent-commands))
                      (t (dotimes (counter number-of-quadrants nil)
                            (setf subsequent-commands (cons (list :ROTATE-ONE-QUADRANT direction) subsequent-commands)))
                         (print subsequent-commands)
                         (values 0 subsequent-commands)))))
      (:ROTATE-ONE-QUADRANT (values (rotate agent (second command) 1) subsequent-commands))
      (:AFFECT (affect-location (find-agent-location agent *current-time*) (second command) (third command))
               (values 0 subsequent-commands))   
      ; Added 3/18/91 DED                      
      (OTHERWISE (cerror "Resume to use NULL-ACTION" "Unknown mice command") (values 1 subsequent-commands)))))

;;; ---------------------------------------------------------------------------

(defun compute-direction-and-cost-list (loc1 loc2 agent direction-list)
  (let* ((delta-x (- (location$x loc2) (location$x loc1)))
         (delta-y (- (location$y loc2) (location$y loc1)))
         (direction-x (cond ((zerop delta-x) nil)
                            ((plusp delta-x) :EAST)
                            (t :WEST)))
         (time-x (find-movement-time agent direction-x))
         (direction-y (cond ((zerop delta-y) nil)
                            ((plusp delta-y) :SOUTH)
                            (t :NORTH)))
         (time-y (find-movement-time agent direction-y))
         (total-cost (ceiling (sqrt (+ (expt (* time-x delta-x) 2) (expt (* time-y delta-y) 2)))))
         (savings (float (- (+ (* time-x (abs delta-x)) (* time-y (abs delta-y))) total-cost)))
         (savings-per-move (ceiling (/ savings (list-length direction-list))))
         (direction-cost-list nil))
    (format t "~%Savings ~a, directions ~a" savings direction-list)
    (do ((directions (reverse direction-list) (rest directions)))
        ((null directions) nil)
      (let* ((direction (first directions))
             (expected-cost (find-movement-time agent direction))
             (savings-this-move (min expected-cost savings-per-move)))
        (setf savings (- savings savings-this-move))
        (setf savings-per-move
              (let ((remaining-directions (1- (list-length directions))))
                (if (zerop remaining-directions)
                    savings
                    (ceiling (/ savings (1- (list-length directions)))))))
        (setf direction-cost-list
              (cons (list direction (truncate (- expected-cost savings-this-move))) direction-cost-list))))
    direction-cost-list))

;;; ---------------------------------------------------------------------------

(defun compute-shortest-track-between-locations (loc1 loc2)
  (let* ((delta-x (- (location$x loc2) (location$x loc1)))
         (delta-y (- (location$y loc2) (location$y loc1)))
         (direction-x (cond ((zerop delta-x) nil)
                            ((plusp delta-x) :EAST)
                            (t :WEST)))
         (direction-y (cond ((zerop delta-y) nil)
                            ((plusp delta-y) :SOUTH)
                            (t :NORTH)))
         (x (location$x loc1))
         (y (location$y loc1))
         (directions nil))
    (loop
      (when (and (= x (location$x loc2)) (= y (location$y loc2))) (return (reverse directions)))
      (let* ((new-x (+ x (cond ((null direction-x) 0) ((eq direction-x :EAST) 1) (t -1))))
             (new-y (+ y (cond ((null direction-y) 0) ((eq direction-y :SOUTH) 1) (t -1))))
             (rev-distance-x
               (get-euclidean-distance new-x y (location$x loc1) (location$y loc1)))
             (rev-distance-y
               (get-euclidean-distance x new-y (location$x loc1) (location$y loc1)))             
             (new-distance-x
               (get-euclidean-distance new-x y (location$x loc2) (location$y loc2)))
             (new-distance-y
               (get-euclidean-distance x new-y (location$x loc2) (location$y loc2))))
        (cond ((not direction-x)
               (setf directions (cons direction-y directions))
               (setf y new-y))
              ((not direction-y)
               (setf directions (cons direction-x directions))
               (setf x new-x))
              ((< (+ rev-distance-y new-distance-y) (+ rev-distance-x new-distance-x))
               (setf directions (cons direction-y directions))
               (setf y new-y))
              (t (setf directions (cons direction-x directions))
                 (setf x new-x)))))))

;;; ---------------------------------------------------------------------------

(defun get-euclidean-distance (x1 y1 x2 y2)
  (sqrt (+ (expt (- x2 x1) 2) (expt (- y2 y1) 2))))

;;; ---------------------------------------------------------------------------

(defun compute-real-time-commands-for-agent (agent)
  (let* ((start-time (get-internal-real-time))
         (commands   (funcall (agent$invocation-function agent) agent))
         (end-time   (get-internal-real-time))
         (elapsed-seconds (/ (- end-time start-time) (float internal-time-units-per-second))))
    (cons `(:REASONING ,(round (* elapsed-seconds *real-time-knob*))) commands)))

;;; ---------------------------------------------------------------------------

(defun compute-link-cost (agent1 agent2)
  (let* ((alist (agent$link-cost-alist agent1))
         (cost (rest (assoc (agent$type agent2) alist))))
    (if cost
        (eval cost)
        (let ((all-cost (rest (assoc :ALL alist))))
          (if all-cost (eval all-cost) 0)))))

;;; ---------------------------------------------------------------------------

(defun compute-unlink-cost (agent1 agent2)
  (let* ((alist (agent$unlink-cost-alist agent1))
         (cost (rest (assoc (agent$type agent2) alist))))
    (if cost
        (eval cost)
        (let ((all-cost (rest (assoc :ALL alist))))
          (if all-cost (eval all-cost) 0)))))

;;; ---------------------------------------------------------------------------

(defun get-grid-element (loc &OPTIONAL (create nil) &KEY x y)
  "Returns a grid element.  Creates a new element when one does not exist and CREATE is true."
  (if (and x y)
      (if (or (eq x :LIMBO) (eq y :LIMBO))
          *removed-agent-grid-location*
          (let ((x-coord (- x (region$x-min (simulation-data$overall-region *simulation-data*))))
                (y-coord (- y (region$y-min (simulation-data$overall-region *simulation-data*)))))
            (when (coords-in-grid-bounds x-coord y-coord)
              (if (aref (simulation-data$grid *simulation-data*) x-coord y-coord)
                  (aref (simulation-data$grid *simulation-data*) x-coord y-coord)
                  (when create (setf (aref (simulation-data$grid *simulation-data*)
                                           x-coord y-coord) (make-grid-element)))))))
      (if (not-limbo-p loc)
          (let ((x-coord (- (location$x loc) (region$x-min (simulation-data$overall-region *simulation-data*))))
                (y-coord (- (location$y loc) (region$y-min (simulation-data$overall-region *simulation-data*)))))
            (when (coords-in-grid-bounds x-coord y-coord)
            (if (aref (simulation-data$grid *simulation-data*) x-coord y-coord)
                (aref (simulation-data$grid *simulation-data*) x-coord y-coord)
                (when create (setf (aref (simulation-data$grid *simulation-data*)
                                         x-coord y-coord) (make-grid-element))))))
          *removed-agent-grid-location*)))

;;; ---------------------------------------------------------------------------

(defun coords-in-grid-bounds (x y)
  ;Added because aref on TI's does not always check bounds correctly.
  (and (>= x 0)
       (>= y 0)
       (< x (first (array-dimensions (simulation-data$grid *simulation-data*))))
       (< y (second (array-dimensions (simulation-data$grid *simulation-data*))))))

;;; ---------------------------------------------------------------------------

(defun not-limbo-p (location)

  "NOT-LIMBO-P location
Returns nil if either coordinate of the location is :LIMBO, otherwise it returns location."

  (if (or (eq (location$x location) :LIMBO) (eq (location$y location) :LIMBO))
      nil
      location))

;;; ***************************************************************************
;;; EOF
