;;;************************************************************************
;;; *mac-graphics.lisp		L. Hohmann & Young-pa So
;;;
;;; Macintosh based graphics for MICE.  Based on similar routines for
;;; X-Windows (CLX) written by D. Musliner
;;;
;;;************************************************************************

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.


; Not a member of any package quite yet...

(provide 'mac-graphics)

;(require 'di-graphics)

(proclaim '(optimize (speed 3) (safety 1)))

;;; ---------------------------------------------------------------------------
;;; MAC graphics global constants.

(defvar *mac-window* nil
  "Holds the single global instance of the MICE graphics window")

(defparameter *mac-pixels-per-inch* 36
  "The number of pixels per inch used for a window")

(defun inches-to-pixels (inch-value)
  (* *mac-pixels-per-inch* inch-value))

(defun pixels-to-inches (pixel-value)
  (/ pixel-value *mac-pixels-per-inch*))


;; width and height are the width and the height of the window in pixels.
;; --> watch for an allowance for the height of the window title bar
;;
(defobject *mice-mac-graphics-window* *window*)
(proclaim '(object-variable *mice-mac-graphics-window* 
            width height y-mag-factor x-mag-factor y-offset x-offset
            x-min y-min x-max y-max))

(defobfun (exist *mice-mac-graphics-window*) (init-list)
  (let* ((window-width-in-inches  (getf init-list :width-in-inches))
         (window-height-in-inches (getf init-list :height-in-inches))
         (the-size (make-point (inches-to-pixels window-width-in-inches)
                               (inches-to-pixels window-height-in-inches))))
    
    (have 'x-mag-factor (getf init-list :x-mag-factor nil))
    (have 'y-mag-factor (getf init-list :y-mag-factor nil))
    (have 'x-offset (getf init-list :x-offset nil))
    (have 'y-offset (getf init-list :y-offset nil))
    (have 'x-min (getf init-list :x-min nil))
    (have 'y-min (getf init-list :y-min nil))
    (have 'x-max (getf init-list :x-max nil))
    (have 'y-max (getf init-list :y-max nil))
    (usual-exist 
     (init-list-default init-list
                        :window-size the-size
                        :window-title "MICE Graphics Window"
                        :close-box-p nil
                        :window-type :document-with-grow))))
                            
(defobfun (set-window-size *mice-mac-graphics-window*) 
          (h &optional v &aux new-size)
  (usual-set-window-size (setf new-size (make-point h v)))

  ; now reset all the mice related graphics information  
  ; note that magnification factor is based on a pixel 
  ; measurement
  ;
  (setf x-mag-factor (/ (point-h new-size)  (- x-max x-min)))
  (let ((temp-y-mag-factor (/ (point-v new-size) (- y-max y-min))))
    (if (< y-mag-factor 0)
      (setf y-mag-factor (- temp-y-mag-factor))
      (setf y-mag-factor temp-y-mag-factor))))
 
     
(defun dump-mac ()
  (cond (*mac-window*
         (format t "Variable information for *mac-window*:~%")
         (format t "    *mac-window*    :  ~a~%" *mac-window*)
         (format t "    width-in-inches :  ~a~%" (get-width-in-inches))
         (format t "    width-in-pixels :  ~a~%" (get-width-in-pixels))

         (format t "    height-in-inches:  ~a~%" (get-height-in-inches))
         (format t "    height-in-pixels:  ~a~%" (get-height-in-pixels))

         (format t "    x-mag-factor    :  ~a~%" (get-x-mag-factor))
         (format t "    y-mag-factor    :  ~a~%" (get-y-mag-factor))

         (format t "    x-offset        :  ~a~%" (get-x-offset))
         (format t "    y-offset        :  ~a~%" (get-y-offset)))
        (t
         (format t "*mac-window* is unbound"))))

(defun get-width-in-inches ()
  (let ((size-in-pixels (ask *mac-window* (window-size))))
    (round (/ (point-h size-in-pixels) *mac-pixels-per-inch*))))
  
(defun get-width-in-pixels ()
  (point-h (ask *mac-window* (window-size))))
  
(defun get-height-in-inches ()
  (let ((size-in-pixels (ask *mac-window* (window-size))))
    (round (/ (point-v size-in-pixels) *mac-pixels-per-inch*))))
  
(defun get-height-in-pixels ()
  (point-v (ask *mac-window* (window-size))))
  
(defun get-y-mag-factor ()
  (ask *mac-window* y-mag-factor))
(defun set-y-mag-factor (new-y-mag-factor)
  (ask *mac-window* (setf y-mag-factor new-y-mag-factor)))

(defun get-x-mag-factor ()
  (ask *mac-window* x-mag-factor))

(defun get-x-offset ()
  (ask *mac-window* x-offset))

(defun get-y-offset ()
  (ask *mac-window* y-offset))

;; ---------------------------------------------------------------------------
;; pass in width and height in inches, and coords to fit w/in that screen.
;; we use the xy-mag to scale the dimensions, and the xy-offset to offset for
;; nonzero xmin ymin.
;;
;; For development on the mac, we use 36 points per inch instead of
;; of the normal 72 (==> 72 points make the window too big for an SE.)
;;
(defun di-mac-initialize-graphics (width height xmin ymin xmax ymax)
  
  ; if there is a previously open window, close it
  (when *mac-window*
    (when (typep *mac-window* *window*)
      (ask *mac-window* (window-close)))
    (setf *mac-window* nil))
  
  (setf *mac-window* 
        (oneof *mice-mac-graphics-window*
               :width-in-inches   width
               :height-in-inches  height
               :x-min             xmin
               :y-min             ymin
               :x-max             xmax
               :y-max             ymax
               
               ; calculate the xy-magnification factors and scale the
               ; coordinate system to fit it into the window
               ;
               :x-offset      (- xmin)
               :y-offset      (- ymin)

               ; note that magnification factor is based on a pixel 
               ; measurement
               ;
               :x-mag-factor (/ (inches-to-pixels width)  (- xmax xmin))
               :y-mag-factor (/ (inches-to-pixels height) (- ymax ymin))))

  (ask *mac-window* (set-window-font '(9 :plain))))

;;; ---------------------------------------------------------------------------
(defun di-mac-deinitialize-graphics ()
  (when *mac-window*
    (when (typep *mac-window* *window*)
      (ask *mac-window* (window-close)))
    (setf *mac-window* nil)))
 
;;; ---------------------------------------------------------------------------
(defun di-mac-label-drawing (label)
  (ask *mac-window* (set-window-title label)))

;;; ---------------------------------------------------------------------------
(defun di-mac-draw-in-background ( &aux temp)
  (message-dialog "di-mac-draw-in-background not implemented yet."))

;;; ---------------------------------------------------------------------------
(defun di-mac-draw-in-foreground ( &aux temp)
  (message-dialog "di-mac-draw-in-foreground not implemented yet."))


;;; ---------------------------------------------------------------------------
;;; To invert the Y axis to conform to my PS-like standard,
;;; 	new miny = total-height - miny - rect-height
;;; note should make filled and label and other options be &key options
;;; and include the ignore-other-keys flag, so that difff graphics packages
;;; can choose to support these or not, depending, but all calls need the
;;; basic rectangle and will give no errors on any syntax beyond that.
;;;
(defun di-mac-draw-rectangle (xmin ymin xmax ymax 
                                    &key (filled nil) (label nil)
                                    &allow-other-keys)
  
  (setf xmin (round (* (+ xmin (get-x-offset)) (get-x-mag-factor))))
  (setf xmax (round (* (+ xmax (get-x-offset)) (get-x-mag-factor))))
  
  (cond ((< (get-y-mag-factor) 0) 
         ; when negative Y mag, dont invert Y axis (for MICE)
         (setf ymin (round (- (* (+ ymin (get-y-offset)) (get-y-mag-factor)))))
         (setf ymax (round (- (* (+ ymax (get-y-offset)) (get-y-mag-factor)))))
         
         )
        (T
         (setf ymin (round (* (+ ymin (get-y-offset)) (get-y-mag-factor))))
         (setf ymax (round (* (+ ymax (get-y-offset)) (get-y-mag-factor))))
         
         (setf ymin (- (get-height-in-pixels) height ymin))))
  
  ;(format t "di-mac-draw-rectangle drawing rect ~A ~A ~A ~A~%" xmin ymin xmax ymax)
  (ask *mac-window*
    ; mac frame-rect is left top right bottom
    ;
    (frame-rect xmin ymin xmax ymax))

  (when label
    (let* ((pen-x (+ 2 xmin))
           (pen-y (- ymax 1)))
      (ask *mac-window* (move-to pen-x pen-y))
      (ask *mac-window* (princ label *mac-window*))))
)



#|
  (draw-rectangle *pm* *pmgc* xmin ymin width height filled)
  (if label
    (draw-glyphs *pm* *pmgc* 
                 (round (+ xmin 
                           (/ width 2)
                           (- (/ (text-width *pmgc* label) 2))))
                 (+ ymin (round (/ height 2)) 4) 
                 label)))

|#



;;; ---------------------------------------------------------------------------
;;; - fits circle w/in rectangle coords given.
;;;
(defun di-mac-draw-circle (xmin ymin xmax ymax
                                &key (filled nil) (label nil)
                                &allow-other-keys &aux width height)
  
  (setf xmin (round (* (+ xmin (get-x-offset)) (get-x-mag-factor))))
  (setf xmax (round (* (+ xmax (get-x-offset)) (get-x-mag-factor))))
  (setf width (- xmax xmin))
  (cond ((< (get-y-mag-factor) 0) ;; when negative Y mag, dont invert Y axis (for MICE)
         (setf ymin (round (- (* (+ ymin (get-y-offset)) (get-y-mag-factor)))))
         (setf ymax (round (- (* (+ ymax (get-y-offset)) (get-y-mag-factor)))))
         (setf height (- ymax ymin))
         )
        (T
         (setf ymin (round (* (+ ymin (get-y-offset)) (get-y-mag-factor))))
         (setf ymax (round (* (+ ymax (get-y-offset)) (get-y-mag-factor))))
         (setf height (- ymax ymin))
         (setf ymin (- (get-height-in-pixels) height ymin))))
  ;(format t "di-mac-draw-circle drawing circle ~A ~A ~A ~A~%"
  ;        xmin ymin xmax ymax)
  
  (ask *mac-window*
    (frame-oval xmin ymin xmax ymax))

  (when label
    (let* ((pen-x (+ 2 xmin))
           (pen-y (- ymax 1)))
      (ask *mac-window* (move-to pen-x pen-y))
      (ask *mac-window* (princ label *mac-window*))))
)
  
#|
  (if label
    (draw-glyphs *pm* *pmgc* 
                 (round (+ xmin 
                           (/ width 2)
                           (- (/ (text-width *pmgc* label) 2))))
                 (+ ymin (round (/ height 2)) 4) 
                 label))
  
|#



;;; ---------------------------------------------------------------------------
;;;
(defun mac-convert-x (x) 
  (round (* (+ x (get-x-offset)) (get-x-mag-factor))))

(defun mac-convert-y (y) 
  (cond ((< (get-y-mag-factor) 0)
         (round (- (* (+ y (get-y-offset)) (get-y-mag-factor)))))
        (T
         (- (get-height-in-pixels)
            (round (* (+ y (get-y-offset)) (get-y-mag-factor)))))))

;;; ---------------------------------------------------------------------------
;;;
(defun di-mac-draw-segments (&rest seglist)
  (setf seglist (map-even #'mac-convert-x seglist))
  (setf seglist (map-odd #'mac-convert-y seglist))
  (apply #'internal-mac-draw-segments seglist))

(defun internal-mac-draw-segments (&rest seglist)
  (cond (seglist
         (ask *mac-window*
           (move-to (first seglist) (second seglist))
           (line-to (third seglist) (fourth seglist)))
         (apply #'internal-mac-draw-segments (rest (rest (rest (rest seglist))))))
        (T nil)))

;;; ---------------------------------------------------------------------------
;;;
(defun di-mac-draw-triangle (x1 y1 x2 y2 x3 y3 &key (filled nil) (label nil)
                                &allow-other-keys &aux point-list)
  (setf point-list (list x1 y1 x2 y2 x3 y3 x1 y1))
  (setf point-list (map-even #'mac-convert-x point-list))
  (setf point-list (map-odd #'mac-convert-y point-list))
  
  ;(apply #'internal-mac-draw-segments point-list)
  ;(setf point-list (mapcar #'round point-list))
  (ask *mac-window* (move-to (first point-list) (second point-list)))
  (ask *mac-window* (line-to (third point-list) (fourth point-list)))
  (ask *mac-window* (line-to (fifth point-list) (sixth point-list)))
  (ask *mac-window* (line-to (seventh point-list) (eighth point-list)))

  (when label
    (let* ((pen-x (max-convert-x (- x1 2)))
           (pen-y (mac-convert-y (- y1 2))))
      (ask *mac-window* (move-to pen-x pen-y))
      (ask *mac-window* (princ label *mac-window*))))
)
  ;; need to fill the triangle as necessary...
  

;;; ---------------------------------------------------------------------------
;;;
(defun di-mac-draw-lines (point-list &key (filled nil) (label nil)
                                     &allow-other-keys)
  (setf point-list (map-even #'mac-convert-x point-list))
  (setf point-list (map-odd #'mac-convert-y point-list))
  (apply #'internal-mac-draw-segments point-list)
  ;;(draw-lines *pm* *pmgc* point-list :fill-p filled))
  )


;;; ---------------------------------------------------------------------------
;;;
(defun di-mac-center-text (x y text)
  (setf x (mac-convert-x x))
  (setf y (mac-convert-y y))
  ;(format t "Put ~a at positions x=~a y=~a" text x y))

  (when text
    (let* ((pen-x (+ 2 x))
           (pen-y (+ 2 y)))
      (ask *mac-window* (move-to pen-x pen-y))
      (ask *mac-window* (princ text *mac-window*))))
  
;;(draw-glyphs *pm* *pmgc* (- x (/ (text-width *pmgc* text) 2)) y text)))
)


;;; ---------------------------------------------------------------------------
;;; Function clear-window
;;; -- a support function to be used during development
;;;
(defun clear-window ()
  (ask *mac-window*
    (fill-region *white-pattern* (clip-region))
    (window-draw-grow-icon)))

;;; ---------------------------------------------------------------------------
;;; EOF

