;;; icons.lisp		- D. Musliner
;;; ---------------------------------------------------------------------------
;;; THE ICON FUNCTIONS
;;; - all should expect to receive at least the location in the first arg,
;;;	and at least include &allow-other-keys, unless they use :agent.
;;; - if called to draw an agent, will get the agent structure passed in
;;;	with :agent key.
;;; - if called to draw a grid location, will not have a :agent.
;;; ---------------------------------------------------------------------------
;;; icons are drawn using di-graphics device independent graphics call, with
;;; that caveat that the Y axis is reversed from the normal di-graphics usage:
;;; the origin of the Y axis for these functions is normally at the bottom
;;; of the screen, but the MICE window has that reversed by calling 
;;; (di-invert-y-axis) when it initializes the window.
;;; - so the icon functions must draw as though the Y axis increases as you
;;; 	go down the screen.
;;; ---------------------------------------------------------------------------
;;; A major difference from the previous icons is that, in this setup,
;;; MICE grid locations are defined to be 1x1, so you just make your icon 
;;; function fit that scale and the di-graphics routines will take care of
;;; any other scaling that needs to be done to fit the grid into the physical
;;; window size specified in the di-initialize-graphics call
;;; ---------------------------------------------------------------------------
;;; All of the icons so far except the clocks take key options for filling
;;;	and labelling with a string.  The labels can be very useful for telling
;;;	same-shape agents apart!
;;; The hero icon is the one good example of an icon function that looks inside
;;;	the agent structure it is representing and changes its form for the
;;;	status of the agent... the others could certainly do that, but they
;;;	are more generic forms.  I'd suggest creating specialized icon functions
;;;	for, say, agents whose icons are filled whenever they are linked to
;;;	something, for instance.  This is a more modular approach to causing
;;;	the icon change than actually having the linking action alter the
;;;	icon function used to draw the agent.

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.

;;;            This work has been sponsored in part by:
;;;               the NSF (IRI-9010645, IRI-9015423)
;;;         the University of Michigan Rackham Graduate School
;;;

;(in-package 'mice)

;(use-package 'di-graphics)

;;; ---------------------------------------------------------------------------
(defun rectangle-icon (location &key (filled nil) (label nil) 
			&allow-other-keys)
  (cond ((not *graphics?*) 
	 (if label (char label 0) #\R))
  	(T (let* ((locx (location$x location))
	 	  (locy (location$y location)))
  	 	 (di-draw-rectangle locx locy (1+ locx) (1+ locy) 
			:filled filled :label label)))))

;;; ---------------------------------------------------------------------------
(defun square-icon (location &key (filled nil) (label nil) 
			&allow-other-keys)
  (rectangle-icon location :filled filled :label label))

;;; ---------------------------------------------------------------------------
(defun small-rectangle-icon (location &key (filled nil) (label nil) 
			&allow-other-keys)
  (cond ((not *graphics?*) 
	 (if label (char label 0) #\r))
  	(T (let* ((locx (location$x location))
	 	  (locy (location$y location)))
  		 (di-draw-rectangle (+ locx .2) (+ locy .2) 
			(+ locx .8) (+ locy .8) 
			:filled filled :label label)))))

;;; ---------------------------------------------------------------------------
(defun small-square-icon (location &key (filled nil) (label nil) 
			&allow-other-keys)
  (small-rectangle-icon location :filled filled :label label))

;;; ---------------------------------------------------------------------------
(defun circle-icon (location &key (filled nil) (label nil) 
			&allow-other-keys)
  (cond ((not *graphics?*) 
	 (if label (char label 0) #\C))
  	(T (let* ((locx (location$x location))
	 	  (locy (location$y location)))
  		 (di-draw-circle locx locy (1+ locx) (1+ locy) 
			:filled filled :label label)))))

;;; ---------------------------------------------------------------------------
(defun small-circle-icon (location &key (filled nil) (label nil) 
			&allow-other-keys)
  (cond ((not *graphics?*) 
	 (if label (char label 0) #\c))
  	(T (let* ((locx (location$x location))
	 	  (locy (location$y location)))
  		 (di-draw-circle (+ locx .2) (+ locy .2) 
			(+ locx .8) (+ locy .8) 
			:filled filled :label label)))))

;;; ---------------------------------------------------------------------------
(defun triangle-icon (location &key (filled nil) (label nil) 
			&allow-other-keys)
  (cond ((not *graphics?*) 
	 (if label (char label 0) #\T))
  	(T (let* ((locx (location$x location))
	 	  (locy (location$y location)))
  		 (di-draw-triangle locx (+ locy 1) 
			(+ locx 1) (+ locy 1)
			(+ locx .5) locy
		 	:filled filled)

  		 (if label (di-center-text (+ locx .5) (+ locy .9) label))))))

;;; ---------------------------------------------------------------------------
(defun small-triangle-icon (location &key (filled nil) (label nil) 
		&allow-other-keys)
  (cond ((not *graphics?*) 
	 (if label (char label 0) #\t))
  	(T (let* ((locx (location$x location))
	 	  (locy (location$y location)))
  		 (di-draw-triangle (+ locx .2) (+ locy .8)
				(+ locx .8) (+ locy .8)
				(+ locx .5) (+ locy .2)
		 	:filled filled)
  		 (if label (di-center-text (+ locx .5) (+ locy .7) label))))))

;;; ---------------------------------------------------------------------------
(defun tri-in-tri-icon (location &key &allow-other-keys)
  (small-triangle-icon location)
  (triangle-icon location))

;;; ---------------------------------------------------------------------------
(defun tri-in-sqr-icon (location &key &allow-other-keys)
  (small-triangle-icon location)
  (square-icon location))

;;; ---------------------------------------------------------------------------
(defun tri-in-circle-icon (location &key &allow-other-keys)
  (small-triangle-icon location)
  (circle-icon location))

;;; ---------------------------------------------------------------------------
;;; - this one doesnt look very good.
(defun circle-in-tri-icon (location &key &allow-other-keys)
  (small-circle-icon location)
  (triangle-icon location))

;;; ---------------------------------------------------------------------------
(defun circle-in-sqr-icon (location &key &allow-other-keys)
  (small-circle-icon location)
  (square-icon location))

;;; ---------------------------------------------------------------------------
(defun circle-in-circle-icon (location &key &allow-other-keys)
  (small-circle-icon location)
  (circle-icon location))

;;; ---------------------------------------------------------------------------
;;; - this one doesnt look very good.
(defun sqr-in-tri-icon (location &key &allow-other-keys)
  (small-square-icon location)
  (triangle-icon location))

;;; ---------------------------------------------------------------------------
(defun sqr-in-sqr-icon (location &key &allow-other-keys)
  (small-square-icon location)
  (square-icon location))

;;; ---------------------------------------------------------------------------
(defun sqr-in-circle-icon (location &key &allow-other-keys)
  (small-square-icon location)
  (circle-icon location))

;;; ---------------------------------------------------------------------------
;;; - for non-graphical clocks, use first digit of time.

(defun clock-icon (location &key (hour 3) &allow-other-keys)
  (let* ((x-center (+ .5 (location$x location)))
         (y-center (+ .5 (location$y location)))
         (radius   .5)
         (rover2   .25)
         (rover2xsqrt3 (* (sqrt 3) rover2)))

  (cond ((not *graphics?*) 
	 (char (format nil "~A" hour) 0))
  	(T (circle-icon location)
	
        (di-draw-segments x-center y-center
	  (+ x-center
	   (case hour
		(1 rover2) (2 rover2xsqrt3) (3 radius) 
		(4 rover2xsqrt3) (5 rover2) (6 0)
		(7 (- rover2)) (8 (- rover2xsqrt3)) (9 (- radius)) 
		(10 (- rover2xsqrt3)) (11 (- rover2)) (12 0) (T 0)))
	  (+ y-center
	   (case hour
		(1 (- rover2xsqrt3)) (2 (- rover2)) (3 0) 
		(4 rover2) (5 rover2xsqrt3) (6 radius)
		(7 rover2xsqrt3) (8 rover2) (9 0) (10 (- rover2))
		(11 (- rover2xsqrt3)) (12 (- radius)) (T 0))))))))

;;; ---------------------------------------------------------------------------
;;; - for non-graphical clocks, use first digit of hour.

(defun square-clock-icon (location &key (hour 3) &allow-other-keys)
  (let* ((x-left (location$x location))
         (y-top (location$y location))
         (radius   .5)
         (rover2   .25))

  (cond ((not *graphics?*) 
	 (char (format nil "~A" hour) 0))
  	(T (rectangle-icon location)
	(di-draw-segments
	  (+ x-left radius) (+ y-top  radius)
	  (+ x-left
	     (case hour
		(1 (+ radius rover2)) (2 (* 2 radius)) 
		(3 (* 2 radius)) (4 (* 2 radius)) 
		(5 (+ radius rover2)) (6 radius) (7 rover2) (8 0) 
		(9 0) (10 0) (11 rover2) (12 radius) (T 0)))
	  (+ y-top
	     (case hour
		(1 0) (2 rover2) (3 radius) (4 (+ radius rover2)) 
		(5 (* 2 radius)) (6 (* 2 radius)) (7 (* 2 radius))
		(8 (+ radius rover2)) (9 radius) (10 rover2) 
		(11 0) (12 0) (T 0))))))))

;;; ---------------------------------------------------------------------------
(defun hero-icon (location &key (agent nil) (filled nil) (label nil)
			&allow-other-keys)
  (cond ((not *graphics?*) 
	 (if label (char label 0) #\H))
   	(T
  (let* ((orientation (find-agent-orientation agent))
	 (linked? (find-agent-linkages agent))
  	 (s .34)
	 (halfs .17)
	 (r .1)				;; length of tiny fingers
	 (-r (- r))
	 (longarm (- s r))		;; length of arm linked (else r)
	 (minx (+ (location$x location) s))
	 (miny (+ (location$y location) s))
         (topy (+ miny s)))

  	(di-draw-rectangle minx miny (+ minx s) (+ miny s) :filled filled)
    (if label (di-center-text (+ minx halfs) (+ miny halfs) label))
    (case orientation
	(:SOUTH 
	 (if linked? 
		(di-draw-segments
			(+ minx halfs) topy
			(+ minx halfs) (+ topy longarm)
		  	(+ minx halfs -r) (+ topy longarm r)
			(+ minx halfs -r) (+ topy longarm)
			(+ minx halfs -r) (+ topy longarm)
			(+ minx halfs r) (+ topy longarm)
			(+ minx halfs r) (+ topy longarm)
			(+ minx halfs r) (+ topy longarm r))
		(di-draw-segments
			(+ minx halfs) topy
			(+ minx halfs) (+ topy r)
		  	(+ minx halfs -r) (+ topy r r)
			(+ minx halfs -r) (+ topy r)
			(+ minx halfs -r) (+ topy r)
			(+ minx halfs r) (+ topy r)
			(+ minx halfs r) (+ topy r)
			(+ minx halfs r) (+ topy r r))))
	(:NORTH 
	 (if linked?
               (di-draw-segments
			(+ minx halfs) miny
			(+ minx halfs) (- miny longarm)
		  	(+ minx halfs -r) (- miny longarm r)
			(+ minx halfs -r) (- miny longarm)
			(+ minx halfs -r) (- miny longarm)
			(+ minx halfs r) (- miny longarm)
			(+ minx halfs r) (- miny longarm)
			(+ minx halfs r) (- miny longarm r))
		  (di-draw-segments
			(+ minx halfs) miny
			(+ minx halfs) (- miny r)
		  	(+ minx halfs -r) (- miny r r)
			(+ minx halfs -r) (- miny r)
			(+ minx halfs -r) (- miny r)
			(+ minx halfs r) (- miny r)
			(+ minx halfs r) (- miny r)
			(+ minx halfs r) (- miny r r))))
	(:WEST
	 (if linked?
		  (di-draw-segments
			minx (+ miny halfs)
			(- minx longarm) (+ miny halfs)
			(- minx longarm r) (+ miny halfs r)
			(- minx longarm) (+ miny halfs r)
			(- minx longarm) (+ miny halfs r)
			(- minx longarm) (+ miny halfs -r)
			(- minx longarm) (+ miny halfs -r)
			(- minx longarm r) (+ miny halfs -r))
		  (di-draw-segments
			minx (+ miny halfs)
			(- minx r) (+ miny halfs)
			(- minx r r) (+ miny halfs r)
			(- minx r) (+ miny halfs r)
			(- minx r) (+ miny halfs r)
			(- minx r) (+ miny halfs -r)
			(- minx r) (+ miny halfs -r)
			(- minx r r) (+ miny halfs -r))))
	(:EAST
	 (if linked?
		  (di-draw-segments
			(+ minx s) (+ miny halfs)
			(+ minx s longarm) (+ miny halfs)
			(+ minx s longarm r) (+ miny halfs r)
			(+ minx s longarm) (+ miny halfs r)
			(+ minx s longarm) (+ miny halfs r)
			(+ minx s longarm) (+ miny halfs -r)
			(+ minx s longarm) (+ miny halfs -r)
			(+ minx s longarm r) (+ miny halfs -r))
		  (di-draw-segments
			(+ minx s) (+ miny halfs)
			(+ minx s r) (+ miny halfs)
			(+ minx s r r) (+ miny halfs r)
			(+ minx s r) (+ miny halfs r)
			(+ minx s r) (+ miny halfs r)
			(+ minx s r) (+ miny halfs -r)
			(+ minx s r) (+ miny halfs -r)
			(+ minx s r r) (+ miny halfs -r)))))
))))

;;; new in icons.lisp
;;; ---------------------------------------------------------------------------

(defun jet-icon (location &key (agent nil) (label nil)
			&allow-other-keys)
  (cond ((not *graphics?*) 
	 (if label (char label 0) #\J))
   	(T
  (let  ((orientation (find-agent-orientation agent))
	 (locx (location$x location))
	 (locy (location$y location))
	 (wingtipy .32)
	 (wingjointx .44)
	 (wingjointy .6)
	 (tailtipx .33)
	 (tailjointy .1)
	 rotation
	 segments)

  (setf segments (list
	0 (- 1 wingtipy) wingjointx (- 1 wingtipy)
	wingjointx (- 1 wingtipy) wingjointx (- 1 tailjointy)
	wingjointx (- 1 tailjointy) tailtipx 1
	tailtipx 1 (- 1 tailtipx) 1
	(- 1 tailtipx) 1 (- 1 wingjointx) (- 1 tailjointy)
	(- 1 wingjointx) (- 1 tailjointy) (- 1 wingjointx) 
	(- 1 wingtipy)
	(- 1 wingjointx) (- 1 wingtipy) 1 (- 1 wingtipy)
	1 (- 1 wingtipy) (- 1 wingjointx) (- 1 wingjointy)
	(- 1 wingjointx) (- 1 wingjointy) .5 0 
	.5 0 wingjointx (- 1 wingjointy)
	wingjointx (- 1 wingjointy) 0 (- 1 wingtipy)))

  (setf rotation
    (case orientation
	(:NORTH 0)
	(:SOUTH pi)
	(:EAST (/ pi 2))
	(:WEST (- (/ pi 2)))
	(:NORTHEAST (/ pi 4))
	(:NORTHWEST (- (/ pi 4)))
	(:SOUTHEAST (* 3 (/ pi 4)))
	(:SOUTHWEST (- (* 3 (/ pi 4))))))
  (setf segments (di-translate-segments segments -.5 -.5))
  (setf segments (di-rotate-segments segments rotation))
  (setf segments (di-translate-segments segments (+ .5 locx) (+ .5 locy)))
  (apply #'di-draw-segments segments)

  (if label (di-center-text (- locx .2) (+ locy .9)  label))
))))

;;; ---------------------------------------------------------------------------
;;; used to blank out a single grid location, not normally used for an agent
;;; or a grid location.

;(defun blank-icon (location &key (filled nil) (label nil) &allow-other-keys)
;  (cond ((not *graphics?*) (format nil " "))
;	(T (di-swap-colors)
;	   (rectangle-icon location :filled T)
;	   (di-swap-colors))))
;
;;; ---------------------------------------------------------------------------
;;; - null icon, draws nothing.

(defun null-icon (location &key (filled nil) &allow-other-keys))
  


