;; graphics.lisp	D. Musliner
;; - graphics routines for MICE

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.

;;;            This work has been sponsored in part by:
;;;               the NSF (IRI-9010645, IRI-9015423)
;;;         the University of Michigan Rackham Graduate School
;;;

;(in-package 'MICE)
;
;(export '(initialize-Mice-window
;	  deinitialize-Mice-window
;	  draw-board-configuration))
;

(require 'di-graphics)

;(use-package '(di-graphics))

(proclaim '(optimize (speed 3) (safety 1)))

;;; ***************************************************************************
;;; set graphics mode and default to using graphics display.

(let ((mode nil))
#+TI
  (setf mode 'TI)
#-TI
  (setf mode 'X)
#+:apple
  (setf mode 'MAC)
  (di-set-graphics-mode mode))

(defvar *graphics?* T)

(defvar *display-width* 7)	;; default width in inches of MICE graphics.

;;; ---------------------------------------------------------------------------
;;; - opens window to have same aspect ratio as the mice grid,so rectangle
;;; 	icons come out square and the circle icons always fit right inside
;;;	them on all di-graphics outputs (rather than becoming ovals on some
;;;	output devices and circles on others).

(defun initialize-mice-graphics-window (&key (width *display-width*) 
					&allow-other-keys)

  (declare (ignore default-location))

  (let ((xsize (1+ (- (region$x-max
                            (simulation-data$overall-region *simulation-data*))
                         (region$x-min
                            (simulation-data$overall-region *simulation-data*)))
                 ))
        (ysize (1+ (- (region$y-max
                            (simulation-data$overall-region *simulation-data*))
                         (region$y-min
                            (simulation-data$overall-region *simulation-data*)))
                 )))

  (di-initialize-graphics width (* width (/ ysize xsize)) 0 0 xsize ysize)
  (di-invert-Y-axis)
  (scan-for-drawable-grid-elements)
))

;;; ---------------------------------------------------------------------------
(defun deinitialize-mice-graphics-window () 
  (di-deinitialize-graphics))

;;; ---------------------------------------------------------------------------

(defun return-agent-structure-list (agent-assoc-list)
  (mapcar #'(lambda (pair) (rest pair)) agent-assoc-list))

;;; ---------------------------------------------------------------------------

;;; runs the agents' draw functions with first arg as the agent itself, 
;;; second arg as the time, and rest of args as specified in the rest of the
;;; draw function list.

(defun draw-board-configuration 
	(&OPTIONAL (time *current-time*)
		   (agent-list (return-agent-structure-list *all-agents*)))

  ;(format t ".")
  (di-start-drawing)		;; clear the drawing buffer


  (dolist (agent agent-list) (draw-agent agent time))
  (draw-grid time)

  (di-label-drawing (format nil "MICE   Time = ~A" time))

  (di-finish-drawing)		;; copy drawing buffer to window.
  ;(format t ".~%")
)

;;; ---------------------------------------------------------------------------
;;; replace in graphics.lisp
;;; ---------------------------------------------------------------------------

(defun draw-agent (agent time)
		;; if agent already removed by time, dont draw.
  (cond ((and (agent$removal-time agent) (> time (agent$removal-time agent)))
         nil)
		;; nil status means agent not created yet.
        ((not (find-agent-status agent time)) 
	 nil)
 	((agent$draw-function agent)
	  (apply (first (agent$draw-function agent))
		(list* (find-agent-location agent time)
                       :AGENT agent :TIME time
		       (rest (agent$draw-function agent)))))
	((eq (agent$type agent) :BLUE) 
	 (rectangle-icon (find-agent-location agent time)))
	((eq (agent$type agent) :RED) 
	 (circle-icon (find-agent-location agent time)))
	(T 
	 (cerror t "Unable to draw agent ~a of type ~a"
                 (agent$name agent) (agent$type agent)))))

;;; ---------------------------------------------------------------------------
;;; This version of draw-grid only draws the elements that have been
;;; determined to have a draw-function by scan-for-drawable-grid-elements.
;;; Thus, if you ever change a grid element from a blank to something drawable,
;;; you'd need to rerun the scan routine.
;;; - this saves massively during running of a simulation, since we dont
;;; 	check every grid element each time.
;;; - should probably extend this functionality to the print-grid routine also.

(defun draw-grid (time)
  (let  (grid-el loc)
     	(dolist (grid-el-loc *drawable-grid-elements*)
		(setf grid-el (first grid-el-loc))
		(setf loc (make-location :x (second grid-el-loc) :y (third grid-el-loc)))
		(apply (first (grid-element$draw-function grid-el))
		       (list* loc (rest (grid-element$draw-function grid-el)))))
))

(defvar *drawable-grid-elements* nil)

;;; ---------------------------------------------------------------------------
(defun scan-for-drawable-grid-elements ()
  (setf *drawable-grid-elements* nil)
  (let ((loc (make-location)) grid-el)
    (do ((x (region$x-min (simulation-data$overall-region *simulation-data*)) (1+ x)))
        ((> x (region$x-max (simulation-data$overall-region *simulation-data*))))
      (do ((y (region$y-min (simulation-data$overall-region *simulation-data*)) (1+ y)))
          ((> y (region$y-max (simulation-data$overall-region *simulation-data*))))
        (setf (location$x loc) x)
        (setf (location$y loc) y)
        (when (and (setf grid-el (get-grid-element loc))
                   (grid-element$draw-function grid-el))
	      (push (list grid-el x y) *drawable-grid-elements*))))
))
  
;;; ---------------------------------------------------------------------------

(defmacro redisplay-graphics (&rest args) `(di-redisplay-drawings ,@args))

;;; ---------------------------------------------------------------------------
(defun save-grid-ps (&key (file nil) (time *current-time*))
"save-grid-ps &key (time *current-time*) (file time.lps)
  - Saves PostScript file which draws the MICE grid at time.
  - file defaults to the time with suffixed '.lps' (for Lisp PostScript)."

  (let  ((last-mode *graphics-mode*))
	(di-set-graphics-mode 'ps)
	(if file
	  (setf *ps-filename* file)
	  (setf *ps-filename* (format nil "~A" time)))
	(initialize-mice-graphics-window)
	(draw-board-configuration time)
	(di-deinitialize-graphics)
	(di-set-graphics-mode last-mode)))

;;; ---------------------------------------------------------------------------
;;; - these old redisplay functions are only used with save/restore-history
;;;	functions
;;; - there is no new equivalent of the redisplay-agents-history b/c the di-graphics
;;;	save does not distinguish the calls used to draw each agent.
;;; ---------------------------------------------------------------------------
(defun redisplay-agents-history (&OPTIONAL agent-list)
  (unless agent-list (setf agent-list (mapcar #'(lambda (pair) (rest pair)) *all-agents*)))
  (initialize-mice-graphics-window)
  (dotimes (i (1+ *current-time*))
    (draw-agents-board-configuration i agent-list)))

(defun redisplay-history (&KEY (sleep-time 0) (start 0) (end *current-time*) (breaks nil))
  (initialize-mice-graphics-window :DEFAULT-LOCATION t)
  (do ((i start (1+ i)))
      ((> i end) nil)
    (when (>= i start)
      (draw-board-configuration i)
      (if (member i breaks)
          (progn
            (format t "~%Break at time ~a.~%Hit any key to continue. " i)
            (read-char)
            (format t "Continuing...~%"))
          (sleep sleep-time)))))

;;; ---------------------------------------------------------------------------
; overwrite this definition to include the abort keyword, so it doesn't hang

;(defun xlib::close-buffer (buffer)
;  ;; Close the host connection in BUFFER
;;  (declare (type buffer xlib::buffer))
;  (unless (null (xlib::buffer-output-stream buffer))
;    (xlib::wrap-buf-output buffer              
;      (funcall (xlib::buffer-close-function buffer) buffer :ABORT :ABORT))
;    (setf (xlib::buffer-dead buffer) t)
;    ;; Zap pointers to the streams, to ensure they're GC'd
;    (setf (xlib::buffer-output-stream buffer) nil)
;    (setf (xlib::buffer-input-stream buffer) nil)
;    )
;  nil)

;;; ---------------------------------------------------------------------------
