;; extint.lisp
;; - external interface for mice agents

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.

;;;            This work has been sponsored in part by:
;;;               the NSF (IRI-9010645, IRI-9015423)
;;;         the University of Michigan Rackham Graduate School
;;;


;(in-package 'MICE)

;(export '(move-agent
;          null-action
;          compute-new-location
;          live-agent-at-location-p
;          scan-for-agent-grid-descriptions
;          scan-mice-region
;          read-and-reset-scan-data
;          legal-region-p
;          agent-allowed-in-region-p
;          legal-location-p
;          agent-allowed-in-location-p
;          location-in-region-p
;          coordinates-in-region-p
;          regions-overlap-p
;          find-movement-time
;          find-agent-location
;          find-agent-status
;          find-agent-linkages
;          find-agent-other
;          find-agent-orientation
;          agent-structure
;          link
;          create-link
;          unlink
;          remove-link))
;
(proclaim '(optimize (speed 3) (safety 1)))

;;; ***************************************************************************

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;        STRUCTURES IN AGENT-MICE INTERFACE
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Grid-description is the external equivalent of MICE's grid-element defstruct.
;;; It is used in the external interface in order to limit the information passed
;;; to an agent when it scans a region.

(defxstruct (grid-description :EXPORT (:CONC-NAME "GRID-DESCRIPTION$"))
  (agents   nil)
  (coordinates nil :TYPE location)
  (features nil))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;        FUNCTIONS IN AGENT-MICE INTERFACE
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun move-agent (agent direction &OPTIONAL time)

  "MOVE-AGENT agent direction
Move an agent in the direction given."

  (when direction 
    (let* ((old-location (find-agent-location agent *current-time*))
           (new-location (compute-new-location old-location direction (find-agent-orientation agent *current-time*))))

;; DJM	(format t "move ~A from old loc ~A to new loc ~A~%" agent old-location new-location)
      ;Add move action to action list
      (setf (agent-state$action-history (current-state agent))
            (acons (agent$current-time agent)
                   (make-agent-action :ACTION (cons :MOVE direction) :TIME (agent$current-time agent))
                   (agent-state$action-history (current-state agent))))
      
      ;Update new location
      (when (legal-location-p new-location :AGENT agent)
        (setf (agent$location agent) new-location))
      
      ;Update time
      (setf (agent$current-time agent) (+ (agent$current-time agent) (if time time (find-movement-time agent direction)))))))

;;; ---------------------------------------------------------------------------

(defun null-action (agent time-steps)

  "NULL-ACTION agent time-steps
Advance agent's clock by given time-steps."

  (when (not (equalp (agent$location agent)
                     (find-agent-location agent (1- (agent$current-time agent)))))
    (setf (agent$state-history agent)
          (acons (agent$current-time agent)
                 (make-agent-state :LOCATION (agent$location agent)
                                   :STATUS (agent-state$status (current-state agent)))
                 (agent$state-history agent))))
  (setf (agent$current-time agent) (+ (agent$current-time agent) time-steps)))

;;; ---------------------------------------------------------------------------

(defun affect-location (agent-location grid-location changes)

   "AFFECT-LOCATION location changes
Change grid element at grid-location to values in changes.  Agent-location is
the current location of the calling agent.  Grid-location is a location or direction.
Changes is (:FEATURES (...) :DRAW-FUNCTION (...)) for the grid-location."

   (let ((new-location nil)
      ;  (new-grid-element nil)
         (features nil)
         (draw-fn nil)
         (feature-changes nil)
         (draw-changes nil))
      ;(format t "~%grid-location = ~a" grid-location)
      ;(format t "~%changes = ~a ~%" changes)
      ;(format t "~%first (changes) = ~a~%" (first changes))
      (case (first changes)
               (:FEATURES (setf features (second changes))
                          (setf feature-changes 't))
               (:DRAW-FUNCTION (setf draw-fn (second changes))
                               (setf draw-changes 't))
               (nil nil)
               (OTHERWISE (error "Unown parameter to AFFECT - ignored." )))
      (case (third changes)
               (:FEATURES (setf features (fourth changes))
                          (setf feature-changes 't))    
               (:DRAW-FUNCTION (setf draw-fn (fourth changes))
                               (setf draw-changes 't))
               (nil nil)
               (OTHERWISE (error "Unkown parameter to AFFECT - ignored." )))
      (cond ((location-p grid-location) (setf new-location grid-location))
            (t (setf new-location (compute-new-location agent-location grid-location)))) 
      ;(setf new-grid-element (get-grid-element new-location))
      (let ((x-coord (- (location$x new-location) (region$x-min (simulation-data$overall-region *simulation-data*))))
            (y-coord (- (location$y new-location) (region$y-min (simulation-data$overall-region *simulation-data*)))))
         ;(format t "~%grid= ~a " (simulation-data$grid *simulation-data*)) 
         ;(format t "~%arrayp = ~a " (arrayp (simulation-data$grid *simulation-data*)))
         (when feature-changes
            (setf (grid-element$features (get-grid-element (make-location :x x-coord :y y-coord) t)) features))  
         ;(format t "~%draw-fn = ~a " draw-fn)   
         (when draw-changes
              (setf (grid-element$draw-function (get-grid-element (make-location :x x-coord :y y-coord) t)) 
                       ; :.33-GREY-CIRCLE  
                        draw-fn
               ))
         )))

;;; ----------------------------------------------------------------------------

(defun compute-new-location (old-location direction &OPTIONAL orientation)

  "COMPUTE-NEW-LOCATION old-location direction
Compute a new location from an old location and direction."

  (when (and (member direction '(:FORWARD :BACKWARD :LEFT :RIGHT))
             (not orientation))
    (cerror "Continue to return old location"
            "Compute-new-location in direction ~A with orientation ~A not allowed"
            direction orientation))
  (let ((new-location
         (case direction
           (:NORTH (make-location :X (location$x old-location)
                                  :Y (1- (location$y old-location))))
           (:SOUTH (make-location :X (location$x old-location)
                                  :Y (1+ (location$y old-location))))
           (:EAST  (make-location :X (1+ (location$x old-location))
                                  :Y (location$y old-location)))
           (:WEST  (make-location :X (1- (location$x old-location))
                                  :Y (location$y old-location)))
           (:FORWARD  (compute-new-location old-location orientation))
           (:BACKWARD (compute-new-location
                        old-location
                        (case orientation
                          (:NORTH :SOUTH)
                          (:SOUTH :NORTH)
                          (:EAST  :WEST )
                          (:WEST  :EAST )
                          (otherwise nil))))
           (:LEFT     (compute-new-location
                        old-location
                        (case orientation
                          (:NORTH :WEST )
                          (:SOUTH :EAST )
                          (:EAST  :NORTH)
                          (:WEST  :SOUTH)
                          (otherwise nil))))
           (:RIGHT    (compute-new-location
                        old-location
                        (case orientation
                          (:NORTH :EAST )
                          (:SOUTH :WEST )
                          (:EAST  :SOUTH)
                          (:WEST  :NORTH)
                          (otherwise nil))))
           (otherwise (copy-location old-location)))))
    new-location))

;;; ---------------------------------------------------------------------------

(defun live-agent-at-location-p (grid-elem)
  (and (not (null (grid-element$agents grid-elem)))
       (some #'(lambda (agent)
                 (null (agent$removal-time agent)))
             (grid-element$agents grid-elem))))

;;; ---------------------------------------------------------------------------

(defun agents-at-location (grid-elem)
  (grid-element$agents grid-elem))

;;; ---------------------------------------------------------------------------

(defun read-and-reset-received-messages (&key (reset-value :empty) (agent *current-agent*))
  (let ((current-messages (agent$receive-message-buffer agent)))
    (setf (agent$receive-message-buffer agent) reset-value)
    current-messages))

;;; ---------------------------------------------------------------------------

(defun read-and-reset-scanned-data (&key (reset-value nil) (agent *current-agent*))
  (let ((current-data (agent$scan-data-buffer agent)))
    (setf (agent$scan-data-buffer agent) reset-value)
    current-data))

;;; ---------------------------------------------------------------------------

(defun scan-for-agent-grid-descriptions (agent)
  (scan-mice-region nil
                    (agent$location agent)
                    :INTERESTING-P #'live-agent-at-location-p
                    :OBSTRUCTED-BY (sensor-data$obstructed-by (first (agent$sensors agent)))
                    :X-MIN (+ (location$x (find-agent-location agent *current-time*))
                              (region$x-min (sensor-data$range (first (agent$sensors agent)))))
                    :Y-MIN (+ (location$y (find-agent-location agent *current-time*))
                              (region$y-min (sensor-data$range (first (agent$sensors agent)))))
                    :X-MAX (+ (location$x (find-agent-location agent *current-time*))
                              (region$x-max (sensor-data$range (first (agent$sensors agent)))))
                    :Y-MAX (+ (location$y (find-agent-location agent *current-time*))
                              (region$y-max (sensor-data$range (first (agent$sensors agent)))))))

;;; ---------------------------------------------------------------------------

(defun scan-with-sensor (agent sensor)
  (cond
    ((eq (sensor-data$orientation-sensitive-p sensor) :UNKNOWN)
     (setf (sensor-data$orientation-sensitive-p sensor)
           (or (/= (region$x-min (sensor-data$range sensor))
                   (region$y-min (sensor-data$range sensor)))
               (/= (region$x-max (sensor-data$range sensor))
                   (region$y-max (sensor-data$range sensor)))))
     (scan-with-sensor agent sensor))
    ((sensor-data$orientation-sensitive-p sensor)
     (let ((agent-orientation (find-agent-orientation agent))
           (dfront (- (region$y-min (sensor-data$range sensor))))
           (dback  (region$y-max (sensor-data$range sensor)))
           (dleft  (- (region$x-min (sensor-data$range sensor))))
           (dright (region$x-max (sensor-data$range sensor))))
       (scan-mice-region nil
                         (find-agent-location agent *current-time*)
                         :INTERESTING-P (sensor-data$interesting-p sensor)
                         :OBSTRUCTED-BY (sensor-data$obstructed-by sensor)
                         :X-MIN (+ (location$x (find-agent-location agent *current-time*))
                                   (- (case agent-orientation
                                        (:NORTH dleft)
                                        (:SOUTH dright)
                                        (:EAST  dback)
                                        (:WEST  dfront))))
                         :Y-MIN (+ (location$y (find-agent-location agent *current-time*))
                                   (- (case agent-orientation
                                        (:NORTH dfront)
                                        (:SOUTH dback)
                                        (:EAST  dleft)
                                        (:WEST  dright))))
                         :X-MAX (+ (location$x (find-agent-location agent *current-time*))
                                   (case agent-orientation
                                     (:NORTH dright)
                                     (:SOUTH dleft)
                                     (:EAST  dfront)
                                     (:WEST  dback)))
                         :Y-MAX (+ (location$y (find-agent-location agent *current-time*))
                                   (case agent-orientation
                                     (:NORTH dback)
                                     (:SOUTH dfront)
                                     (:EAST  dright)
                                     (:WEST  dleft))))))
    (t (scan-mice-region nil
                         (find-agent-location agent *current-time*)
                         :INTERESTING-P (sensor-data$interesting-p sensor)
                         :OBSTRUCTED-BY (sensor-data$obstructed-by sensor)
                         :X-MIN (+ (location$x (find-agent-location agent *current-time*))
                                   (region$x-min (sensor-data$range sensor)))
                         :Y-MIN (+ (location$y (find-agent-location agent *current-time*))
                                   (region$y-min (sensor-data$range sensor)))
                         :X-MAX (+ (location$x (find-agent-location agent *current-time*))
                                   (region$x-max (sensor-data$range sensor)))
                         :Y-MAX (+ (location$y (find-agent-location agent *current-time*))
                                   (region$y-max (sensor-data$range sensor)))))))

;;; ---------------------------------------------------------------------------

(defun scan-mice-region (scan-region-or-regions sensor-location
                         &KEY (interesting-p #'live-agent-at-location-p)
                         (obstructed-by nil)
                         x-min y-min x-max y-max)

  "SCAN-MICE-REGION scan-regions sensor-location &KEY (interesting-p #'live-agent-at-location-p)
   (obstructed-by nil) x-min y-min x-max y-max

Scan-mice-region takes a list of regions, scan-regions, and a sensor-location and scans the
mice grid for interesting (according to interesting-p) and visible grid elements.
It returns a list of grid-descriptions that meet the criteria."

  (let ((grid-list nil))
    (if scan-region-or-regions
        (if (listp scan-region-or-regions)
            (dolist (region scan-region-or-regions)
              (let ((x-min (max (region$x-min region)
                                (region$x-min (simulation-data$overall-region *simulation-data*))))
                    (y-min (max (region$y-min region)
                                (region$y-min (simulation-data$overall-region *simulation-data*))))
                    (x-max (min (region$x-max region)
                                (region$x-max (simulation-data$overall-region *simulation-data*))))
                    (y-max (min (region$y-max region)
                                (region$y-max (simulation-data$overall-region *simulation-data*)))))
                (setf grid-list
                      (cons (get-visible-grids (location$x sensor-location) (location$y sensor-location)
                                               x-min x-max y-min y-max interesting-p obstructed-by)
                            grid-list))))
            (let ((x-min (max (region$x-min scan-region-or-regions)
                              (region$x-min (simulation-data$overall-region *simulation-data*))))
                  (y-min (max (region$y-min scan-region-or-regions)
                              (region$y-min (simulation-data$overall-region *simulation-data*))))
                  (x-max (min (region$x-max scan-region-or-regions)
                              (region$x-max (simulation-data$overall-region *simulation-data*))))
                  (y-max (min (region$y-max scan-region-or-regions)
                              (region$y-max (simulation-data$overall-region *simulation-data*)))))
              (setf grid-list
                    (get-visible-grids (location$x sensor-location) (location$y sensor-location)
                                       x-min x-max y-min y-max interesting-p obstructed-by))))
        (let ((x-min (max x-min (region$x-min (simulation-data$overall-region *simulation-data*))))
              (y-min (max y-min (region$y-min (simulation-data$overall-region *simulation-data*))))
              (x-max (min x-max (region$x-max (simulation-data$overall-region *simulation-data*))))
              (y-max (min y-max (region$y-max (simulation-data$overall-region *simulation-data*)))))
          (setf grid-list
                (get-visible-grids (location$x sensor-location) (location$y sensor-location)
                   x-min x-max y-min y-max interesting-p obstructed-by))))
    grid-list))

;;; ---------------------------------------------------------------------------

(defun get-visible-grids (x-sensor y-sensor x-min x-max y-min y-max interesting-p obstructed-by)

  "GET-VISIBLE-GRIDS x-sensor y-sensor x-min x-max y-min y-max interesting-p obstructed-by

 Returns grid-descriptions for all interesting locations within the region defined by 
 (x-min x-max y-min y-max) that are visible from the sensor location (x-sensor y-sensor)."

  (multiple-value-bind (interesting-grids obstructions)
      (get-interesting-and-obstructing x-min x-max y-min y-max interesting-p obstructed-by)
    (dolist (interest-point interesting-grids)
      (when (some #'(lambda (obstruction-data)
                      (let ((obstruction (first obstruction-data))
                            (obstruction-diameter (second obstruction-data)))
                        (obstructed-p x-sensor y-sensor
                                      (location$x (grid-description$coordinates interest-point))
                                      (location$y (grid-description$coordinates interest-point))
                                      (location$x obstruction) (location$y obstruction)
                                      :DIAMETER obstruction-diameter)))
                  obstructions)
        (setf interesting-grids (remove interest-point interesting-grids))))
    interesting-grids))

;;; ---------------------------------------------------------------------------

(defun get-interesting-and-obstructing (x-min x-max y-min y-max interesting-p obstructed-by)

  "GET-INTERESTING-AND-OBSTRUCTING x-min x-max y-min y-max interesting-p obstructed-by

Searches the grid in the region specified by (x-min x-max y-min y-max).  Returns two lists,
the first is grid-descriptions for all interesting-p location, the second is locations that
obstruct the sensor (obstructed-by)."

  (let ((interesting-grids nil)
        (obstructions      nil))
    (do ((x x-min (1+ x)))
        ((> x x-max))
      (do ((y y-min (1+ y)))
          ((> y y-max))
        (let* ((grid-elem (get-grid-element nil nil :X x :Y y)))          
          ;;; If there is a grid element at the location specified, add it to the interesting-grids
          ;;; list if it is interesting-p.  Add its location to the obstructions list if it obstructs
          ;;; the sensor.
          (when grid-elem
            (when (and interesting-p
                       (funcall interesting-p grid-elem))
              (setf interesting-grids (cons (make-grid-description
                                              :AGENTS (copy-list (grid-element$agents grid-elem))
                                              :COORDINATES (make-location :X x :Y y)
                                              :FEATURES (copy-list (grid-element$features grid-elem)))
                                            interesting-grids)))
            (when obstructed-by
              (let ((width (funcall obstructed-by grid-elem)))
                (when (numberp width)
                  (setf obstructions (cons (list (make-location :X x :Y y) width) obstructions)))))))))
    (values interesting-grids obstructions)))

;;; ---------------------------------------------------------------------------
  
(defun legal-region-p (region &KEY (agent nil) x-min x-max y-min y-max)

  "LEGAL-REGION-P region &KEY (agent nil)
Indicate whether a region is within the grid boundary.  Returns allowable portion
of the region (if any) or nil if completely illegal.
If the AGENT parameter is included, check if the agent is allowed in the location."

  (when (and (let* ((overall-region (simulation-data$overall-region *simulation-data*))
                    (xmin (max (if x-min x-min (region$x-min region)) (region$x-min overall-region)))
                    (ymin (max (if y-min y-min (region$y-min region)) (region$y-min overall-region)))
                    (xmax (min (if x-max x-max (region$x-max region)) (region$x-max overall-region)))
                    (ymax (min (if y-max y-max (region$y-max region)) (region$y-max overall-region))))
               (if (or (> xmin xmax) (> ymin ymax))
                   (setf region nil)
                   (if region
                       (progn (setf (region$x-min region) xmin)
                              (setf (region$y-min region) ymin)
                              (setf (region$x-max region) xmax)
                              (setf (region$y-max region) ymax))
                       (setf region t)))
               region)
             (or (null agent)
                 (agent-allowed-in-region-p region agent :X-MIN x-min :X-MAX x-max :Y-MIN y-min :Y-MAX y-max)))
    region))

;;; ---------------------------------------------------------------------------

(defun agent-allowed-in-region-p (region agent &KEY x-min x-max y-min y-max)

  "AGENT-ALLOWED-IN-REGION-P region agent
Returns non-nil if agent is allowed in all locations of the region.
Assumes that the region is within the grid boundary and checks if the agent is allowed there."
  
  (let ((okx? t))
    (do ((x (if x-min x-min (region$x-min region)) (1+ x)))
        ((or (not okx?) (> x (if x-max x-max (region$x-max region)))) okx?)
      (setf okx?
            (let ((oky? t))
              (do ((y (if y-min y-min (region$y-min region)) (1+ y)))
                  ((or (not oky?) (> y (if y-max y-max (region$y-max region)))) oky?)
                (setf oky?
                      (let ((grid-elem (get-grid-element nil nil :X x :Y y)))
                        (or (null grid-elem)
                            
                            ;;; Agent's type is not one of the location's blocking types.
                            (and (not (member
                                        (agent$type agent)
                                        (rest (assoc :BLOCKED-TYPES (grid-element$features grid-elem)))))
                                 
                                 ;;; Grid location does not block all agents.
                                 (not (member
                                        :ALL
                                        (rest (assoc :BLOCKED-TYPES (grid-element$features grid-elem)))))
                                 
                                 ;;; Other agents already there allow the agent to be there.
                                 ;*** Commented out since it keeps agents from following each other when ***
                                 ;*** one is right behind the other.                                     ***
                                 ;*** (not (some #'(lambda (other-agent) (member (agent$type agent)
                                 ;*** (agent$blocked-by-types other-agent)))
                                 ;***            (live-agents (grid-element$agents grid-elem))))        ***
                                 ))))))))
    okx?))

;;; ---------------------------------------------------------------------------
  
(defun legal-location-p (location &KEY (agent nil))

  "LEGAL-LOCATION-P location &KEY (agent nil)
Indicate whether a location is within the grid boundary.
If the AGENT parameter is included, check if the agent is allowed in the location."

  (and (location-in-region-p location (simulation-data$overall-region *simulation-data*))
       (or (null agent)
           (agent-allowed-in-location-p location agent))))

;;; ---------------------------------------------------------------------------

(defun agent-allowed-in-location-p (location agent)

  "AGENT-ALLOWED-IN-LOCATION-P location agent
Assumes that the location is within the grid boundary and checks if the agent is allowed there."

  (let ((grid-elem (get-grid-element location)))
    (or (null grid-elem)

              ;;; Agent's type is not one of the location's blocking types.
        (and (not (member
                    (agent$type agent)
                    (rest (assoc :BLOCKED-TYPES (grid-element$features grid-elem)))))

             ;;; Grid location does not block all agents.
             (not (member
                    :ALL
                    (rest (assoc :BLOCKED-TYPES (grid-element$features grid-elem)))))

             ;;; Subcomponents are allowed in their new locations.
             (or (null (agent$sub-components agent))
                 (every #'(lambda (child-and-link-type) 
                            (legal-location-p 
                              (compute-new-location (agent$location agent)
                                                    (compute-link-direction
                                                      (agent$orientation agent)
                                                      (rest child-and-link-type)))
                              :AGENT (first child-and-link-type)))
                        (agent$sub-components agent)))
             ))))

;;; ---------------------------------------------------------------------------

(defun location-in-region-p (loc reg)
  (coordinates-in-region-p (location$x loc) (location$y loc) reg))

(defun coordinates-in-region-p (x y reg)
  (coordinates-in-bounds-p x y (region$x-min reg) (region$y-min reg) (region$x-max reg) (region$y-max reg)))

(defun coordinates-in-bounds-p (x y x-min y-min x-max y-max)
  (and (>= x x-min) (>= y y-min) (<= x x-max) (<= y y-max)))

;;; ---------------------------------------------------------------------------

(defun obstructed-p (x11 y11 x12 y12 xob yob &KEY (diameter 1))

  "OBSTRUCTED-P x11 y11 x12 y12 xob yob

X11 Y11 are the observer's position, X12 Y12 are the observed position,
and XOB YOB are the obstruction's position."
  (when (or (< diameter 0) (> diameter 1))
    (error "Diameter must be within range of 0 to 1.~%"))
  (cond ((and (= x11 x12) (= y11 y12))
         nil) ; can't be obstructed when looking at myself
        ((and (= xob x12) (= yob y12))
         nil) ; obstruction at observed position cannot block
        ((and (= xob x11) (= yob y11))
         nil) ; obstruction at observer's position cannot block
        ((= x11 x12) ; looking north-south
         (and (= xob x11)  ; in line
              (within-range-p yob (min y11 y12) (max y11 y12)))) ; and between observer and observed
        ((= y11 y12) ; looking east-west
         (and (= yob y11)  ; in line
              (within-range-p xob (min x11 x12) (max x11 x12)))) ; and between observer and observed
        ((or (< xob (min x11 x12)) (< yob (min y11 y12)) (> xob (max x11 x12)) (> yob (max y11 y12)))
         nil) ; no way the segments can intersect
        ; at this point, we know that the slope will be > 0 but < infinity
        ; what we do is find the intersection between the line of observation and each of
        ; the lines surrounding the obstruction (its grid region) and determine whether the
        ; intersection point is on both segments.  If so, the observation is obstructed.
        (t (let* ((slope (/ (float (- y12 y11)) (float (- x12 x11))))
                  (constant (- y11 (* slope x11)))
                  (intersection-point nil)
                  (radius (/ diameter 2)))      ;radius of obstruction
;             (format t "~%Slope ~a and Constant ~a" slope constant)
             (or
               ; line below the obstruction
               (progn (setf intersection-point  ; in this case, compute x
                            (/ (- (- yob radius) constant) slope))
                      (and (within-range-p intersection-point (- xob radius) (+ xob radius))
                           (within-range-p intersection-point (min x11 x12) (max x11 x12))))
               ; line above the obstruction
               (progn (setf intersection-point  ; in this case, compute x
                            (/ (- (+ yob radius) constant) slope))
                      (and (within-range-p intersection-point (- xob radius) (+ xob radius))
                           (within-range-p intersection-point (min x11 x12) (max x11 x12))))
               ; line left of the obstruction
               (progn (setf intersection-point  ; in this case, compute y
                            (+ (* slope (- xob radius)) constant))
                      (and (within-range-p intersection-point (- yob radius) (+ yob radius))
                           (within-range-p intersection-point (min y11 y12) (max y11 y12))))
               ; line right of the obstruction
               (progn (setf intersection-point  ; in this case, compute y
                            (+ (* slope (+ xob radius)) constant))
                      (and (within-range-p intersection-point (- yob radius) (+ yob radius))
                           (within-range-p intersection-point (min y11 y12) (max y11 y12)))))))))

(defun within-range-p (value lower upper)
  (and (>= value lower) (<= value upper)))

;;; ---------------------------------------------------------------------------

(defun regions-overlap-p (reg1 reg2)
  (coordinates-in-bounds-p
    (region$x-min reg1)
    (region$y-min reg1)
    (- (region$x-min reg2) (- (region$x-max reg1) (region$x-min reg1)))
    (- (region$y-min reg2) (- (region$y-max reg1) (region$y-min reg1)))
    (region$x-max reg2)
    (region$y-max reg2)))

;;; ---------------------------------------------------------------------------

(defun find-movement-time (agent direction)

  "FIND-MOVEMENT-TIME agent direction
Return the amount of time necessary to move in the given direction."

  (let ((move-data (agent$move-data agent)))
    (case direction
      (:NORTH (eval (move-data$north move-data)))
      (:SOUTH (eval (move-data$south move-data)))
      (:EAST  (eval (move-data$east move-data)))
      (:WEST  (eval (move-data$west move-data)))
      (otherwise 1))))

;;; ---------------------------------------------------------------------------

(defun find-agent-location (agent &OPTIONAL (time *current-time*))

  "FIND-AGENT-LOCATION agent time
Returns location of agent at given time.
It is assumed that an agent is in its new location the time-step immediately following
the the time at which it notifies MICE of its intention to move.  The agent is then
not scheduled again for the number of time steps it takes the agent to move.  Thus, the
time cost of moving is paid in 'recovery' at the new location rather than in a delay
before the move is completed."

  (cond
    ;;; If time >= agent's current time, or
    ;;; if the agent does not have a state history, or
    ;;; if the agent is currently attempting to move but the move hasn't been resolved yet
    ;;; then its current location should be returned.
    ((or (>= time (agent$current-time agent))
         (not (agent$state-history agent))
         (and (first (first (agent-state$action-history (current-state agent))))
              (> time
                 (first (first (agent-state$action-history (current-state agent)))))))
     (agent$location agent))

    ;;; If there is some state history element before the time desired, return the location contained
    ;;; in the nearest state history element created before the time in question.
    ((assoc-if #'(lambda (past) (<= past time)) (agent$state-history agent))
     (do ((tt time (1- tt)))
         ((assoc tt (agent$state-history agent))
          (agent-state$location (rest (assoc tt (agent$state-history agent)))))))

    ;;; Otherwise, nil.
    (t nil)))

;;; ---------------------------------------------------------------------------

(defun find-agent-status (agent &OPTIONAL (time *current-time*))

  "FIND-AGENT-STATUS agent time
Returns status of agent at given time.
It is assumed that an agent is in its new status the time-step immediately following
the the time at which it notifies MICE of its intention to change.  The agent is then
not scheduled again for the number of time steps it takes the agent to change.  Thus, the
time cost of changing is paid in 'recovery' at the new location rather than in a delay
before the change is completed."

  (cond
    ;;; If time >= agent's current time, or
    ;;; if the agent does not have a state history, or
    ;;; if the agent is currently attempting to change but the change hasn't been resolved yet
    ;;; then its current status should be returned.
    ((or (>= time (agent$current-time agent))
         (not (agent$state-history agent))
         (and (first (first (agent-state$action-history (current-state agent))))
              (> time
                 (first (first (agent-state$action-history (current-state agent)))))))
     (agent$current-status agent))

    ;;; If there is some state history element before the time desired, return the status contained
    ;;; in the nearest state history element created before the time in question.
    ((assoc-if #'(lambda (past) (<= past time)) (agent$state-history agent))
     (do ((tt time (1- tt)))
         ((assoc tt (agent$state-history agent))
          (agent-state$status (rest (assoc tt (agent$state-history agent)))))))

    ;;; Otherwise, nil.
    (t nil)))

;;; ---------------------------------------------------------------------------

(defun find-agent-linkages (agent &OPTIONAL (time *current-time*))
  (cond
    ;;; If time >= agent's current time, or
    ;;; if the agent does not have a state history, or
    ;;; if the agent is currently attempting to change but the change hasn't been resolved yet
    ;;; then its current status should be returned.
    ((or (>= time (agent$current-time agent))
         (not (agent$state-history agent))
         (and (first (first (agent-state$action-history (current-state agent))))
              (> time
                 (first (first (agent-state$action-history (current-state agent)))))))
     (if (agent$sub-components agent) t nil))

    ;;; If there is some state history element before the time desired, return the status contained
    ;;; in the nearest state history element created before the time in question.
    ((assoc-if #'(lambda (past) (<= past time)) (agent$state-history agent))
     (do ((tt time (1- tt)))
         ((assoc tt (agent$state-history agent))
          (agent-state$linkages (rest (assoc tt (agent$state-history agent)))))))

    ;;; Otherwise, nil.
    (t nil)))

;;; ---------------------------------------------------------------------------

(defun find-agent-other (agent &OPTIONAL (time *current-time*))

  "FIND-AGENT-OTHER agent time
Returns contents of the OTHER field for an agent at given time.  The contents of
this field are user defined."

  (cond
    ;;; If time >= agent's current time, or
    ;;; if the agent does not have a state history, or
    ;;; if the agent is currently attempting to change but the change hasn't been resolved yet
    ;;; then its current status should be returned.
    ((or (>= time (agent$current-time agent))
         (not (agent$state-history agent))
         (and (first (first (agent-state$action-history (current-state agent))))
              (> time
                 (first (first (agent-state$action-history (current-state agent)))))))
     (find-agent-current-other-state-information agent))

    ;;; If there is some state history element before the time desired, return the status contained
    ;;; in the nearest state history element created before the time in question.
    ((assoc-if #'(lambda (past) (<= past time)) (agent$state-history agent))
     (do ((tt time (1- tt)))
         ((assoc tt (agent$state-history agent))
          (agent-state$other (rest (assoc tt (agent$state-history agent)))))))

    ;;; Otherwise, nil.
    (t nil)))

;;; ---------------------------------------------------------------------------

(defun find-agent-orientation (agent &OPTIONAL (time *current-time*))

  "FIND-AGENT-ORIENTATION agent time
Mirrors find-agent-location in returning orientation of agent at given time."

  (cond
    ;;; If time >= agent's current time, or
    ;;; if the agent does not have a state history, or
    ;;; if the agent is currently attempting to move but the move hasn't been resolved yet
    ;;; then its current orientation should be returned.
    ((or (>= time (agent$current-time agent))
         (not (agent$state-history agent))
         (and (first (first (agent-state$action-history (current-state agent))))
              (> time
                 (first (first (agent-state$action-history (current-state agent)))))))
     (agent$orientation agent))

    ;;; If there is some state history element before the time desired, return the orientation contained
    ;;; in the nearest state history element created before the time in question.
    ((assoc-if #'(lambda (past) (<= past time)) (agent$state-history agent))
     (do ((tt time (1- tt)))
         ((assoc tt (agent$state-history agent))
          (agent-state$orientation (rest (assoc tt (agent$state-history agent)))))))

    ;;; Otherwise, nil.
    (t nil)))

;;; ---------------------------------------------------------------------------

(defun agent-structure (agent-symbol)

  "AGENT-STRUCTURE agent-symbol
Returns the agent structure associated with agent-symbol."

  (if (and (symbolp agent-symbol) (assoc agent-symbol *all-agents*))
      (rest (assoc agent-symbol *all-agents*))
      agent-symbol))

;;; ---------------------------------------------------------------------------

(defun channel-structure (channel-name)
  "Returns the channel structure associated with channel-name."
  (cond ((channel-p channel-name) channel-name)
	(t (find (intern (string-upcase (string channel-name)))
		 *mice-channels* :test #'equal :key #'channel$name))))


;;; ---------------------------------------------------------------------------

(defun link (parent children link-type)

  "LINK parent children link-type
Create a link between a parent agent and a child (or list of children)."

  (if (listp children)
      (dolist (child children)
        (create-link parent child link-type))
      (create-link parent children link-type)))

;;; ---------------------------------------------------------------------------

(defun create-link (parent child link-type)

  "CREATE-LINK parent child link-type
Logs an intended link between the parent and child agents. The link is applied in
the function check-link only after all moves have been made and resolved."

;Get agent structures from their symbols.
  (when *link-verbose* (print "hello from create-link!"))
  (setf parent (agent-structure parent))
  (setf child (agent-structure child))

  (setf *attempted-links*
        (cons (make-link :COMPOSITE-AGENT parent
                         :SUB-COMPONENT child
                         :TYPE link-type)
              *attempted-links*)))

;;; --------------------------------------------------------------------------------  

(defun check-link (link)

  "CHECK-LINK parent child link-type
Checks a link between the parent and child agents."

  ;(let ((parent (agent-structure (link$composite-agent link)))
  ;      (child  (agent-structure (link$sub-component link)))
  ;      (link-type (agent-structure (link$type link))))
  (when *link-verbose* (print "check-link!"))

 (let ((parent (link$composite-agent link))
       (child  (link$sub-component   link))
       (link-type (link$type         link)))
  (if (and (agent-p parent) (agent-p child))    ;Verify agents are legal.
  
      ;Verify that constraints of desired link are satisfied.
      (if (cond ((member link-type '(:NORTH :SOUTH :EAST :WEST))
                 (equalp (agent$location child)
                         (compute-new-location (agent$location parent) link-type)))
                ((eq link-type :NEXT-TO)
                 (some #'(lambda (direction)
                           (equalp (agent$location child) (compute-new-location (agent$location parent) direction)))
                       '(:NORTH :SOUTH :EAST :WEST)))
                (t
                 (when *link-verbose* (format 't "parent loc ~a~%" (agent$location parent)))
                 (when *link-verbose* (format 't "child loc ~a~%" (agent$location child)))
                 (when *link-verbose* (format 't "parent orientation ~a~%" (agent$orientation parent)))
                 (when *link-verbose* (format 't "link-type ~a~%" link-type))
                 (equalp (agent$location child)
                         (compute-new-location
                           (agent$location parent)
                           (compute-link-direction (agent$orientation parent) link-type)))))
          (progn

            ;Check if child already has a parent agent.
            (when (agent$super-component child)
              (when (not *can-overwrite-links*)
                (cerror "Overwrite existing link with new link."
                        "Attempt to create parent-child link between ~a and ~a.  ~a already has parent ~a."
                        parent child child (agent$super-component child)))
              (remove-link (first (agent$super-component child)) child))

;            (when (eq link-type :NEXT-TO)
;              (setf link-type (find-direction (agent$location parent) (agent$location child) parent)))
;            (setf (agent$super-component child) (cons parent link-type))
;            (setf (agent$sub-components parent) (acons child link-type (agent$sub-components parent)))
;            (setf *links* (cons (make-link :COMPOSITE-AGENT parent :SUB-COMPONENT child :TYPE link-type) *links*))
;            (handle-event :LINK (list parent child))
          ;link checks out
          't)
          ;Constraints not satisfied.
          (progn
             (format 't "link between ~a and ~a failed because agent ~a in wrong place." parent child child)
             nil))
;          (cerror "Ignore create link request."
;                  "Constraints of desired link between ~a and ~a not satisfied."
;                 parent child))

      (cerror "Ignore create link request."
              "Agent ~a not found." (if (agent-p parent) child parent)))))

;;; ---------------------------------------------------------------------------



;;; ------------------------------------------------------------------------------------

(defun unlink (parent children)

  "UNLINK parent children
Remove the link between a parent agent and a child (or list of children)."

  (if (listp children)
      (dolist (child children)
        (remove-link parent child))
      (remove-link parent children)))

;;; ---------------------------------------------------------------------------

(defun remove-link (parent child)

  "REMOVE-LINK parent child
Removes the link between the parent and child agents."

  ;Get agent structures from their symbols.
  (setf parent (agent-structure parent))
  (setf child (agent-structure child))
  
  (if (eq parent (first (agent$super-component child)))    ;Verify that link exists.
      (progn
        (setf (agent$super-component child) nil)
        (setf (agent$sub-components parent) (remove (assoc child (agent$sub-components parent))
                                                    (agent$sub-components parent)))
        (do* ((link-list *links* (rest link-list))
              (a-link (first link-list) (first link-list)))
             ((or (null a-link)
                  (and (eq parent (link$composite-agent a-link))
                       (eq child (link$sub-component a-link))))
              (if (null a-link)
                  (error "Could not find link between ~a and ~a in *links*.~%" parent child) 
                  (setf *links* (remove a-link *links*)))))
        (handle-event :UNLINK (list parent child)))
      (cerror "Ignore unlink command."
              "Attempt to remove non-existent parent-child link between ~a and ~a."
              parent child)))

;;; ---------------------------------------------------------------------------

(defun rotate (agent direction number-of-quadrants)

  "ROTATE agent direction number-of-quadrants
Change the orientation of the agent by rotating it :RIGHT or :LEFT through the number of quadrants.
ROTATE returns the time cost of the rotation (the number of quadrants * agent's rotate cost/quadrant)."

  (cond ((member direction '(:NORTH :SOUTH :EAST :WEST))
         (multiple-value-bind (new-direction number-quadrants)
             (find-rotation-for-absolute-direction agent direction)
           (rotate agent new-direction number-quadrants)))
        ((null direction) (move-data$rotate-one-quadrant (agent$move-data agent)))
        (t (let ((rotation-steps (mod number-of-quadrants 4))
                 (old-orientation (find-agent-orientation agent))
                 (new-orientation (find-agent-orientation agent)))
             (setf (agent$orientation agent)
                   (dotimes (counter rotation-steps new-orientation)
                     (setf new-orientation
                           (if (eq direction :LEFT)
                               (case new-orientation
                                 (:NORTH :WEST)
                                 (:EAST :NORTH)
                                 (:SOUTH :EAST)
                                 (:WEST :SOUTH))
                               (case new-orientation
                                 (:NORTH :EAST)
                                 (:EAST :SOUTH)
                                 (:SOUTH :WEST)
                                 (:WEST :NORTH))))))
             
             ;;; Check that the change in orientation does not cause a conflict for any linked agents.
             (when (not (legal-location-p (agent$location agent) :AGENT agent))
               (handle-event :ROTATE-FAIL (list agent))
               (setf (agent$orientation agent) old-orientation))
             
             (* number-of-quadrants (eval (move-data$rotate-one-quadrant (agent$move-data agent))))))))

;;; ---------------------------------------------------------------------------

(defun find-rotation-for-absolute-direction (agent direction)
  (let ((orientation (find-agent-orientation agent)))
    (case direction
      (:NORTH (case orientation
                (:NORTH    (values nil    nil))
                (:SOUTH    (values :LEFT  2))
                (:EAST     (values :LEFT  1))
                (:WEST     (values :RIGHT 1))
                (otherwise (values nil    nil))))
      (:SOUTH (case orientation
                (:NORTH    (values :LEFT  2))
                (:SOUTH    (values nil    nil))
                (:EAST     (values :RIGHT 1))
                (:WEST     (values :LEFT  1))
                (otherwise (values nil    nil))))
      (:EAST  (case orientation
                (:NORTH    (values :RIGHT 1))
                (:SOUTH    (values :LEFT  1))
                (:EAST     (values nil    nil))
                (:WEST     (values :LEFT  2))
                (otherwise (values nil    nil))))
      (:WEST  (case orientation
                (:NORTH    (values :LEFT  1))
                (:SOUTH    (values :RIGHT 1))
                (:EAST     (values :LEFT  2))
                (:WEST     (values nil    nil))
                (otherwise (values nil    nil)))))))

;;; ***************************************************************************
;;; EOF


