;; domain.lisp

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.

;;;            This work has been sponsored in part by:
;;;               the NSF (IRI-9010645, IRI-9015423)
;;;         the University of Michigan Rackham Graduate School
;;;

;(in-package 'MICE)

;(export '(compute-agent-authority))

;(use-package '(umass-extended-lisp))

(proclaim '(optimize (speed 3) (safety 1)))

;;; ***************************************************************************

(defun create-agent (&REST args)
  (let* ((new-agent
           (apply #'make-agent
                  (append (when (not (member :INVOCATION-FUNCTION args))
                            '(:INVOCATION-FUNCTION simple-agent))
                          args)))
         (new-agent-name
           (intern
             (if (agent$name new-agent)
                 (agent$name new-agent)
                 (setf (agent$name new-agent) (make-agent-name))))))
    (setf (agent$name new-agent) new-agent-name)
    (setf *agent-schedule-queue* (cons new-agent *agent-schedule-queue*))
    (setf *all-agents* (acons new-agent-name new-agent *all-agents*))
    (when *verbose?* (format t "Created agent ~a.~%" new-agent-name))
    ;(when (not *mice-graphics-window*) (format t "Created agent ~a.~%" new-agent-name))
    new-agent))

;;; ---------------------------------------------------------------------------

#+TI
(defun create-red-agent (&REST args)
  (agent$name (apply #'create-agent (append '(:TYPE :RED
                                              :CREATION-TIME 0
                                              :CAPTURED-BY-TYPES (:BLUE)
                                              :CAPTURE-TYPES ()
                                              :DRAW-FUNCTION :FILLED-RECTANGLE)
                                            args))))
#-TI
(defun create-red-agent (&REST args)
  (agent$name (apply #'create-agent (append args
                                            '(:TYPE :RED
                                              :CREATION-TIME 0
                                              :CAPTURED-BY-TYPES (:BLUE)
                                              :CAPTURE-TYPES ()
                                              :DRAW-FUNCTION :FILLED-RECTANGLE)))))

;;; ---------------------------------------------------------------------------

#+TI
(defun create-blue-agent (&REST args)
  (agent$name (apply #'create-agent (append '(:TYPE :BLUE
                                              :CREATION-TIME 0
                                              :CAPTURED-BY-TYPES ()
                                              :CAPTURE-TYPES (:RED)
                                              :DRAW-FUNCTION :FILLED-CIRCLE)
                                            args))))
#-TI
(defun create-blue-agent (&REST args)
  (agent$name (apply #'create-agent (append args
                                            '(:TYPE :BLUE
                                              :CREATION-TIME 0
                                              :CAPTURED-BY-TYPES ()
                                              :CAPTURE-TYPES (:RED)
                                              :DRAW-FUNCTION :FILLED-CIRCLE)))))

;;; ---------------------------------------------------------------------------

(defun create-channel (&REST args)
  (let* ((new-channel (apply #'make-channel args))
         (new-channel-name 
	  (intern (if (channel$name new-channel)
		      (string-upcase (string (channel$name new-channel)))
		    (setf (channel$name new-channel) (make-channel-name))))))
    (setf (channel$name new-channel) new-channel-name)
    (setf *mice-channels* (cons new-channel *mice-channels*))
    (when *verbose?* (format t "Created channel ~a.~%" new-channel-name))
    new-channel))

  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;                     TERMINATION PREDICATES
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun mice-continue-p ()
  (if (not (or ;(no-red-agents-p *agent-schedule-queue*)
             (no-agent-has-moved-p *agent-schedule-queue*)
             (>= (simulation-data$overall-time *simulation-data*) *time-limit*)))
      t
      (progn
        (when *move-count*
          (format t "~%Successful agent moves:       ~a" *move-count*))
        (when *move-back-count*
          (format t "~%Moves undone due to conflict: ~a" *move-back-count*))
        (when *attempted-move-count*
          (format t "~%Total moves attempted:        ~a" *attempted-move-count*))
        (when *remote-hosts*
          (if (>= (simulation-data$overall-time *simulation-data*) *time-limit*)
              (progn
                (format t "~%**** RUN TERMINATING -- TIME LIMIT REACHED ****~%")
                (format t "If you can't learn to cooperate, you won't be allowed to play again!"))
              (format t "***** CONGRATULATIONS!!! You are part of a winning team! ****")))
        (dolist (host *remote-hosts*)
          (if (>= (simulation-data$overall-time *simulation-data*) *time-limit*)
              (request-slave-char host "You ran out of time.  Think you can cooperate better next time (Y or N)?")
              (request-slave-char host "***** Congratulations, you won.  Are you proud of yourself (Y or N)?"))
          (terminate-slave-mice host))
        nil)))

;;; ---------------------------------------------------------------------------

;;; These functions loop through the agent list and count all of the extra actions
;;; attempted by agents.  This is done by looking at the action-history of each
;;; state in each agent's state-history and counting all but one action from the 
;;; action history for that state.

(defun count-extra-actions ()
  (apply #'+ (mapcar #'(lambda (agent) (times-undone agent)) *agent-schedule-queue*)))

(defun times-undone (agent)
  (apply #'+ (mapcar #'(lambda (astate) (extra-actions astate)) (agent$state-history agent))))

(defun extra-actions (astate)
  (if (agent-state$action-history (rest astate))
      (1- (length (agent-state$action-history (rest astate))))
      0))

;;; ---------------------------------------------------------------------------

(defun no-red-agents-p (agents)
  (no-agents-of-types-p agents '(:RED)))

(defun no-cup-agents-p (agents)
  (no-agents-of-types-p agents '(:CUP)))

(defun no-agents-of-types-p (agent-list-to-check agent-type-list)
  (notany #'(lambda (agent) (and (member (agent$type agent) agent-type-list)
                                 (not (eq (agent-state$status (rest (first (agent$state-history agent))))
                                          :INACTIVATED))))
          agent-list-to-check))

;;; ---------------------------------------------------------------------------

(defun no-agent-has-moved-p (agent-list &KEY (how-long (if *add-randomness*
                                                           (* 5 *longest-move-time*)
                                                           *longest-move-time*)))
  (cond ((null agent-list) t)
	((or (not (agent$location (first agent-list))) (not (agent$state-history (first agent-list))))
	 (error "Agent ~a is missing a location or state-history." (first agent-list)))
        ((and (equalp (agent$location (first agent-list))
                      (find-agent-location (first agent-list) (- (agent$current-time (first agent-list)) how-long)))
              (or (not *add-randomness*)
                  (do* ((agent (first agent-list))
                        (time (agent$current-time (first agent-list)) (1- time))
                        (ok (equalp (agent$location agent) (find-agent-location agent time))
                            (when ok (equalp (agent$location agent) (find-agent-location agent time)))))
                       ((= time (- (agent$current-time agent) how-long)) ok))))
         (no-agent-has-moved-p (rest agent-list) :HOW-LONG how-long))
        (t nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;                    AGENT OVERLAP PREDICATES
;;
;; The following functions are generic agent interaction functions.
;; They are used to apply domain specific agent overlap effects.  
;; When two agents swap locations or occupy the same spot, 
;; ds-overlap-effect will be called.  If the agents have been given one
;; of these functions to explain their interaction in the environment
;; file, then it will be applied to both agents at that time.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun decr-strgth-1st-agent (agent1 agent2)
  "Decrements the strength of agent1"
  (declare (ignore agent2))
  (setf (rest (assoc :STRENGTH (agent$domain-variables agent1))) 
        (1- (rest (assoc :STRENGTH (agent$domain-variables agent1))))))

;;; ---------------------------------------------------------------------------

(defun decr-strgth-2nd-agent (agent1 agent2)
  "Decrements the strength of agent2"
  (declare (ignore agent1))
  (setf (rest (assoc :STRENGTH (agent$domain-variables agent2))) 
        (1- (rest (assoc :STRENGTH (agent$domain-variables agent2))))))

;;; ---------------------------------------------------------------------------

(defun decr-strgth-both (agent1 agent2)
  "Decrements the strength of both agents"
  (setf (rest (assoc :STRENGTH (agent$domain-variables agent1))) 
        (1- (rest (assoc :STRENGTH (agent$domain-variables agent1)))))
  (setf (rest (assoc :STRENGTH (agent$domain-variables agent2))) 
        (1- (rest (assoc :STRENGTH (agent$domain-variables agent2))))))

;;; ---------------------------------------------------------------------------

(defun death-1st-agent (agent1 agent2)
  "Kills agent1 by dropping strength to 0"
  (declare (ignore agent2))
  (setf (rest (assoc :STRENGTH (agent$domain-variables agent1))) 0))

;;; ---------------------------------------------------------------------------

(defun death-2nd-agent (agent1 agent2)
  "Kills agent2 by dropping strength to 0"
  (declare (ignore agent1))
  (setf (rest (assoc :STRENGTH (agent$domain-variables agent2))) 0))

;;; ---------------------------------------------------------------------------

(defun death-both (agent1 agent2)
  "Kills both agents by dropping strength to 0"
  (setf (rest (assoc :STRENGTH (agent$domain-variables agent1))) 0)
  (setf (rest (assoc :STRENGTH (agent$domain-variables agent2))) 0))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;;                     DEATH PREDICATES
;;
;; These predicates are used to determine when an agent is dead.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;(defun hole-remove-p (agent time)
;  (declare (ignore time))
;  (let ((loc (agent$location agent))
;        (hole nil)
;        (result 't)
;        (agents nil))
;    (dolist (possible-hole holes nil)
;       (when (member (list (location$x loc) (location$y loc)) possible-hole :test #'equal)
;         (setf hole (member (list (location$x loc) (location$y loc)) possible-hole :test #'equal))))
    ;(format t "hole-remove-p: agent= ~a hole= ~a~%" agent hole)
    ;(format t "hole-remove-p: agents = ~a~%" (agents-at-location (get-grid-element (agent$location agent))))
;
;    (dolist (location hole result)
;       ;(format t "location: ~a :")
;       (when (not (some #'(lambda (agent) (eql (agent$type agent) :TILE))
;                     (agents-at-location (get-grid-element (make-location :X (first location) :Y (second location))))))
;           (setf result nil)))))

    ;(when (every #'(lambda (location) (some #'(lambda (agent) (eql (agent$type agent) :TILE))
    ;                 (agents-at-location (get-grid-element (make-location :X (first location) :Y (second location))))))
    ;                  hole))))

;--------------
;         (agents (agents-at-location (get-grid-element (agent$location agent)))))
;    (print (list 'hole-remove-p--  agents))
;    (when (some #'(lambda (agent) (eql (agent$type agent) :TILE)) agents)
;      (format t "Removing hole at ~a~%" (agent$location agent))
;      (setf number_of_holes (- number_of_holes 1))
;      't)))

;(defun hole-remove-function (agent time)
;  (declare (ignore time))
;  (let ((hole nil)
;        (agents nil)
;        (loc (agent$location agent)))
;    (dolist (possible-hole holes nil)
;       (when (member (list (location$x loc) (location$y loc)) possible-hole :test #'equal)          
;         (setf hole (member (list (location$x loc) (location$y loc)) possible-hole :test #'equal))))
;    (dolist (location hole nil) 
;             (setf agents (append agents (agents-at-location 
;                      (get-grid-element (make-location :X (first location) :Y (second location)))))))
;    (setf number_of_holes (- number_of_holes 1))
;    (setf number_of_tiles (- number_of_tiles 1))
;    (setf score (+ score (agent$authority agent)))
;    (format t "Score: ~a~%" score)
;    agents))

;(defun tileworld nil
;  (print 'hello)
;  (format t "~%number of holes?")
;  (setf initial_number_of_holes (string-to-int (read-line)))
;  (format t "~%max hole size?")
;  (setf max_hole_size (string-to-int (read-line)))
;  (format t "~%max number of holes?")
;  (setf max_holes (string-to-int (read-line)))
;  (format t "~%number of tiles?")
;  (setf number_of_tiles (string-to-int (read-line)))
;  (setf max_tiles number_of_tiles)
;  (format t "~%number of obstacles?")
;  (setf number_of_obstacles (string-to-int (read-line)))
;  (mice "lewis:damouth;tileworld.env")
;)

(defun stagnant-or-strgth-0-p (agent time)
  (or (agent-strgth-0-p agent time)
      (stagnant-agent-p agent time)))

;;; ---------------------------------------------------------------------------

(defun agent-strgth-0-p (agent time)
  (declare (ignore time))
  (<= (rest (assoc :STRENGTH (agent$domain-variables agent))) 0))

;;; ---------------------------------------------------------------------------

(defun stagnant-agent-p (agent time)
  "True if agent has been in the same location for max-stagnant-time"
  (do* ((how-long (rest (assoc :MAX-STAGNANT-TIME (agent$domain-variables agent))))
        (location (find-agent-location agent time))
        (time-to-check time (1- time-to-check))
        (not-moved t (when not-moved (equalp location (find-agent-location agent time-to-check)))))
       ((= time-to-check (- time how-long))
        (when (>= (- time (agent$creation-time agent)) how-long) not-moved))))

;;; ---------------------------------------------------------------------------

(defun captured-agent-p (agent time)
  (let ((surrounding-agents (find-neighboring-agents agent *agent-schedule-queue* time)))
    (and (>= (length surrounding-agents) 4)
         (every #'(lambda (direction)
                    (let ((new-location (compute-new-location (agent$location agent) direction)))
                      (some #'(lambda (a)
                                (and (equalp (agent$location a) new-location)
                                     (member (agent$type a) (agent$captured-by-types agent))))
                            surrounding-agents)))
                '(:NORTH :SOUTH :EAST :WEST)))))

;;; ---------------------------------------------------------------------------

(defun cup-in-basket-p (agent time)
  (and (eq (agent$type agent) :CUP)
       (let ((sharing-agents (find-agents-in-location (agent$location agent) *agent-schedule-queue* time)))
         (some #'(lambda (agent) (eq (agent$type agent) :BASKET)) sharing-agents))))

;;; ---------------------------------------------------------------------------

(defun find-neighboring-agents (agent others time)
  (cond ((null others) nil)
        ((= (+ (abs (- (location$x (find-agent-location agent time))
                       (location$x (find-agent-location (first others) time))))
               (abs (- (location$y (find-agent-location agent time))
                       (location$y (find-agent-location (first others) time)))))
            1)
         (cons (first others) (find-neighboring-agents agent (rest others) time)))
        (t (find-neighboring-agents agent (rest others) time))))

;;; ---------------------------------------------------------------------------

(defun ds-position-effect (agent1 agent2 time)
  "Domain specific agent overlap effect"
  ;Applies effect of two agents occupying the same location for the 
  ;elapsed-time.  In the case of a switch, elapsed time is assumed to be 1.
  ; NOTE: any user-supplied overlap function should return t if it moves any agent!
  (declare (ignore time))
  (let ((type-fcn (or (assoc (agent$type agent2) (agent$overlap-predicates agent1))
                      (assoc :ALL (agent$overlap-predicates agent1)))))
    (when type-fcn (funcall (rest type-fcn) agent1 agent2))))

;;; ---------------------------------------------------------------------------

(defun ds-overlap-effect (agent1 agent2 time)
  "Domain specific agent overlap effect"
  ;Applies effect of two agents occupying the same location for the 
  ;elapsed-time.  In the case of a switch, elapsed time is assumed to be 1.
  ; NOTE: any user-supplied overlap function should return t if it moves any agent!
  (let ((type-fcn (or (when (and (agent$collision-function agent1) (not (consp (agent$collision-function agent1))))
                        (setf (agent$collision-function agent1)
                              (acons :ALL (agent$collision-function agent1) nil))
                        (assoc :ALL (agent$collision-function agent1)))
                      ; the above case is for compatibility and will eventually be outmoded
                      ; it allows a single function to be given, assumed for :ALL
                      (assoc (agent$type agent2) (agent$collision-function agent1))
                      (assoc :ALL (agent$collision-function agent1)))))
    (if type-fcn
        (funcall (eval (rest type-fcn)) agent1 agent2 time)
        (default-collision-function agent1 agent2 time))))

;(defun ds-link-effect (agent1 agent2 time)
;  "Domain specific agent link effect"
  ; returns success of agent1 trying to link to agent2
;  (let ((type-fcn (or (when (and (agent$link-function agent1) (not (consp (agent$link-function agent1))))
;                        (setf (agent$link-function agent1)
;                              (acons :ALL (agent$link-function agent1) nil))
;                        (assoc :ALL (agent$link-function agent1)))
;                      ; the above case is for compatibility and will eventually be outmoded
;                      ; it allows a single function to be given, assumed for :ALL
;                      (assoc (agent$type agent2) (agent$link-function agent1))
;                      (assoc :ALL (agent$link-function agent1)))))
;    (if type-fcn
;        (funcall (eval (rest type-fcn)) agent1 agent2 time)
;        (default-link-function agent1 agent2 time))))
;
;(defun default-link-function (agent1 agent2 time)
;  (declare (ignore agent1 agent2 time))
;
;  "DEFAULT-LINK-FUNCTION agent1 agent2"
;  (print "hello from default-link-function!")
;  
;  't)

(defun default-collision-function (agent1 agent2 time)
                                                
  "DEFAULT-COLLISION-FUNCTION agent1 agent2 time

If the agents block each other, moves them back to where they were at previous
time, otherwise leaves them alone."

  (when (or (member (agent$type agent1) (agent$blocked-by-types agent2))
            (member (agent$type agent2) (agent$blocked-by-types agent1)))
    (let ((conflict-agents
            (moved-agents (union (list agent1 agent2) (union (superior-agents agent1) (superior-agents agent2)))
                          time)))
      (handle-event :MOVE-BACK conflict-agents)
      (move-agents-to-location-at-given-time conflict-agents (1- time))
      t)))

;;; ---------------------------------------------------------------------------

(defvar *already-preprocessed* -1)

(defun authority-collision-function (agent1 agent2 time 
                                     &optional (authority nil) (currently-moved-agents nil) (strong-direction nil))

  "AUTHORITY-COLLISION-FUNCTION agent1 agent2 time

(By Dan) If the agents block each other, then it moves the lower authority agent
back to where it was at the previous time.  If it still overlaps, then
it moves that agent in the direction away from the direction the higher
authority agent moved from if that direction is not blocked.  If that
direction is blocked by an even  higher authority agent, then it returns the original
agent back to where it was at the previous time.  Otherwise, if that direction
is blocked by an agent with lower authority, the first lower authority agent
is pushed onto the (blocking) lower authority agent, and the function recurses
to resolve that overlap.  In this way, a stack of agents, all with lower authority
that the original pushing agent, may be pushed along.  The original authority value
is propagated in the argument 'authority'.  If this process pushes an agent into
an agent with an authority higher than 'authority', the entire push fails and
all agents moved are moved back. This means that if any agent in a stack has 
higher authority than the original pushing agent, the stack may not be pushed 
by that agent."
  ;(when (> *already-preprocessed* time) (setf *already-preprocessed* -1))
  ;(format t "~%~a ~a" *already-preprocessed* time)
  ;(unless (= *already-preprocessed* time)
  ;  (preprocess *agent-schedule-queue*)
  ;  (setf *already-preprocessed* time))
  (when *collision-verbose* (format t "~%invoking authority-collision-function -- ~a ~a ~a ~a" agent1 agent2 time authority))
  (when (or (member (agent$type agent1) (agent$blocked-by-types agent2))
            (member (agent$type agent2) (agent$blocked-by-types agent1)))
    (let* ((superiors-agent1 (superior-agents agent1))
           (superiors-agent2 (superior-agents agent2))
           (agent1-authority (if authority authority (agent-list-authority superiors-agent1))) 
           (agent2-authority (agent-list-authority superiors-agent2))
           (strong-agent (cond ((> agent1-authority agent2-authority) agent1)
                                ((> agent2-authority agent1-authority) agent2)
                                (t nil)))
           (strong-agent-authority (if authority authority 
                                       (when strong-agent (agent-list-authority (superior-agents strong-agent)))))
           (strong-superiors (when strong-agent (if (eq strong-agent agent1) superiors-agent1 superiors-agent2)))
           (weak-agent (when strong-agent (if (eq strong-agent agent1) agent2 agent1)))
           (weak-superiors (when strong-agent (if (eq strong-agent agent1) superiors-agent2 superiors-agent1)))
           (conflict-agents (when strong-agent (moved-agents weak-superiors time)))
           ;(strong-current-location (when strong-agent (agent$location strong-agent)))
           ;(strong-previous-location (when strong-agent (find-agent-location strong-agent (1- time))))
           (strongs-direction (if strong-direction strong-direction
                                  (when strong-agent (find-agent-direction strong-agent time))))
           (push-location (when strongs-direction 
                           (compute-new-location (agent$location (first weak-superiors)) strongs-direction)))
	   ;(weak-agent-auth-info (when strong-agent (rest (assoc :AUTH-INFO (agent$domain-variables weak-agent)))))
	   ;(strong-agent-auth-info (when strong-agent (rest (assoc :AUTH-INFO (agent$domain-variables strong-agent)))))
	   )
      (cond 
        ((not strong-agent)
         ;; agents are of equal authority
                 (default-collision-function agent1 agent2 time))
        
        ((member agent2 (find-switched-agents agent1 *agent-schedule-queue* time))
         ;;  agent1 and agent2 tried to switch positions (and they block each other)
         (when *collision-verbose* (format t "~%agents ~a and ~a tried to switch location and bounced off" agent1 agent2))
         (move-agents-to-location-at-given-time weak-superiors (1- time))
	 t)
         ;(authority-collision-function strong-agent weak-agent time))

        ((not (equalp (agent$location agent1) (agent$location agent2))) 
         ;;  an agent has already been moved back.    
         (when *collision-verbose* (print "function returns nil"))
         nil)

        ((and conflict-agents
              (not (pushing-agent-in-location-p (find-agent-location weak-agent (1- time)) weak-agent nil time))
              (not (equalp (agent$location weak-agent) (find-agent-location weak-agent (1- time)))))
        ;; if weaker moved, move it back to its previous location
         (when *collision-verbose* (format t "~%moving weaker one -~a- back~%" conflict-agents))
         (handle-event :MOVE-BACK conflict-agents)
         (move-agents-to-location-at-given-time conflict-agents (1- time))
	 t)

        ((and conflict-agents
              (pushing-agent-in-location-p (find-agent-location weak-agent (1- time)) weak-agent nil time)
              (not (equalp (agent$location weak-agent) (find-agent-location weak-agent (1- time))))
              (not (find-agents-in-location (compute-new-location (agent$location weak-agent) strongs-direction) 
                                            *agent-schedule-queue* time)))
         ;; weaker agent was pushed into location and can be pushed by strong agent without pushing another block
         (when *collision-verbose* (format t "~%weaker one -~a- being pushed ~a" conflict-agents strongs-direction))
         (move-agent-to-location (first conflict-agents) 
                                 (compute-new-location (agent$location weak-agent) strongs-direction)))

	((and conflict-agents		
	      (pushing-agent-in-location-p (find-agent-location weak-agent (1- time)) weak-agent nil time)
	      (not (equalp (agent$location weak-agent) (find-agent-location weak-agent (1- time))))
	      (or (equalp (find-agent-location weak-agent (1- time)) (compute-new-location (agent$location weak-agent)
											    strongs-direction))
		  (not strongs-direction)))
	 ;; weak agent was pushed into location from opposite direction; need to push stack back
	 (when *collision-verbose* (format t "~%*** starting to push stack back ***"))
	 (let* ((loc (find-agent-location weak-agent (1- time)))
		(next-agents (find-agents-in-location loc *agent-schedule-queue* time))
		(direction (reverse-direction (find-agent-direction weak-agent time)))
		(in-loc-agents nil)
                (in-line-agents nil)
                (next-push-location (compute-new-location loc direction)))
           (unless authority (setf authority strong-agent-authority))
           (move-agent-to-location (first weak-superiors) loc)
           (apply-link-constraints)
           (unless currently-moved-agents 
             (setf currently-moved-agents (moved-agents strong-superiors time)))
           (setf currently-moved-agents 
                 (append currently-moved-agents (moved-agents weak-superiors time)))	
           (when *collision-verbose* (format t "~%currently-moved2 ~a ~%" currently-moved-agents))
           (dolist (next-agent next-agents)
             (cond ((equalp (find-agent-location next-agent (1- time)) loc)
                    ;; agent being pushed was previously in same location
                    (push next-agent in-loc-agents))
                   ((equalp (find-agent-location next-agent (1- time)) 
                            (compute-new-location push-location strongs-direction))
                    ;; agent being-pushed moved into location from next location in line with stack
                    (push next-agent in-line-agents))))
           (dolist (next-agent in-line-agents)
             (when *collision-verbose* (format t "~%*** agent ~a pushed." (first (superior-agents next-agent))))
	     ;; collision function not recursively called for in-line agents because they came from a square away.
	     ;; first they are moved back to where they were; they are actually 'pushed' in the next cycle
             (move-agent-to-location (first (superior-agents next-agent)) next-push-location))
           (dolist (next-agent in-loc-agents)
             (when *collision-verbose* (format t "~%*** recursing..."))
             (authority-collision-function (first weak-superiors) next-agent 
                                           time authority currently-moved-agents direction)))
	 t)
	   

        ((weaker-agents-in-location-p push-location strong-agent authority time)
         ;; weaker agents are in a position to be pushed; add temporary authority and call recursively 
         (let* ((next-agents (find-agents-in-location push-location *agent-schedule-queue* time))
                (in-loc-agents nil)
                (in-line-agents nil)
		(result nil)
                (next-push-location (compute-new-location push-location strongs-direction)))
           (unless authority (setf authority strong-agent-authority))
           (move-agent-to-location (first weak-superiors) push-location)
           (apply-link-constraints)
           (unless currently-moved-agents 
             (setf currently-moved-agents (moved-agents strong-superiors time)))
           (setf currently-moved-agents 
                 (append currently-moved-agents (moved-agents weak-superiors time)))	
           (when *collision-verbose* (format t "~%currently-moved ~a ~%" currently-moved-agents))
           (dolist (next-agent next-agents)
             (cond ((equalp (find-agent-location next-agent (1- time)) push-location)
                    ;; agent being pushed was previously in same location
                    (push next-agent in-loc-agents))
                   ((equalp (find-agent-location next-agent (1- time)) 
                            (compute-new-location push-location strongs-direction))
                    ;; agent being-pushed moved into location from next location in line with stack
                    (push next-agent in-line-agents))))
           (dolist (next-agent in-line-agents)
             (when *collision-verbose* (format t "~%*** agent ~a pushed." (first (superior-agents next-agent))))
	     ;; collision function not recursively called for in-line agents because they came from a square away.
	     ;; first they are moved back to where they were; they are actually 'pushed' in the next cycle
             (move-agent-to-location (first (superior-agents next-agent)) next-push-location))
           (dolist (next-agent in-loc-agents result)
             (when *collision-verbose* (format t "~%*** recursing..."))
             (setf result (authority-collision-function (first weak-superiors) next-agent 
                                           time authority currently-moved-agents strongs-direction))))) 
						
        ((or (not strongs-direction)
             (not (legal-location-p push-location :AGENT (first weak-superiors)))
             (blocking-agent-in-location-p push-location (first weak-superiors)))
         ;; push fails; have to move agent back along with currently-moved-agents
         (let ((moved-strong (moved-agents (union (list strong-agent) strong-superiors) time))) 
           (when *collision-verbose* (format t "moving ~a back ~%" moved-strong))
           (handle-event :MOVE-BACK moved-strong)
           (move-agents-to-location-at-given-time moved-strong (1- time))       
           (when currently-moved-agents
             (handle-event :MOVE-BACK currently-moved-agents)
             (when *collision-verbose* (format 't "now moving back ~a~%" currently-moved-agents))
             (move-agents-to-location-at-given-time currently-moved-agents (1- time))))
	 t) 

        (t ;; the weaker agent is pushed 
         (move-agent-to-location (first weak-superiors) push-location)
         (when *collision-verbose* (format t "pushing weaker one, ~a ~%" (first weak-superiors)))
         (apply-link-constraints))))))

(defun stronger-agent-in-location-p (loc strong-agent authority time)
  ; returns t iff any agent in loc is stronger than (max authority (agent$authority strong-agent)) ; 12/91 DED
  (let ((agents-in-loc (find-agents-in-location loc *agent-schedule-queue* time))
        (maximum-authority -100)
        (new-authority (if authority authority (agent-list-authority (superior-agents strong-agent))))) 
    (when agents-in-loc
      (dolist (agent agents-in-loc nil)
        (let ((agent-authority (agent-list-authority (superior-agents agent))))
          (when (> agent-authority maximum-authority)
            (setf maximum-authority agent-authority))))
      (if (> maximum-authority new-authority)
          maximum-authority
          nil))))

(defun pushing-agent-in-location-p (loc strong-agent authority time)
  ; returns t iff any agent in loc is stronger than (max authority (agent$authority strong-agent)) ; 12/91 DED
  ; and is moving towards strong-agent
  (let ((agents-in-loc (find-agents-in-location loc *agent-schedule-queue* time))
        (maximum-authority -100)        
        (new-authority (if authority authority (agent-list-authority (superior-agents strong-agent)))))
    (when agents-in-loc
      (dolist (agent agents-in-loc nil)
        (let ((agent-authority (agent-list-authority (superior-agents agent))))
          (when (and (> agent-authority maximum-authority)	;
                     (eq (find-agent-direction agent time) (find-agent-direction strong-agent time)))
            (setf maximum-authority agent-authority))))
      (if (> maximum-authority new-authority)
          maximum-authority
          nil))))

(defun weaker-agents-in-location-p (loc strong-agent authority time)
  ; returns t iff all agents in loc are weaker than either authority or (agent$authority strong-agent)
  (let ((agents-in-loc (find-agents-in-location loc *agent-schedule-queue* time))
        (maximum-authority -100)
        (new-authority (if authority authority (agent-list-authority (superior-agents strong-agent))))) ; added 12/91 DED
    (when agents-in-loc
      (dolist (agent agents-in-loc nil)
        (let ((agent-authority (agent-list-authority (superior-agents agent))))
          (when (> agent-authority maximum-authority)
            (setf maximum-authority agent-authority))))
      (if (< maximum-authority new-authority)
          maximum-authority
          nil))))

(defun find-agent-direction (strong-agent time) ; returns direction from last location to current location
  (let ((strong-previous-location (find-agent-location strong-agent (1- time)))
        (strong-current-location (agent$location strong-agent)))
    (cond ;; in these first four, strong-agent is being "swung" into the weaker agent by its super-component
      ((and (< (location$x strong-previous-location) (location$x strong-current-location))
            (< (location$y strong-previous-location) (location$y strong-current-location))
            (agent$super-component strong-agent))
       (if (= (location$x (agent$location (first (agent$super-component strong-agent))))
              (location$x strong-current-location))
           :EAST   
           :SOUTH))
      ((and (< (location$x strong-previous-location) (location$x strong-current-location))                
            (> (location$y strong-previous-location) (location$y strong-current-location))
            (agent$super-component strong-agent))
       (if (= (location$x (agent$location (first (agent$super-component strong-agent))))
              (location$x strong-current-location))
           :EAST   
           :NORTH))
      ((and (> (location$x strong-previous-location) (location$x strong-current-location))
            (< (location$y strong-previous-location) (location$y strong-current-location))
            (agent$super-component strong-agent))
       (if (= (location$x (agent$location (first (agent$super-component strong-agent))))
              (location$x strong-current-location))
           :WEST   
           :SOUTH))
      ((and (> (location$x strong-previous-location) (location$x strong-current-location))
            (> (location$y strong-previous-location) (location$y strong-current-location))
            (agent$super-component strong-agent))
       (if (= (location$x (agent$location (first (agent$super-component strong-agent))))
              (location$x strong-current-location))
           :West   
           :NORTH))
      ((< (location$x strong-previous-location) (location$x strong-current-location))
       :EAST)     
      ((< (location$y strong-previous-location) (location$y strong-current-location))
       :SOUTH)
      ((> (location$x strong-previous-location) (location$x strong-current-location))
       :WEST)
      ((> (location$y strong-previous-location) (location$y strong-current-location))
       :NORTH))))

(defun preprocess (agents)
  (dolist (agent agents)
    (setf (rest (assoc :AUTH-INFO (agent$domain-variables agent))) nil)))

(defun preprocess1 (agents time)
  (let ((new-agents (sort (copy-tree agents) #'(lambda (a1 a2) (< (compute-agent-authority a2)
								  (compute-agent-authority a1))))))
    (when *collision-verbose* (format t "~%preprocessing at time ~a" time))
    (dolist (agent new-agents)
      (when (and (not (equal (find-agent-location agent (1- time)) (agent$location agent)))
		 (find-agents-in-location (agent$location agent) new-agents (1- time)))
	;; if agent moved and there was an agent in the location 
	(let ((next-agents (find-agents-in-location (agent$location agent) new-agents (1- time)))
	      (direction (find-agent-direction agent time)))
	  (unless (stronger-agent-in-location-p (agent$location agent) agent nil time)
	    (dolist (next-agent next-agents)
	      (when (eq (find-agent-direction next-agent time) direction)
		(when *collision-verbose* (format t "~%moving agent ~a back to be pushed." next-agent))
		(move-agents-to-location-at-given-time (superior-agents next-agent) (1- time))))))))))

(defun reverse-direction (dir)
  (case dir 
    (:NORTH :SOUTH)
    (:SOUTH :NORTH)
    (:EAST :WEST)
    (:WEST :EAST)))


(defun blocking-agent-in-location-p (loc agent)
  (let ((blocked-by-types (agent$blocked-by-types agent))
        (agent-list (ordered-set-difference *agent-schedule-queue* (inferiors agent))))
    (some #'(lambda (other-agent)
              (and (equalp (agent$location other-agent) loc)
                   (member (agent$type other-agent) blocked-by-types)))
          agent-list)))

(defun inferiors (agent)
  (cond ((null agent) nil)
        ((agent$sub-components agent)
         (cons agent (inferiors-list (agent$sub-components agent))))
        (t (list agent))))

(defun inferiors-list (agt-sub-comp-list)
  (cond ((null agt-sub-comp-list) nil)
        (t (append (inferiors (first (first agt-sub-comp-list)))
                   (inferiors-list (rest agt-sub-comp-list))))))

;;; ---------------------------------------------------------------------------

(defun compute-agent-authority (agent)

  "COMPUTE-AGENT-AUTHORITY agent
Returns the value of the agent's authority slot.  If the slot is empty, it first
computes the authority by extracting the agent's number from its name.  If there is
no number in the name, then it returns the default authority of 1."
  (if (agent-p agent)
      ; if it is an agent structure, do this
      (if (agent$authority agent)
          (agent$authority agent)
          (let ((number (parse-integer (string-trim "ABCDEFGHIJKLMNOPQRSTUVWXYZ!$%^&*_-+=<>.?"
                                                    (agent$name agent)) :JUNK-ALLOWED T)))
            (setf (agent$authority agent) (if number number 1))))
      ; if it is just the agent's name, do this
      (parse-integer (string-trim "ABCDEFGHIJKLMNOPQRSTUVWXYZ!$%^&*_-+=<>.?"
                                  agent) :JUNK-ALLOWED T)))

;;; ---------------------------------------------------------------------------

(defun agent-list-authority (agent-list)
  (apply 'max (mapcar #'compute-agent-authority agent-list)))

;;; ---------------------------------------------------------------------------

(defun superior-agents (agent)

  "SUPERIOR-AGENTS agent
Returns a list of agents that are super-components of the agent.
Only the top-most super-components are returned."

; (print (agent$super-component agent))
  (cond ((null agent) nil)
        ; if we have more than one, assume we have a list of lists
        ((and (agent$super-component agent)
              (listp (first (agent$super-component agent))))
         (superior-agents-list (agent$super-component agent)))
        ((agent$super-component agent)
         (if (agent-p (first (agent$super-component agent)))
             (superior-agents (first (agent$super-component agent)))
             (superior-agents (first (first (agent$super-component agent))))))
        (t (list agent))))

(defun superior-agents-list (super-component-list)

  "SUPERIOR-AGENTS-LIST super-component-list
Handles the case of an agent having more than one super-component by taking the 
union of the superior-agents of the agents in the super-component-list."

  (if (null super-component-list)
      nil
      (union (superior-agents (first (first super-component-list)))
             (superior-agents-list (rest super-component-list)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;     Agent Generation (Creation) Routines
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun create-fire-p (agent time)
  "True if agent moved and vacated location is empty."
  (let ((old-loc (find-agent-location agent (1- time))))
    (and (not-limbo-p old-loc)
         (not (equalp (agent$location agent) old-loc))
         (null (live-agents (find-agents-in-location old-loc *agent-schedule-queue* time))))))

;;; ---------------------------------------------------------------------------

(defun create-fire-at-prev-loc (agent time)
  "Creates a new fire agent at the previous location of the creating agent."
  (let ((new-agent 
          (create-agent :LOCATION (find-agent-location agent (1- time))
                        :CURRENT-TIME time
                        :TYPE (agent$type agent)
                        :BLOCKED-BY-TYPES (agent$blocked-by-types agent)
                        :CAPTURE-TYPES (agent$capture-types agent)
                        :CAPTURED-BY-TYPES (agent$captured-by-types agent)
                        :DRAW-FUNCTION (agent$draw-function agent)
                        :SENSORS (agent$sensors agent)
                        :MOVE-DATA (agent$move-data agent)
                        :REMOVE-P (agent$remove-p agent)
                        :ACTIVATE-P (agent$activate-p agent)
                        :INACTIVATE-P (agent$inactivate-p agent)
                        :REMOVE-FUNCTION (agent$remove-function agent)
                        :ACTIVATE-FUNCTION (agent$activate-function agent)
                        :INACTIVATE-FUNCTION (agent$inactivate-function agent)
                        :CREATION-TIME time
                        :DOMAIN-VARIABLES (copy-alist (agent$domain-variables agent))
                        :OVERLAP-PREDICATES (agent$overlap-predicates agent)
                        :INVOCATION-FUNCTION (agent$invocation-function agent))))
    (format t "Created agent ~a from ~a at ~a.~%" (agent$name new-agent) (agent$name agent) time)
    new-agent))

;;; ---------------------------------------------------------------------------

(defun mice-self (agent time)
  (declare (ignore time))
  agent)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;     Agent Activate/Inactivate Predicates
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun inact-for-given-time-p (agent time)
  (or (>= (- time (first (first (agent$state-history agent))))
          (rest (assoc :INACT-LIMIT (agent$domain-variables agent))))
      (= time 0)))

;;; ---------------------------------------------------------------------------

(defun fill-draw-function (agent time)
  (declare (ignore time))
  (setf (agent$draw-function agent) :FILLED-RECTANGLE))

;;; ---------------------------------------------------------------------------

(defun hollow-draw-function (agent time)
  (declare (ignore time))
  (setf (agent$draw-function agent) :HOLLOW-RECTANGLE))

;;; ---------------------------------------------------------------------------

(defun change-to-6-oclock (agent time)
  (declare (ignore time))
  (setf (agent$draw-function agent) :SIXOCLOCK))

;;; ---------------------------------------------------------------------------

(defun above-box-p (agent time)
  (declare (ignore time))
  (let ((grid-elem (get-grid-element nil nil :X (location$x (agent$location agent))
                                    :Y (1+ (location$y (agent$location agent))))))
    (and grid-elem
         (grid-element$agents grid-elem)
         (eq :HOLLOW-RECTANGLE (agent$draw-function (first (grid-element$agents grid-elem)))))))

;;; ---------------------------------------------------------------------------

(defun below-hero-p (agent time)
  (declare (ignore time))
  (let ((grid-elem (get-grid-element nil nil :X (location$x (agent$location agent))
                                     :Y (1- (location$y (agent$location agent))))))
    (and grid-elem
         (grid-element$agents grid-elem)
         (eq :BLUE (agent$type (first (grid-element$agents grid-elem)))))))
                                                           
;(defun remove-goals (agent time)
;  (declare (ignore time))
;  (setf (agent$goals agent) nil)
;  (setf (agent$old-goals agent) nil)
;  (setf (agent$goal-priority-function agent) nil))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;     Obstruction Predicates
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun agent-or-grid-obstruction (grid-elem)
"True if a grid element contains any agents or has an obstruction feature."
  (or (grid-element$agents grid-elem)
      (assoc :OBSTRUCTION (grid-element$features grid-elem))))

;;; ***************************************************************************
;;; EOF
