;;;************************************************************************
;;; di-graphics.lisp		D. Musliner
;;;
;;; - routines for device-independent graphics package allowing
;;;   my routines to uniformly output to Xwindows or to Postscript using the
;;;   same basic drawing routines and resetting an output flag.
;;;	- essentially a beginning of a device independent graphic language
;;;		ala Dobkin's denver system.
;;; - the standard arguments to the functions use the PS standard Y axis 
;;;   (the normal human way) and we'll convert
;;;	as necessary w/in the specific output functions for the various 
;;;	device-dependent calls.
;;; - see di-invert-Y-axis for using inverted Y axis (origin at top)
;;;************************************************************************
;;; the device independent calls do not specify the form of the arguments
;;; both to simplify the macros and to minimize the number of places which
;;; need to change when the semantics of calls are altered.
;;;************************************************************************

(provide 'di-graphics)

;(in-package 'di-graphics :nicknames '(dig))
;
;(export '(x ps ti
;	di-set-graphics-mode
;	di-initialize-graphics
;	di-deinitialize-graphics
;	di-start-drawing
;	di-label-drawing
;	di-finish-drawing
;	di-draw-rectangle
;	di-draw-circle
;	di-draw-segments
;	di-draw-lines
;	di-center-text
;	di-pan-left
;	di-pan-right
;	di-pan-up
;	di-pan-down
;	di-zoom
;	di-invert-Y-axis
;))

;;; ---------------------------------------------------------------------------
(defvar *graphics-mode* nil)
(defvar *graphics-enabled* nil)

(defvar *graphics-initialization* nil)	;; initializing graphics calls.
(defvar *graphics-trace* nil)		;; all drawing graphics calls.
(defvar *graphics-commands* nil)	;; recorded graphics calls for current
					;; drawing.


(defvar *display?* T)	;; should we have any display at all?
			;; - if set to nil, recordings of graphics calls still 
			;; 	made, but not run, so could save/redisplay if 
			;; 	post-exam of run shows we might like to see it.

;;; ---------------------------------------------------------------------------
  ;;(require (format nil "~A-graphics" mode) (format nil "./~A-graphics" mode))

(defun di-set-graphics-mode (mode)
  (setf *graphics-mode* mode)
  (cond ((eq *graphics-mode* 'x) 
         (require 'x-graphics))
        ((eq *graphics-mode* 'mac) 
         (require 'mac-graphics))
  	((eq *graphics-mode* 'ps) 
         (require 'ps-graphics))
  	((eq *graphics-mode* 'ti) 
		(require 'ti-graphics))
	(T (format t "ERROR: Unknown graphics mode ~A~%" *graphics-mode*))))

;;; ---------------------------------------------------------------------------
(defmacro setappend (a b) `(setf ,a (append ,a ,b)))

;;; ---------------------------------------------------------------------------

(defun di-initialize-graphics (&rest args)
  (setf *graphics-enabled* T)
  (setf *graphics-trace* nil)
  (setf *graphics-initialization* nil)
  (setf *graphics-commands* nil)
		;;; default graphics mode to X if not set already.
  (when (not *graphics-mode*) (di-set-graphics-mode 'x))	
  (setappend *graphics-initialization* 
		(list (cons 'di-initialize-graphics args)))
  (cond ((not *display?*))
	((eq *graphics-mode* 'X) 
	 (apply #'di-X-initialize-graphics args))
        ((eq *graphics-mode* 'MAC) 
         (apply #'di-MAC-initialize-graphics args))
  	((eq *graphics-mode* 'PS)
	 (apply #'di-PS-initialize-graphics args))
  	((eq *graphics-mode* 'TI) 
	 (apply #'di-TI-initialize-graphics args))
	(T (format t "ERROR: Unknown graphics mode ~A~%" *graphics-mode*))
))

;;; ---------------------------------------------------------------------------

(defun di-deinitialize-graphics ()
  (setf *graphics-enabled* nil)
  (cond ((not *display?*))
	((eq *graphics-mode* 'X) (di-X-deinitialize-graphics))
        ((eq *graphics-mode* 'MAC) (di-MAC-deinitialize-graphics))
  	((eq *graphics-mode* 'PS) (di-PS-deinitialize-graphics))
  	((eq *graphics-mode* 'TI) (di-TI-deinitialize-graphics))
	(T (format t "ERROR: Unknown graphics mode ~A~%" *graphics-mode*))
))

;;; ---------------------------------------------------------------------------

(defun di-label-drawing (&rest args)
  (setappend *graphics-commands* (list (cons 'di-label-drawing args)))
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) 
	 (apply #'di-X-label-drawing args))
        ((eq *graphics-mode* 'MAC) 
         (apply #'di-MAC-label-drawing args))
  	((eq *graphics-mode* 'PS)
	 (apply #'di-PS-label-drawing args))
  	((eq *graphics-mode* 'TI)
	 (apply #'di-TI-label-drawing args))
))

;;; ---------------------------------------------------------------------------
(defun di-draw-in-foreground (&rest args)
  (setappend *graphics-commands* (list (cons 'di-draw-in-foreground args)))
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) 
	 (apply #'di-X-draw-in-foreground args))
        ((eq *graphics-mode* 'MAC) 
         (applt #'di-MAC-draw-in-foreground args))
  	((eq *graphics-mode* 'PS)
	 (apply #'di-PS-draw-in-foreground args))
  	((eq *graphics-mode* 'TI)
	 (apply #'di-TI-draw-in-foreground args))
))

;;; ---------------------------------------------------------------------------
(defun di-draw-in-background (&rest args)
  (setappend *graphics-commands* (list (cons 'di-draw-in-backgraound args)))
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) 
	 (apply #'di-X-draw-in-background args))
        ((eq *graphics-mode* 'MAC) 
         (apply #'di-MAC-draw-in-background args))
  	((eq *graphics-mode* 'PS)
	 (apply #'di-PS-draw-in-background args))
  	((eq *graphics-mode* 'TI)
	 (apply #'di-TI-draw-in-background args))
))

;;; ---------------------------------------------------------------------------

(defun di-draw-rectangle (&rest args)
  (setappend *graphics-commands* (list (cons 'di-draw-rectangle args)))
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) 
	 (apply #'di-X-draw-rectangle args))
        ((eq *graphics-mode* 'MAC) 
         (apply #'di-MAC-draw-rectangle args))
	((eq *graphics-mode* 'PS)
	 (apply #'di-PS-draw-rectangle args))
	((eq *graphics-mode* 'TI)
	 (apply #'di-TI-draw-rectangle args))
))

;;; ---------------------------------------------------------------------------

(defun di-draw-circle (&rest args)
  (setappend *graphics-commands* (list (cons 'di-draw-circle args)))
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) 
	 (apply #'di-X-draw-circle args))
        ((eq *graphics-mode* 'MAC) 
         (apply #'di-MAC-draw-circle args))
	((eq *graphics-mode* 'PS)
	 (apply #'di-PS-draw-circle args))
	((eq *graphics-mode* 'TI)
	 (apply #'di-TI-draw-circle args))
))

;;; ---------------------------------------------------------------------------

(defun di-draw-triangle (&rest args)
  (setappend *graphics-commands* (list (cons 'di-draw-triangle args)))
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) 
	 (apply #'di-X-draw-triangle args))
        ((eq *graphics-mode* 'MAC) 
         (apply #'di-MAC-draw-triangle args))
	((eq *graphics-mode* 'PS)
	 (apply #'di-PS-draw-triangle args))
	((eq *graphics-mode* 'TI)
	 (apply #'di-TI-draw-triangle args))
))

;;; ---------------------------------------------------------------------------

(defun di-draw-segments (&rest args)
  (setappend *graphics-commands* (list (cons 'di-draw-segments args)))
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) 
	 (apply #'di-X-draw-segments args))
        ((eq *graphics-mode* 'MAC) 
         (apply #'di-MAC-draw-segments args))
	((eq *graphics-mode* 'PS)
	 (apply #'di-PS-draw-segments args))
	((eq *graphics-mode* 'TI)
	 (apply #'di-TI-draw-segments args))
))

;;; new in di-graphics.lisp
;;; ---------------------------------------------------------------------------

(defun di-translate-segments (seglist x y)
"(di-translate-segments segment-list deltaX deltaY)
  - returns list of segments translated by adding deltaX and deltaY."

  (setf seglist (map-even #'+ seglist x))
  (map-odd #'+ seglist y))

;;; new in di-graphics.lisp
;;; ---------------------------------------------------------------------------

(defun di-rotate-segments (seglist rad &aux (return-val nil) a h x y)
"(di-rotate-segments segment-list radians)
  - returns list of segments rotated counter-clockwise around origin by radians
	(unless Y axis inverted, in which case clockwise rotation)."

  (for (seg 0 (length seglist) 2)
	(setf x (nth seg seglist))
	(setf y (nth (1+ seg) seglist))
	(setf a (if (zerop x) 
			(if (> y 0) (/ pi 2) (- (/ pi 2))) 
		     	(atan (/ y x))))
  	(if (and (< x 0) (> y 0)) (+= a pi))
  	(if (and (< x 0) (< y 0)) (+= a pi))
 	(setf h (sqrt (+ (* x x) (* y y))))
	(setf return-val (append return-val (list (* h (cos (+ a rad))) (* h (sin (+ a rad)))))))
  return-val)

;;; ---------------------------------------------------------------------------
;
;(defun di-draw-lines (&rest args)
;  (cond ((eq *graphics-mode* 'X) (di-X-draw-lines args))
;	((eq *graphics-mode* 'PS) (di-PS-draw-lines  args))
;	((eq *graphics-mode* 'TI) (di-TI-draw-lines  args))
;))
;
;;; ---------------------------------------------------------------------------

(defun di-center-text (&rest args)
  (setappend *graphics-commands* (list (cons 'di-center-text args)))
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) 
	 (apply #'di-X-center-text args))
        ((eq *graphics-mode* 'MAC) 
         (apply #'di-MAC-center-text args))
	((eq *graphics-mode* 'PS)
	 (apply #'di-PS-center-text args))
	((eq *graphics-mode* 'TI)
	 (apply #'di-TI-center-text args))
))

;;; ---------------------------------------------------------------------------

(defun di-start-drawing ()
  (setappend *graphics-commands* '(progn (di-start-drawing)))
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) (clear-buffer))
        ((eq *graphics-mode* 'MAC) (clear-window))
	((eq *graphics-mode* 'PS) (di-ps-start-drawing))
	((eq *graphics-mode* 'TI) (di-TI-start-drawing))
))

;;; ---------------------------------------------------------------------------

(defun di-finish-drawing ()
  (setappend *graphics-commands* '((di-finish-drawing)))
  (setappend *graphics-trace* (list *graphics-commands*))
  (setf *graphics-commands* nil)
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) (copy-buffer-to-window))
        ((eq *graphics-mode* 'MAC) (event-dispatch))
	((eq *graphics-mode* 'PS) (di-ps-finish-drawing))
	((eq *graphics-mode* 'TI) (di-TI-finish-drawing))
))

;;; ---------------------------------------------------------------------------

(defun di-pan-right (dist)
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) (-= *x-x-offset* dist))
        ((eq *graphics-mode* 'MAC) (-= *mice-mac-x-offset* dist))
	((eq *graphics-mode* 'PS) (-= *ps-x-offset* dist))
	((eq *graphics-mode* 'TI) (-= *TI-x-offset* dist))
))

;;; ---------------------------------------------------------------------------

(defun di-pan-left (dist)
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) (+= *x-x-offset* dist))
        ((eq *graphics-mode* 'MAC) (+= *mice-mac-x-offset* dist))
	((eq *graphics-mode* 'PS) (+= *ps-x-offset* dist))
	((eq *graphics-mode* 'TI) (+= *TI-x-offset* dist))
))

;;; ---------------------------------------------------------------------------

(defun di-pan-up (dist)
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) (+= *x-y-offset* dist))
        ((eq *graphics-mode* 'MAC) (+= *mice-mac-y-offset* dist))
	((eq *graphics-mode* 'PS) (+= *ps-y-offset* dist))
	((eq *graphics-mode* 'TI) (+= *TI-y-offset* dist))
))

;;; ---------------------------------------------------------------------------

(defun di-pan-down (dist)
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) (-= *x-y-offset* dist))
        ((eq *graphics-mode* 'MAC) (-= *mice-mac-y-offset* dist))
	((eq *graphics-mode* 'PS) (-= *ps-y-offset* dist))
	((eq *graphics-mode* 'TI) (-= *TI-y-offset* dist))
))

;;; ---------------------------------------------------------------------------

(defun di-zoom (factor)
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) 
		(*= *x-x-mag* factor) (*= *x-y-mag* factor))
        ((eq *graphics-mode* 'MAC) 
          `(*= *ps-x-mag* ,factor) (*= *mice-mac-y-magnification-factor* factor))
	((eq *graphics-mode* 'PS) 
		(*= *ps-x-mag* factor) (*= *ps-y-mag* factor))
	((eq *graphics-mode* 'TI) 
		(*= *TI-x-mag* factor) (*= *TI-y-mag* factor))
))

;;; ---------------------------------------------------------------------------
;;; - this specifies that the user's Y axis should be treated as starting
;;; 	at the top of the screen instead of the bottom.

(defun di-invert-Y-axis ()
  (setappend *graphics-initialization* '((di-invert-Y-axis)))
  (cond ((not *display?*))
  	((eq *graphics-mode* 'X) (setf *x-y-mag* (- *x-y-mag*)))
        ((eq *graphics-mode* 'MAC) (set-y-mag-factor (- (get-y-mag-factor)))) 
	((eq *graphics-mode* 'PS) (di-ps-invert-Y-axis))
	((eq *graphics-mode* 'TI)  (setf *ti-y-mag* (- *ti-y-mag*)))
))

;;; ---------------------------------------------------------------------------
;;; Each element of the *graphics-trace* is a progn list of the calls which 
;;; make one drawing (starting w/ start-drawing and ending w/ finish-drawing).
;;; - all of the initialization functions are put into *graphics-initialization*

;;; ---------------------------------------------------------------------------
;;; Function di-redisplay-drawings
;;; - note this shadows the global vars *graphics-trace* and
;;;   	*graphics-initialization* so that the drawing call for redisplay wont
;;;   	clobber the values for the redisplay, so can redisplay multiple times in
;;;	a row.

(defun di-redisplay-drawings (&key (initlist *graphics-initialization*) 
				   (tracelist *graphics-trace*)
				   (sleep-time 0) (start 0) 
				   (end nil) (breaks nil))

  (let  ((*graphics-initialization* initlist)
	 (*graphics-trace* tracelist))

  	(if (not end) 
	    (setf end (length tracelist))
	    (++ end))
  	(dolist (init initlist) (eval init))
  	(for (index start end 1)
		(eval (nth index tracelist))
      		(if (member index breaks)
          	  (progn
            	  	(format t "~%Break at time ~a.~%Hit any key to continue. " index)
            	  	(read-char)
            	  	(format t "Continuing...~%"))
          	  (sleep sleep-time)))))

;;; ---------------------------------------------------------------------------
(defun di-save-drawings (filename)
  (with-open-file (stream (merge-pathnames filename "*.dig")
                          :direction :output
                          :if-exists :new-version)
  	(format stream "(setf *graphics-initialization* '~S)~%" 
		*graphics-initialization*)
  	(format stream "(setf *graphics-trace* '~S)~%" *graphics-trace*)
))


(defun di-restore-drawings (filename)
  (load (merge-pathnames filename "*.dig")))

;;; ---------------------------------------------------------------------------
