; Communication.lisp

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.

;;;            This work has been sponsored in part by:
;;;               the NSF (IRI-9010645, IRI-9015423)
;;;         the University of Michigan Rackham Graduate School
;;;


(defun clear-messages (agent)
  (setf (agent$receive-queue agent) nil))

(defun communication-region (agent channel time)
  (let ((range (channel$range channel))
	(ag-location (find-agent-location agent time)))
    (cond
     ((null range)
      (simulation-data$overall-region *simulation-data*))
     ((eq (channel$orientation-sensitive-p channel) :UNKNOWN)
      (setf (channel$orientation-sensitive-p channel)
	(or (/= (region$x-min range) (region$y-min range))
	    (/= (region$x-max range) (region$y-max range))))
      (communication-region agent channel time))
     ((channel$orientation-sensitive-p channel)
      (let ((agent-orientation (find-agent-orientation agent))
	    (dfront (- (region$y-min range)))
	    (dback  (region$y-max range))
	    (dleft  (- (region$x-min range)))
	    (dright (region$x-max range)))
	(legal-region-p (make-region :X-MIN (+ (location$x ag-location)
					       (- (case agent-orientation
						    (:NORTH dleft)
						    (:SOUTH dright)
						    (:EAST  dback)
						    (:WEST  dfront))))
				     :Y-MIN (+ (location$y ag-location)
					       (- (case agent-orientation
						    (:NORTH dfront)
						    (:SOUTH dback)
						    (:EAST  dleft)
						    (:WEST  dright))))
				     :X-MAX (+ (location$x ag-location)
					       (case agent-orientation
						 (:NORTH dright)
						 (:SOUTH dleft)
						 (:EAST  dfront)
						 (:WEST  dback)))
				     :Y-MAX (+ (location$y ag-location)
					       (case agent-orientation
						 (:NORTH dback)
						 (:SOUTH dfront)
						 (:EAST  dright)
						 (:WEST  dleft)))))))
     (t 
      (legal-region-p (make-region :X-MIN (+ (location$x ag-location) (region$x-min range))
				   :Y-MIN (+ (location$y ag-location) (region$y-min range))
				   :X-MAX (+ (location$x ag-location) (region$x-max range))
				   :Y-MAX (+ (location$y ag-location) (region$y-max range))))))))

;;
(defun mark-message-status (messages status)
  (mapcar #'(lambda (m) (setf (message$status m) status)) messages))

(defun send-back-messages (channel messages)
  "If the 'failure-message-priority' of the channel is positive,
   change the priority of the messages to 'failure-message-priority' and
   send back to the speaker."
  (let ((failure-priority (channel$failure-message-priority channel)))
    (when (>= failure-priority 0)
      (dolist (m messages)
	(setf (message$priority m) failure-priority)
	(push m (agent$receive-queue (agent-structure (message$speaker m))))))))

(defun transmit-messages (channels time)
  (dolist (channel channels)
    (restrict-within-channel-capacity channel)
    (deliver-on-going-messages channel time)))

;;
;;
;;

(defun restrict-within-channel-capacity (channel)
  "Apply communication capacity constraints."
  (let* ((capacity (channel$capacity channel))
	 (new-messages (channel$message-buffer channel))
	 (n-messages (length new-messages))
	 (to-be-sent nil)
	 (not-to-be-sent nil)
	 classified-messages)

    (if (null capacity) (setf capacity most-positive-fixnum))
    
    (cond ((<= n-messages capacity)	; Sufficient capacity
	   (setf to-be-sent new-messages))
	  ((setf classified-messages 
	     (mapcar #'(lambda (sequence) (sort sequence #'> :key #'message$priority)) 
		     (classify-list new-messages :key #'message$speaker)))
	   (multiple-value-setq (to-be-sent not-to-be-sent)
	       (select-n-from-classification capacity (randomize classified-messages)))
	   (mark-message-status not-to-be-sent :over-capacity)
	   (send-back-messages channel not-to-be-sent)))
    
    (mark-message-status to-be-sent :on-the-way)
    (setf (channel$on-going-messages channel)
      (append (channel$on-going-messages channel) to-be-sent))
    (setf (channel$message-buffer channel) nil)))


(defun deliver-on-going-messages (channel time)
  "Apply communication delay and other communication constraints."
  (let ((messages nil))
    (multiple-value-bind (org-messages leftovers)
        (select-if #'(lambda (m) (elapsed-delay-p m channel time)) (channel$on-going-messages channel))
      (setf (channel$on-going-messages channel) leftovers)
      (setf messages (expand-broadcast-messages org-messages channel)))
    
    (multiple-value-bind (out-of-range-messages leftovers)
        (select-if #'(lambda (m) (out-of-range-message-p m channel time)) messages)
      (when out-of-range-messages
        (mark-message-status out-of-range-messages :out-of-range)
        (send-back-messages channel out-of-range-messages)
        (setf messages leftovers)))
    
    (multiple-value-bind (obstructed-messages leftovers)
        (select-if #'(lambda (m) (obstructed-message-p m channel time)) messages)
      (when obstructed-messages
        (mark-message-status obstructed-messages :obstructed)
        (send-back-messages channel obstructed-messages)
        (setf messages leftovers)))
    
    (multiple-value-bind (failure-messages leftovers)
        (select-if #'(lambda (m) (failed-message-p m channel time)) messages)
      (when failure-messages
        (mark-message-status failure-messages :failure)
        (send-back-messages channel failure-messages)
      	(setf messages leftovers)))

    ;; Successful delivery
    (mark-message-status messages :success)
    (dolist (m messages)
      (setf (agent$receive-queue (agent-structure (message$hearer m)))
	(append (agent$receive-queue (agent-structure (message$hearer m))) (list m))))
    
    messages))

(defun expand-broadcast-messages (org-messages channel)
  "Expand orginal messages which have broadcast messages (:hearer :all)
   to duplicated messages for each hearer of the channel."
  (let ((messages nil)
	(copied-message nil))
    (dolist (m org-messages)
      (cond ((equal (message$hearer m) :all)
	     (dolist (h (channel$agents channel))
	       (setf copied-message (copy-message m))
	       (setf (message$hearer copied-message) h)
	       (push copied-message messages)))
	    (t
	      (push m messages))))
    messages))

(defun elapsed-delay-p (message channel time)
  "True if the message in the queue was sent before channel-delay."
  (<= (+ (message$time-created message) (channel$delay channel)) time))
	 
(defun failed-message-p (message channel time)
  "True if the message is failed to be delivered.
   Failure is determined based on the reliablity of channel.
   Reliability is the probability of successful transmission."
  (declare (ignore message time))
  (if (> (random 1.0) (channel$reliability channel)) t nil))

(defun out-of-range-message-p (message channel time)
  (let* ((speaker (agent-structure (message$speaker message)))
	 (speaker-region (communication-region speaker channel time))
	 (hearer (agent-structure (message$hearer message)))
	 (hearer-location (find-agent-location hearer time)))
    (null (location-in-region-p hearer-location speaker-region))))

(defun obstructed-message-p (message channel time)
  (let ((speaker (agent-structure (message$speaker message)))
	(hearer (agent-structure (message$hearer message))))
    (obstruct-between-p speaker hearer channel time)))

(defun obstruct-between-p (speaker hearer channel time)
  "True if there are any communication obstructions between speaker and hearer
   using the channel at 'time'."
  (let* ((obstructed-by (channel$obstructed-by channel))
	 (speaker-location (find-agent-location speaker time))
	 (hearer-location (find-agent-location hearer time))
	 (region (communication-region speaker channel time))
	 (x-min (region$x-min region))
	 (x-max (region$x-max region))
	 (y-min (region$y-min region))
	 (y-max (region$y-max region)))
    (multiple-value-bind (interesting-grids obstructions)
	;; nil for interesting-p to get only obstructions
	(get-interesting-and-obstructing x-min x-max y-min y-max nil obstructed-by)
      (declare (ignore interesting-grids))
      (some #'(lambda (obstruction-data)
		(let ((obstruction (first obstruction-data))
		      (obstruction-diameter (second obstruction-data)))
		  (obstructed-p (location$x speaker-location) (location$y speaker-location)
				(location$x hearer-location) (location$y hearer-location)
				(location$x obstruction) (location$y obstruction)
				:DIAMETER obstruction-diameter)))
	    obstructions))))

;;
;;
;;

(defun send-message (agent channel-name type content &key (priority 0) (hearer :all))
  "Just put the message in the sending queue of the agent.
   The message will be transfered to the hearer's receive-queue by MICE.
   The undelivered messages will be sent back to the speaker's receive-queue."
  (let ((channel (channel-structure channel-name)))
    (setf (channel$message-buffer channel)
      (append (channel$message-buffer channel)
	      (list (make-message :speaker (agent$name (agent-structure agent))
				  :hearer hearer 
				  :type type 
				  :content content 
				  :priority priority
				  :time-created *current-time*
				  :channel-name channel-name
				  :status :created))))))

(defun recv-messages (agent channel-name &key (count most-positive-fixnum) (clear nil))
  "Receive 'count' messages from the 'channel'. 
   If clear is set to 't', the remaining messages of the channel is removed."
  (multiple-value-bind (channel-messages others)
      (select-if #'(lambda (m) (equal (message$channel-name m) channel-name)) (agent$receive-queue agent))
    (setf count (min count (length channel-messages)))
    (setf channel-messages (stable-sort channel-messages #'>= :key #'message$priority))
    (let ((interesting (subseq channel-messages 0 count)))
      (setf (agent$receive-queue agent) 
	(if clear others (append (nthcdr count channel-messages) others)))
      (setf (agent$receive-message-buffer agent) interesting); to be read by the read-and-reset-received-messages function
      interesting)))

(defun select-messages (messages &key (type :all) (status :success))
  "Returns two values as multiple values. The first value is a list of selected
   messages satisfying the type and the status of a message. The second value
   is the list of remaining messages."
  (let ((match-p #'(lambda (msg) (and (or (equal type :all) (equal type (message$type msg)))
				      (or (equal status :all) (equal status (message$status msg)))))))
    (cond ((listp messages)
	   (select-if match-p messages))
	  (t
	   (values nil nil)))))

;;
;; Utility functions
;;

(defmacro domain (agent var)
  "Domain variable access function."
  `(cdr (assoc ,var (agent$domain-variables ,agent))))

(defsetf domain (agent var) (new-value)
  "Domain variable update function."
  `(setf (cdr (assoc ,var (agent$domain-variables ,agent))) ,new-value))

(defun min-key (sequence &key (key #'identity))
  "Returns a element with minimum key value."
  (let ((x (first sequence)))
    (dolist (m (cdr sequence) x)
      (if (< (funcall key m) (funcall key x)) (setf x m)))))

(defun max-key (sequence &key (key #'identity))
  "Returns a element with maximum key value."
  (let ((x (first sequence)))
    (dolist (m (cdr sequence) x)
      (if (> (funcall key m) (funcall key x)) (setf x m)))))

(defun select-if (pred sequence &key (key #'identity))
  "Return two values as multiple values. The first value is the list of the 
   subset of sequence whose 'key' satisfies the 'pred'. The second value is
   the list of remaining elements."
  (values (remove-if-not pred sequence :key key)
	  (remove-if pred sequence :key key)))

(defmacro assoc-insert (key datum alist)
  "Like acons, but doesn't simply push new key-datum pair on - replaces old one if it exists"
  `(if (assoc ,key ,alist)
       (progn (setf (rest (assoc ,key ,alist)) ,datum) ,alist)
       (setf ,alist (acons ,key ,datum ,alist))))

(defun classify-list (sequence &key (key #'identity))
  "Classify elements of the list based on the key of the element.
   eg. (classify-list '((a 3) (b 4) (c 5) (a 7) (c 8)) :key #'car) 
       ==> ( ((a 3) (a 7))    ((b 4))   ((c 5) (c 8)) )"
  (let ((classification nil))
    (dolist (m sequence)
      (let* ((group (assoc (funcall key m) classification :key key)))
	(cond (group (setf (cdr group) (append (cdr group) (list m))))
	      (t (setf classification (append classification (list (list m))))))))
    classification))

(defun select-n-from-classification (n classification)
  "Return the selected list of elements from each classification
   and the rest elements as multiple values. The selection starts from the 
   first element of first class, then the first of second class, and so on."
  (let ((total (length (apply #'append classification)))
	(selected nil)
	(others nil)
	element)
    (do ((i 1) (column 0 (1+ column))) 
	((> i total) (values (reverse selected) (reverse others)))
      (dolist (class classification)
	(when (setf element (nth column class))
	  (if (<= i n) (push element selected) (push element others))
	  (incf i))))))

(defun randomize (sequence)
  "Randomize the order of the sequence and return the randomized list."
  (let ((randomized nil)
	s)
    (do ((n (length sequence) (- n 1)))
	((<= n 0) randomized)
      (setf s (random n))
      (push (nth s sequence) randomized)
      (setf sequence (append (subseq sequence 0 s) (nthcdr (1+ s) sequence))))))

(defun firstn (n sequence)
  "Return multiple values. First value is list of first n element of the list
   and the second value is the list of remaining elements.
   i.e. (0 .. n-1) (n ... )"
  (values (subseq sequence 0 n) (nthcdr n sequence)))
  