;;;; board.lisp
;;;; **************************************************************************
;;;; **************************************************************************
;;;; *                                                                        *
;;;; *            ENVIRONMENT CONSTRAINT FUNCTIONS (CONTROL BOARD)            *
;;;; *                                                                        *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Thomas Anthony Montgomery
;;;             Department of Electrical Engineering and Computer Science
;;;             University of Michigan
;;;             Ann Arbor, Michigan 48103.
;;;
;;;
;;;      1991 The Regents of the University of Michigan
;;;                  All Rights Reserved
;;;
;;;
;;;            This work has been sponsored in part by:
;;;               the NSF (IRI-9010645, IRI-9015423)
;;;         the University of Michigan Rackham Graduate School
;;;
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.


;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  06-06-89 Header Created.  (MONTY)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

;(in-package 'MICE)

;(export '(print-board-configuration))

(proclaim '(optimize (speed 3) (safety 1)))

;;; --*--
;;; ***************************************************************************

(defun control-board ()

  "CONTROL-BOARD nil
Applies the environmental constraints on agent actions and displays the board configuration."

  ; Control-board represents the constraints of the board
  ; such as
  ; 2 pieces cannot pass through each other
  ; and
  ; 2 pieces cannot be in the same spot at the same time
  (let ((min-time                                        ;min-time is the time at which the
          (if *agent-schedule-queue*
              (apply #'min                                    ;next agent's action will occur
                     (mapcar #'(lambda (agent)
                                 (agent$current-time agent))
                             *agent-schedule-queue*))
              nil)))
    ; This code only performs work when none of the agents can perform any
    ; more actions (i.e. they have all advanced their clocks which indicate
    ; their next oportunity to act past the current simulation time).
    (cond ((not min-time) nil)
          ((> min-time (simulation-data$overall-time *simulation-data*))
           (catch-up-board (simulation-data$overall-time *simulation-data*) min-time)
           (setf (simulation-data$overall-time *simulation-data*) min-time)
           )
          (T nil))))

;;; ---------------------------------------------------------------------------

(defun catch-up-board (schedule-time until-time)

  "CATCH-UP-BOARD schedule-time until-time
Updates board values from last scheduled time to time given.
This is done by simulating time steps for all times
between the schedule-time and the until-time NOT including the
schedule time."

  (do ((time (1+ schedule-time) (1+ time))
       (first-pass? t nil))
      ((> time until-time))

    ;;; Resolve agent actions.
    (when first-pass?
      (check-for-movement *agent-schedule-queue* time :ATTEMPTED-MOVES)
      (resolve-actions (copy-list *agent-schedule-queue*) time)
      (update-agent-grid-locations *agent-schedule-queue* time)
      (check-for-movement *agent-schedule-queue* time :SUCCESSFUL-MOVES))

    (apply-environment-constraints *agent-schedule-queue* time)
    (transmit-messages *mice-channels* time) ; Jaeho's

    (if (not *graphics?*)
        	(when (print-board-configuration-p time) 
			(print-board-configuration time))
        	(when first-pass? (draw-board-configuration time)))))

;;; ---------------------------------------------------------------------------

(Defun resolve-actions (agent-list time)
  (setf agent-list (sort agent-list #'(lambda (a1 a2) (< (compute-agent-authority a2)
                                                         (compute-agent-authority a1)))))
  (check-locations-for-legality agent-list time)
  (resolve-status-changes agent-list time :INSTANTANEOUS? nil)
  (apply-link-constraints)
  (apply-overlap-effects agent-list time)
  (setf *temp-authority* nil)
  (apply-attempted-links time)
;  (resolve-passing-through-each-other agent-list time)
;  (resolve-overlaps agent-list time)
;  (update-state-histories agent-list time)
  )

;;; ---------------------------------------------------------------------------

(defun apply-environment-constraints (agent-list time)

  "APPLY-ENVIRONMENT-CONSTRAINTS agent-list time
Determine position effects, creations, activations, inactivations, and removals for time `time'."

  (let ((mice-actions nil))
    (determine-position-effects agent-list time)
  
    ;;; Buffer agent status changes.
    (setf mice-actions
          (append (return-create-new-agents-actions agent-list time)
                  (return-activate-agents-actions agent-list time)
                  (return-inactivate-agents-actions agent-list time)
                  (return-remove-agents-actions agent-list time)))
  
    (check-buffered-actions mice-actions time)

    (apply-status-changes mice-actions time)
    (verify-state time)))

;;; ---------------------------------------------------------------------------

(defun check-buffered-actions (action-list time)
"Group the actions buffered by MICE into sets of actions that have interdependencies.
Combine interdependent actions into composite actions where appropriate."
  (declare (ignore action-list time)))

(defun apply-status-changes (mice-actions time)
  (mapc #'eval mice-actions)
  (resolve-status-changes *agent-schedule-queue* time :INSTANTANEOUS? t))

(defun verify-state (time)
  (resolve-overlaps *agent-schedule-queue* time :ERROR-IF-CONFLICT t)
  (update-state-histories *agent-schedule-queue* time))

;;; ---------------------------------------------------------------------------

(defun check-for-movement (agent-list time event-type)

  "CHECK-FOR-MOVEMENT agent-list time event-type
Call handle-event for moved agents if there is a fcn associated with the event-type"

  (when (assoc event-type *event-function-list*)
    (handle-event event-type (moved-agents agent-list time))))

;;; ---------------------------------------------------------------------------

(defun moved-agents (agent-list time)

  "MOVED-AGENTS agent-list time
Find all agents that have moved in the last time step."

  (cond ((null agent-list) nil)
        ((and (not-limbo-p (find-agent-location (first agent-list) (1- time)))
              (or (not (eql (find-agent-orientation (first agent-list) time)
                            (find-agent-orientation (first agent-list) (1- time))))
                  (not (equalp (find-agent-location (first agent-list) time)
                               (find-agent-location (first agent-list) (1- time))))))
         (cons (first agent-list) (moved-agents (rest agent-list) time)))
        (t (moved-agents (rest agent-list) time))))

;;; ---------------------------------------------------------------------------

(defun apply-link-constraints ()

  "APPLY-LINK-CONSTRAINTS nil
Loops through all established links and resolves agent positions so that they do not
violate the links.  It assumes that the location of the composite agent (parent) is
fixed while the location of the sub-component (child) can be adjusted."

  (dolist (alink *links*)
    (let ((desired-child-loc
            (compute-new-location (agent$location (link$composite-agent alink))
                                  (compute-link-direction (agent$orientation (link$composite-agent alink))
                                                          (link$type alink)))))
      (when (not (equalp desired-child-loc
                         (agent$location (link$sub-component alink))))
        (move-agent-to-location (link$sub-component alink) desired-child-loc :REASON :LINK-CONSTRAINT)))))


(defun compute-link-direction (orientation link-type)
  (case link-type
    (:FRONT orientation)
    (:LEFT (case orientation
             (:NORTH :WEST)
             (:EAST :NORTH)
             (:SOUTH :EAST)
             (:WEST :SOUTH)))
    (:RIGHT (case orientation
              (:NORTH :EAST)
              (:EAST :SOUTH)
              (:SOUTH :WEST)
              (:WEST :NORTH)))
    (:BACK (case orientation
             (:NORTH :SOUTH)
             (:EAST :WEST)
             (:SOUTH :NORTH)
             (:WEST :EAST)))
    (:SHARED-LOC nil)
    (:NORTH :NORTH)
    (:SOUTH :SOUTH)
    (:EAST :EAST)
    (:WEST :WEST)
    (otherwise (error "Unexpected link type encountered: ~a.~%" link-type))))

;;; ---------------------------------------------------------------------------

(defun apply-overlap-effects (agent-list time)

  "APPLY-OVERLAP-EFFECTS agent-list time
Apply the effect of agents attempting to switch locations or occupy the same locations.
The default is for agents that block each other to bounce off of each other."

  (let ((changes? (apply-overlap-effects-1 agent-list time)))
    (when changes? (apply-overlap-effects agent-list time))))

(defun apply-overlap-effects-1 (agent-list time)
"Changes made by Dan."
  (cond ((null agent-list) nil)
	((null (rest agent-list)) nil)
        (t (let* ((agent (first agent-list))
                  (switched-agents (find-switched-agents agent (rest agent-list) time))
                  (overlap-agents (find-agents-in-location (find-agent-location agent time)
                                                           (rest agent-list) time))
                  (movement? nil))
             (when switched-agents
               (dolist (switched-agent switched-agents)
                 (setf movement?
                       (or (ds-overlap-effect agent switched-agent time) movement?))))
             (when *collision-verbose* (when overlap-agents (format t "~%overlap: ~a ~a" agent overlap-agents)))
             (when overlap-agents
               (dolist (overlap-agent overlap-agents)
                 (setf movement?
                       (or (ds-overlap-effect agent overlap-agent time) movement?))))
             (if movement?
                 movement?
                 (apply-overlap-effects-1 (rest agent-list) time))))))

;;; ---------------------------------------------------------------------------

(defun apply-attempted-links (time)
  (declare (ignore time))

  "APPLY-ATTEMPTED-LINKS time
Applies attempted links iff the parent-child relationships are right."
  
  (dolist (alink *attempted-links*)
    (let ((parent (link$composite-agent alink))
          (child  (link$sub-component   alink))
          (link-type (link$type         alink)))

      (when (check-link alink)
         (when (eq link-type :NEXT-TO)
                (setf link-type (find-direction (agent$location parent) (agent$location child) parent)))
              (when *link-verbose* (print "linking!"))
              (setf (agent$super-component child) (cons parent link-type))
              (setf (agent$sub-components parent) (acons child link-type (agent$sub-components parent)))
              (setf *links* (cons (make-link :COMPOSITE-AGENT parent :SUB-COMPONENT child :TYPE link-type) *links*))
              (handle-event :LINK (list parent child)))))
  (setf *attempted-links* nil))

(defun determine-position-effects (agent-list time)

  "DETERMINE-POSITION-EFFECTS agent-list time
Determine the effects on agents due to their current position.
NOTE: Currently this assumes that the position effects are due to switching or overlapping
locations with other agents."

  (cond ((null agent-list) nil)
        ((null (rest agent-list)) nil)
        (t (let* ((agent (first agent-list))
                  (switched-agents (find-switched-agents agent (rest agent-list) time))
                  (overlap-agents (find-agents-in-location (find-agent-location agent time)
                                                           (rest agent-list) time)))
             (when switched-agents
               (mapc #'(lambda (switched-agent) (ds-position-effect agent switched-agent time))
                     switched-agents))
             (when overlap-agents
               (mapc #'(lambda (overlap-agent) (ds-position-effect agent overlap-agent time))
                     overlap-agents))
             (determine-position-effects (rest agent-list) time)))))

;;; ---------------------------------------------------------------------------

(defun return-create-new-agents-actions (agent-list time)

  "RETURN-CREATE-NEW-AGENTS-ACTIONS agent-list time
Returns a list of commands that, when evaled, will create new agents and
add them to the schedule-queue."

  (let ((create-actions nil))
    (dolist (agent agent-list)
      (when (and (agent$create-p agent)
                 (funcall (agent$create-p agent) agent time))
        (setf create-actions
              (cons `(funcall #',(agent$create-function agent) ',agent ,time)
                    create-actions))))          

    (if create-actions
        `((let ((new-agents nil)
                (new-agent-list nil))
            (dolist (create-function  ',create-actions)
              (setf new-agents  (eval create-function))
              (setf new-agent-list (append (if (agent-p new-agents) (list new-agents) new-agents)
                                           new-agent-list)))))
        nil)))
               
;;; ---------------------------------------------------------------------------

(defun return-activate-agents-actions (agent-list time)

  (let ((activate-actions nil))
    (dolist (agent agent-list)
      (when (and (member (agent$current-status agent) (list :INACTIVATED :CREATED))
                 (agent$activate-p agent)
                 (funcall (agent$activate-p agent) agent time))
        (when (agent$activate-function agent)
          (setf activate-actions
                (cons `(funcall #',(agent$activate-function agent) ',agent ,time)
                      activate-actions)))
        (setf activate-actions
              (cons `(add-action ',agent ,:ACTIVATED ,time)
                    activate-actions))))
    activate-actions))

;;; ---------------------------------------------------------------------------

(defun current-state (agent)
  (rest (first (agent$state-history agent))))

;;; ---------------------------------------------------------------------------

(defun add-action (agent action time)
  (setf (agent-state$action-history (current-state agent))
        (acons time (make-agent-action :ACTION action :TIME time)
               (agent-state$action-history (current-state agent)))))

;;; ---------------------------------------------------------------------------

(defun return-inactivate-agents-actions (agent-list time)

  (let ((inactivate-actions nil))
    (dolist (agent agent-list)
      (when (and (eq (agent$current-status agent) :ACTIVATED)
                 (agent$inactivate-p agent)
                 (funcall (agent$inactivate-p agent) agent time))
        (when (agent$inactivate-function agent)
          (setf inactivate-actions
                (cons `(funcall #',(agent$inactivate-function agent) ',agent ,time)
                      inactivate-actions)))
        (setf inactivate-actions
              (cons `(add-action ',agent ,:INACTIVATED ,time)
                    inactivate-actions))))
    inactivate-actions))

;;; ---------------------------------------------------------------------------

(defun return-remove-agents-actions (agent-list time)
  (let ((remove-actions nil))
    (dolist (agent agent-list)
      (when (and (agent$remove-p agent)
                 (funcall (agent$remove-p agent) agent time))
        (setf remove-actions
              (cons `(funcall #',(agent$remove-function agent) ',agent ,time)
                    remove-actions))))

    (if remove-actions
        `((let ((removed-agents nil)
                (removed-agent-list nil))
            (dolist (agents-to-remove  ',remove-actions)
              (setf removed-agents  (eval agents-to-remove))
              (setf removed-agent-list (append (if (agent-p removed-agents) (list removed-agents) removed-agents)
                                           removed-agent-list)))
            (when removed-agent-list
              (setf *agent-schedule-queue* (ordered-set-difference *agent-schedule-queue* removed-agent-list))
              (dolist (agent removed-agent-list)
                (setf (agent$removal-time agent) ,time)
                (add-action agent ,:REMOVED ,time)))))
        nil)))
 
;;; ---------------------------------------------------------------------------

(defun remove-agents (agent-list &OPTIONAL (time *current-time*))
  (when agent-list
    (remove-agent (first agent-list) time)
    (remove-agents (rest agent-list) time)))

;;; ---------------------------------------------------------------------------

(defun remove-agent (agent &OPTIONAL (time *current-time*))
  (setf *agent-schedule-queue* (remove agent *agent-schedule-queue*))
  (setf (agent$removal-time agent) time)
  (add-action agent :REMOVED time))

;;; ---------------------------------------------------------------------------

(defun print-board-configuration-p (time)
  "True when time divides evenly by *print-board-interval*"
  (= (/ time *print-board-interval*) (truncate time *print-board-interval*)))

;;; ---------------------------------------------------------------------------

(defun check-locations-for-legality (agent-list time)
  "Loops through all of the agents.  If they are not in legal locations according to 
the grid features, then it either undoes their state change or their last move to resolve."
  (mapc #'(lambda (agt)
                 (when (not (legal-location-p (agent$location agt) :AGENT agt))
                   (cond ((not (eq (agent$current-status agt)
                                   (agent-state$status (current-state agt))))
                          (when *verbose?*
                            (format t "Agent ~a illegally attempted to become ~a in location ~a at ~a.~%"
                                    agt (agent$current-status agt) (agent$location agt) time))
                          (setf (agent$current-status agt) (agent-state$status (current-state agt))))

                         ((not (equalp (agent$location agt)
                                       (agent-state$location (current-state agt))))
                          (when *verbose?*
                            (format t "Agent ~a attempted to move into illegal location ~a at ~a.~%"
                                    agt (agent$location agt) time))
                          (setf (agent$location agt) (agent-state$location (current-state agt)))))))
        agent-list))

;;; ---------------------------------------------------------------------------

(defun get-agent-linkages (agent)
  (if (agent$sub-components agent) t nil))

(defun update-state-histories (agent-list time)

"Update the state histories of all agents whose current state is different from
the state on the top of its state-history list."
  
  (dolist (agent agent-list)
    (let ((curr-state (current-state agent))
	  (agent-linkages (get-agent-linkages agent))
          (other (find-agent-current-other-state-information agent)))
      (when (not (and (equalp (agent$location agent) (agent-state$location curr-state))
                      (eql (agent$current-status agent) (agent-state$status curr-state))
                      (eql (agent$orientation agent) (agent-state$orientation curr-state))
		      (eql agent-linkages (agent-state$linkages curr-state))
                      (equalp other (agent-state$other curr-state))))
        (setf (agent$state-history agent)
              (acons time
                     (make-agent-state :LOCATION (agent$location agent)
                                       :STATUS (agent$current-status agent)
                                       :ORIENTATION (agent$orientation agent)
				       :LINKAGES agent-linkages
                                       :OTHER other)
                     (agent$state-history agent)))))))

(defun find-agent-current-other-state-information (agent)
  (when *other-state-history-information-function*
    (funcall *other-state-history-information-function* agent)))

;;; ---------------------------------------------------------------------------

(defun resolve-status-changes (agent-list time &key (instantaneous? nil))

  "RESOLVE-STATUS-CHANGES agent-list time
Resolve status change actions taken by the agents in agent-list.
NOTE -- THIS ASSUMES ALL STATUS CHANGES ARE SUCCESSFUL AND NON-OVERLAPPING."

  (dolist (agent agent-list)
    (let ((time-action (first (agent-state$action-history (current-state agent)))))
      ;;; Update the current-status of the agent if there is a status change action
      ;;; and the time is right.
      (when (and time-action
                 (= time
                    (if instantaneous? (first time-action) (1+ (first time-action))))
                 (member (agent-action$action (rest time-action)) *status-change-actions*))
        (setf (agent$current-status agent)
              (agent-action$action (rest time-action)))))))

;;; ---------------------------------------------------------------------------

(defun resolve-passing-through-each-other (agent-list time)
  (cond ((null agent-list) nil)
        ((null (rest agent-list)) nil)
        (t (let* ((agent (first agent-list))
                  (switched-agents (find-switched-agents agent (rest agent-list) time))
                  (move-agent-back nil))
             (when switched-agents
               (mapc #'(lambda (switched-agent)
                         (when (member (agent$type switched-agent) (agent$blocked-by-types agent))
                           (move-agent-to-location switched-agent
                                                   (find-agent-location switched-agent (1- time))
                                                   :REASON :SWITCH)
                           (setf move-agent-back t)
                           (handle-event :MOVE-BACK (list switched-agent))))
                     switched-agents)
               (when move-agent-back
                 (move-agent-to-location agent (find-agent-location agent (1- time)) :REASON :SWITCH)
                 (handle-event :MOVE-BACK (list agent))))
             (resolve-passing-through-each-other (ordered-set-difference (rest agent-list) switched-agents) time)))))

;;; ---------------------------------------------------------------------------

(defun find-switched-agents (test-agent agent-list time)
  (cond ((null agent-list) nil)
        ((equalp (find-agent-location test-agent time) (find-agent-location test-agent (1- time))) nil)
        ((and (equalp (find-agent-location test-agent time) (find-agent-location (first agent-list) (1- time)))
              (equalp (find-agent-location test-agent (1- time)) (find-agent-location (first agent-list) time)))
         (cons (first agent-list)
               (find-switched-agents test-agent (rest agent-list) time)))
        (t (find-switched-agents test-agent (rest agent-list) time))))

;;; ---------------------------------------------------------------------------

(defun move-agent-to-location (agent location &KEY reason)
  (when *verbose?*
    (format t "Agent ~a moved to location ~a for reason ~a~%" (agent$name agent) location reason))
  (setf (agent$location agent) location))

;;; ---------------------------------------------------------------------------

(defun resolve-overlaps (agent-list time &KEY (error-if-conflict nil))

  "RESOLVE-OVERLAPS agent-list time
Steps through the agents checking for conflicts."

  (cond ((null agent-list) nil)
        ((null (second agent-list)) nil)
        (t (let* ((same-place-agents (find-agents-in-location (find-agent-location (first agent-list) time)
                                                              (rest agent-list)
                                                              time))
                  (conflict-agents (agent-cannot-share-space-with-others-p (first agent-list) same-place-agents)))
             (cond ((and conflict-agents error-if-conflict)
                    (error "Conflict found after application of environmental constraints."))
                   (conflict-agents
                    (setf conflict-agents (moved-agents (cons (first agent-list) conflict-agents) time))
                    (handle-event :MOVE-BACK conflict-agents)
                    (move-agents-to-location-at-given-time conflict-agents (1- time))
                    (resolve-overlaps *agent-schedule-queue* time))           ;Recurse on all agents if some moved back
                   (t (resolve-overlaps (rest agent-list) time)))))))   ;Otherwise, continue stepping through list.

;;; ---------------------------------------------------------------------------

(defun agent-cannot-share-space-with-others-p (agent other-agents)

  "AGENT-CANNOT-SHARE-SPACE-WITH-OTHERS-P agent other-agents
Returns members of other-agents that cannot share space with agent."

  (cond ((null other-agents) nil)
        ((member (agent$type (first other-agents)) (agent$blocked-by-types agent))
         (cons (first other-agents) (agent-cannot-share-space-with-others-p agent (rest other-agents))))
        (t (agent-cannot-share-space-with-others-p agent (rest other-agents)))))

;;; ---------------------------------------------------------------------------

(defun find-agents-in-location (loc agent-list time)

  "FIND-AGENTS-IN-LOCATION loc agent-list time
Return all agents in agent-list with loc at time"

  (cond ((null agent-list) nil)
        ((equalp loc (find-agent-location (first agent-list) time))
         (cons (first agent-list) (find-agents-in-location loc (rest agent-list) time)))
        (t (find-agents-in-location loc (rest agent-list) time))))

;;; ---------------------------------------------------------------------------

(defun move-agents-to-location-at-given-time (agent-list time)

  "MOVE-AGENTS-TO-LOCATION-AT-GIVEN-TIME agent-list time
Moves agents to the location they were in at time 'time' as well as all of their sub-components."

  (dolist (agent agent-list)
    (move-agent-to-location agent (find-agent-location agent time) :REASON :CONFLICT)
    (setf (agent$orientation agent) (find-agent-orientation agent time))
    (dolist (sub-comp-link (agent$sub-components agent))
      (move-agent-to-location (first sub-comp-link)
                              (find-agent-location (first sub-comp-link) time) :REASON :CONFLICT)
      (setf (agent$orientation (first sub-comp-link))
            (find-agent-orientation (first sub-comp-link) time)))))

;;; ---------------------------------------------------------------------------

;(defun update-agent-grid-locations (agent-list time)
;  (cond ((null agent-list) t)
;        (t (let* ((agent (first agent-list))
;                  (x-loc (location$x (agent$location agent)))
;                  (y-loc (location$y (agent$location agent)))
;                  (grid-elem (get-grid-element x-loc y-loc t)))
;            ; (when (null grid-elem)
;            ;   (setf grid-elem (setf (aref (simulation-data$grid *simulation-data*) x-loc y-loc)
;            ;                         (make-grid-element))))
;             (if (assoc time (grid-element$agent-history grid-elem))
;                 (setf (rest (assoc time (grid-element$agent-history grid-elem)))
;                       (cons agent (rest (assoc time (grid-element$agent-history grid-elem)))))
;                 (setf (grid-element$agent-history grid-elem)
;                       (acons time (list agent) (grid-element$agent-history grid-elem))))
;             (setf (grid-element$agents grid-elem) (rest (assoc time (grid-element$agent-history grid-elem)))))
;           (update-agent-grid-locations (rest agent-list) time))))

(defun update-agent-grid-locations (agent-list time)

  "UPDATE-AGENT-GRID-LOCATIONS agent-list time
Loop through agent-list and update their representation in the grid structure."

  (dolist (agent agent-list)
    (let* ((old-loc (find-agent-location agent (1- time)))
           (old-grid-elem (get-grid-element old-loc t))
           (grid-elem (get-grid-element  (agent$location agent) t)))
      
      ;;; Update agent-history.
      (if (assoc time (grid-element$agent-history grid-elem))
          (setf (rest (assoc time (grid-element$agent-history grid-elem)))
                (cons agent (rest (assoc time (grid-element$agent-history grid-elem)))))
          (setf (grid-element$agent-history grid-elem)
                (acons time (list agent) (grid-element$agent-history grid-elem))))

      ;;; Update current agents list of new agent location.
      (setf (grid-element$agents grid-elem) (rest (assoc time (grid-element$agent-history grid-elem))))

      ;;; Update current agents list of old location.
      (when (not (equalp (agent$location agent) old-loc))
        (setf (grid-element$agents old-grid-elem) (remove agent (grid-element$agents old-grid-elem)))))))

;;; ---------------------------------------------------------------------------

(defun print-board-configuration (time)

  "PRINT-BOARD-CONFIGURATION time
Prints the board configuration as it appeared at the given time."

  (let ((agents *agent-schedule-queue*))
    (format t "~%-------------- TIME ~A --------------~%" time)
    (do ((y (region$y-min (simulation-data$overall-region *simulation-data*)) (1+ y)))
        ((> y (region$y-max (simulation-data$overall-region *simulation-data*))))
      (do ((x (region$x-min (simulation-data$overall-region *simulation-data*)) (1+ x)))
          ((> x (region$x-max (simulation-data$overall-region *simulation-data*))))
        (let ((grid-element (get-grid-element nil t :X x :Y y))
              (agent-in-loc (find-agent-in-location-at-time agents x y time)))
          (format t "~C " 
		 (if agent-in-loc
		     (apply (first (agent$draw-function agent-in-loc))
			       (list* (make-location :x x :y y)
		       		    (rest (agent$draw-function agent-in-loc))))
                     (if (grid-element$draw-function grid-element)
			(apply (first (grid-element$draw-function grid-element))
			       (list* (make-location :x x :y y)
		       		    (rest (grid-element$draw-function grid-element))))
                         #\.)))
          (when agent-in-loc (setf agents (remove agent-in-loc agents)))))
      (terpri))
    (terpri) (terpri)))

;;; ---------------------------------------------------------------------------

(defun find-agent-in-location-at-time (agent-list x y time)

  "FIND-AGENT-IN-LOCATION-AT-TIME agent-list x y time
Returns first agent in agent-list at x,y location at time."

  (cond ((null agent-list) nil)
        ((and (= x (location$x (find-agent-location (first agent-list) time)))
              (= y (location$y (find-agent-location (first agent-list) time))))
         (first agent-list))
        (t (find-agent-in-location-at-time (rest agent-list) x y time))))

;;; ---------------------------------------------------------------------------

(defun handle-event (event-type agent-list)

  "HANDLE-EVENT event-type agent-list
Pass agent-list to the function associated with event-type in the *event-function-list*."

  (let ((event-function (assoc event-type *event-function-list*)))
    (when event-function
      (funcall (rest event-function) agent-list))))

;;; ***************************************************************************
