;; agents.lisp

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.

;;;            This work has been sponsored in part by:
;;;               the NSF (IRI-9010645, IRI-9015423)
;;;         the University of Michigan Rackham Graduate School
;;;

;  THE CODE IN THIS FILE REPRESENTS AN ACCUMULATION OF SOME RELATIVELY SIMPLE
;  MICE AGENT FUNCTIONS WRITTEN OVER THE YEARS.  SOME OF THESE MIGHT NOT BE
;  FULLY COMPATIBLE WITH THE CURRENT MICE SYSTEM.  USE WITH CAUTION!!!!!!!

;(in-package 'MICE)

;(export '(find-direction))

;(use-package '(umass-extended-lisp))

(proclaim '(optimize (speed 3) (safety 1)))

;;; --*--
;;; ***************************************************************************

(defun default-agent (agent)
  (declare (ignore agent))
  :QUIESCENT)

#+TI
(defun read-mice-char ()
  (read-char))

#-TI
(defun read-mice-char ()
  (let ((c (read-char))			;; read the useful char
	(newline (read-char))) 		;; read the newline
    (declare (ignore newline))
    c))

;;; ***************************************************************************
;;; null agent never moves.

(defun null-agent (agent) 
  (declare (ignore agent)) 
  '((:QUIESCENT)))

;;; ***************************************************************************
#+TI
(defun human-move-only-agent (agent)
  (format t "~%Select direction for ~a " (agent$name agent))
  (loop
    (let ((response (char-upcase (read-mice-char))))
      (format t "~a~%" response)
      (case response
        ((#\N #\up-arrow)    (return `((:MOVE :NORTH 0))))
        ((#\S #\down-arrow)  (return `((:MOVE :SOUTH 0))))
        ((#\E #\right-arrow) (return `((:MOVE :EAST 0))))
        ((#\W #\left-arrow)  (return `((:MOVE :WEST 0))))
        (#\G                 (return (if (and (compute-new-location (agent$location agent) (agent$orientation agent))
                                              (grid-element$agents
                                                (get-grid-element
                                                  (compute-new-location (agent$location agent) (agent$orientation agent))
                                                  t)))
                                         `((:LINK ,(agent$name
                                                     (first (grid-element$agents
                                                              (get-grid-element
                                                                (compute-new-location (agent$location agent)
                                                                                      (agent$orientation agent))))))
                                              :FRONT))
                                         `((:REASONING 5)))))
        (#\U                 (return (if (agent$sub-components agent)
                                         `((:UNLINK ,(agent$name (first (first (agent$sub-components agent))))))
                                         `((:REASONING 5)))))
        (#\R                 (return `((:ROTATE :RIGHT 1))))
        (#\L                 (return `((:ROTATE :LEFT 1))))
        (#\                  (return `((:MOVE NIL))))
        (#\Q                 (return :STOP))
        (otherwise
         (format t "~%Illegal entry.~%  Use arrow keys to move~%  <SPACE> to stay put~%  G to Grasp~%  U to Ungrasp~%  R to rotate Right~%  L to rotate Left~%  Q to Quit "))))))

#-TI
(defun human-move-only-agent (agent)
  (format t "~%Select direction for ~a " (agent$name agent))
  (loop
    (let ((response (char-upcase (read-mice-char))))
      (format t "~a~%" response)
      (case response
        ((#\N)    (return `((:MOVE :NORTH 0))))
        ((#\S)  (return `((:MOVE :SOUTH 0))))
        ((#\E) (return `((:MOVE :EAST 0))))
        ((#\W)  (return `((:MOVE :WEST 0))))
        (#\                  (return `((:MOVE nil 0))))
        (#\Q                 (return :STOP))
        (otherwise (format t "~%Illegal entry.  Use arrow keys, <SPACE> to stay put, or Q to quit "))))))

#+TI
(defun remote-agent (agent)
  (let* ((host     (rest (assoc :REMOTE-HOST (agent$domain-variables agent))))
         (response (char-upcase
                     (request-slave-char
                       host
                       (format nil "Select direction for ~a "
                               (agent$name agent))))))
    (loop
      (case response
        ((#\N #\up-arrow)    (return `((:MOVE :NORTH 0))))
        ((#\S #\down-arrow)  (return `((:MOVE :SOUTH 0))))
        ((#\E #\right-arrow) (return `((:MOVE :EAST 0))))
        ((#\W #\left-arrow)  (return `((:MOVE :WEST 0))))
        (#\G                 (return (if (and (compute-new-location (agent$location agent) (agent$orientation agent))
                                              (grid-element$agents
                                                (get-grid-element
                                                  (compute-new-location (agent$location agent) (agent$orientation agent))
                                                  t)))
                                         `((:LINK ,(agent$name
                                                     (first (grid-element$agents
                                                              (get-grid-element
                                                                (compute-new-location (agent$location agent)
                                                                                      (agent$orientation agent))))))
                                              :FRONT))
                                         `((:REASONING 5)))))
        (#\U                 (return (if (agent$sub-components agent)
                                         `((:UNLINK ,(agent$name (first (first (agent$sub-components agent))))))
                                         `((:REASONING 5)))))
        (#\R                 (return `((:ROTATE :RIGHT 1))))
        (#\L                 (return `((:ROTATE :LEFT 1))))
        (#\                  (return `((:REASONING 5))))
        (#\Q                 (return :STOP))
        (otherwise
         (format t "~%Illegal entry.~%  Use arrow keys to move~%  <SPACE> to stay put~%  G to Grasp~%  U to Ungrasp~%  R to rotate Right~%  L to rotate Left~%  Q to Quit "))))))

#-TI
(defun remote-agent (agent)
  (let* ((host     (rest (assoc :REMOTE-HOST (agent$domain-variables agent))))
         (response (char-upcase
                     (request-slave-char
                       host
                       (format nil "Select direction for ~a "
                               (agent$name agent))))))
    (loop
      (case response
        ((#\N)    (return `((:MOVE :NORTH 0))))
        ((#\S)  (return `((:MOVE :SOUTH 0))))
        ((#\E) (return `((:MOVE :EAST 0))))
        ((#\W) (return `((:MOVE :WEST 0))))
        (#\                  (return `((:MOVE nil 0))))
        (#\Q                 (return :STOP))
        (otherwise (setf response
                         (char-upcase
                           (request-slave-char
                             host
                             "Illegal entry.  Use arrow keys, <SPACE> to stay put, or Q to quit "))))))))

(defun print-agents-and-locations nil
  (dolist (agent *agent-schedule-queue*)
    (format 't "~%~a ~a" (agent$name agent) (agent$location agent)))
  (format 't "~%"))

(defun print-agent (agent)
  (format 't "~%agent ~a: location  ~a  orientation   ~a ~%" agent (agent$location agent) (agent$orientation agent))
  (format 't "          authority ~a  subcomponents ~a  super-component ~a~%" 
             (agent$authority agent) (agent$sub-components agent) (agent$super-component agent)))

(defun human-agent (agent)
  (format t "~%Select command for agent ~a~%" (agent$name agent))
  (format t "Move, Affect, Link, Unlink, Scan, Reason, sTop, Direction, Quiescent, or Other ")
  (let ((response (char-upcase (read-mice-char))))
    (cond ((eql response #\M)
           (format t "~%Select direction to move (N, S, E, W). ")
           (setf response (char-upcase (read-mice-char)))
           (format t "~a~%" response)
           (cond ((eql response #\N)
                  `((:MOVE :NORTH 0)))
                 ((eql response #\S)
                  `((:MOVE :SOUTH 0)))
                 ((eql response #\E)
                  `((:MOVE :EAST 0)))
                 ((eql response #\W) 
                  `((:MOVE :WEST 0)))		;
                 (t `((:MOVE nil 0)))))

          ((eql response #\L)
           (format t "~%Select link type: Front, Left, Right, Back, Next-to or Shared-loc. ")
           (let* ((response (char-upcase (read-mice-char)))
                  (link-type (case response
                               (#\F :FRONT)
                               (#\L :LEFT)
                               (#\R :RIGHT)
                               (#\B :BACK)
                               (#\N :NEXT-TO)
                               (#\S :SHARED-LOC)
                               (otherwise (cerror "Default to :NEXT-TO." "Illegal link type selected.")
                                          :NEXT-TO))))
             (format t "~%Enter agent to create ~a link with. " link-type)
             (setf response (string-upcase (read-line)))
             ;`((:LINK ,(intern response 'mice) ,link-type))))
             `((:LINK ,(intern response 'user) ,link-type))))

          ((eql response #\A)
           (format t "~%Relative or Absolute location? ")
           (setf response (char-upcase (read-mice-char)))
           (format t "~a~%" response)
           (cond ((eql response #\R)
                  (format t "~%Enter N,S,E,W, or H)ere ")
                  (setf response (char-upcase (read-mice-char)))
                  (let ((direction (case response
                                     (#\N :NORTH)
                                     (#\E :EAST)
                                     (#\S :SOUTH)
                                     (#\W :WEST)
                                     (#\H :HERE))))
		    `((:AFFECT ,direction (:DRAW-FUNCTION :FILLED-RECTANGLE
                                           :FEATURES ,(acons :BLOCKED-TYPES (list :ALL) nil))))))))

          ((eql response #\U)
           (format t "~%Enter agent to unlink from. ")
           (setf response (read-line))
           ;`((:UNLINK ,(intern (string-upcase response) 'mice))))
           `((:UNLINK ,(intern (string-upcase response) 'user))))

          ((eql response #\S)
           (format t "~%Enter time spent scanning. ")
           (setf response (read-line))
           `((:SCAN ,(string-to-int response))))

          ((eql response #\R)
           (format t "~%Enter time spent reasoning. ")
           (setf response (read-line))
           `((:REASONING ,(string-to-int response))))

          ((eql response #\T)
           (format t "~%Agent terminating.~%")
           :STOP)

          ((eql response #\Q)
           (format t "~%Agent quiescent.~%")
           :QUIESCENT)

          ((eql response #\D)
           (format t "~%Enter direction of rotation. (Right, Left) ")
           (let ((direction (if (eql (char-upcase (read-mice-char)) #\L) :LEFT :RIGHT)))
             (format t "~%Enter number of quadrants to rotate ~a. " direction)
             `((:ROTATE ,direction ,(- (char-int (read-mice-char)) (char-int #\0))))))

          ((eql response #\O)
           (format t "~%Enter list of commands: ")
           `,(read))

          (t (format t "~%Illegal selection.~%")))))

(defun string-to-int (str)
  (let* ((char-offset (char-int #\0))
         (tens-factor 1)
         (int 0)
         (str (string-trim '(#\Space #\Tab #\Newline) str))
         (str-len-1 (1- (length str))))
    (dotimes (position (1+ str-len-1) int)
      (setf int (+ int (* tens-factor (- (char-int (aref str (- str-len-1 position))) char-offset))))
      (setf tens-factor (* tens-factor 10)))
    int))

;;; ---------------------------------------------------------------------------

(defun shy-agent (agent)
  (let ((human-locs (find-human-locations agent)))
    (if human-locs
        (let* ((walls (list (make-location :X (region$x-min (simulation-data$overall-region *simulation-data*))
                                           :Y (location$y (agent$location agent)))
                            (make-location :X (region$x-max (simulation-data$overall-region *simulation-data*))
                                           :Y (location$y (agent$location agent)))
                            (make-location :X (location$x (agent$location agent))
                                           :Y (region$y-max (simulation-data$overall-region *simulation-data*)))
                            (make-location :X (location$x (agent$location agent))
                                           :Y (region$y-min (simulation-data$overall-region *simulation-data*)))))
               (best-direction (find-furthest-move-direction agent (append human-locs walls))))
          `((:MOVE ,best-direction)))
        `((:SCAN :ALL)))))

(defun find-human-locations (agent)
  (declare (ignore agent))
  (let ((scan-data (read-and-reset-scanned-data))
        (found-locations nil))
    (when scan-data
      (dolist (grid-descr scan-data)
        (when (agent-type-at-loc-p grid-descr :HUMAN)
          (setf found-locations (cons (grid-description$coordinates grid-descr) found-locations))))
      found-locations)))

(defun agent-type-at-loc-p (grid-descr agt-type)
  (some #'(lambda (agt) (eql (agent$type agt) agt-type))
        (grid-description$agents grid-descr)))

(defun find-furthest-move-direction (agt locs-to-avoid)
  (let ((best-dir nil)
        (best-value 0))
    (dolist (dir '(:NORTH :SOUTH :EAST :WEST))
      (let* ((new-loc (compute-new-location (agent$location agt) dir))
             (new-value (shortest-dist new-loc locs-to-avoid)))
        (when (legal-location-p new-loc :AGENT agt)
          (cond ((> new-value  best-value)
                 (setf best-value new-value)
                 (setf best-dir dir))
                ((= new-value best-value)
                 (when (> 5 (random 10))
                   (setf best-dir dir)))
                (t nil)))))
    best-dir))

(defun dist-sum (from-loc to-locs)
  (let ((total 0))
    (dolist (to-loc to-locs)
      (setf total (+ (abs (- (location$x from-loc) (location$x to-loc)))
                     (abs (- (location$y from-loc) (location$y to-loc)))
                     total)))
    total))

(defun shortest-dist (from-loc to-locs)
  (let ((shortest 10000)
        (new-dist 0))
    (dolist (to-loc to-locs)
      (setf new-dist (+ (abs (- (location$x from-loc) (location$x to-loc)))
                        (abs (- (location$y from-loc) (location$y to-loc)))))
      (when (< new-dist shortest)
        (setf shortest new-dist)))
    shortest))

;;; ---------------------------------------------------------------------------

(defvar *simple-agents* nil)

(defvar *communication-delay* 1)

(defxstruct (simple-agent :EXPORT (:CONC-NAME "SIMPLE-AGENT$") (:PRINT-FUNCTION simple-agent-print-function))
  (name                nil)
  (goals               nil            :TYPE list)
  (old-goals           nil            :TYPE list)
  (goal-priority-function :NEARNESS)
  (world-model         nil)                     ; this will be a structure.
  (last-scan-time      nil)
  (scan-interval       1             :TYPE integer)
  (organizational-model nil)
  (message-buffer      nil)
  )

(defun simple-agent-print-function (agent stream print-level)
  (declare (ignore print-level))
  (format stream "~A" (simple-agent$name agent)))

;;;   A goal object indicates a potential goal position for an agent,
;;;   the time that the position was determined, the priority of getting
;;;   there, and the type of objective the goal represents.

(defstruct (goal-object (:CONC-NAME "GOAL-OBJECT$"))
  (location (make-location)     :TYPE location)
  (region   (make-region)       :TYPE region)
  (time     *current-time*      :TYPE integer)
  (priority 0                   :TYPE integer)
  (last-pursued-time 0)
  (goal-description nil)
  (type nil))

(defstruct (simple-agent-model (:CONC-NAME "SIMPLE-AGENT-MODEL$"))
  (location      (make-location)     :TYPE location)
  (time          *current-time*       :TYPE integer)
  (type          nil)
  (captured-by-types nil)
  (characteristics nil))

(defun create-new-simple-agent (mice-agent)
  (let ((goal-priority (second (assoc :GOAL-PRIORITY-FUNCTION (agent$domain-variables mice-agent)))))
    (make-simple-agent :NAME (agent$name mice-agent)
                       :GOALS (second (assoc :GOALS (agent$domain-variables mice-agent)))
                       :GOAL-PRIORITY-FUNCTION (if goal-priority goal-priority :NEARNESS))))

(defun simple-agent (mice-agent)
  (let ((top-goal t)
        (commands nil)
        (agent (rest (assoc (agent$name mice-agent) *simple-agents*))))
    (unless agent
      (setf agent (create-new-simple-agent mice-agent))
      (setf *simple-agents* (acons (agent$name mice-agent) agent *simple-agents*)))
    (do ()
        ((or (not top-goal) commands))
      (setf commands (append commands (revise-goals agent)))
      (unless commands
        (multiple-value-bind (current-top-goal current-commands)
            (pursue-top-goal agent)
          (setf top-goal current-top-goal)
          (setf commands (append commands current-commands)))))
    commands))

(defun revise-goals (agent)
  (let ((scanned-data (read-and-reset-scanned-data)))
    (cond (scanned-data
           (let ((time-to-assimilate-info (revise-model-and-goals agent scanned-data)))
             (unless (zerop time-to-assimilate-info)
               `((:REASONING ,time-to-assimilate-info)))))
          ((time-to-scan-p agent) `((:SCAN :ALL)))
          (t nil))))

(defun pursue-top-goal (agent)
  ; Revised to return top-goal and a list of commands
  (let ((top-goal (find-top-goal agent)))
    (if top-goal
        (case (goal-object$type top-goal)
          (:MOVE (values top-goal (list (get-command-for-move-toward-goal agent top-goal))))
          (:RANDOM-MOVE (values top-goal (list (get-command-for-random-move agent))))
          (t (cerror "Undefined goal type ~a~%" top-goal)))
        (values nil `((:NULL-ACTION))))))

(defun get-command-for-move-toward-goal (agent goal)
  (let* ((old-location (get-agent-location agent))
         (direction    (find-direction old-location (when goal (goal-object$location goal)) agent)))
    `(:MOVE ,direction)))

(defun get-command-for-random-move (agent)
  (let* ((blocked-directions (find-blocked-directions agent (get-agent-location agent)))
         (free-directions (ordered-set-difference '(:NORTH :SOUTH :EAST :WEST) blocked-directions))
         (direction    (when free-directions (nth (random (list-length free-directions)) free-directions))))
    `(:MOVE ,direction)))

(defun get-agent-location (agent)
  ; technically, this should be based on the agent's perception of itself
  ; but for now, assume perfect information by looking it up
  (declare (ignore agent))
  (agent$location *current-agent*))

(defun find-direction (old-loc new-loc agent)
  "Returns the most preferable direction to go from old-loc to new-loc for agent"
  (when (and new-loc (not (equalp old-loc new-loc)))
    (let ((blocked-directions (find-blocked-directions agent old-loc)))
      (if (not *add-randomness*)
          (if (>= (abs (- (location$x old-loc) (location$x new-loc)))
                  (abs (- (location$y old-loc) (location$y new-loc))))
              (if (minusp (- (location$x old-loc) (location$x new-loc)))
                  (first (ordered-set-difference '(:EAST) blocked-directions))
                  (first (ordered-set-difference '(:WEST) blocked-directions)))
              (if (minusp (- (location$y old-loc) (location$y new-loc)))
                  (first (ordered-set-difference '(:SOUTH) blocked-directions))
                  (first (ordered-set-difference '(:NORTH) blocked-directions))))
          ; try some tricky randomness to this, giving preference to making
          ; up the farther distance
          (let ((total (+ (abs (- (location$x old-loc) (location$x new-loc)))
                          (abs (- (location$y old-loc) (location$y new-loc))))))
            (first (ordered-set-difference
                     (if (<= (1+ (random total)) (abs (- (location$x old-loc) (location$x new-loc))))
                         (if (minusp (- (location$x old-loc) (location$x new-loc)))
                             (if (minusp (- (location$y old-loc) (location$y new-loc)))
                                 '(:EAST :SOUTH :NORTH :WEST)
                                 '(:EAST :NORTH :SOUTH :WEST))
                             (if (minusp (- (location$y old-loc) (location$y new-loc)))
                                 '(:WEST :SOUTH :NORTH :EAST)
                                 '(:WEST :NORTH :SOUTH :EAST)))
                         (if (minusp (- (location$y old-loc) (location$y new-loc)))
                             (if (minusp (- (location$x old-loc) (location$x new-loc)))
                                 '(:SOUTH :EAST :WEST :NORTH)
                                 '(:SOUTH :WEST :EAST :NORTH))
                             (if (minusp (- (location$x old-loc) (location$x new-loc)))
                                 '(:NORTH :EAST :WEST :SOUTH)
                                 '(:NORTH :WEST :EAST :SOUTH))))
                     blocked-directions)))))))

(defun find-blocked-directions (agent old-loc)
  (when *cautious?*
    (let ((blocked-directions nil))
      (dolist (direction '(:NORTH :SOUTH :EAST :WEST))
        (when (not (legal-and-not-occupied-p (compute-new-location old-loc direction)
                                             (simple-agent$world-model agent)
                                             ; assume agent knows the same blocked-by-types as MICE
                                             (agent$blocked-by-types *current-agent*)))
          (setf blocked-directions (cons direction blocked-directions))))
      blocked-directions)))

(defun find-top-goal (agent)
  (find-top-priority-goal (simple-agent$goals agent)))

(defun find-top-priority-goal (goals)
  (when goals
    (if *add-randomness*
        (find-top-priority-goal-with-randomness goals goals)
        (find-top-priority-goal-exactly (rest goals) (list (first goals))))))

(defun find-top-priority-goal-exactly (goals best)
  (cond ((null goals) (nth (random (length best)) best))
        ((< (goal-object$priority (first goals)) (goal-object$priority (first best)))
         (let ((new-best (first goals)))
           (find-top-priority-goal-exactly (rest goals) (list new-best))))
        ((= (goal-object$priority (first goals)) (goal-object$priority (first best)))
         (setf best (cons (first goals) best))
         (find-top-priority-goal-exactly (rest goals) best))
        (t (find-top-priority-goal-exactly (rest goals) best))))

(defun find-top-priority-goal-with-randomness (goals all-goals)
  (cond ((null all-goals) nil)
        ((null goals) (find-top-priority-goal-with-randomness all-goals all-goals))
        ((zerop (goal-object$priority (first goals))) (first goals))
        ((zerop (random (goal-object$priority (first goals))))
         (first goals))
        (t (find-top-priority-goal-with-randomness (rest goals) all-goals))))

(defun find-nearest-goal (start-loc goals best)
  (cond ((null goals) best)
        ((null best) (find-nearest-goal start-loc (rest goals) (first goals)))
        ((first-closer-to-second-than-third-p (goal-object$location (first goals)) (goal-object$location best) start-loc)
         (find-nearest-goal start-loc (rest goals) (first goals)))
        ((not (first-closer-to-second-than-third-p (goal-object$location best)
                                                   (goal-object$location (first goals))
                                                   start-loc))
         (find-nearest-goal start-loc (rest goals) (if (zerop (random 2)) (first goals) best)))
        (t (find-nearest-goal start-loc (rest goals) best))))

(defun first-closer-to-second-than-third-p (first second third)
  ;  because agents cannot move diagonally, just sum the x and y
  (< (+ (abs (- (location$x first) (location$x third)))
        (abs (- (location$y first) (location$y third))))
     (+ (abs (- (location$x second) (location$x third)))
        (abs (- (location$y second) (location$y third))))))

(defun revise-model-and-goals (agent scanned-data)
  ; this function will evolve
  ; for now, let's worry about blue agents
  ;  1. scan for other agents
  ;  2. make goals for every red agent
  (cond ((eq (agent$type *current-agent*) :BLUE)
         (setf (simple-agent$world-model agent) (update-world-model agent scanned-data))
         (setf (simple-agent$last-scan-time agent) *current-time*)
         (make-goals-for-red-agents agent (simple-agent$world-model agent)))
        (t 0)))

(defun make-goals-for-red-agents (agent seen-agents)
  (let ((red-agents (return-red-agents seen-agents)))
    (when red-agents
      (dolist (ra red-agents)
        (create-or-update-goals agent ra))))
  (remove-obsolete-goals agent)
  0) ; the time could be some function of the number of red-agents seen....

(defun return-red-agents (ra-list)
  (cond ((null ra-list) nil)
        ((eq (simple-agent-model$type (first ra-list)) :RED)
         (cons (first ra-list) (return-red-agents (rest ra-list))))
        (t (return-red-agents (rest ra-list)))))

(defun create-or-update-goals (agent goal-agent)
  (let ((goal-descriptions (goal-descriptions-for-agent-toward-goal-agent agent goal-agent)))
    (dolist (goal-description goal-descriptions)
      (let ((old-goal (find-existing-goal-for-goal-description (simple-agent$goals agent) goal-description)))
        (if old-goal
            (update-old-goal old-goal goal-agent agent)
            (setf (simple-agent$goals agent)
                  (cons (create-new-goal goal-agent agent :DESCRIPTION goal-description)
                        (simple-agent$goals agent))))))))

(defun remove-obsolete-goals (agent)
  (let ((current-goals nil)
        (obsolete-goals nil)
        (last-scan-time (simple-agent$last-scan-time agent)))
    (when last-scan-time
      (dolist (goal (simple-agent$goals agent))
        (if (= (goal-object$time goal) last-scan-time)
            (setf current-goals (cons goal current-goals))
            (setf obsolete-goals (cons goal obsolete-goals)))
        )
      (setf (simple-agent$goals agent) current-goals)
      (setf (simple-agent$old-goals agent) obsolete-goals))))

(defun find-existing-goal-for-goal-description (old-goals goal-description)
  (cond ((null old-goals) nil)
        ((equalp goal-description (goal-object$goal-description (first old-goals)))
         (first old-goals))
        (t (find-existing-goal-for-goal-description (rest old-goals) goal-description))))

(defun update-old-goal (old-goal goal-agent current-agent)
  (setf (goal-object$location old-goal)
        (generate-goal-location-from-agent-and-description goal-agent (goal-object$goal-description old-goal)))
  (setf (goal-object$time old-goal) *current-time*)
  (setf (goal-object$priority old-goal) (compute-goal-priority old-goal current-agent))
  )
  
(defun create-new-goal (goal-agent current-agent &KEY description)
  (let ((new-goal (make-goal-object
                   :LOCATION (generate-goal-location-from-agent-and-description
                              goal-agent description)
                   :TIME     *current-time*
                   :TYPE     :MOVE
                   :GOAL-DESCRIPTION description)))
    (setf (goal-object$priority new-goal) (compute-goal-priority new-goal current-agent))
    new-goal))

(defun check-messages (agent)
  (when (simple-agent$message-buffer agent)
    (let ((new-buffer nil))
      (dolist (msg (simple-agent$message-buffer agent))
        (if (<= (first msg) *current-time*) ; timestamp indicates arrival
            (let ((org-for-prey (rest (assoc (first (second msg)) (simple-agent$organizational-model agent)))))
              (setf org-for-prey
                    (update-organization-for-prey
                      org-for-prey
                      (third msg)
                      (second msg)
                      (fourth msg)))
              (setf (simple-agent$organizational-model agent)
                    (my-acons (first (second msg)) org-for-prey (simple-agent$organizational-model agent))))
            (setf new-buffer (cons msg new-buffer))))
      (setf (simple-agent$message-buffer agent) (nreverse new-buffer)))))

(defun my-acons (key datum alist)
  "Like acons, but doesn't simply push new key-datum pair on - replaces old one if it exists"
  (let ((entry (assoc key alist)))
    (if entry
        (progn (setf (rest entry) datum) alist)
        (acons key datum alist))))

(defun compute-goal-priority (goal agent)
  (case (simple-agent$goal-priority-function agent)
    (:RECENCY (goal-object$last-pursued-time goal))
    (:NEARNESS (+ (abs (- (location$x (goal-object$location goal)) (location$x (get-agent-location agent))))
                  (abs (- (location$y (goal-object$location goal)) (location$y (get-agent-location agent))))))
    (:NEAR-AND-RECENT (+ (goal-object$last-pursued-time goal)
                         (abs (- (location$x (goal-object$location goal)) (location$x (get-agent-location agent))))
                         (abs (- (location$y (goal-object$location goal)) (location$y (get-agent-location agent))))))
    (:DISTRIBUTED-CONTROL
     (check-messages agent)
     (let* ((prey (first (goal-object$goal-description goal)))
            (org-for-prey (rest (assoc prey (simple-agent$organizational-model agent))))
            (distance (+ (abs (- (location$x (goal-object$location goal)) (location$x (get-agent-location agent))))
                         (abs (- (location$y (goal-object$location goal)) (location$y (get-agent-location agent)))))))
       (communicate-goal-to-others agent (goal-object$goal-description goal) distance)
       (setf org-for-prey 
             (update-organization-for-prey org-for-prey agent (goal-object$goal-description goal) distance))
       (setf (simple-agent$organizational-model agent)
             (my-acons prey org-for-prey (simple-agent$organizational-model agent)))
       (compute-goal-priority-from-organization (goal-object$goal-description goal) agent org-for-prey)))))

; Note - this function still only approximates the correct functionality

(defun compute-goal-priority-from-organization (goal agent sorted-list)
  (let ((current-best-worst (find-best-worst-goal sorted-list)))
    (if (equalp agent (third current-best-worst))
        (if (equalp goal (first current-best-worst))
            1
            0)
        (compute-goal-priority-from-organization
          goal
          agent
          (remove-agents-and-matching-goals (third current-best-worst) (second (first current-best-worst)) sorted-list)))))

(defun remove-agents-and-matching-goals (agent goal-side goal-list)
  (cond ((null goal-list) nil)
        ((equalp agent (third (first goal-list)))
         (remove-agents-and-matching-goals agent goal-side (rest goal-list)))
        ((equalp goal-side (second (first (first goal-list))))
         (remove-agents-and-matching-goals agent goal-side (rest goal-list)))
        (t (cons (first goal-list)
                 (remove-agents-and-matching-goals agent goal-side (rest goal-list))))))

(defun find-best-worst-goal (goal-list)
  (let ((best-worst nil)
        (seen-agents nil))
    (dolist (goal goal-list)
      (unless (member (third goal) seen-agents)
        (setf seen-agents (cons (third goal) seen-agents))
        (setf best-worst goal)))
    best-worst))

(defun update-organization-for-prey (sorted-list agent goal distance)
  (sort (insert-or-modify-goal-distance-agent goal distance agent sorted-list)
        #'(lambda (gda1 gda2)
            (< (second gda1) (second gda2)))))

(defun communicate-goal-to-others (agent goal distance)
  (dolist (to-agent *simple-agents*)
    (unless (equalp (first to-agent) (simple-agent$name agent))
      (setf (simple-agent$message-buffer (rest to-agent))
            (append (simple-agent$message-buffer (rest to-agent))
                    (list (list (+ *current-time* *communication-delay*) goal agent distance)))))))

(defun insert-or-modify-goal-distance-agent (goal distance agent old-stuff)
  (cond ((null old-stuff) (list (list goal distance agent)))
        ((and (equalp goal (first (first old-stuff)))
              (equalp agent (third (first old-stuff))))
         (setf (second (first old-stuff)) distance)
         old-stuff)
        (t (cons (first old-stuff)
                 (insert-or-modify-goal-distance-agent goal distance agent (rest old-stuff))))))

(defun generate-goal-location-from-agent-and-description (goal-agent description)
  (if description
      (compute-new-location (simple-agent-model$location goal-agent) (second description))
      (simple-agent-model$location goal-agent)))

(defun goal-descriptions-for-agent-toward-goal-agent (agent goal-agent)
  ; generally, we want to work toward one of the sides of the goal agent
  ; so we make a goal for each side
  ; exceptions:  when we are far away from the agent, only work toward the center unless it is already trapped
  ;              when another blue-agent is already on the side, don't go for it (unless we are that agent)
  (cond ((not goal-agent) nil)
        ((far-apart-p (get-agent-location agent) (simple-agent-model$location goal-agent))
         (when (agent-not-surrounded-p goal-agent (simple-agent$world-model agent))
           (list (list (simple-agent-model$characteristics goal-agent) :CENTER))))
        (t (build-goal-description-for-directions agent goal-agent '(:NORTH :SOUTH :EAST :WEST)))))

(defun agent-not-surrounded-p (simple-agent-model all-simple-agent-models)
  (let ((location (simple-agent-model$location simple-agent-model)))
    (or (legal-and-not-occupied-p
          (compute-new-location location :NORTH) all-simple-agent-models (simple-agent-model$captured-by-types simple-agent-model))
        (legal-and-not-occupied-p
          (compute-new-location location :SOUTH) all-simple-agent-models (simple-agent-model$captured-by-types simple-agent-model))
        (legal-and-not-occupied-p
          (compute-new-location location :EAST)  all-simple-agent-models (simple-agent-model$captured-by-types simple-agent-model))
        (legal-and-not-occupied-p
          (compute-new-location location :WEST)  all-simple-agent-models (simple-agent-model$captured-by-types simple-agent-model)))))

(defun legal-and-not-occupied-p (location simple-agent-model-list blocking-agent-types)
  (and (legal-location-p location)
       (notany #'(lambda (am)
                   (and (equalp location (simple-agent-model$location am))
                        (member (simple-agent-model$type am) blocking-agent-types)))
               simple-agent-model-list)))

;;; NOTE:  THIS CONTAINS A CONSTANT THAT SHOULD BE A SPECIAL GLOBAL!

(defun far-apart-p (loc1 loc2)
  (> (+ (abs (- (location$x loc1) (location$x loc2)))
        (abs (- (location$y loc1) (location$y loc2))))
     30))


(defun time-to-scan-p (agent)
  ; this will have to be based on perception goals
  (or (not (simple-agent$goals agent))
      ; for now, find the last time scanned
      (or (not (simple-agent$last-scan-time agent))
          (>= (- *current-time* (simple-agent$last-scan-time agent))
              (simple-agent$scan-interval agent)))))

(defun find-most-recent-goal-creation-time (goals &optional time)
  (cond ((null goals) time)
        ((not time) (find-most-recent-goal-creation-time 
                     (rest goals) (goal-object$time (first goals))))
        ((> (goal-object$time (first goals)) time)
         (find-most-recent-goal-creation-time (rest goals) (goal-object$time (first goals))))
        (t (find-most-recent-goal-creation-time (rest goals) time))))

(defun create-or-update-simple-agent-model (agent agent-to-model)
  (let ((old-simple-agent-model (simple-agent-model-matches-agent agent-to-model (simple-agent$world-model agent))))
    (if old-simple-agent-model (update-simple-agent-model old-simple-agent-model agent-to-model)
        (create-simple-agent-model agent-to-model))))

(defun simple-agent-model-matches-agent (agent-to-model simple-agent-models)
  (some #'(lambda (am) (when (equalp (simple-agent-model$characteristics am) 
                                     (find-agent-characteristics agent-to-model))
                         am))
        simple-agent-models))

(defun update-simple-agent-model (model agent)
  (setf (simple-agent-model$location model) (find-agent-location agent))
  (setf (simple-agent-model$time model) *current-time*)
  model)

(defun create-simple-agent-model (agent)
  (make-simple-agent-model :LOCATION (find-agent-location agent *current-time*)
                    :TIME     *current-time*
                    :CHARACTERISTICS (find-agent-characteristics agent)
                    :TYPE (agent$type agent)
                    :CAPTURED-BY-TYPES (agent$captured-by-types agent)))
                 
;;;  NOTE:  We drop goals that are currently satisfied by some blue-agent (except if we are that agent)
;;;  if that blue-agent moves, then a new goal will be formed ...

(defun build-goal-description-for-directions (agent goal-agent directions)
  (cond ((null directions) nil)
        (t (let ((new-location (compute-new-location (simple-agent-model$location goal-agent) (first directions))))
             (cond ((and new-location
                         (or (legal-and-not-occupied-p new-location
                                                       (simple-agent$world-model agent)
                                                       (simple-agent-model$captured-by-types goal-agent))
                             (and (legal-location-p new-location)
                                  ; it's I!
                                  (equalp (get-agent-location agent) new-location))))
;                        (let ((blue-agent-there 
;                               (blue-agent-in-location (simple-agent$world-model agent) new-location)))
;                          (or (not blue-agent-there)
;                              (simple-agent-model-matches-agent agent (list blue-agent-there)))))
                    (cons (list (simple-agent-model$characteristics goal-agent) (first directions))
                          (build-goal-description-for-directions agent goal-agent (rest directions))))
                   (t (build-goal-description-for-directions agent goal-agent (rest directions))))))))

(defun blue-agent-in-location (world-model new-location)
  (cond ((null world-model) nil)
        ((and (eq (simple-agent-model$type (first world-model)) :BLUE)
              (equalp (simple-agent-model$location (first world-model)) new-location))
         (first world-model))
        (t (blue-agent-in-location (rest world-model) new-location))))

(defun update-world-model (agent scanned-data)
  (let ((found-agents nil)
        (model-agents nil))
    (dolist (grid-descr scanned-data)
      (setf found-agents (append (live-agents (grid-description$agents grid-descr)) found-agents)))
    (dolist (new-agent found-agents)
      (setf model-agents (cons (create-or-update-simple-agent-model agent new-agent) model-agents)))
    model-agents))

(defun live-agents (agent-list)
  (cond ((null agent-list) nil)
        (t (let ((agent (first agent-list)))
             (if (not (member (agent-state$status (current-state agent))
                              (list :INACTIVATED :REMOVED)))
                 (cons agent (live-agents (rest agent-list)))
                 (live-agents (rest agent-list)))))))

(defun find-agent-characteristics (agent)
  (agent$name agent))

(defun agent-in-region-p (agent region &KEY (time nil))
  (let ((agent-location (if time (find-agent-location agent time) (agent$location agent))))
    (when agent-location (location-in-region-p agent-location region))))


;;; ***************************************************************************
;;; EOF

