;;;; -*- Mode: Fi:Common-Lisp -*-
;;;; 
;;;; $Id$
;;;; $Source$
;;;; 
;;;; Description       : Strategies for the Predator-Prey domain 
;;;;                     using Contract Net protocol
;;;; 
;;;; Original author(s): Jaeho Lee <jaeho@bigtwohearted.engin.umich.edu>
;;;; Organization      : University of Michigan DAI Lab
;;;; Created On        : Sat Jan 11 13:41:10 1992
;;;; Last Modified On  : Mon Jan 13 17:37:55 1992
;;;; Last Modified By  : Jaeho Lee <jaeho@woundedknee.engin.umich.edu>
;;;; 
;;;; Copyright (C) 1992 University of Michigan.
;;;; 
;;;; HISTORY 
;;;; 11-Jan-1992		Jaeho Lee (on bigtwohearted.engin.umich.edu)
;;;;    After the communication capability of MICE has been made, revised
;;;;    and modified by Jaeho Lee.
;;;;    
;;;; 04-Oct-1991		
;;;;     Originally written by Jaeho Lee and Young-Pa So for the project
;;;;     of EECS 598-4 in fall, 1991. 
;;;;    

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.


(defvar *cnet-strategy-list*)

(defvar *wandering-strategy*)
(defvar *surround-strategy*)
(defvar *news-strategy*)
(defvar *relax-strategy*)

;;
;; Wandering strategy
;;

(setf *wandering-strategy*
  (make-cnet-strategy
   :name 'wandering-strategy
   :precondition
   #'(lambda (cognition)
       "Prey is NOT in the sensing range of all four agents"
       (let ((reported-locations (mapcar #'cdr (cognitive$prey-locations cognition))))
	 (null (some #'specified-location-p reported-locations))))
   :task-announcement (make-cnet-task-announcement
		       :task-abstraction 'wandering-strategy
		       :bid-specification '(agent-location) ; List of functions
		       :expiration-time nil)
   :task-bid (make-cnet-task-bid)
   :task-award (make-cnet-task-award
		:task-name 'wandering-strategy
		:task-parameters nil	; (World boundary)
		:role-name nil
		:report-specification (make-cnet-report-specification
				       :report-condition '(prey-found)
				       :report-timeout nil))
   :bid-rating
   #'(lambda (cognition strategy role-name bid-message) ; Bid contains agent- and prey-location (possibly nil)
       "The distance from the agent location to the center of the specified region is the bid measure."
       (declare (ignore cognition))
       (let* ((overall-region (simulation-data$overall-region *simulation-data*))
	      (goal-region (funcall (cnet-strategy-role-parameter strategy) role-name overall-region))
	      (goal-location (center-of goal-region))
	      (agent-location (first (cnet-task-bid-node-abstraction (message$content bid-message)))))
	 (distance-between goal-location agent-location)))
   :task-parameter
   #'(lambda (cognition)
       "Returns the overall world region (region structure)."
       (declare (ignore cognition))
       (list (simulation-data$overall-region *simulation-data*)))
   :roles '(:area1 :area2 :area3 :area4)
   :role-parameter
   #'(lambda (role-name overall-region)
       "Returns assigned region of each role"
       (case role-name
	 (:area1 (quarter-region overall-region 1))
	 (:area2 (quarter-region overall-region 2))
	 (:area3 (quarter-region overall-region 3))
	 (:area4 (quarter-region overall-region 4))
	 (otherwise nil)))))

;;
;; Surround Strategy
;;

(setf *surround-strategy*
  (make-cnet-strategy
   :name 'surround-strategy
   :precondition
   #'(lambda (cognition)
       "At least 1, at most 3 agents have the prey in its sensor range"
       (let* ((reported-locations (mapcar #'cdr (cognitive$prey-locations cognition)))
	      (sense-count (my-count-if #'specified-location-p reported-locations)))
	 (and (>= sense-count 1) (<= sense-count 3))))
   :task-announcement (make-cnet-task-announcement
		       :task-abstraction 'surround-strategy
		       :bid-specification '(prey-relative-location) ; function returns (theta, radius)
		       :expiration-time nil)
   :task-bid (make-cnet-task-bid)
   :task-award (make-cnet-task-award
		:task-name 'surround-strategy
		:task-parameters nil	; (prey-location, distance from the prey)
		:role-name nil
		:report-specification (make-cnet-report-specification
				       :report-condition '(timeout)
				       :report-timeout 5))
   :task-parameter
   #'(lambda (cognition)
       (list (fix-prey-location (cognitive$prey-locations cognition)) 2))
   :bid-rating
   #'(lambda (cognition strategy role-name bid-message) ; Bid contains agent- and prey-location --possibly nil
       "The distance from the agent location to the specified location is the bid measure."
       (let* ((prey-location (fix-prey-location (cognitive$prey-locations cognition)))
	      (goal-location (funcall (cnet-strategy-role-parameter strategy) role-name prey-location))
	      (agent-location (first (cnet-task-bid-node-abstraction (message$content bid-message)))))
	 (distance-between goal-location agent-location)))
   :roles '(:north :east :west :south)
   :role-parameter
   #'(lambda (role-name prey-location)
       "Returns assigned location of each role"
       (compute-boundary-location prey-location role-name))))

;;
;; NEWS Strategy
;;

(setf *news-strategy*
  (make-cnet-strategy
   :name 'news-strategy
   :precondition
   #'(lambda (cognition)
       "Prey is in the sensing range of all four agents"
       (let ((reported-locations (mapcar #'cdr (cognitive$prey-locations cognition))))
	 (>= (my-count-if #'specified-location-p reported-locations) 4)))
   :task-announcement (make-cnet-task-announcement
		       :task-abstraction 'news-strategy
		       :bid-specification '(prey-relative-location) ; function returns (theta, radius)
		       :expiration-time nil)
   :task-bid (make-cnet-task-bid)
   :task-award (make-cnet-task-award
		:task-name 'news-strategy
		:task-parameters nil	; (prey-location, distance from the prey)
		:role-name nil
		:report-specification (make-cnet-report-specification
				       :report-condition '(precond-failure timeout)
				       :report-timeout 5))
   :bid-rating
   #'(lambda (cognition strategy role-name bid-message)
       "The distance from the agent location to the specified location is the bid measure."
       (let* ((prey-location (fix-prey-location (cognitive$prey-locations cognition)))
	      (goal-location (funcall (cnet-strategy-role-parameter strategy) role-name prey-location))
	      (agent-location (first (cnet-task-bid-node-abstraction (message$content bid-message)))))
	 (distance-between goal-location agent-location)))
   :task-parameter
   #'(lambda (cognition)
       (list (fix-prey-location (cognitive$prey-locations cognition)) 1))
   :roles '(:north :east :west :south)
   :role-parameter
   #'(lambda (role-name prey-location)
       "Returns assigned location of each role"
       (compute-boundary-location prey-location role-name))))

;;
;; Relax Strategy
;;

(setf *relax-strategy*
  (make-cnet-strategy
   :name 'relax-strategy
   :precondition
   #'(lambda (cognition)
       "Prey on wall or corener and agents are surrounding it"
       (let ((prey-location (fix-prey-location (cognitive$prey-locations cognition)))
             (reported-locations (mapcar #'cdr (cognitive$prey-locations cognition))))
	 (and (>= (my-count-if #'specified-location-p reported-locations) 4)
	      (wall-side-location-p prey-location))))
   :task-announcement (make-cnet-task-announcement
		       :task-abstraction 'relax-strategy
		       :bid-specification '(prey-relative-location) ; function returns (theta, radius)
		       :expiration-time nil)
   :task-bid (make-cnet-task-bid)
   :task-award (make-cnet-task-award
		:task-name 'relax-strategy
		:task-parameters nil	; (prey-location, distance from the prey)
		:role-name nil
		:report-specification (make-cnet-report-specification
				       :report-condition '(timeout)
				       :report-timeout 3))
   :bid-rating
   #'(lambda (cognition strategy role-name bid-message) ; Bid contains agent- and prey-location --possibly nil
       "The distance from the agent location to the specified location is the bid measure."
       (let* ((prey-location (fix-prey-location (cognitive$prey-locations cognition)))
	      (goal-location (funcall (cnet-strategy-role-parameter strategy) role-name prey-location))
	      (agent-location (first (cnet-task-bid-node-abstraction (message$content bid-message)))))
	 (distance-between goal-location agent-location)))
   :task-parameter
   #'(lambda (cognition)
       (list (fix-prey-location (cognitive$prey-locations cognition)) 3))
   :roles '(:north :east :west :south)
   :role-parameter
   #'(lambda (role-name prey-location)
       "Returns assigned location of each role"
       (compute-boundary-location prey-location role-name))))

(setf *cnet-strategy-list* (list *relax-strategy* *news-strategy* *surround-strategy* *wandering-strategy*))
