;;;; -*- Mode: Fi:Common-Lisp -*-
;;;; 
;;;; $Id$
;;;; $Source$
;;;; 
;;;; Description       : Manger agent for the Predator-Prey doamin
;;;;                     using Contract Net protocol
;;;; 
;;;; Original author(s): Jaeho Lee <jaeho@bigtwohearted.engin.umich.edu>
;;;; Organization      : University of Michigan DAI Lab
;;;; Created On        : Sat Jan 11 13:42:51 1992
;;;; Last Modified On  : Mon Jan 13 17:38:02 1992
;;;; Last Modified By  : Jaeho Lee <jaeho@woundedknee.engin.umich.edu>
;;;; 
;;;; Copyright (C) 1992 University of Michigan.
;;;; 
;;;; HISTORY 
;;;; 11-Jan-1992		Jaeho Lee (on bigtwohearted.engin.umich.edu)
;;;;    After the communication capability of MICE has been made, revised
;;;;    and modified by Jaeho Lee.
;;;;    
;;;; 04-Oct-1991		
;;;;     Originally written by Jaeho Lee and Young-Pa So for the project
;;;;     of EECS 598-4 in fall, 1991. 
;;;;    

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.


(defvar *debug-cnet* "set it to t if want to see what the system is doing")

(defun manager-contractor-agent (agent)
  "The agent which plays the role of manager and contractor together."
  (append (manager-agent agent)	(contractor-agent agent)))

(defun manager-agent (manager)
  "The manager agent of the CNET."
  (if *debug-cnet* (format t "~%Time ~a: ### Manager is activated." *current-time*))
  (if (= *current-time* 0) (cognitive-initialization))
  (let* ((cognition (agent-cognitive manager))
	 (cnet-channel (cognitive$channel cognition))
	 (recv-data (read-and-reset-received-messages :reset-value :empty)))

    (setf (cognitive$commands cognition) nil)

    (if (eq :empty recv-data)
      (push `(:RECV ,cnet-channel) (cognitive$commands cognition))
      (setf (cognitive$messages cognition) (append (cognitive$messages cognition) recv-data)))
    
    (case (cognitive$state cognition)
      ((:build-and-announce-task nil)
       (build-and-announce-task manager cognition))
      (:collect-bids-and-award-tasks
       (collect-bids-and-award-tasks manager cognition))
      (:collect-task-reports
       (collect-task-reports manager cognition))
      (otherwise
       (format t "~%*** Something is wrong with the manager state")))
    
    (reverse (cognitive$commands cognition))))

;;
;; :build-and-announce-task state
;;

(defun build-and-announce-task (manager cognition)
  "Decide best strategy based on the current manager information,
   and announce it."
  (decide-strategy cognition)
  (task-announce manager cognition)
  (format t "~%Time ~a:(Build and announce task: ~a)" 
	  *current-time* (cnet-strategy-name (cognitive$strategy cognition)))
  (setf (cognitive$state cognition) :collect-bids-and-award-tasks))

(defun decide-strategy (cognition)
  "Select the best strategy applicable to current state.
   Note that the decision should be made only based on the
   manager's local information gathered from the contractors."
  (dolist (strategy (cognitive$strategy-list cognition) nil)
    (if (funcall (cnet-strategy-precondition strategy) cognition) 
      (return (setf (cognitive$strategy cognition) strategy)))))

(defun task-announce (manager cognition)
  "Broadcast the task."
  (declare (ignore manager))
  (let ((strategy (cognitive$strategy cognition))
	(cnet-channel (cognitive$channel cognition))
	(contract-id (gensym "Contract-")))
    (push (list :SEND cnet-channel `task-announcement (build-announcement contract-id strategy))
	  (cognitive$commands cognition))))

(defun build-announcement (contract-id strategy)
  "Make task-announcement format and fill the necessary slot. 
   The format is defined in the strategy definition."
  (let ((task (copy-cnet-task-announcement (cnet-strategy-task-announcement strategy))))
    (setf (cnet-task-announcement-contract-id task) contract-id)
    task))

;;
;; :collect-bids-and-award-tasks state
;;

(defun collect-bids-and-award-tasks (manager cognition)
  "Collect bids from the contractors and assign each role of the 
   strategy to best bidder. It is assumed that the manager can process
   bids and award tasks in one time unit. Otherwise the manager spend 
   two time units to collect bids and to award tasks. 
   The manager also updates its local information about other agent
   based on the 'node-abstraction' of the bid." 
  (format t "~%Time ~a:(Collect bids and award tasks: ~a)" 
	  *current-time* (cnet-strategy-name (cognitive$strategy cognition)))
  (collect-bids manager cognition)
  (update-manager-information-from-bid cognition)
  (when (all-bids manager cognition)
    (cond ((task-award manager cognition)
	   (setf (cognitive$state cognition) :collect-task-reports))
	  (t
	   (build-and-announce-task manager cognition)
	   (setf (cognitive$state cognition) :collect-bids-and-award-tasks)))))

(defun collect-bids (manager cognition)
  "Collect bids from the contractors and save it."
  (declare (ignore manager))
  (multiple-value-bind (bid-messages remaining-messages)
      (select-messages (cognitive$messages cognition) :type 'task-bid)
    (setf (cognitive$messages cognition) remaining-messages)
    (setf (cognitive$bid-collect cognition) (append bid-messages (cognitive$bid-collect cognition)))
    (cognitive$bid-collect cognition)))

(defun all-bids (manager cognition)
  "Return t if all bids are arrived from contractors."
  (declare (ignore manager))
  (let* ((bid-collect (cognitive$bid-collect cognition))
	 (bidden-p #'(lambda (contractor) 
		       (member-if #'(lambda (bid) (equalp (message$speaker bid) (agent$name contractor))) 
				  bid-collect))))
    (every bidden-p (cognitive$contractors cognition))))

(defun clear-bids (cognition)
  (setf (cognitive$bid-collect cognition) nil))

(defun task-award (manager cognition)
  "Assign the best role for each contractor based on bid-rating 
   function define in the :bid-rating function slot of the strategy."
  (block task-award
    (let* ((strategy (cognitive$strategy cognition))
	   (bid-messages (cognitive$bid-collect cognition))
	   (roles (copy-list (cnet-strategy-roles strategy)))
	   (role-bid-match nil)
	   (best-bid nil)
	   (bid-ratings nil))
      (dolist (role roles)
	(setf bid-ratings 
	  (mapcar #'(lambda (bid) (funcall (cnet-strategy-bid-rating strategy) cognition strategy role bid)) 
		  bid-messages))
	(when (some #'null bid-ratings)
	  (clear-bids cognition)
	  (return-from task-award nil))
	(setf best-bid (best-bid-rating bid-ratings bid-messages))
	(assoc-insert role best-bid role-bid-match)
	(setf bid-messages (remove best-bid bid-messages)))
      (distribute-tasks manager cognition role-bid-match)
      (clear-bids cognition)
      'successful-awards)))

(defun best-bid-rating (bid-ratings bid-messages)
  "Return a bid-message from bid-messages which has maximum rating.
   The best is the one with the MINIMUM rating"
  (cdr (first (sort (pairlis bid-ratings bid-messages) #'< :key #'car))))

(defun distribute-tasks (manager cognition role-bid-match)
  "Copy the task-award format from the selected strategy and
   fill necessary slots of the format. The task specifying the role within the
   strategy is sent to the corresponding  bidder."
  (dolist (role-bid role-bid-match)
    (let* ((cnet-channel (cognitive$channel cognition))
	   (strategy (cognitive$strategy cognition))
	   (role (car role-bid))
	   (bid (cdr role-bid))
	   (task (copy-cnet-task-award (cnet-strategy-task-award strategy)))
	   (contractor (message$speaker bid)))
      (setf (cnet-task-award-task-parameters task) 
	(funcall (cnet-strategy-task-parameter strategy) cognition))
      (setf (cnet-task-award-contract-id task) (gensym "Task-"))
      (setf (cnet-task-award-task-name task) (cnet-strategy-name strategy))
      (setf (cnet-task-award-role-name task) role)
      (setf (cnet-task-award-speaker task) manager)
      (setf (cnet-task-award-hearer task) contractor)
      (if *debug-cnet* (format t "~%Manager assigns ~a to ~a" role contractor))
      (push (list :SEND cnet-channel 'task-award task :hearer contractor)
	    (cognitive$commands cognition)))))

(defun update-manager-information-from-bid (cognition)
  "Based on the bids from the contractors, update manager's information."
  (let ((bid-messages (cognitive$bid-collect cognition)))
    (dolist (b bid-messages)
      (update-agent-location-from-bid cognition b)
      (update-prey-location-from-bid cognition b))))

(defun update-agent-location-from-bid (cognition bid-message)
  "If the bid specify agent's location, update manager's information."
  (declare (ignore manager))
  (let* ((new-location (first (cnet-task-bid-node-abstraction (message$content bid-message))))
	 (key (message$speaker bid-message)))
    (assoc-insert key new-location (cognitive$agent-locations cognition))))

(defun update-prey-location-from-bid (cognition bid-message)
  "If the bid specify prey's location, update manager's information."
  (declare (ignore manager))
  (let* ((new-location (second (cnet-task-bid-node-abstraction (message$content bid-message))))
	 (key (message$speaker bid-message)))
    (when new-location (assoc-insert key new-location (cognitive$prey-locations cognition)))))

;;
;; :collect-task-reports state
;;

(defun collect-task-reports (manager cognition)
  "Collect reports from the contractors and update manager's 
   local information accordingly."
  (format t "~%Time ~a:(Collect Task Reports: ~a)" 
	  *current-time* (cnet-strategy-name (cognitive$strategy cognition)))
  (when (collect-reports manager cognition)
    (update-manager-information-from-report cognition)
    (clear-reports cognition)
    (build-and-announce-task manager cognition)
    (setf (cognitive$state cognition) :collect-bids-and-award-tasks)))

(defun collect-reports (manager cognition)
  "Collect reports from the contractors and save it."
  (declare (ignore manager))
  (multiple-value-bind (report-msgs remaining-messages)
      (select-messages (cognitive$messages cognition) :type 'task-report)
    (setf (cognitive$messages cognition) remaining-messages)
    (setf (cognitive$report-collect cognition)
      (append report-msgs (cognitive$report-collect cognition)))
    (cognitive$report-collect cognition)))

(defun update-manager-information-from-report (cognition)
  "Based on the reports from the contractors, update manager's information."
  (let ((report-messages (cognitive$report-collect cognition)))
    (dolist (r (reverse report-messages))
      (update-agent-location-from-report cognition r)
      (update-prey-location-from-report cognition r))))

(defun update-agent-location-from-report (cognition report-message)
  "Update manager's information of agent locations."
  (let* ((new-location (cnet-task-report-agent-location (message$content report-message)))
	 (key (message$speaker report-message)))
    (assoc-insert key new-location (cognitive$agent-locations cognition))))

(defun update-prey-location-from-report (cognition report-message)
  "Update manager's information of prey locations."
  (let* ((new-location (cnet-task-report-prey-location (message$content report-message)))
	 (key (message$speaker report-message)))
    (when new-location (assoc-insert key new-location (cognitive$prey-locations cognition)))))

(defun clear-reports (cognition)
  (setf (cognitive$report-collect cognition) nil))

