;;;; -*- Mode: Fi:Common-Lisp -*-
;;;; 
;;;; $Id$
;;;; $Source$
;;;; 
;;;; Description       : Contractor agent for the Predator-Prey doamin
;;;;                     using Contract Net protocol
;;;; 
;;;; Original author(s): Jaeho Lee <jaeho@bigtwohearted.engin.umich.edu>
;;;; Organization      : University of Michigan DAI Lab
;;;; Created On        : Sat Jan 11 13:43:34 1992
;;;; Last Modified On  : Fri Jan 24 18:12:07 1992
;;;; Last Modified By  : Jaeho Lee <jaeho@osprey.engin.umich.edu>
;;;; 
;;;; Copyright (C) 1992 University of Michigan.
;;;; 
;;;; HISTORY 
;;;; 11-Jan-1992		Jaeho Lee (on bigtwohearted.engin.umich.edu)
;;;;    After the communication capability of MICE has been made, revised
;;;;    and modified by Jaeho Lee.
;;;;    
;;;; 04-Oct-1991		
;;;;     Originally written by Jaeho Lee and Young-Pa So for the project
;;;;     of EECS 598-4 in fall, 1991. 
;;;;    

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.


(defstruct contractor-goal
  "The goal structure."
  name					; Strategy Name
  manager				; Manager Name
  parameters				; Strategy Parameters
  role					; Role Name
  precond				; Precondition
  report-condition			; (Success, Precond-Failure, Timeout)
  timeout				; time to report
  create-time				; goal creation time
  report?				; Decides whether to send the report to manager or not
  )

;;
;; Agents
;;

(defun contractor-agent (agent)
  "Ccontractor agent invocation function."
  (if *debug-cnet* (format t "~%Time ~a: ### Contractor (~a) is activated." *current-time* agent))
  (if (= *current-time* 0) (cognitive-initialization))
  (let* ((cognition (agent-cognitive agent))
	 (next-scan-time (+ (cognitive$last-scan-time cognition) (cognitive$scan-interval cognition)))
	 (next-recv-time (+ (cognitive$last-recv-time cognition) (cognitive$recv-interval cognition)))
	 (cnet-channel (cognitive$channel cognition))
	 (scan-data (read-and-reset-scanned-data :reset-value :empty))
	 (recv-data (read-and-reset-received-messages :reset-value :empty)))

    (setf (cognitive$commands cognition) nil)

    (when (>= *current-time* next-scan-time)
      (push `(:SCAN :ALL) (cognitive$commands cognition))
      (setf (cognitive$last-scan-time cognition) *current-time*))

    (when (>= *current-time* next-recv-time)
      (push `(:RECV ,cnet-channel) (cognitive$commands cognition))
      (setf (cognitive$last-recv-time cognition) *current-time*))
    
    (when (null (eq :empty scan-data))
      (setf (cognitive$scan-data cognition) scan-data))

    (when (null (eq :empty recv-data))
      (setf (cognitive$messages cognition) (append (cognitive$messages cognition) recv-data)))

    (when (cognitive$messages cognition)
      (multiple-value-bind (task-announcement-msgs remaining-msg)
	  (select-messages (cognitive$messages cognition) :type 'TASK-ANNOUNCEMENT)
	(setf (cognitive$messages cognition) remaining-msg)
	(when task-announcement-msgs
	  (push (list :SEND cnet-channel 'TASK-BID
		      (create-bid agent cognition (message$content (first task-announcement-msgs)))
		      :hearer (message$speaker (first task-announcement-msgs)))
		(cognitive$commands cognition))))

      (multiple-value-bind (task-award-msgs remaining-msg)
	  (select-messages (cognitive$messages cognition) :type 'TASK-AWARD)
	(setf (cognitive$messages cognition) remaining-msg)
	(when task-award-msgs
	  (pop (cognitive$goal-stack cognition))
	  (push (create-goal (first task-award-msgs)) (cognitive$goal-stack cognition)))))

      ;; Now pursue the current goal of the agent
      (append (reverse (cognitive$commands cognition))
	      (pursue-current-goal agent cognition))))

;;
;;

(defun wandering-agent (agent)
  (let ((scan-data (read-and-reset-scanned-data)))
    (if scan-data
      (let* ((agent-blockers (agent$blocked-by-types agent))
	     (agent-location (find-agent-location agent))
	     (loc-x (location$x agent-location))
	     (loc-y (location$y agent-location))
	     (valid-directions-and-locations
	      (list (list :EAST (make-location :X (1+ loc-x) :Y loc-y))
		    (list :WEST (make-location :X (1- loc-x) :Y loc-y))
		    (list :NORTH (make-location :X loc-x :Y (1- loc-y)))
		    (list :SOUTH (make-location :X loc-x :Y (1+ loc-y))))))
	(dolist (grid-desc scan-data)
	  (let ((valid-dir-and-loc
		 (some #'(lambda (valid-one)
			   (when (equalp (second valid-one) (grid-description$coordinates grid-desc))
			     valid-one))
		       valid-directions-and-locations)))

	    (when (and valid-dir-and-loc
		       (some #'(lambda (near-agent)
				 (member (agent$type near-agent) agent-blockers))
			     (grid-description$agents grid-desc)))

	      (setf valid-directions-and-locations
		(delete valid-dir-and-loc valid-directions-and-locations :TEST 'equalp)))))

	`((:MOVE ,(first (nth (random (list-length valid-directions-and-locations))
			      valid-directions-and-locations)))))
      ;; If no scan-data...
      `((:SCAN :ALL)))))

;;
;;
;;

(defun create-bid (agent cognition announce-msg)
  "BID creation function."
  (let ((scan-data (cognitive$scan-data cognition)))
    (make-cnet-task-bid
     :hearer (cnet-task-announcement-speaker announce-msg)
     :speaker (agent$name agent)
     :contract-id (cnet-task-announcement-contract-id announce-msg)
     :node-abstraction (list (find-agent-location agent) (locate-prey scan-data)))))


(defun create-goal (task-award-msg)
  "Create goal structure."
  (let ((task-award (message$content task-award-msg)))
    (make-contractor-goal
     :name (cnet-task-award-task-name task-award)
     :manager (cnet-task-award-speaker task-award)
     :parameters (cnet-task-award-task-parameters task-award)
     :role (cnet-task-award-role-name task-award)
     :precond nil
     :report-condition (cnet-report-specification-report-condition
			(cnet-task-award-report-specification task-award))
     :timeout (cnet-report-specification-report-timeout
	       (cnet-task-award-report-specification task-award))
     :create-time *current-time*
     :report? t)))


(defun pursue-current-goal (agent cognition)
  "Current goal execution function."
  (let ((current-goal (first (cognitive$goal-stack cognition))))
    (cond ((null current-goal)
	   ;; If no goals on stack behave instinctively.
	   ;; That is, if one sees a prey, pursue, try to catch it
	   (instinctive-behavior agent cognition))
	  (t
	   ;; Check to see if the current-goal is satisfied and see if report is needed.
	   ;; Current goal is satisfied when the goal is achieved or its precondition
	   ;; is violated or goal has timed out, in which case must report
	   ;; Finally, pursue the current-goal
	   (let* ((report? (and current-goal (contractor-goal-report? current-goal)))
		  (report (build-report agent cognition current-goal))
		  (strategy-name (contractor-goal-name current-goal))
		  (strategy-parms (contractor-goal-parameters current-goal))
		  (role (contractor-goal-role current-goal))
		  (report-command nil)
		  (behavior-command nil))
	     (when (and report? report)
	       (setf report-command (report-to-manager agent (contractor-goal-manager current-goal) report))
	       (setf (contractor-goal-report? (first (cognitive$goal-stack cognition))) nil))
	     (setf behavior-command
	       (case strategy-name
		 (Wandering-strategy (Wandering-behavior agent cognition role strategy-parms))
		 (Surround-strategy  (Surround-behavior  agent cognition role strategy-parms))
		 (NEWS-strategy      (NEWS-behavior      agent cognition role strategy-parms))
		 (Relax-strategy     (Relax-behavior     agent cognition role strategy-parms))
		 (otherwise (format t "Contractor: Unknown Strategy ~a~%" strategy-name) nil)))
	     (append report-command behavior-command))))))


(defun report-to-manager (agent manager-name report)
  "Make a list of Mice commands for sending the report."
  (list (list :SEND (first (agent$channels agent)) 'TASK-REPORT report :hearer manager-name)))


(defun build-report (agent cognition current-goal)
  "Check to see if current goal is satisfied
   Returns a report structure when the report condition for
   the goal (task) is satisfied
   Different Strategies have different report conditions
   For example : 1) When Goal is achieved successfully
	         2) When the precondition of the Goal is violated
		 3) When the the Goal (task) has timed out."
  (let* ((scan-data (cognitive$scan-data cognition))
	 (agent-loc (find-agent-location agent))
	 (prey-loc (locate-prey scan-data))
	 (goal-name (contractor-goal-name current-goal))
	 (goal-distance (second (contractor-goal-parameters current-goal)))
	 (goal-role (contractor-goal-role current-goal))
	 (goal-report-condition (contractor-goal-report-condition current-goal))
	 (timeout (and (member 'Timeout goal-report-condition)
		       (contractor-goal-timeout current-goal)
		       (>= *current-time* (+ (contractor-goal-create-time current-goal)
					     (contractor-goal-timeout current-goal)))))
	 (report (make-cnet-task-report :agent-location agent-loc :prey-location prey-loc)))

    (case goal-name
      (Wandering-strategy
       (cond (prey-loc (setf (cnet-task-report-result report) 'Success))
	     (timeout  (setf (cnet-task-report-result report) 'Timeout))
	     (t (setf report nil))))
      (NEWS-strategy
       (cond ((null prey-loc) (setf (cnet-task-report-result report) 'Precond-Failure))
	     ((same-location-p prey-loc agent-loc) 
	      (setf (cnet-task-report-result report) 'Success))
	     (timeout (setf (cnet-task-report-result report) 'Timeout))
	     (t (setf report nil))))
      (Surround-strategy
       (cond ((null prey-loc) (setf report nil))
	     ((same-location-p agent-loc (prey-relative-location prey-loc goal-distance goal-role))
	      (setf (cnet-task-report-result report) 'Success))
	     (timeout (setf (cnet-task-report-result report) 'Timeout))
	     (t (setf report nil))))
      (Relax-strategy
       (cond ((null prey-loc) (setf report nil))
	     ((>= (distance-between prey-loc agent-loc) goal-distance)
	     (setf (cnet-task-report-result report) 'Success))
	     (timeout (setf (cnet-task-report-result report) 'Timeout))
	     (t (setf report nil))))
      (otherwise 
       (format "current-goal-satisfied: Unknown goal name ~a~%" goal-name)
       (setf report nil)))
    
    report))


(defun prey-relative-location (prey-loc distance direction)
  "Calculate the XY coordinate of the point which is 'distance' away
   in 'direction' direction from the prey location 'prey-loc'."
  (let* ((prey-X (location$x prey-loc))
	 (prey-Y (location$y prey-loc)))
    (case direction
      (:NORTH (make-location :x prey-X :y (- prey-Y distance)))
      (:SOUTH (make-location :x prey-X :y (+ prey-Y distance)))
      (:EAST (make-location :x (+ prey-X distance) :y prey-Y))
      (:WEST (make-location :x (- prey-X distance) :y prey-Y))
      (otherwise (format "prey-relative-location: Bad Direction ~a~%" direction)))))


(defun move-to-goal (agent-loc prey-loc direction distance &optional (detour nil))
  (let* ((goal-loc (get-relative-loc prey-loc direction distance))
	 (move (move-toward agent-loc goal-loc)))
    (if detour 
      (detour-move move agent-loc prey-loc)
      move)))


;;
;; Strategy Dependent Behaviors -- They should return Mice commands
;; 

(defun instinctive-behavior (agent cognition)
"
    Strategy Name  : Instinct
    Precondition   : None
    Description	   : If the prey is sensed pursue, if not wander about random
		     Default behavior of the agent when there is no higher goals
    Roles	   : None
    Report	   : None
"
  (let ((scan-data (cognitive$scan-data cognition)))
    (if scan-data
      (let* ((agent-loc (find-agent-location agent))
	     (prey-loc (locate-prey scan-data))
	     (move (if prey-loc (move-toward agent-loc prey-loc) (random-direction))))
	`((:MOVE ,move)))
      `((:SCAN :ALL)))))


(defun Wandering-behavior (agent cognition area parameters)
"
    Strategy Name  : Wander
    Precondition   : Prey is out of sensor range for all predators
    Description	   : Disect a given area into 4 sub-areas and have each
		     predator assigned a distinct area to search for the prey
    Roles	   : :area1 :area2 :area3 :area4
    Report	   : Must report when prey is found
"
  (let* ((scan-data (cognitive$scan-data cognition))
	 (agent-location (find-agent-location agent))
	 (prey-loc (locate-prey scan-data))
	 (possible-moves '(:NORTH :EAST :WEST :SOUTH))
	 (domain-region (first parameters))
	 (region (case area
		   (:area1 (quarter-region domain-region 1))
		   (:area2 (quarter-region domain-region 1))
		   (:area3 (quarter-region domain-region 1))
		   (:area4 (quarter-region domain-region 1))
		   (otherwise (format t "Wandering-behavior: Unknown role ~a~%" area) nil)))
	 (prev-move (cognitive$prev-move cognition))
	 (next-move nil))
    (cond (prey-loc
	   (instinctive-behavior agent cognition))
	  (t
	   (cond ((location-in-region-p agent-location region)
		  (setf possible-moves 
		    (remove (opposite-move prev-move) (valid-moves agent-location region)))
		  (setf next-move (nth (random (length possible-moves)) possible-moves))
		  (setf (cognitive$prev-move cognition) next-move))
		 (t
		  (setf next-move (move-toward agent-location (center-of region)))))
	   `((:MOVE ,next-move))))))

	
(defun Surround-behavior (agent cognition direction parameters)
"
  Strategy Name : Surround
  Precondition  : At least 1 at most 3 agents have the prey in their sensor range
  Description   : Predators are to surround the prey at a given distance
  Roles	        : NORTH, EAST, WEST, SOUTH
  Report	: Report when successful or timeout.
"
  (let* ((current-goal (first (cognitive$goal-stack cognition)))
	 (scanned-prey-loc (locate-prey (cognitive$scan-data cognition)))
	 (default-prey-loc (first parameters))
	 (distance-from-prey (second parameters))
	 (agent-loc (find-agent-location agent))
	 (prey-loc (if scanned-prey-loc scanned-prey-loc default-prey-loc)))
    (setf (first (contractor-goal-parameters current-goal)) prey-loc)
    `((:MOVE ,(move-to-goal agent-loc prey-loc direction distance-from-prey)))))


(defun NEWS-behavior (agent cognition direction parameters)
"
    Strategy Name : NEWS
    Precondition  : Prey is within sensor range
    Description	  : Surround all four directions of the prey
    Roles	  : North, East, West, South
    Report	  : Must report when Prey is out of sensor range
"
  (let* ((scanned-prey-loc (locate-prey (cognitive$scan-data cognition)))
	 (default-prey-loc (cognitive$last-location cognition))
	 (distance-from-prey (second parameters))
	 (agent-loc (find-agent-location agent))
	 (prey-loc (if scanned-prey-loc scanned-prey-loc default-prey-loc)))
    (setf (cognitive$last-location cognition) prey-loc)
    `((:MOVE ,(move-to-goal agent-loc prey-loc direction distance-from-prey t)))))


(defun Relax-behavior (agent cognition direction parameters)
"
    Strategy Name  : Relax
    Precondition   : Prey is on the wall and all Predators are sensing it
    Description	   : Predators gets back from the prey at a given distance
    Roles          : Identical roles
    Report         : Report when successful or timeout
"
  (declare (ignore direction))
  (let* ((current-goal (first (cognitive$goal-stack cognition)))
	 (scanned-prey-loc (locate-prey (cognitive$scan-data cognition)))
	 (default-prey-loc (first parameters))
	 (distance-from-prey (second parameters))
	 (agent-loc (find-agent-location agent))
	 (prey-loc (if scanned-prey-loc scanned-prey-loc default-prey-loc)))
    (setf (first (contractor-goal-parameters current-goal)) prey-loc)
    (if (>= (distance-between agent-loc prey-loc) distance-from-prey)
      `((:MOVE ,(move-toward agent-loc prey-loc)))
      `((:MOVE ,(move-toward prey-loc agent-loc))))))

;;
;;
;;

(defun locate-prey (scan-data)
  "If the prey has been sensed, return its location, otherwise return nil."
  (block found-prey
    (dolist (grid-desc scan-data nil)
      (dolist (agent (mapcar #'agent-structure (grid-description$agents grid-desc)))
	(when (eq (agent$type agent) :PREY)
	  (return-from found-prey (find-agent-location agent)))))
    (return-from found-prey nil)))

(defun detour-move (next-move agent-loc prey-loc)
  "When taking the next move in the direction 'next-move' from 'agent-loc',
   if the prey is blocking, then move orthogonal direction
   to the 'next-move' direction. This function is used in the
   NEWS strategy so as to prevent agents from pushing around the prey all the time."
  (let ((my-X (location$x agent-loc))
	(my-Y (location$y agent-loc))
	(prey-X (location$x prey-loc))
	(prey-Y (location$y prey-loc)))
    (and (eq next-move :NORTH) (= my-X prey-X) (= (1- my-Y) prey-Y)
	 (setf next-move (random-direction '(:EAST :WEST))))
    (and (eq next-move :SOUTH) (= my-X prey-X) (= (1+ my-Y) prey-Y)
	 (setf next-move (random-direction '(:EAST :WEST))))
    (and (eq next-move :EAST) (= (1+ my-X) prey-X) (= my-Y prey-Y)
	 (setf next-move (random-direction '(:NORTH :SOUTH))))
    (and (eq next-move :WEST) (= (1- my-X) prey-X) (= my-Y prey-Y)
	 (setf next-move (random-direction '(:NORTH :SOUTH))))
    next-move))

(defun get-relative-loc (target-loc direction distance)
  "Calculate the coordinate of the location 'distance' apart from
   the 'target-loc' in 'direction' direction."
  (let ((x (location$x target-loc))
	(y (location$y target-loc)))
    (case direction
      (:NORTH (make-location :x x :y (- y distance)))
      (:SOUTH (make-location :x x :y (+ y distance)))
      (:EAST  (make-location :x (+ x distance) :y y))
      (:WEST  (make-location :x (- x distance) :y y)))))

(defun opposite-move (move)
  (cond (move
	 (case move
	   (:NORTH :SOUTH)
	   (:SOUTH :NORTH)
	   (:WEST :EAST)
	   (:EAST :WEST)
	   (otherwise (format t "opposite-move: Unknown move ~a~%" move) nil)))
	(t nil)))

(defun valid-moves (loc region)
  (let ((moves '(:NORTH :EAST :WEST :SOUTH))
	(x (location$x loc))
	(y (location$y loc)))
    (when (< (1- x) (region$x-min region)) (setf moves (remove :WEST moves)))
    (when (> (1+ x) (region$x-max region)) (setf moves (remove :EAST moves)))
    (when (< (1- y) (region$y-min region)) (setf moves (remove :NORTH moves)))
    (when (> (1+ y) (region$y-max region)) (setf moves (remove :SOUTH moves)))
    moves))


(defun random-direction (&optional (directions '(:EAST :NORTH :WEST :SOUTH)))
  (nth (random (length directions)) directions))

(defun move-toward (from to)
  (let ((from-X (location$x from))
	(from-Y (location$y from))
	(to-X (location$x to))
	(to-Y (location$y to))
	(directions '(:WEST :EAST :SOUTH :NORTH)))
    (when (<= from-X to-X)
      (setf directions (remove :WEST directions))
      (when (< (- to-X from-X) (abs (- from-Y to-Y)))
	(setf directions (remove :EAST directions))))
    (when (>= from-X to-X)
      (setf directions (remove :EAST directions))
      (when (< (- from-X to-X) (abs (- from-Y to-Y)))
	(setf directions (remove :WEST directions))))
    (when (<= from-Y to-Y)
      (setf directions (remove :NORTH directions))
      (when (< (- to-Y from-Y) (abs (- from-X to-X)))
	(setf directions (remove :SOUTH directions))))
    (when (>= from-Y to-Y)
      (setf directions (remove :SOUTH directions))
      (when (< (- from-Y to-Y) (abs (- from-X to-X)))
	(setf directions (remove :NORTH directions))))
    (if directions
      (nth (random (list-length directions)) directions)
      nil)))





