;;;; -*- Mode: Fi:Common-Lisp -*-
;;;; 
;;;; $Id$
;;;; $Source$
;;;; 
;;;; Description       : Basci architecture for the Predator-Prey doamin
;;;;                     using Contract Net protocol
;;;; 
;;;; Original author(s): Jaeho Lee <jaeho@bigtwohearted.engin.umich.edu>
;;;; Organization      : University of Michigan DAI Lab
;;;; Created On        : Sat Jan 11 13:43:34 1992
;;;; Last Modified On  : Mon Jan 13 17:38:16 1992
;;;; Last Modified By  : Jaeho Lee <jaeho@woundedknee.engin.umich.edu>
;;;; 
;;;; Copyright (C) 1992 University of Michigan.
;;;; 
;;;; HISTORY 
;;;; 11-Jan-1992		Jaeho Lee (on bigtwohearted.engin.umich.edu)
;;;;    After the communication capability of MICE has been made, revised
;;;;    and modified by Jaeho Lee.
;;;;    
;;;; 04-Oct-1991		
;;;;     Originally written by Jaeho Lee and Young-Pa So for the project
;;;;     of EECS 598-4 in fall, 1991. 
;;;;    

;  Copyright 1991, 1992
;  Regents of the University of Michigan
;  
;  Permission is granted to copy and redistribute this software so long as
;  no fee is charged, and so long as the copyright notice above, this
;  grant of permission, and the disclaimer below appear in all copies made.
;  
;  This software is provided as is, without representation as to its fitness
;  for any purpose, and without warranty of any kind, either express or implied,
;  including without limitation the implied warranties of merchantability and fitness
;  for a particular purpose.  The Regents of the University of Michigan shall not
;  be liable for any damages, including special, indirect, incidental, or
;  consequential damages, with respect to any claim arising out of or in
;  connection with the use of the software, even if it has been or is hereafter
;  advised of the possibility of such damages.


;;
;; Mice setting
;;

(defun mice-continue-p ()
  "When the prey is captured, stop the mice excution."
  (some #'(lambda (agent) (eq (agent$type agent) :PREY)) *agent-schedule-queue*))

;;
;; Agent Cognition
;;

(defvar *cognitive-agents* nil "User-accessable cognition part of the agents")
(setf *cognitive-agents* nil)

(defxstruct (cognitive :export (:conc-name "COGNITIVE$") (:print-function cognitive-print-function))
  (name			nil)
  (commands             nil)
  ;; sensing
  (scan-data            nil)
  (scan-interval        1)
  (last-scan-time	-1)
  ;; communication
  (channel              nil)
  (messages             nil)
  (recv-interval        1)
  (last-recv-time       -1)
  ;; Manager specific information
  (state                nil)
  (strategy             nil)
  (strategy-list        nil)
  (contractors          nil)
  (bid-collect          nil)
  (report-collect       nil)
  (agent-locations      nil)
  (prey-locations       nil)
  ;; Contractor specific information
  (goal-stack           nil)
  (prev-move            nil)
  (last-location        (make-location :x 0 :y 0))
  )

(defun cognitive-print-function (agent stream print-level)
  (declare (ignore print-level))
  (format stream "~A" (cognitive$name agent)))

(defun agent-cognitive (mice-agent)
  (let ((agent (rest (assoc (agent$name mice-agent) *cognitive-agents*))))
    (unless agent
      (setf agent (make-cognitive :name (agent$name mice-agent)
				  :channel (first (agent$channels mice-agent))
				  :contractors (mapcar #'agent-structure '(pred1 pred2 pred3 pred4))
				  :strategy-list *cnet-strategy-list*))
      (setf *cognitive-agents* (acons (agent$name mice-agent) agent *cognitive-agents*)))
    agent))

(defun cognitive-initialization ()
  (setf *cognitive-agents* nil))

;;
;; Data structure
;;

(defstruct cnet-task-announcement
  hearer
  speaker
  contract-id
  task-abstraction
  eligibility-specification
  bid-specification
  expiration-time)

(defstruct cnet-task-bid
  hearer
  speaker
  contract-id
  node-abstraction)

(defstruct cnet-task-award
  hearer
  speaker
  contract-id
  task-name				; Strategy name
  task-parameters			; Strategy parameters
  role-name
  (report-specification nil :type cnet-report-specification))

(defstruct cnet-report-specification
  report-condition			; list of report condition names (prey-found timeout failure)
  report-timeout)

(defstruct cnet-task-report
  result
  (agent-location nil :type location)
  (prey-location nil :type location))	; possibly nil if out of sensing range

(defstruct (cnet-strategy (:print-function (lambda (struct stream depth)
					     (declare (ignore depth))
					     (format stream "~A" (cnet-strategy-name struct)))))
  "Strategy is defined declararively. Hence the architecture of
   CNET needs minimum modification even if the strategy is changed."
  name
  (precondition nil :type function)	; a predicate specifying necessary conditions to run this strategy
  (task-announcement nil :type cnet-task-announcement)
  (task-bid nil :type cnet-task-bid)
  (task-award nil :type cnet-task-award)

  (bid-rating nil :type function)	; a function which rates the bid for the role.
					; this function has three argument (manager strategy role-name bid)
					; smaller rating value means high eligibility.

  roles					; list of role names of this strategy.

  (task-parameter nil :type function)	; a function which returns task-specific parameters

  (role-parameter nil :type function))	; a function which returns role-specific parameters

;;
;; Utility functions

(defun my-count-if (test sequence &key (key #'identity))
  (if (null sequence) 0
      (count-if test sequence :key key)))

(defmacro extract-if (pred from &key (key #'identity))
  "Destructively extract and return elements whose 'key'
   satisfies the 'pred' from the list, 'from'"
  `(let ((selection (remove-if-not ,pred ,from :key ,key)))
     (if ,from (setf ,from (remove-if ,pred ,from :key ,key)))
     selection))

(defun compute-boundary-location (old-location direction)
  "Compute a new boundary location from the old location and direction."
  (let ((boundary-region (simulation-data$overall-region *simulation-data*)))
    (case direction
      (:NORTH (make-location :X (location$x old-location)
			     :Y (region$y-min boundary-region)))
      (:SOUTH (make-location :X (location$x old-location)
			     :Y (region$y-max boundary-region)))
      (:EAST  (make-location :X (region$x-max boundary-region)
			     :Y (location$y old-location)))
      (:WEST  (make-location :X (region$x-min boundary-region)
			     :Y (location$y old-location)))
      (otherwise (copy-location old-location)))))


(defun specified-location-p (location)
  "Return t if the location structure is filled with specfic numbers."
  (and (location-p location) (numberp (location$x location)) (numberp (location$y location))))

(defun same-location-p (loc1 loc2)
  "Retrun t if the two location is identical location."
  (and (specified-location-p loc1)
       (specified-location-p loc2)
       (equal (location$x loc1) (location$x loc2))
       (equal (location$y loc1) (location$y loc2))))

(defun wall-side-location-p (location)
  "Return t if the given location is just next of the wall.
   If the give location is nil, just return nil"
  (and (specified-location-p location)
       (notevery #'legal-location-p (mapcar #'(lambda (dir) (compute-new-location location dir))
					    '(:NORTH :SOUTH :EAST :WEST)))))

(defun blocked-location-p (location other-locations)
  "Return t if the give location is surrounded by the wall or other-locations.
  If the location or others-locations are nil, just return nil."
  (let ((surroundings
	 (mapcar #'(lambda (dir) (compute-new-location location dir)) '(:NORTH :SOUTH :EAST :WEST))))
    (every #'(lambda (sl)
	       (or (null (legal-location-p sl))
		   (member-if #'(lambda (ol) (equalp ol sl)) other-locations)))
	   surroundings)))

(defun distance-between (loc1 loc2)
  "Return the city-block-distance between two locations.
   If any of the location value is nil, then return nil."
  (cond ((and loc1 loc2)
	 (let ((x1 (location$x loc1))
	       (y1 (location$y loc1))
	       (x2 (location$x loc2))
	       (y2 (location$y loc2)))
	   (if (every #'numberp (list x1 y1 x2 y2))
	     (+ (abs (- x2 x1)) (abs (- y2 y1)))
	     nil)))
	(t nil)))

(defun center-of (reg)
  "Return the center coordinate of the region as an location"
  (let ((xmin (region$x-min reg))
	(ymin (region$y-min reg))
	(xmax (region$x-max reg))
	(ymax (region$y-max reg)))
    (make-location :x (/ (+ xmin xmax) 2) :y (/ (+ ymin ymax)))))

(defun quarter-region (overall-region number)
  "Return the region corresponding to the number.
	+-------+-------+
	|	|       |
	|   2   |   1   |
	|       |       |
	+-------+-------+
	|	|	|
	|   3   |   4   |
	|       |       |
	+-------+-------+
"
  (let* ((xmin (region$x-min overall-region))
	 (xmax (region$x-max overall-region))
	 (ymin (region$y-min overall-region))
	 (ymax (region$y-max overall-region))
	 (xwid (- xmax xmin))
	 (ywid (- ymax ymin)))
    (case number
      (1 (make-region :x-min (+ xmin (/ xwid 2) -1)
		      :y-min ymin
		      :x-max xmax
		      :y-max (+ ymin (/ ywid 2) +1)))
      (2 (make-region :x-min xmin
		      :y-min ymin
		      :x-max (+ xmin (/ xwid 2) +1)
		      :y-max (+ ymin (/ ywid 2) +1)))
      (3 (make-region :x-min xmin
		      :y-min (+ ymin (/ ywid 2) -1)
		      :x-max (+ xmin (/ xwid 2) +1)
		      :y-max ymax))
      (4 (make-region :x-min (+ xmin (/ xwid 2) -1)
		      :y-min (+ ymin (/ ywid 2) -1)
		      :x-max xmax
		      :y-max ymax))
      (otherwise nil))))

(defun fix-prey-location (assoc-locations)
  "From a list of prey locations, return the first specified location."
  (let ((specified-pair (rassoc-if #'specified-location-p assoc-locations)))
    (if specified-pair (cdr specified-pair))))
