/*
 * Constants used in the simulator
 */

#include <math.h>

extern int debug;

#ifndef NULL
#define NULL 0
#endif

#define DEG_TO_RAD (M_PI / 180.0)
#define RAD_TO_DEG (180.0 / M_PI)

/* angular units are rads */

#define ANG45 M_PI_4
#define ANG90 M_PI_2
#define ANG180 M_PI
#define ANG270 (M_PI + M_PI_2)
#define ANG360 (M_PI + M_PI)

#define deg_to_ang(d) (d * DEG_TO_RAD)
#define ang_to_deg(d) (d * RAD_TO_DEG)
#define rad_to_ang(d) (d)
#define ang_to_rad(d) (d)

/* radians to wheel encoder readings */

#define RAD_TO_RAW (float)(4095.0 / (M_PI * 2.0))
#define RAW_TO_RAD (float)((M_PI * 2.0) / 4095.0)

/* angular velocity to differential wheel velocity in mm/sec */

#define ANG_VEL_TO_WHEEL_DIFF 300.0

/* 
 * Definitions of some common angular operations
 */

#define add_ang(a,b) norm_angle(a+b)
#define add2_ang(a,b) norm2_angle(a+b)
#define sub_ang(a,b) norm_angle(a-b)
#define sub2_ang(a,b) norm2_angle(a-b)

#define add_dang(a,b) norm_dangle(a+b)
#define add2_dang(a,b) norm2_dangle(a+b)
#define sub_dang(a,b) norm_dangle(a-b)
#define sub2_dang(a,b) norm2_dangle(a-b)

extern float norm_angle(float a); /* between 0 and 2PI */
extern float norm2_angle(float a); /* between -PI and PI */
extern float norm_dangle(float a); /* between 0 and 2PI */
extern float norm2_dangle(float a); /* between -PI and PI */

/* floor and ceiling ops */

#define FLOOR(x) ((int)(x))
#define CEILING(x) ((int)((x) + 1.0))
#define ROUND(x) ((int)((x) + 0.5))

/* max and min and abs */

#ifndef MAX
#define MAX(x,y) ((x) > (y) ? (x) : (y))
#endif
#ifndef MIN
#define MIN(x,y) ((x) > (y) ? (y) : (x))
#endif
#define ABS(x) ((x) > 0.0 ? (x) : (- (x)))
#define SIGNUM(x) ((x) > 0.0 ? 1 : ((x) < 0.0 ? -1 : 0))
#define FSIGNUM(x) ((x) > 0.0 ? 1.0 : ((x) < 0.0 ? -1.0 : 0.0))


/* Communication types */

#define TROBOT_PORT 8101	/* port number by default */
#define UROBOT_PORT 8102	/* port number by default */
#define TCP 0			/* protocol number */
#define UDP 1			/* protocol number */
#define UNIX 4			/* protocol number */
#define TTYPORT 5		/* protocol number */

extern int connected;		/* 1 if connected to Flakey, 0 if not */

/* packet constants */

#define flakey_data_waiting (check_in_buf() > 0)

#define SYNC1char 0x1F
#define SYNC2char 0x2A
#define ARGchar   0x3B
#define NARGchar  0x1B
#define SARGchar  0x2B
#define ENDchar   0x4C

#define VISIONpac 0x50
#define MOTORpac  0x70
#define ECHOpac   0x40
#define DEPTHpac  0x60
#define SONARpac  0x00


/* some useful read/write macros */


#define force_client_output() write_out_buf()
#define clear_client_input() clear_in_buf()
#define client_data_waiting() (check_in_buf(0) > 0)
#define client_data_waiting_timeout(x) (check_in_buf(x) > 0)
#define write_byte_to_client(b) put_buf_byte(b)
#define read_byte_from_client() get_data_byte()
#define wait_client_data() wait_in_buf()


/* command numbers */

#define PULSE_COM   0
#define OPEN_COM    1
#define CLOSE_COM   2
#define POLLING_COM 3
#define ENABLE_COM  4
#define SETA_COM    5
#define SETV_COM    6
#define SETO_COM    7
#define MOVE_COM    8
#define ROTATE_COM  9
#define SETRV_COM   10
#define VEL_COM	    11
#define HEAD_COM    12
#define DHEAD_COM   13
#define DCHEAD_COM  22
#define DROTATE_COM 14
#define SAY_COM	    15
#define VISION_COM  16
#define V_WINDOW_COM 17
#define UDP_COM    20
#define RVEL_COM   21
#define STEP_COM   64

#define HIGH_POWER_ARG 1
#define NORMAL_POWER_ARG  0
#define ENABLE_ARG        1
#define DISABLE_ARG       0
#define MOTOR_ARG         1		
#define SONAR_ARG         2		
#define MOTOR_SONAR_ARG   3		
#define VISION_ARG        4		
#define C_VISION_ARG      8		
#define NONE_ARG          0
#define ALL_ARG     037774 
#define FRONT_ARG   011454 
#define REAR_ARG    004100
#define SIDES_ARG   022220
#define LEFT_ARG    022000 
#define RIGHT_ARG   000220
#define BX_ARG      0x1000			; window parameters
#define EX_ARG      0x2000
#define DY_ARG      0x3000
#define THRESH_ARG  0x4000
#define STEP_START_ARG  1
#define STEP_STOP_ARG   0
#define STEP_STEP_ARG   -1


#define STATUS_NO_HIGH_POWER 0	/* status */
#define STATUS_NOT_SERVOING  1
#define STATUS_STOPPED       2
#define STATUS_MOVING        3
#define STATUS_NOT_CONNECTED 4

/* sonar arrays */

#define LEFT_SONAR_MASK  0x024
#define RIGHT_SONAR_MASK 0x900
#define FRONT_SONAR_MASK 0xCCF

/* extern float sonar_spec_array[][5]; */

#define sonar_spec_x(i) (sonar_spec_array[i][0])
#define sonar_spec_y(i) (sonar_spec_array[i][1])
#define sonar_spec_dx(i) (sonar_spec_array[i][2])
#define sonar_spec_dy(i) (sonar_spec_array[i][3])
#define sonar_spec_th(i) (sonar_spec_array[i][4])

/* control type values */

#define VEL 0
#define HEAD 1

extern incr_velocity(float inc); /* increment the velocity setpoint */
extern incr_head(float inc);	/* increment the heading setpoint, in degrees */
extern set_control(int type,float value); /* set these control values */
extern float target_vel();	/* returns targeted velocity */
extern float target_head();	/* returns targeted heading, in degrees */

/* Fuzzy sets */

extern float
f_or(float a, float b);

extern float
f_and(float a, float b);


extern float
f_not(float a);


extern float
straight_down(float x, float a, float b);


extern float
up_straight(float x, float a, float b);

extern float
f_smaller(float a, float b,float delta);

extern float
f_greater(float a, float b,float delta);

extern float
f_eq(float a, float b,float delta);


/* some drawing routines */

extern
draw_sonar_pt(float x,float y,int n); /* x,y sonar reading; n size of box */

extern
draw_centered_rect(float x, float y, float w, float h);	/* draw centered on x,y */
extern
draw_rect(float x,float y,float dx,float dy); /* draws a rectangle using rw coords */


/*
 * point types
 */

#define PT_POS 1

