/* Some sonar segment fns to compute intersection of sonars
 *
 * Uses a fan of line segs to simulate the beam width of the sonar
 *
 */

#include <math.h>
#include <stdio.h>
#include "structures.h"
#include "constants.h"
#include "parameters.h"

#define bet(x1,y1,x2,y2,x,y) (((x1 == x2) || ((x >= x1) ? (x < x2) : (x >= x2))) \
        && ((y1 == y2) || ((y >= y1) ? (y < y2) : (y >= y2))))

#define CP4 0.7071067811865
#define SP4 0.7071067811865

/*-------------------Globals--------------------------*/

static float x,y;		/* origin of sonar */
static int n;			/* number of beam segs - 1 */

static float sa[20], sb[20], sc[20]; /* beam fan line params */
static float xp[20], yp[20];	/* beam fan seg points */

static float r, xi, yi, t;	/* min intersection range and pt */


/*-------------------Setup----------------------------*/


setup_sonar(num,range,fx,fy,th,dth,incs)
     float range, fx, fy, th, dth; /* fx, fy are flakey location
				    * num is sonar number
				    * range is max range in mm
				    * th is Flakey angle in rads
				    * dth is beam half-angle in rads
				    * incs is half number of fan segs 
				    */
{
  float cs, sn;
  int i;
  float dx, dy, xp1, yp1, xp2, yp2;

  n = incs + incs;
  th = -th;

  cs = cos(th);
  sn = sin(th);
  x = fx + (cs * sonar_spec_x(num)) + (sn * sonar_spec_y(num));
  y = fy + (cs * sonar_spec_y(num)) - (sn * sonar_spec_x(num));

  th -= dth;
  cs = cos(th);			/* calc of first beam seg */
  sn = sin(th);
  xp1 = x + (cs * sonar_spec_dx(num) * range) + (sn * sonar_spec_dy(num) * range);
  yp1 = y + (cs * sonar_spec_dy(num) * range) - (sn * sonar_spec_dx(num) * range);

  th += dth + dth;
  cs = cos(th);			/* calc of last beam seg */
  sn = sin(th);
  xp2 = x + (cs * sonar_spec_dx(num) * range) + (sn * sonar_spec_dy(num) * range);
  yp2 = y + (cs * sonar_spec_dy(num) * range) - (sn * sonar_spec_dx(num) * range);

  dx = (xp2 - xp1) / n;
  dy = (yp2 - yp1) / n;

  for(i = 0; i <= n; i++)	/* calculate line and seg params for fan */
    {
      xp[i] = xp1;  
      yp[i] = yp1;  
      sa[i] = y - yp1;
      sb[i] = xp1 - x;
      sc[i] = (yp1 * x) - (xp1 * y);
      xp1 += dx;
      yp1 += dy;
    }

  r = 10000001;

}



/*----------------------Check one segment------------------------*/

float
check_seg(s)			
     seg *s;
{
  float x1 = s->x1, y1 = s->y1, x2 = s->x2, y2 = s->y2,
     d = s->a, e = s->b, f = s->c, th = s->th;
  int i;
  float xx, yy, a, b, c;
  float norm;

  for (i = 0; i <= n; i++)	/* loop through all beam segs */
    {
      a = sa[i];
      b = sb[i];
      c = sc[i];
      
      norm = (e * a) - (d * b);
      if (norm == 0.0)
	{ xx = yy = 0; }
      else
	{ xx = ((f * b) - (e * c)) / norm;
	  yy = ((c * d) - (a * f)) / norm;
	}

      if ( bet(x,y,xp[i],yp[i],xx,yy) &&
	   bet(x1,y1,x2,y2,xx,yy) )
	{
	  a = ((xx - x) * (xx - x)) + ((yy - y) * (yy - y));
	  if (a < r)
	    { r = a;
	      t = th;
	      xi = xx;
	      yi = yy;
	    }
	}
    }
  return(r);
}


float
return_seg_ang()		/* returns the min segment angle */
{
  float s;
  float fy, fx, a;

  fy = yi - y;  fx = xi - x;
  a = atan2(fy,fx);
  s = rad_to_ang(a);
  s = norm_seg_angle(t-s);
  if (s > ANG90) s = ANG180 - s;
  return(s);
  
}

