/*  EXPLORE.P  */


/*
SPECIFICATION
-------------

This module exports one routine, explore, which is used by
PLANNING_BUG.PL. It implements a simple form of breadth-first search,
which Bug uses to wander around its world and build up a map in its AWM.
The idea is that the bug moves around taking successive retinal
"snapshots" and merging them into a complete (hopefully) map, until it
finds that it can't move any further because of walls or other obstacles
all the way round.

The routine exported is
    explore( n )
This explores until it can find no more points to visit, or until Bug
has moved more (approximately) than n steps.
*/


/*
IMPLEMENTATION
--------------

'explore' is a breadth-first search, where the neighbours of a given
point are those points on the periphery of Bug's vision to which it
can find paths. See comments inside the routine.                
*/


needs exec;
needs awm;
needs awm_path;
needs path_to_moves;


vars awm;
/*  The AWM used to hold the map being constructed. */

vars been;
/*  A list holding all positions the bug has so far visited.  */

vars steps;
/*  The number of positions visited.  */

/*
All these variables are used by 'explore', and preserve information
only within one call.
*/


define explore( n );
    lvars n;
    lvars here, neighs, neigh, moves, new_heading;
    vars q, path, endpoint;

    new_awm( `?` ) -> awm;
    [% awm_position(awm) %] -> been;
    0 -> steps;
    [] -> q;
    /*  Initialise awm, been, steps and q.  */

    until steps > n do
        look_around();
        /*  Look in all directions, merging retinal images with awm.  */

        bug_message( 'Looking for neighbours' );
        neighbours() -> neighs;
        bug_message( 'Found ' >< neighs );
        /*  Find points on the periphery of vision which are to be
            explored later.
        */

        for neigh in neighs do
            if not( q matches [ == neigh == ] ) then
                [ ^^q ^neigh ] -> q;
            endif;
        endfor;
        explore_debug( 'Queue:' );
        explore_debug( q );
        /*  Append to the queue those points which aren't already
            in it.
        */

        if q = [] then return endif;
        /*  Return if no more points to explore.  */

        q --> [ ?endpoint ??q ];
        bug_message('Taking next entry: '><endpoint);
        /*  Take next point to move to.  */

        awm_position( awm ) -> here;
        bug_message('Finding path from '><here><' to '><endpoint);
        expl_safe_path( here, endpoint ) -> path;
        explore_debug( 'Path:' );
        explore_debug( path );
        explore_debug( 'Heading:' );
        /*  Find a path to it.  */

        explore_debug( awm_forwardvector(awm) );
        path_to_moves( awm_forwardvector(awm), path ) -> new_heading -> moves;
        explore_debug( 'Moves:' );
        explore_debug( moves );
        /*  Convert that into a list of moves.  */           

        follow_moves( moves );
        /*  And follow those moves.  */            
    enduntil
enddefine;


/*  look_around():
        Turn right round, and merge the retinal image with awm
        after each right-angle rotation.
*/
define look_around();
    lvars i;
    for i to 4 do
        awm_merge_retina( awm, retina() );
        bug_message( 'Looking around' );
        exec( "left" );
        awm_left( awm );
    endfor;
enddefine;


/*  neighbours():
        Return a list of up to four elements each of which is a point
        ahead of Bug near the edge of its vision. Endpoints are
        considered for each of Bug's four possible orientations, but
        only if it can find a path to them.
*/
define neighbours();
    lvars i;
    [%  for i to 4 do
            far_point();
            awm_left( awm );
        endfor;
    %]
enddefine;


/*  far_point():
        Return one such endpoint, for Bug's current heading.

        A more intelligent version of this routine would not discard
        the endpoint if no path exists, but would try other points
        around it.
*/
define far_point();
    lvars endpoint;
    awm_position(awm) ++ 5***awm_forwardvector(awm) -> endpoint;
    if not( member(endpoint,been) ) and
       expl_safe_path( awm_position(awm), endpoint ) /= false then
        endpoint
    endif;
enddefine;


/*  expl_safe_path( here, there ):
        Returns false if there is no path between here and there,
        otherwise the path as a list of points.
*/
define expl_safe_path( here, there );
    lvars here, there;

    define safe(awm,x,y);
        lvars awm, x, y;
        awm(x,y) = ` `
    enddefine;

    awm_path( awm, here, there, safe );
enddefine;


/*  follow_moves( path ):
        Path is a list of position-changing moves. Obey them, updating
        the expected position in the AWM after each. Add each new
        position to been, so that explore doesn't consider it as
        a new point to visit. Also, increment steps.
*/
define follow_moves( moves );
    lvars moves;
    lvars m, here;
    for m in moves do
        bug_message( 'Following a path' );
        exec(m);
        awm_move( awm, move );
        awm_position(awm) -> here;
        if not( member( here, been ) ) then
            [ ^here ^^been ] -> been;
        endif;
        1 + steps -> steps;
    endfor;
enddefine;


/*
Debugging
---------
*/


vars explore_trace = false;
/*  Make this true for debug output in the view window.  */      


define explore_debug( message );
    if explore_trace then bug_view (pr(message); 1.nl) endif;
enddefine;
