;;; -*- Mode:Common-Lisp; Package:NISP; Base:10 -*-
;;; NILS Control Structures.  Depends on BASE and IO for macros.
;;; Copyright (C) 1988, Drew McDermott, Yale University (see "copyright" file).
(IN-PACKAGE :NISP)

;; EARROR: (earror fun val -msgs-) does a msg then opens a break loop.
;; If user continues normally, evaluates and returns val.
;; 5.17.88 added -NOCONTINUE and -NOVALUE options
(DEFMACRO EARROR (FNAME DEFAULTVAL . MSGSTUFF)
    `(PROGN (TTYMSG 0 "ERROR -- " ',FNAME " broken"
		    0 ,@MSGSTUFF T)
	    ,(COND ((EQ DEFAULTVAL '-NOCONTINUE)
		    `(ERROR "Fatal EARROR break"))
		   ((EQ DEFAULTVAL '-NOVALUE)
		    `(CERROR "I will attempt to proceed" "EARROR break"))
		   (T
		    `(MULTIPLE-VALUE-BIND (USE-EARROR-DEFAULTVAL EARROR-USERVAL)
					  (EARROR-EAR)
                        (IF USE-EARROR-DEFAULTVAL
			    ,DEFAULTVAL
			    EARROR-USERVAL)))   ))   )

(PROCLAIM '(SPECIAL NOW-LOADING*))

(DEFUN EARROR-EAR ()
   (LET ((NOW-LOADING* NIL))
      (CERROR "You will be prompted for return value"
	      "EARROR break")
      (SRMMSG TTYOUT*
	      0 "Type RETURN <value> to proceed; OK for default value: " T)
      (CLEAR-INPUT TTYIN*)
      (LET ((R (SRMLINEREAD TTYIN*)))
	 (COND ((OR (NULL R) (EQ (CAR R) 'OK))
		(VALUES T NIL))
	       ((EQ (CAR R) 'RETURN)
		(VALUES NIL (EVAL (CADR R))))
	       (T
		(VALUES NIL (EVAL (CAR R))))   ))))

;;; --------------------------- MAPPERS

(SUBR-SYNONYM 'MAPELTLIST 'MAPCAR)
(SUBR-SYNONYM 'MAPELEMLIST 'MAPCAR)
(SUBR-SYNONYM 'MAPTAILLIST 'MAPLIST)
(SUBR-SYNONYM 'MAPELTDO 'MAPC)
(SUBR-SYNONYM 'MAPELEMDO 'MAPC)
(SUBR-SYNONYM 'MAPTAILDO 'MAPL)
(SUBR-SYNONYM 'MAPELTCONC 'MAPCAN)
(SUBR-SYNONYM 'MAPELEMCONC 'MAPCAN)
(SUBR-SYNONYM 'MAPTAILCONC 'MAPCON)

(DEFMACRO MAPTAILAPPEND (PROC . LISTS) (MAPPEND-EXPAND PROC LISTS T)   )
(DEFMACRO MAPELTAPPEND (PROC . LISTS) (MAPPEND-EXPAND PROC LISTS NIL)  )
(DEFMACRO MAPELEMAPPEND (PROC . LISTS) (MAPPEND-EXPAND PROC LISTS NIL)  )

;; 10.2.87: Denys patch: LAMBDA body may begin with declarations, or with
;;  forms that expand into declarations; PROGN doesn't allow for this.
(DEFUN MAPPEND-EXPAND (PROC LISTS TAIL-SW)
   `(,(IF TAIL-SW 'MAPCON 'MAPCAN)
     #',(LET ((VARS (NSYMS (LENGTH LISTS))))
	     `(LAMBDA ,VARS
		      (COPY-LIST ,(CONS-FUNCALL PROC VARS)) ))
     . ,LISTS)   )

(DEFUN UNFQUOT (FF)
   (COND ((ATOM FF) FF)
	 ((MEMQ (CAR FF) '(FUNCTION FUNKTION QUOTE)) 
	  (CADR FF))
	 (T FF)   )) 

(DEFUN CONS-FUNCALL (F ARGL)
   (COND ((AND (IS-PAIR F) (MEMQ (CAR F) '(FUNCTION FUNKTION QUOTE)))
          `(,(CADR F) . ,ARGL))
         ((AND (IS-PAIR F) (EQ (CAR F) '\\))
          `((LAMBDA . ,(CDR F)   ) . ,ARGL))
         (T `(FUNCALL ,F . ,ARGL))   ))

(DEFMACRO MAPELTCOLLECT (PROC . LISTS) (MAPCOL-EXPAND PROC LISTS NIL)   )
(DEFMACRO MAPELEMCOLLECT (PROC . LISTS) (MAPCOL-EXPAND PROC LISTS NIL)   )
(DEFMACRO MAPTAILCOLLECT (PROC . LISTS) (MAPCOL-EXPAND PROC LISTS T)   )

(DEFUN MAPCOL-EXPAND (PRED ARGS TAIL-SW)
   (LET ((VARS (NSYMS (LENGTH ARGS))))
      `(,(IF TAIL-SW 'MAPCON 'MAPCAN)
        #'(LAMBDA ,VARS
             (COND (,(CONS-FUNCALL PRED VARS) (LIST ,(LASTELT VARS)))
                   (T NIL)   ))
        . ,ARGS)   ))
; Note: MAP..COLLECT could be redefined to return multiple values for
; multiple lists, but then it might cons up a list to be discarded.

(DEFMACRO MAPTAILOR (PROC . LISTS) (MAPCOR-EXPAND PROC LISTS T)   )
(DEFMACRO MAPELTOR (PROC . LISTS) (MAPCOR-EXPAND PROC LISTS NIL)   )
(DEFMACRO MAPELEMOR (PROC . LISTS) (MAPCOR-EXPAND PROC LISTS NIL)   )

(DEFUN MAPCOR-EXPAND (PROC LISTS TAIL-SW)
   (LET ((ATOMS (NSYMS (LENGTH LISTS))))
   `(DO ,(MAPDOVARS ATOMS LISTS)
        ((NOT ,(MAPFTEST ATOMS))
         (VALUES . ,(MAPCAR (\\ (X) (IGNORE X) 'NIL   ) ATOMS)))
      (COND (,(MAPTEST PROC ATOMS TAIL-SW) (RETURN (VALUES . ,ATOMS)))   ))))

(SUBR-SYNONYM 'MAPELTSOME 'SOME)
(SUBR-SYNONYM 'MAPELEMSOME 'SOME)

(DEFMACRO MAPTAILSOME (PRED . LISTS)
   (LET ((ATOMS (NSYMS (LENGTH LISTS))))
      `(DO (\!RES!
            . ,(MAPDOVARS ATOMS LISTS))
           ((NOT ,(MAPFTEST ATOMS)) NIL)
         (SETQ \!RES! ,(MAPTEST PRED ATOMS T))
         (COND (\!RES! (RETURN \!RES!))   ))))

(SUBR-SYNONYM 'MAPELTAND 'EVERY)
(SUBR-SYNONYM 'MAPELEMAND 'EVERY)

(DEFMACRO MAPTAILAND (PRED . LISTS)
   (LET ((ATOMS (NSYMS (LENGTH LISTS))))
      `(DO ,(MAPDOVARS ATOMS LISTS)
           ((NOT ,(MAPFTEST ATOMS)) T)
         (COND ((NOT ,(MAPTEST PRED ATOMS T)) (RETURN NIL))   ))   ))

(DEFMACRO MAPELTREDUCE (PROC IDENT . LISTS) (REDUCE-EXPAND PROC IDENT LISTS NIL)  )
(DEFMACRO MAPELEMREDUCE (PROC IDENT . LISTS) (REDUCE-EXPAND PROC IDENT LISTS NIL)  )
(DEFMACRO MAPTAILREDUCE (PROC IDENT . LISTS) (REDUCE-EXPAND PROC IDENT LISTS T)  )

(DEFUN REDUCE-EXPAND (PROC IDENT LISTS TAIL-SW)
   (LET ((ATOMS (NSYMS (LENGTH LISTS))))
      `(DO ((\!RES! ,IDENT ,(CONS-FUNCALL PROC
                                          `(\!RES! . ,(MAPARGS ATOMS TAIL-SW))))
            . ,(MAPDOVARS ATOMS LISTS))
           ((NOT ,(MAPFTEST ATOMS)) \!RES!)   )))

(DEFUN MAPDOVARS (ATOMS LISTS)
   (MAPCAR (\\ (A L) `(,A ,L (CDR ,A))   )
           ATOMS LISTS)   )

(DEFUN MAPFTEST (ATOMS)
   (COND ((NULL ATOMS) 'T)
         ((NULL (CDR ATOMS)) (CAR ATOMS))
         (T (CONS 'AND ATOMS))   ))

(DEFUN MAPTEST (PROC ATOMS TAIL-SW)
   (CONS-FUNCALL PROC (MAPARGS ATOMS TAIL-SW))   )

(DEFUN MAPARGS (ATOMS TAIL-SW)
   (MAPCAR (\\ (A) (IF TAIL-SW A `(CAR ,A)) )
           ATOMS)   )

(DEFMACRO SELQ (&REST B) `(CASE . ,B))

;; BIND: "Like LET, but binds dynamically." [NISP Manual p.41]
;; VAX version was wrong: (DM BIND (ARGS\: . B) `(LET . ,B))
;; This version pervasively affects references (but not bindings) ** Wright

(DEFMACRO BIND (VARS-N-VALS &BODY BODY)
  ;; use normal let
  `(LET ,VARS-N-VALS
     ;; if bindings, declare variables to be special
     ,@(IF VARS-N-VALS
         `((LISP:DECLARE (SPECIAL
                          ,@(MAPCAR #'(LAMBDA (VAR-N-VAL)
                                        (IF (CONSP VAR-N-VAL)
                                            (CAR VAR-N-VAL)
                                            VAR-N-VAL))
                                    VARS-N-VALS))))
         NIL)
     ,@BODY))


(DEFMACRO FLABELS (&REST STUFF) `(LABELS . ,STUFF))

(DEFMACRO REPEAT (&REST BODY) `(LOOP ,@BODY))

;Super LOOP macro.  Syntax: (LOOP [FOR (variable-specs)] -statements-)
;variable-spec = symbol | (sym init-val [bump]) | (sym IN list)
;         (The last of these is data-driven and can be generalized.)
;statement = WHILE test | UNTIL test | RESULT [IS] value | action
;Semantics: The variables are initialized, and statements are executed.
;If a test indicates termination, then the next RESULT value is evaluated
;and returned as the value of the loop.  The default value is NIL.
;(Instead of RESULT, you may write RESULT = or RETURN.  Also, any other atoms
;are ignored, so you may write DEFAULT RESULT IS or ELSE RETURN, etc.)
;At the end of the statements, if no test has succeeded, the variables
;are bumped and the statements are re-evaluated.
;Example:
;(LOOP FOR ((N 1 (+ N 1)) (X IN L) Y)
; WHEN L EMPTY RESULT IS 'NOMORE  ; Note implicit test of L
; UNTIL (> N 10) WHEN RETURN 'MORE-THAN-TEN
;   (!= Y (REVERSE X))
;   (PRINT (LIST N Y))
; REPEAT)

; This is upward-compatible with CL's eunuchy version.  It may not
; be compatible with various implementations, but worry about that later.
(DEFMACRO LOOP (&REST STUFF)
   (LET ((L STUFF) (VARS NIL) (FIRSTS NIL) DECLARATIONS)
      (COND ((EQ (CAR L) 'FOR)
             (SETF VARS (ANALYZE-LOOP-VARS (CADR L)))
	     (MULTIPLE-VALUE-SETQ (DECLARATIONS L) 
				  (DECLARATIONS-SEPARATE (CDDR L)))
             (SETF L (APPEND (CADR VARS) L (CADDR VARS)))
             (SETF VARS (CAR VARS)))
	    (T
             (MULTIPLE-VALUE-SETQ (DECLARATIONS L) 
				  (DECLARATIONS-SEPARATE L)))   )
      (DO ((BODY NIL BODY)
           (FOUNDTEST NIL FOUNDTEST)
           (RETNUM 1 RETNUM)
           (RETURNS NIL RETURNS)
           (PENDINGRETS NIL PENDINGRETS))
          ((NULL L)
           (STUFFRETS PENDINGRETS NIL)
           (COND ((NOT FOUNDTEST)
                  (SRMMSG (ERROUT)
                          "Warning-- loop without a test "
                          `(LOOP . ,STUFF) T))   )
	   ;; 5.16.88 Denys patch put ,@DECLARATIONS before ,@FIRSTS
           `(PROG ,VARS ,@DECLARATIONS
		  ,@FIRSTS
                  L_O_O_P ,@BODY (GO L_O_O_P) . ,RETURNS))
        (COND ((MEMQ (CAR L) '(FIRST BEFORE))
	       (SETF FIRSTS (LIST (CADR L)))
	       (SETF L (CDDR L)))
	      ((EQ (CAR L) ':INFINITE)
	       (SETF FOUNDTEST T)
	       (SETF L (CDR L)))
	      ((MEMQ (CAR L) '(WHILE UNTIL))
               (SETF FOUNDTEST T)
               (LET ((CONDITIONAL
                      `(COND (,(COND ((EQ (CAR L) 'WHILE) `(NOT ,(CADR L)))
                                     (T (CADR L))   )
                              (GO ,(SYMBOL RET (< RETNUM))))   )))
                  (SETF PENDINGRETS (CONS (CDADR CONDITIONAL) PENDINGRETS))
                  (SETF L (CDDR L))
                  (SETF BODY (APPEND BODY (LIST CONDITIONAL)))   ))
               ((MEMQ (CAR L) '(ELSE-RESULT-IS WHEN-RESULT-IS RESULT RETURN))
                (COND ((MEMQ (CADR L) '(IS = EQUALS))
                       (SETF L (CDR L)))   )
                (COND (PENDINGRETS
                       (COND ((OR (NULL (CDR PENDINGRETS))
                                  (ATOM (CADR L))
                                  (EQ (CAADR L) 'QUOTE))
                              (STUFFRETS PENDINGRETS (CADR L)))
                             (T
                              (SETF RETURNS `(,(SYMBOL RET (< RETNUM))
                                              (RETURN ,(CADR L))
                                              . ,RETURNS)))   ))
                      (T (SRMMSG (ERROUT) T
                                 "Warning -- unused return " (CADR L) T))   )
                (SETF RETNUM (FX+ RETNUM 1))
                (SETF PENDINGRETS NIL)
                (SETF L (CDDR L)))
               ((ATOM (CAR L)) (SETF L (CDR L)))
               (T (SETF BODY (APPEND BODY (LIST (CAR L))))
                  (SETF L (CDR L)))   ))))

; Analyze loop variables, returning a list of three things:
; variable bindings; initialization for variables; cleanups for variables.
(DEFUN ANALYZE-LOOP-VARS (BDGS)
   (LET ((VARS NIL) (VARSYMS SYMS*)
         (FRONTSTEPS NIL) (BACKSTEPS NIL))
      (DO ((BDGS BDGS (CDR BDGS)))
          ((NULL BDGS) (LIST VARS FRONTSTEPS BACKSTEPS))
         (COND ((ATOM (CAR BDGS))
                (SETF VARS (APPEND VARS (LIST (CAR BDGS)))))
               ((AND (SYMBOLP (CADAR BDGS))
                     (GET (CADAR BDGS) 'LOOPVARSPEC))
                (LET ((LOOPVARSPEC
                       (FUNCALL (GET (CADAR BDGS) 'LOOPVARSPEC)
                                (CAR BDGS) VARSYMS)))
                   (SETF VARS (APPEND VARS (CAR LOOPVARSPEC)))
                   (SETF FRONTSTEPS (APPEND FRONTSTEPS (CADR LOOPVARSPEC)))
                   (SETF BACKSTEPS (APPEND BACKSTEPS (CADDR LOOPVARSPEC)))
                   (SETF VARSYMS (CADDDR LOOPVARSPEC))   ))
               (T (SETF VARS
                      (APPEND VARS `((,(CAAR BDGS) ,(CADAR BDGS)))))
                  (COND ((CDDAR BDGS)
                         (SETF BACKSTEPS
                             (APPEND BACKSTEPS
                                     `((SETF ,(CAAR BDGS)
                                             ,(CADDAR BDGS))))))   ))   )
      )))
; PENDINGRETS is a list of places that want to return from the loop.
; To be precise, each element of PENDINGRETS is a tail of a COND
; clause, whose CAR is of the form (GO <returntag>).  If it turns out
; we don't need a special return tag, then we go back and change these
; to (RETURN <value>).
(DEFUN STUFFRETS (PENDINGRETS VAL)
   (DO ((PL PENDINGRETS (CDR PL)))
       ((NULL PL) NIL)
      (SETF (CAR (CAR PL)) `(RETURN ,VAL))   ))

;Data structure used by loop variable definers (IN, =)
(DEFUN LOOPVARCODE (BDGS FRONTCODE BACKCODE REMVARS)
   (LIST BDGS FRONTCODE BACKCODE REMVARS)   )

;;Code to allow you to say (LOOP FOR ((var IN list)) ...)
;; Nov.4.87 modified
;; We are slipping an initial value for the variable in here.
(DEFUN IN-LOOP-SPEC (BDG VARSYMS)
   (LOOPVARCODE `((,(CAR BDG) ,(COND ((CDDDR BDG) (CADDDR BDG)) (T 'NIL)   ))
		  (,(CAR VARSYMS) ,(CADDR BDG)))
                `(WHILE ,(CAR VARSYMS)
                   (SETF ,(CAR BDG) (CAR ,(CAR VARSYMS))))
                `((SETQ ,(CAR VARSYMS) (CDR ,(CAR VARSYMS))))
                (CDR VARSYMS))   )

(SETF (PROP 'LOOPVARSPEC 'IN) #'IN-LOOP-SPEC)

;Code to allow you to say (LOOP FOR ((var = init [TO final] [BY incr])) ...)
;If present, final is computed at beginning.
;WARNING: Positive increment assumed unless incr is negative constant!!
;If both incr and final are omitted, variable is only initialized.
;If only incr omitted, defaults to 1.
;If only final is omitted, variable is incremented but not tested.
(DEFUN INCR-LOOP-SPEC (BDG VARSYMS)
   (LET ((VAR (CAR BDG)) (INIT (CADDR BDG))
         (FINAL (COND ((MEMQ 'TO BDG) (CADR (MEMQ 'TO BDG)))
                       (T NIL)   ))
         (INCR (COND ((MEMQ 'BY BDG) (CADR (MEMQ 'BY BDG)))
                     ((MEMQ 'TO BDG) 1)
                     (T NIL)   )))
      (LOOPVARCODE `((,VAR ,INIT)
                     ,@(INCLUDE-IF FINAL `(,(CAR VARSYMS) ,FINAL)))
                   (COND (FINAL
                          `(UNTIL
                             (,(COND ((AND (NUMBERP INCR) (< INCR 0)) 'FX<)
                                     (T 'FX>)   )
                              ,VAR
                              ,(CAR VARSYMS))))
                         (T NIL)   )
                   (COND (INCR `((SETF ,VAR (FX+ ,VAR ,INCR))))
                         (T NIL)   )
                   (COND (FINAL (CDR VARSYMS))
                         (T VARSYMS)   ))   ))

(SETF (PROP 'LOOPVARSPEC '=) #'INCR-LOOP-SPEC)
(SETF (PROP 'LOOPVARSPEC 'FROM) #'INCR-LOOP-SPEC)

(DEFMACRO FOR (&REST STUFF)
   (LET (VARS LISTS (WHENCLAUSE NIL) MAINCLAUSE DECLARATIONS)
      (MULTIPLE-VALUE-SETQ (VARS LISTS MAINCLAUSE)
                           (FOR-ANALYZE STUFF))
      (MULTIPLE-VALUE-SETQ (DECLARATIONS MAINCLAUSE)
			   (DECLARATIONS-SEPARATE MAINCLAUSE))
      (SETF WHENCLAUSE
            (MAPELTOR (\\ (C) (CAR-EQ C 'WHEN)   ) MAINCLAUSE))
      (COND (WHENCLAUSE
             (SETF WHENCLAUSE (CAR WHENCLAUSE))
             (SETF MAINCLAUSE (REMOVE1Q WHENCLAUSE MAINCLAUSE)))   )
      (COND ((AND VARS
                  (= (LEN MAINCLAUSE) 1)
                  (IS-PAIR (CAR MAINCLAUSE))
                  (MEMQ (CAAR MAINCLAUSE) '(SAVE SPLICE FILTER)))
             (SETF MAINCLAUSE (CAR MAINCLAUSE))
             (COND (WHENCLAUSE
                    (WHENCLAUSE-FOR VARS LISTS 
				    WHENCLAUSE MAINCLAUSE DECLARATIONS))
                   (T
                    (NOWHEN-FOR VARS LISTS MAINCLAUSE DECLARATIONS))   ))
            (T
             (EARROR FOR NIL "Bad syntax: " `(FOR . ,STUFF)))   )))

(DEFUN NOWHEN-FOR (VARS LISTS MAINCLAUSE DECLARATIONS)
   (SELQ (CAR MAINCLAUSE)
      (SAVE
       `(MAPELTLIST (\\ ,VARS ,@DECLARATIONS . ,(CDR MAINCLAUSE)   ) . ,LISTS))
      (SPLICE
       `(MAPELTCONC (\\ ,VARS ,@DECLARATIONS . ,(CDR MAINCLAUSE)   ) . ,LISTS))
      (T
       `(MAPELTCONC
           (\\ ,VARS
              ,@DECLARATIONS
              (LET ((\!RES! (PROGN . ,(CDR MAINCLAUSE))))
                 (COND (\!RES! (LIST \!RES!))
                       (T NIL)   ))   )
           . ,LISTS))   ))

(DEFUN WHENCLAUSE-FOR (VARS LISTS WHENCLAUSE MAINCLAUSE DECLARATIONS)
  (LET ((WHENTEST (COND ((= (LEN (CDR WHENCLAUSE)) 1) (CADR WHENCLAUSE))
                        (T `(PROGN . ,(CDR WHENCLAUSE)))   )))
    (COND ((EQ (CAR MAINCLAUSE) 'SAVE)
           (COND ((AND (= (LEN VARS) 1)
                       (EQ (CADR MAINCLAUSE) (CAR VARS)))
                  `(MAPELTCOLLECT
                      (\\ (,(CAR VARS)) ,@DECLARATIONS ,WHENTEST   )
                      ,(CAR LISTS)))
                 (T
                  `(MAPELTCONC
                      (\\ ,VARS 
			  ,@DECLARATIONS
			  (COND (,WHENTEST (LIST ,(CADR MAINCLAUSE)))
                                (T NIL)   ))
                      . ,LISTS))   ))
         ((EQ (CAR MAINCLAUSE) 'SPLICE)
          `(MAPELTCONC
              (\\ ,VARS 
		  ,@DECLARATIONS
		  (COND (,WHENTEST ,(CADR MAINCLAUSE)) (T NIL)   ))
              . ,LISTS))
         (T ; filter
          `(MAPELTCONC
              (\\ ,VARS 
		  ,@DECLARATIONS
		  (COND (,WHENTEST
                         (LET ((\!RES! ,(CADR MAINCLAUSE)))
                            (COND (\!RES! (LIST \!RES!)) (T NIL)   )))
                        (T NIL)   ))
              . ,LISTS))   )))

(DEFMACRO FORALL (&REST STUFF)
   (MULTIPLE-VALUE-BIND (VARS LISTS BODY) (FOR-ANALYZE STUFF)
      `(MAPELTAND (\\ ,VARS . ,BODY) . ,LISTS)   ))

(DEFMACRO EXISTS (&REST STUFF)
   (MULTIPLE-VALUE-BIND (VARS LISTS BODY) (FOR-ANALYZE STUFF)
      `(MAPELTOR (\\ ,VARS . ,BODY) . ,LISTS)   ))

(DEFUN FOR-ANALYZE (STUFF)
      (DO ((L STUFF (CDR L))
           (VARS NIL (CONS (CAAR L) VARS))
           (LISTS NIL (CONS (CADDAR L) LISTS)))
          ((OR (NULL L)
               (ATOM (CAR L))
               (NOT (EQ (CADAR L) 'IN)))
           (VALUES (DREVERSE VARS) (DREVERSE LISTS) L))   ))

; Return two values: declarations, everything else
(DEFUN DECLARATIONS-SEPARATE (BODY)
   (DO ((B BODY (CDR B))
	(DECLARATIONS NIL (CONS (CAR B) DECLARATIONS)))
       ((OR (ATOM (CAR B))
	    (NOT (MEMQ (CAAR B) '(DECLARE IGNORE))))
	(VALUES (DREVERSE DECLARATIONS) B))   ))

(DEFUN CAR-EQ (X Y) (AND (IS-PAIR X) (EQ (CAR X) Y))   )

(DEFMACRO INTERCEPT (LABEL &REST BODY)
   `(CATCH ',LABEL . ,BODY)   )

(DEFMACRO PASS (LABEL VALUE) `(THROW ',LABEL ,VALUE)   )

(DEFMACRO ONE-VALUE (X) `(VALUES ,X)   )

(SUBR-SYNONYM 'LIST->VALUES 'VALUES-LIST)

(DEFMACRO MULTIPLE-VALUE-LET (VARS E &REST L)
   `(MULTIPLE-VALUE-BIND ,VARS ,E ,@(IGNORE-CONVERT L))   )

; (DATAFUN master sym def) defines a new procedure and puts it
; on the property list of sym under the indicator master.
; master is used by some function (often with the name master)
; for data-driven hacks.  def is an ordinary function definition,
; with the name omitted.  E.g., (DATAFUN PRINTMACRO COND (DEFUN ...)).
; If the def is just another symbol, then that means that the
; two symbols behave equivalently.
; If it is of the form (FUNCTION name), then that function is
; used.
; Some of the complexity of this machinery is wasted, but we'll
; leave it as is in case we ever need it.
(DEFMACRO DATAFUN (MASTER SYM DEF)
   (LET (FUNAME)
      (COND ((ATOM DEF)
             (SETF FUNAME (SYMBOL (< DEF) - (< MASTER)))
             `(DECLARE-DATAFUN ',FUNAME ',MASTER ',SYM NIL))
            ((MEMQ (CAR DEF) '(FUNCTION FUNKTION))
             `(DECLARE-DATAFUN ',(CADR DEF) ',MASTER ',SYM NIL))
            (T
             (SETF FUNAME (SYMBOL (< SYM) - (< MASTER)))
             `(*SPLICE*
                (SYMBOLFUN
                   (,(CAR DEF) ,FUNAME . ,(CDR DEF)))
                (DECLARE-DATAFUN ',FUNAME ',MASTER
                                 ',SYM T)))   )))

; The DATAFUN property of a function name is a list of the form
; (master sym1 sym2 ...)
; This specifies that this function is to be used for the given
;master and symbols.  sym1, if non-(), is the "main symbol"--
; funame= (SYMBOL (< sym1) - (< master)).

(DEFUN DECLARE-DATAFUN (FUNAME MASTER SYM MAIN)
   (LET ((SPEC (PROP 'DATAFUN FUNAME)))
      (COND ((NULL SPEC)
             (SETF SPEC (LIST MASTER NIL))
             (SETF (PROP 'DATAFUN FUNAME) SPEC))   )
      (COND (MAIN (SETF (CADR SPEC) SYM))
            (T
             (SETF (CDDR SPEC) (ADJOINQ SYM (CDDR SPEC))))   )
      (LET ((ATTACHFN (PROP 'ATTACH-DATAFUN MASTER)))
         (COND (ATTACHFN
                (FUNCALL ATTACHFN MASTER SYM FUNAME))
               (T
                (SETF (PROP MASTER SYM) (SYMBOL->FUN FUNAME)))   ))))

; 87.9.30: Add to CONTROL.L and CONTROL.T
(DEFMACRO DATAFUN-TABLE (NAME IND)
  `(NEEDED-BY-MACROS
      (DEFVAR ,NAME (MAKE-EQ-HASH-TABLE))
      (DATAFUN ATTACH-DATAFUN ,IND
         (DEFUN (IND SYM FNAME)
	    (IGNORE IND)
            (SETF (TABLE-ENTRY ,NAME SYM) (SYMBOL->FUN FNAME))   )))   )

