/* 
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#ifdef ultrix
#include <fcntl.h>
#else
#include <sys/fcntl.h>
#endif
#include <sys/param.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include "sound.h"
#include "rplayd.h"
#include "cache.h"
#ifdef MMAP
#include <sys/mman.h>
#endif /* MMAP */
#include "ulaw.h"
#include "hash.h"
#include "server.h"
#include "buffer.h"
#include "connection.h"

SOUND		*sounds = NULL;
int		sound_count = 0;

/*
 * read the rplay configuration file and load the hash table
 */
#ifdef __STDC__
void	sound_read(char *filename)
#else
void	sound_read(filename)
char	*filename;
#endif
{
	FILE	*fp;
	char	buf[MAXPATHLEN], *p;

	fp = fopen(filename, "r");
	if (fp == NULL)
	{
		report("warning: cannot open %s\n", filename);
		/*
		 * I guess it's ok to not have any local sounds
		 */
		return;
	}

	while (fgets(buf, sizeof(buf), fp) != NULL)
	{
		switch (buf[0])
		{
		case '#':
		case ' ':
		case '\t':
		case '\n':
			continue;
		}
		p = strchr(buf, '\n');
		if (p)
		{
			*p = '\0';
		}
		sound_insert(buf, SOUND_READY);
	}

	fclose(fp);
}

BUFFER	*sound_list_create()
{
	SOUND	*s;
	int	n;
	BUFFER	*start, *b;

	b = buffer_create();
	start = b;
	strcat(b->buf, "+sounds\r\n");
	b->nbytes += strlen(b->buf);

	for (s = sounds; s; s = s->list)
	{
		if (s->status != SOUND_READY)
		{
			continue;
		}
		n = strlen(s->name) + 2;
		if (b->nbytes + n > BUFFER_SIZE)
		{
			b->next = buffer_create();
			b = b->next;
		}
		strcat(b->buf, s->name);
		strcat(b->buf, "\r\n");
		b->nbytes += n;
	}

	if (b->nbytes + 3 > BUFFER_SIZE)
	{
		b->next = buffer_create();
		b = b->next;
	}
	strcat(b->buf, ".\r\n");
	b->nbytes += 3;
	
	return start;
}

#ifdef __STDC__
SOUND	*sound_insert(char *path, int status)
#else
SOUND	*sound_insert(path, status)
char	*path;
int	status;
#endif
{
	SOUND	*s;
	char	*hash_val;

	s = sound_create();
	if (s == NULL)
	{
		report("out of memory\n");
		done(1);
	}

	s->path = strdup(path);
	s->hash_key = strdup(hash_name(s->path));
	s->name = s->path[0] == '/' ? strrchr(s->path, '/') + 1 : s->path;
	s->status = status;

	hash_val = hash_get(s->hash_key);
	if (hash_val == NULL)
	{
		hash_put(s->hash_key, (char *)s);

		s->list_prev = NULL;
		s->list = sounds;
		if (sounds)
		{
			sounds->list_prev = s;
		}
		sounds = s;
	}
	else
	{
		SOUND	*ss, *prev = NULL;

		for (ss = (SOUND *)hash_val; ss; prev = ss, ss = ss->next)
		{
			if (strcmp(ss->path, s->path) == 0)
			{
				free((char *)s);
				return ss;
			}
		}
		prev->next = s;
		s->prev = prev;
	}

	return s;
}

/*
 * create a sound
 */
#ifdef __STDC__
SOUND	*sound_create(void)
#else
SOUND	*sound_create()
#endif
{
	SOUND	*s;

	s = (SOUND *)malloc(sizeof(SOUND));
	if (s == NULL)
	{
		return s;
	}
	s->path = NULL;
	s->name = NULL;
	s->hash_key = NULL;
	s->buf = NULL;
	s->start = NULL;
	s->stop = NULL;
	s->size = 0;
	s->next = NULL;
	s->prev = NULL;
	s->list = NULL;
	s->list_prev = NULL;
	s->status = SOUND_NULL;
	s->count = 0;

	return s;
}

/*
 * lookup the sound name in the hash table and load
 * the sound file if necessary
 */
#ifdef __STDC__
SOUND	*sound_lookup(char *name, int mode)
#else
SOUND	*sound_lookup(name, mode)
char	*name;
int	mode;
#endif
{
	SOUND		*s;
	int		has_extension = 0, has_pathname = 0;
	struct stat	st;
	
	s = (SOUND *)hash_get(hash_name(name));
	if (s == NULL)
	{
		if (debug)
		{
			report("%s not found\n", name);
		}
		
		if (mode == SOUND_DONT_FIND || mode == SOUND_DONT_COUNT)
		{
			return NULL;
		}
		
		/*
		 * see if a local file can be loaded
		 */
		if (stat(name, &st) == 0)
		{
			if (debug)
			{
				report("loading local file %s\n", name);
			}
			s = sound_insert(name, SOUND_READY);
		}
		else if (mode == SOUND_FIND)
		{
			if (name[0] == '/')
			{
				if (debug)
				{
					report("not searching for %s\n", name);
				}
				return NULL;
			}
			if (servers)
			{
				if (debug)
				{
					report("searching for %s\n", name);
				}
				s = sound_insert(cache_name(name), SOUND_SEARCH);
				connection_server_open(servers, s);
			}
			else
			{
				return NULL;
			}
		}
	}
	else
	{
		/*
	 	 * make sure the sound found has the correct path
	 	 * and dot extension
	 	 */
		if (name[0] == '/')
		{
			has_pathname++;
		}
		else if (strrchr(name, '.'))
		{
			has_extension++;
		}
		for (; s; s = s->next)
		{
			if (has_pathname && strcmp(name, s->path))
			{
				continue;
			}
			if (has_extension && strcmp(name, s->name))
			{
				continue;
			}
			break;
		}
		if (s == NULL)
		{
			if (has_pathname)
			{
				report("pathname %s not in hash table\n", name);
				if (stat(name, &st) == 0)
				{
					if (debug)
					{
						report("loading local file %s\n", name);
					}
					s = sound_insert(name, SOUND_READY);
				}
				else
				{
					return NULL;
				}
			}
			else if (has_extension)
			{
				report("sound with extension %s not in hash table\n", name);
				return NULL;
			}
		}
	}
	
	if (mode == SOUND_DONT_COUNT)
	{
		return s;
	}
	else
	{
		sound_count++;
		s->count = sound_count;
	}

	switch (s->status)
	{
	case SOUND_NOT_READY:
	case SOUND_SEARCH:
		return s;
	}

	if (s->buf == NULL)
	{
		return sound_load(s) == 0 ? s : NULL;
	}
	else
	{
		return s;
	}
}

/*
 * load the sound using mmap or malloc
 */
#ifdef __STDC__
int	sound_load(SOUND *s)
#else
int	sound_load(s)
SOUND	*s;
#endif
{
	int		fd;
	struct stat	st;
	char		*buf;
#ifndef MMAP
	int		n;
#endif /* MMAP */

	if (s->buf != NULL)
	{
		return 0;
	}

	fd = open(s->path, O_RDONLY, 0);
	if (fd < 0)
	{
		if (debug)
		{
			report("open %s: %s\n", s->path, sys_errlist[errno]);
		}
		return -1;
	}

	if (fstat(fd, &st) < 0)
	{
		if (debug)
		{
			report("fstat %s: %s\n", s->path, sys_errlist[errno]);
		}
		return -1;
	}

#ifdef MMAP
	buf = mmap(0, (int)st.st_size, PROT_READ, MAP_SHARED, fd, 0);
	if (buf == (caddr_t)-1)
	{
		if (debug)
		{
			report("file=%s size=%s mmap: %s\n", s->path, st.st_size, sys_errlist[errno]);
		}
		return -1;
	}
#else /* MMAP */
	buf = (char *)malloc((int)st.st_size);
	if (buf == NULL)
	{
		report("out of memory\n");
		done(1);
	}
	n = read(fd, buf, st.st_size);
	if (n != st.st_size)
	{
		if (debug)
		{
			report("read %s (%d): %s\n", s->path, st.st_size, sys_errlist[errno]);
		}
		return -1;
	}
#endif /* MMAP */

	close(fd);
	
	return sound_map(s, buf, (int)st.st_size);
}

#ifdef __STDC__
void	sound_unmap(SOUND *s)
#else
void	sound_unmap(s)
SOUND	*s;
#endif
{
	if (s->buf)
	{
#ifdef MMAP
		if (munmap(s->buf, s->size) < 0)
		{
			report("munmap: %s\n", sys_errlist[errno]);
			done(1);
		}
#else /* MMAP */
		free(s->buf);
#endif /* MMAP */
	}

	s->buf = NULL;
	s->size = 0;
}

#ifdef __STDC__
int	sound_map(SOUND *s, char *buf, int size)
#else
int	sound_map(s, buf, size)
SOUND	*s;
char	*buf;
int	size;
#endif
{
	long	ulaw_hdr_size;
	char	*p;

	s->buf = buf;
	s->size = size;
	
	/*
	 * make sure it is a ulaw audio file
	 */
	if (PORTABLE_LONG(s->buf) != ULAW_MAGIC)
	{
		if (debug)
		{
			report("%s no ulaw header\n", s->path);
		}
		p = strrchr(s->path, '.');
		if (!p || strcmp(p, ".au") && strcmp(p, ".u") && strcmp(p, ".ul"))
		{
			if (debug)
			{
				report("%s no sound extension\n", s->path);
			}
			return -1;
		}
		ulaw_hdr_size = 0;
	}
	else
	{
		/*
		 * ignore most of the header
		 */
		ulaw_hdr_size = PORTABLE_LONG(&s->buf[4]);
		if (ulaw_hdr_size < ULAW_HDRSIZE)
		{
			if (debug)
			{
				report("%s not a ulaw file\n", s->path);
			}
			return -1;
		}
	}

	s->start = s->buf + ulaw_hdr_size;
	s->stop = s->buf + s->size - 1;

	return 0;
}

#ifdef __STDC__
void	sound_delete(SOUND *s)
#else
void	sound_delete(s)
SOUND	*s;
#endif
{
	sound_unmap(s);

	if (s->next == NULL)
	{
		hash_delete(s->hash_key);
		if (s->list_prev)
		{
			s->list_prev->list = s->list;
		}
		else
		{
			sounds = s->list;
		}
		if (s->list)
		{
			s->list->list_prev = s->list_prev;
		}
	}
	else
	{
		if (s->prev)
		{
			s->prev->next = s->next;
			if (s->next)
			{
				s->next->prev = s->prev;
			}
		}
		else if (s->next)
		{
			if (s->list_prev)
			{
				s->list_prev->list = s->next;
			}
			if (s->list)
			{
				s->list->list_prev = s->next;
			}
			hash_put(s->next->hash_key, (char *)s->next);
		}
		else
		{
			if (s->list_prev)
			{
				s->list_prev->list = s->list;
			}
			if (s->list)
			{
				s->list->list_prev = s->list_prev;
			}
		}
	}

	if (unlink(s->path) < 0)
	{
		if (debug)
		{
			report("sound_delete: unlink %s: %s\n", s->path,
				sys_errlist[errno]);
		}
	}

	free((char *)s);
}

/*
 * free memory used by all sounds except the ones that have
 * events associated with them
 */
void	sound_cleanup()
{
	SOUND		*s1, *s2;
	EVENT		*e;
	CONNECTION	*c;
	int		n;

	if (debug)
	{
		report("cleaning up sounds\n");
	}
	
	for (s1 = sounds; s1; s1 = s1->list)
	{
		for (s2 = s1; s2; s2 = s2->next)
		{
			if (s2->status == SOUND_READY && s2->buf)
			{	
				n = 0;
				for (c = connections; c && n == 0; c = c->next)
				{
					for (e = c->event; e && n == 0; e = e->next)
					{
						switch (e->type)
						{
						case EVENT_WRITE_FIND:
						case EVENT_READ_FIND_REPLY:
						case EVENT_WRITE_GET:
						case EVENT_READ_GET_REPLY:
						case EVENT_READ_SOUND:
						case EVENT_WRITE_SOUND:
							if (e->sound == s2)
							{
								n++;
							}
							break;
						}
					}
				}
				if (n == 0)
				{
					sound_unmap(s2);
				}
			}
		}
	}
}
