/* search.c: Exterior hash table implementation.
 *	Copyright (c) August 5, 1993 by Darren Senn
 *					(sinster@scintilla.santa-clara.ca.us)
 *	Permission granted for any use.
 */

#include "conf.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "search.h"

#define PRIME 1993		/*  That's right!  This is a prime year!  :)  */
/* Other "prime" choices:
   1993    1997    1999    2003    2011    2017    2027    2029    2039    2053 
   2063    2069    2081    2083    2087    2089    2099    2111    2113    2129 
   2131    2137    2141    2143    2153    2161    2179    2203    2207    2213 
   2221    2237    2239    2243    2251    2267    2269    2273    2281    2287 
   2293    2297    2309    2311    2333    2339    2341    2347    2351    2357 
   2371    2377    2381    2383    2389    2393    2399    2411    2417    2423 
   2437    2441    2447    2459    2467    2473    2477    2503    2521    2531 
   2539    2543    2549    2551    2557    2579    2591    2593    2609    2617 
   2621    2633    2647    2657    2659    2663    2671    2677    2683    2687 
   2689    2693    2699    2707    2711    2713    2719    2729    2731    2741 
   2749    2753    2767    2777    2789    2791    2797    2801    2803    2819 
   2833    2837    2843    2851    2857    2861    2879    2887    2897    2903 
   2909    2917    2927    2939    2953    2957    2963    2969    2971    2999 
*/

struct node {
	ENTRY data;
	struct node *next;
};

static struct node *__hs_array[PRIME];
static struct node *__hs_head = NULL;
static struct node *__hs_cache = NULL;

#ifdef __STDC__
static int hash(ENTRY e)
#else
static int hash(e)
ENTRY e;
#endif
{
	int sum;
	char *p;

	for (sum = 0, p = (char *)e.key; *p; sum += *p++);
	return(sum % PRIME);
}

#ifdef __STDC__
static struct node *newnode(void)
#else
static struct node *newnode()
#endif
{
	struct node *f = NULL;

	if (__hs_head) {
		f = __hs_head;
		__hs_head = __hs_head->next;
	}
	return(f);
}

#ifdef __STDC__
int hcreate(int num)
#else
int hcreate(num)
int num;
#endif
{
	register int i;

	if (__hs_cache) {
		free((char *)__hs_cache);
		__hs_cache = NULL;
	}

	if (!num) return(0);
	__hs_cache = (struct node *)calloc(num, sizeof(struct node));
	if (!__hs_cache) return(0);

	memset((char *)__hs_array, 0, sizeof(__hs_array));
	for (i = 1; i < num; i++)
		__hs_cache[i-1].next = &__hs_cache[i];
	__hs_cache[num - 1].next = NULL;
	__hs_head = __hs_cache;

	return(num);
}

#ifdef __STDC__
ENTRY *hsearch(ENTRY e, ACTION a)
#else
ENTRY *hsearch(e, a)
ENTRY e;
ACTION a;
#endif
{
	int i;
	struct node *f, *g;

	for (g = NULL, f = __hs_array[(i = hash(e))];
		f && strcmp((char *)f->data.key, (char *)e.key);
		g = f, f = f->next);
	if (a == ENTER) {
		if (!f)	{
			f = __hs_array[i];
			__hs_array[i] = newnode();
			if (!__hs_array[i]) {
				__hs_array[i] = f;
				return(NULL);
			}
			__hs_array[i]->next = f;
			f = __hs_array[i];
		}
		f->data = e;
	} else if (a == DELETE && f) {
		if (g)
			g->next = f->next;
		else	__hs_array[i] = f->next;
		f->next = __hs_head;
		__hs_head = f;
	}
	return(f ? &f->data : NULL);
}
