/* 
   Copyright (c) 1992 by Harald Ljoen, Norwegian Telecom Research.
   This code may be used and redistributed as described in the terms of the GNU
   General Public Licence as published by the Free Software Foundation.


   File:       pcmAlaw.c
   Purpose:    convert from linear or u-law PCM to and from A-law PCM
   Reference:  CCITT recommendation G.711
   Author:     Harald Ljoen, Norwegian Telecom Research
               E-mail: harald.ljoen@tf.tele.no, or hlj@nta.no
               Phone: +47 63 80 91 70, Fax: +47 63 81 00 76
   Functions:
         int pcmA_set_bits(int nbits)
            nbits: the number of bits in linear samples that are input to
               lin2pcmA() and returned from pcmA2lin(). Can be set to anything
               between 13 and CHAR_BITS*sizeof(int), where CHAR_BITS is the number
               of bit in a char. Default is 16 bits per linear sample. Note that
               this does not effect the precision, which is always 13 bits, rather,
               it determines which bits (lsb's or msb's) that are not used. Normally,
               it's convenient to set nbits to either 16 (the default) or the number
               of bits in your int. 
            returns: the number of bits that will be used for future calls to
               lin2pcmA() and pcmA2lin().
         
         unsigned char lin2pcmA(int samp)
            samp: linear sample
            returns: A-law coded PCM sample

         int pcmA2lin(unsigned char pcm)
            pcm: A-law coded PCM sample
            returns: linear sample

         unsigned char pcmA2u(unsigned char AlawSample)
            AlawSample: A-law coded PCM sample
            returns: u-law coded PCM sample

         unsigned char pcmu2A(unsigned char ulawSample)
            ulawSample: u-law coded PCM sample
            returns: A-law coded PCM sample
      
*/

#define CHAR_BITS 8           /* Change this if your char is not 8 bits wide */
#define MAX_MAGNITUDE ((unsigned)(1 << (12 + plusbits)) - 1)

static int plusbits = 3;            /* default 13 + 3 = 16 bits/sample */

int pcmA_set_bits(nbits)
int nbits;
{
   if ((nbits >= 13) && (nbits <= CHAR_BITS*sizeof(int))) plusbits = nbits - 13;
   return(plusbits + 13);
}

unsigned char lin2pcmA(samp)
int samp;
{
/*
   Quantization segment limits:  0..32..64..128..256..512..1024..2048..4096
   Quantization step sizes:       2   2   4    8    16   32    64    128
*/
   int seg, seglimit, sign, mant;
   unsigned char pcmA;

   sign = (samp >= 0);
   if (!sign) samp = -samp;
   if (samp > MAX_MAGNITUDE) samp = MAX_MAGNITUDE;
   seglimit = 1 << (5+plusbits);
   seg = 0;
   while (seglimit <= samp) {
      seglimit <<= 1;
      seg++;
   }
   mant = (samp >> ((seg == 0) ? 1 + plusbits : seg + plusbits)) & 0xf;
   pcmA = (sign << 7) | (seg << 4) | mant;
   return(pcmA ^ 0x55);
}

int pcmA2lin(pcm)
unsigned char pcm;
{
   int samp, mant, seg, sign;

   pcm ^= 0x55;
   mant = pcm & 0xf;
   seg = (pcm >> 4) & 0x7;
   if (seg > 0) mant |= 0x10;
   sign = !(pcm >> 7);

   samp = ((mant << 1) + 1);
   if (seg > 1) samp <<= seg - 1;
   samp = sign ? -samp : samp;
   return(samp << plusbits);
}

unsigned char pcmA2u(AlawSample)
unsigned char AlawSample;
{
   unsigned char ulawSample, sign = AlawSample & 0x80;

   AlawSample = (AlawSample ^ 0x55) & 0x7f;
   if      (AlawSample > 78) ulawSample = AlawSample;
   else if (AlawSample > 62) ulawSample = AlawSample + 1;
   else if (AlawSample > 46) ulawSample = AlawSample + 2;
   else if (AlawSample > 44) ulawSample = AlawSample + 3;
   else if (AlawSample > 30) ulawSample = AlawSample + 4;
   else if (AlawSample > 28) ulawSample = AlawSample + 5;
   else if (AlawSample > 26) ulawSample = AlawSample + 6;
   else if (AlawSample > 24) ulawSample = AlawSample + 7;
   else if (AlawSample >  7) ulawSample = AlawSample + 8;
   else                      ulawSample = 2 * AlawSample + 1;
   return((~ulawSample & 0x7f) | sign);
}

unsigned char pcmu2A(ulawSample)
unsigned char ulawSample;
{
   unsigned char AlawSample, sign = ulawSample & 0x80;

   ulawSample = ~ulawSample & 0x7f;
   if      (ulawSample > 79) AlawSample = ulawSample;
   else if (ulawSample > 63) AlawSample = ulawSample - 1;
   else if (ulawSample > 48) AlawSample = ulawSample - 2;
   else if (ulawSample > 47) AlawSample = ulawSample - 3;
   else if (ulawSample > 35) AlawSample = ulawSample - 4;
   else if (ulawSample > 34) AlawSample = ulawSample - 5;
   else if (ulawSample > 33) AlawSample = ulawSample - 6;
   else if (ulawSample > 32) AlawSample = ulawSample - 7;
   else if (ulawSample > 16) AlawSample = ulawSample - 8;
   else                      AlawSample = ulawSample/2;
   return((AlawSample | sign) ^ 0x55);
}
