/*
  title:    ascii.c
  purpose:  read fview feature data that has been saved in ascii format.

  author:   Gareth Lee.
  date:     01-05-94
  modified: 08-06-94

  Copyright (C) 1994 Gareth Lee.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  changes:
  08-06-94: Changes made to ReadASCIIHeader() making textual header record
    handling algorithm more robust.
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <math.h>
#include <assert.h>

#include "../fviewio.h"
#include "../paths.h"

#define VERSION "1.2"

/*****************************************************************************/

char* filename;                                          /* filename to load */

FviewFeatureHeader ffh;                               /* fview header record */
char  ascii_header[10240];                            /* ascii header record */
char  header[1024];                                         /* header record */
char* h[20];                              /* fields within the header record */

#ifdef SUNOS
extern char* optarg;
#endif

/*****************************************************************************/

/*
  CheckExtension: compare the final few characters of `fname' with those in
    `extn' if they are identical then return 1 else 0
*/
int CheckExtension(char* fname, char* extn)
{
  int i;
  int flen = strlen(fname);
  int elen = strlen(extn);

  if (elen > flen)
    return 0;
  for (i = 1; i <= elen; i++)
    if (fname[flen-i] != extn[elen-i])
      return 0;
  return 1;
}

/*****************************************************************************/

/*
  OpenFile: open `fname' for reading, piping through an appropriate
    uncompressing filter if the correct extensions are encountered.
*/
FILE* OpenFile(char* fname, int* piped)
{
  char cmd[1024];

  if (CheckExtension(fname, GUNZIP_EXTN))
  {
    sprintf(cmd, "%s %s", GUNZIP, fname);
    *piped = 1;
    return popen(cmd, "r");                 /* open file piped through UNZIP */
  }
  if (CheckExtension(fname, UNCOMPRESS_EXTN))
  {
    sprintf(cmd, "%s %s", UNCOMPRESS, fname);
    *piped = 1;
    return popen(cmd, "r");            /* open file piped through UNCOMPRESS */
  }
  *piped = 0;
  return fopen(fname, "r");                      /* open file conventionally */
}

/*****************************************************************************/

/*
  IndexHeader: decompose the header record into a sequence of character fields
    separated by white-space.  Place the indexing pointers into an array of
    pointer to char.
*/
int IndexHeader(char *hdr, char **idx)
{
  char *p;
  int was_space = 1;
  int param = 0;

  for (p = hdr; *p; p++)
  {
    if (isspace(*p))
    {
      *p = '\0';           /* convert white space into terminating character */
      was_space = 1;
    }
    else
    {
      if (was_space)
        idx[param++] = p;    /* (*p) is not whitespace but previous char was */
      was_space = 0;
    }
  }
  return param;                   /* return the number of parameters located */
}

/*****************************************************************************/

/*
  ParseHeader: process the fields within the header record and generate error
    messages if any fail to comply.
*/
int ParseHeader(int nfields, char* field[])
{
  int records;

  if (strcmp(field[0], "fview") != 0)
  {
    fprintf(stderr, "ascii: error: file format dubious\n");
    exit(-1);
  }
  if (sscanf(field[1], "%d", &ffh.vector_dimension) != 1)
  {
    fprintf(stderr, "ascii: error: failed to read dimension field.\n");
    exit(-1);
  }
  if (sscanf(field[2], "%d", &ffh.number_observations) != 1)
  {
    fprintf(stderr, "ascii: error: failed to read observations field.\n");
    exit(-1);
  }
  if (sscanf(field[3], "%d", &records) != 1)
  {
    fprintf(stderr, "ascii: error: failed to read textual header field.\n");
    exit(-1);
  }

  /* extract the logged field if available */
  if (nfields > 4)
  {
    if (strcmp(field[4], "logged") == 0 || strcmp(field[4], "LOGGED") == 0)
      ffh.logged = FVIEW_FEAT_LOGGED;
    else
    {
      if (strcmp(field[4], "linear") == 0 || strcmp(field[4], "LINEAR") == 0)
        ffh.logged = FVIEW_FEAT_LINEAR;
      else
        ffh.logged = FVIEW_FEAT_UNKNOWN;
    }
  }
  else
    ffh.logged = FVIEW_FEAT_UNKNOWN;

  /* extract the colour preference field if available */
  if (nfields > 5)
  {
    if (sscanf(field[5], "%d", &ffh.color_preference) != 1)
      ffh.color_preference = FVIEW_COLR_NOPREF;
  }
  else
    ffh.color_preference = FVIEW_COLR_NOPREF;

  return records;
}

/*****************************************************************************/

/*
  ReadASCIIHeader: read in the specified number of records from the ascii file
    and concatenate them into a single ascii_header record for saving within
    the fview format file.  Return the total size.
*/
int ReadASCIIHeader(FILE* fd, int records)
{
  int i, c, len;
  int size = 0;
  char buf[1024];

  (void) fgetc(fd);         /* discard newline at the end of the header line */

  for (i = 0; i < records; i++)
  {
    len = 0;
    c = fgetc(fd);
    buf[len++] = c;
    while (c != '\n')
    {
      c = fgetc(fd);
      buf[len++] = c;
    }
    buf[len] = '\0';
    
    strcpy(ascii_header + size, buf);  /* append `buf' to the `ascii_header' */
    size += len;
  }
  return (size + 1);                /* allow for final terminating character */
}

/*****************************************************************************/

/*
  WriteHeaderRecord: write out an fview header record using the information
    previously compiled.
*/
void WriteHeaderRecord(FILE* fd)
{
  int i;
  char magic[10];

  /* put magic number into header record */
  strcpy(magic, FVIEW_FEATURE_MAGIC);
  for (i = 0; i < 8; i++)
    ffh.magic[i] = magic[i];

  if (fwrite(&ffh, sizeof(FviewFeatureHeader), 1, fd) != 1)
  {
    fprintf(stderr, "ascii: error: unable to write header record\n");
    exit(-1);
  }
  if (fwrite(ascii_header, sizeof(char), ffh.info_size, fd) != ffh.info_size)
  {
    fprintf(stderr, "ascii: error: unable to write ascii header record\n");
    exit(-1);
  }
}

/*****************************************************************************/

/*
  WriteVectorRecord: copy a feature vector from ascii input to fview (binary)
    output.
*/
void WriteVectorRecord(FILE* ifd, FILE* ofd, int t)
{
  int i;
  static double* db = NULL;

  if (db == NULL)
  {
    db = (double *) malloc(ffh.vector_dimension * sizeof(double));
    if (db == NULL)
    {
      fprintf(stderr, "ascii: error: malloc failed\n");
      exit(-1);
    }
  }

  for (i = 0; i < ffh.vector_dimension; i++)
  {
    if (fscanf(ifd, "%lf", &db[i]) != 1)
    {
      fprintf(stderr,
                "ascii: error: failed to read coefficient %d from input\n", t);
      exit(-1);
    }
  }

  if (fwrite(db,sizeof(double),ffh.vector_dimension,ofd)!=ffh.vector_dimension)
  {
    fprintf(stderr, "ascii: error: unable to write feature vector (t=%d)\n",t);
    exit(-1);
  }
}

/*****************************************************************************/

int main(int argc, char *argv[])
{
  int t;
  FILE* fd;                                             /* input file handle */
  int opt;                                                /* getopt() option */
  int piped;                                /* data is being read via a pipe */
  int params;                   /* number of parameters in the header record */
  int ascii_header_recds;                 /* number of lines of ascii header */

  /* read command line options */
  while ((opt = getopt(argc, argv, "f:")) != -1)
    switch (opt)
    {
    case 'f':
      filename = optarg;
      break;
    case '?':
      fprintf(stderr, "usage: ascii -f file_name\n");
      fprintf(stderr, "ascii: option -%c not supported\n", opt);
      exit(-1);
    }
  
  fd = OpenFile(filename, &piped);
  if (fd == NULL)
  {
    fprintf(stderr, "ascii: error: cannot open file %s\n", filename);
    fflush(stderr);
    exit(-1);
  }

  /* read the header record and index it to locate the parameters */
  fscanf(fd, "%[^\n]", header);
  params = IndexHeader(header, h);
  if (params < 4 || params > 6)
  {
    fprintf(stderr, "ascii: error: header deformed.\n");
    exit(-1);
  }
  ascii_header_recds = ParseHeader(params, h);
  ffh.info_size = ReadASCIIHeader(fd, ascii_header_recds);

  /* write fview format records to `stdout' */
  WriteHeaderRecord(stdout);
  for (t = 0; t < ffh.number_observations; t++)
    WriteVectorRecord(fd, stdout, t);

  if (piped)
    pclose(fd);
  else
    fclose(fd);

  /* write a banner indicating success */
  fprintf(stderr, "ascii (v%s): read %d x %d dimensional vectors\n",
                      VERSION,  ffh.number_observations, ffh.vector_dimension);
}

/*****************************************************************************/

/* end of ascii.c */
