/*
  title: fview.c
  purpose: To display two and three dimensional projections of speech
    trajectories through a high dimensional filter-band space.  With
    bindings to an Xview interface defined in interface.c.

  authors:  Gareth Lee & Nicole Duball.
  date:     16-08-93
  modified: 29-06-94

  Copyright (C) 1994 Gareth Lee and Nicole Duball.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  changes:
  05-09-93: fview.c based on sp_proj3.c.  But with the addition of variable
    dimension feature prediction.  The dimension is read from the first
    entry in the list file all other entries must have the same dimension.
    Band selection widgets automatically customize themselves to suit the
    dimension that is currently selected.
  07-09-93: The SpecifyWindow has been rationalized with the removal of
    various controlling variables (vector_one_on etc).
  13-11-93: Support for a SearchPath added and used by all driver/config/list
    file loads.  The path is read from an environment variable.  The default
    list filename can also be overrided using the -list command line option.
  11-02-94: Compression() modified so as to become more senitive to
    compression_alpha values approaching one.  It now compresses using
    alpha^{3} whereas previous it simply used alpha.
  15-02-94: logic problem fixed within RefreshGraphics() which was leading
    to core-dumps/bus-errors when front-end/list-files where combination
    were incorrectly set.
  30-03-94: OrthogonalizeBases() modified to remove calls to Projection()
    and RefreshGraphics() since this causes a double refresh when called from
    SpecifyProjectCallback();
  31-03-94: Randomize() updated to improve the random selction algorithm for
    basis vectors.
  01-04-94: OpenDataFile() improved to detect listfile entries starting with
    '~' and substitute the $HOME environment vatiable.
  16-04-94: General refinements made during port to i386 Linux.  #define'd
    LINUX variable introduced to maintain bacward compatibility with SunOS.
  04-05-94: Bug in OpenConfigFile() resulting in in correct handling of
    comment characters wihin config files corrected.
  29-06-94: Bug in Initialize() corrected so that program behaves sensibly
    when loading list files with zero entries.
*/

/* Standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <signal.h>

/* X lib headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

/* XView headers */
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/notice.h>
#include <xview/cursor.h>

typedef REAL real;

/* Header files associated with other compilation units */
#include "projectors.h"
#include "fviewio.h"
#include "fview.h"
#include "interface.h"
#include "callbacks.h"
#include "entropy.h"
#include "ctable.h"
#include "paths.h"

#ifdef SUNOS
/* System call prototypes (apparently unsupported by SunOS header files) */
extern int fclose(FILE *fd);
extern int pclose(FILE*);
#endif
#ifdef LINUX
extern FILE* popen(char*, char*);
extern int pclose(FILE*);
#endif

/* Variable declarations */
int dimension = 0;                /* number of dimensions in feature vectros */
Display *display;
int screen_num;
Window win;
unsigned long valuemask; 
GC gc;
unsigned int width, height;                         /* window size in pixels */
char *progname;                          /* name this program was invoked by */
int iterations = ITERATIONS;
int samples = 0;
real angle = ANGLE;
real *w1, *w2, *w3;                            /* projection basis vectors */
real *stw1, *stw2, *stw3;               /* stored projection basis vectors */
SpeechRecord *spdata[MAX_SAMPLES];
int size = 0;
int up_down = 0;
int left_right = 0;

/* two axes around which to rotate during animation sequences */
int rotation_dim1 = ROTATION_DIM1;
int rotation_dim2 = ROTATION_DIM2;

DisplayMode display_mode;
WorkProc ExecWorkProc = NULL;
int utterance_index = 0;
int highlighted = 0;                      /* no points highlighted initially */
int max_vertices = 0;  /* maximum number of vertices that can be highlighted */
int auto_scale = 1;            /* automatic trajectory scaling to fit window */
char *list_fname;                                  /* current list file name */
char *list_fpath;                                  /* current list path name */
int list_entries = 0;                     /* number of entries in list_names */
char* list_names[MAX_LIST_LENGTH];             /* names taken from list file */
int sample_display_info = 0; /* texual info should be displayed when loading */
int sample_assume_logged = 0;   /* assumed FVIEW_FEAT_UNKNOWN data to logged */
int compress_raw_mode = 1;    /* by default display samples as they are read */
int compress_scale_mode = 0;  /* scales all traj by their greatest magnitude */
real compress_alpha = 0.5;                      /* compression control value */
int display_label_mode = 1;          /* controls whether traces are labelled */
int display_origin_mode = 0;   /* display the origin within the display area */
int global_ptset_mode = 0;     /* display data as an (unconnected) point set */
int spectrogram_stretch_mode = 0; /* display all spectrograms to diff widths */
int spectrogram_energy_mode = 0;   /* overlay energy profile on spectrograms */
FilterRecord Formats;      /* Record for def format converter + root of list */
FilterRecord Frontends;           /* Record for def front-end program + root */
FilterRecord *FormatFilter = &Formats;        /* Chosen Format filter option */
FilterRecord *FrontendFilter = &Frontends;  /* Chosen Frontend filter option */
char *SearchPath[1024];          /* Path to search for driver and list files */
char *ColorString = NULL;       /* ASCII string containing RGB colors for 3D */
char *UserHomeDir;                             /* home directory of the user */
real CompressionMinValue;    /* minimum value of any datum (for compression) */
real CompressionMaxValue;    /* maximum value of any datum (for compression) */

/*****************************************************************************/

/*
  FviewSignalHandler: function name and interface changed in v1.2 to ensure
    strict compliance with stdio.h.
*/
void FviewSignalHandler(int x)
{
  /* Shutdown X system */
  XUnloadFont(display, FontInfo->fid);
  XFreeGC(display, gc);
  XCloseDisplay(display);

  exit(-1);
}

/*****************************************************************************/

/*
  CreateGlobalVectors: allocate global vectors which depend on feature
    vector dimension.
*/
void CreateGlobalVectors(void)
{
  w1   = (real *) malloc(dimension * sizeof(real));
  w2   = (real *) malloc(dimension * sizeof(real));
  w3   = (real *) malloc(dimension * sizeof(real));
  stw1 = (real *) malloc(dimension * sizeof(real));
  stw2 = (real *) malloc(dimension * sizeof(real));
  stw3 = (real *) malloc(dimension * sizeof(real));

  /* Check malloc() calls suceeded */
  if (w1==NULL||w2==NULL||w3==NULL||stw1==NULL||stw2==NULL||stw3==NULL)
  {
    printf("Error: during allocation of global basis vectors\n");
    Quit(-1);
  }
}

/*****************************************************************************/

/*
  FreeGlobalVectors: free global vectors which depend on feature
    vector dimension.  Set all pointers to NULL as an error check.
*/
void FreeGlobalVectors(void)
{
  free(w1);
  free(w2);
  free(w3);
  w1 = w2 = w3 = NULL;

  free(stw1);
  free(stw2);
  free(stw3);
  stw1 = stw2 = stw3 = NULL;
}

/*****************************************************************************/

/*
  StringDup: Duplicate the source string into a buffer which is malloced to
    exactly the correct size.
  */
char *StringDup(char *source)
{
  char *cp;
  cp = (char *) malloc(strlen(source) + 1);
  if (cp == NULL)
  {
    printf("Error: during StringDup of '%s'\n", source);
    Quit(-1);
  }
  strcpy(cp, source);
  return cp;
}

/*****************************************************************************/

/*
  CreateDefaultConverters: Initialize fields in the default converters
    records.
*/
void CreateDefaultConverters(FilterRecord *dfmt, FilterRecord *dfrt)
{
  /* Setup the default format converter */
  dfmt->name = StringDup(DEF_FORMAT_FILTER);
  dfmt->program = "(Internal)";
  dfmt->path = NULL;
  dfmt->format = 'F';
  dfmt->def_list = StringDup(DEFAULT_LIST_NAME);
  dfmt->index = 0;
  dfmt->options = NULL;
  dfmt->next = NULL;

  /*
    Setup the NULL frontend.  Note: this should be the *only* frontend record
    accpeting 'F' format features
  */
  dfrt->name = StringDup(DEF_FRONTEND_FILTER);
  dfrt->program = "(Internal)";
  dfrt->format = 'F';    
  dfrt->def_list = NULL;
  dfrt->index = 0;
  dfrt->options = NULL;
  dfrt->next = NULL;
}

/*****************************************************************************/

/*
  CreateConfigFields: partition a record consisting of `separator_char'
    isolated fields into a sequence of ASCIIZ accessed by an array of
    char*.
*/
int CreateConfigFields(char* cbuf, char* cptr[])
{
  const char separator_char = ':';     /* character used to separate records */

  int i;
  int fcnt = 0;
  int was_separator = 1;              /* assume a separator_char at cbuf[-1] */

  for (i = 0; cbuf[i]; i++)
    if (cbuf[i] == separator_char)
    {
      cbuf[i] = '\0';        /* convert separator_char to a terminating char */
      was_separator = 1;
    }
    else
    {
      if (was_separator)
        cptr[fcnt++] = &cbuf[i];  /* save address of first char within field */
      was_separator = 0;
    }
  return fcnt;                                   /* number of fields located */
}

/*****************************************************************************/

/*
  OpenConfigFile: read in a filter program list from a configuration file.
*/
FilterRecord* OpenConfigFile(char *fname, int fixed)
{
  int i, fields;
  char *field[1024], fmt;
  FILE *fd;
  char buf[1024];
  FilterRecord *root = NULL;
  FilterRecord *fptr;
  OptionRecord *optr;

  assert(fixed == 3 || fixed == 4);   /* check valid fixed field number used */
  
  /* Try all directories along the search path when opening a config file */
  for (i = 0; SearchPath[i] != NULL; i++)
  {
    (void) sprintf(buf, "%s/%s", SearchPath[i], fname);
    if ((fd = fopen(buf, "r")) != NULL)
      break;
  }

  /* Check that the open suceeded somewhere along the path */
  if (fd == NULL)
  {
    printf("Error: Cannot locate config file %s\n", fname);
    Quit(-1);
  }

  while (1)
  {
    /*
      read a complete record up to a carriage return and skip to the start of
      the next record.
    */
    if (fscanf(fd, "%[^\n]\n", buf) != 1)
      break;

    /* records starting with the '#' character are comments, so ignore them */
    if (buf[0] == '#')
      continue;
    
    /* parse record and split into ASCIIZ fields */
    fields = CreateConfigFields(buf, field);
    assert(((fields - fixed) % 3) == 0);

    /* Allocate a FilterRecord */
    if (root == NULL)
    {
      fptr = (FilterRecord*) malloc(sizeof(FilterRecord));
      assert(fptr != NULL);
      root = fptr;
    }
    else
    {
      fptr->next = (FilterRecord*) malloc(sizeof(FilterRecord));
      assert(fptr->next != NULL);
      fptr = fptr->next;
    }
    
    /* set up compulsary fields */
    fptr->name = StringDup(field[0]);
    fptr->program = StringDup(field[1]);
    fptr->path = NULL;

    /* store format field */
    fmt = field[2][0];
    if (isalpha(fmt))
      fptr->format = toupper(fmt);       /* store upper case form of formats */
    else
    {
      printf("Error: format field '%c' incorrect\n", fmt);
      Quit(-1);
    }
    assert(isupper(fptr->format));

    fptr->def_list = (fixed == 4) ? StringDup(field[3]) : NULL;
    fptr->options = NULL;
    fptr->next = NULL;

    /* process remaining fields in groups of three */
    for  (i = fixed; i < fields; i += 3)
    {
      /* Allocate an OptionRecord */
      if (fptr->options == NULL)
      {
        fptr->options = (OptionRecord*) malloc(sizeof(OptionRecord));
        assert(fptr->options != NULL);
        optr = fptr->options;
      }
      else
      {
        optr->next = (OptionRecord*) malloc(sizeof(OptionRecord));
        assert(optr->next != NULL);
        optr = optr->next;
      }

      /* initialise the name, command and value fields from the record */
      optr->name = StringDup(field[i]);
      optr->command = StringDup(field[i+1]);
      optr->value = StringDup(field[i+2]);
      optr->next = NULL;
    }
  }  
  fclose(fd);

  return root;
}

/*****************************************************************************/

/*
  CheckConverterList: Checks that all converter programs cited in a
    FilterRecord linked-list can be accessed in the directory system by
    searching directories along the SearchPath.  Initialises the path field
    withi the FilterRecord when the filter is located otherwise produces a
    fatal error if any driver cannot be located.
*/
void CheckConverterList(FilterRecord *fptr)
{
  int index;
  FILE *fd;
  char *cp;
  char prog[1024];
  char buf[1024];
  
  while (fptr != NULL)
  {
    /* Separate off trailing parameters before checking */
    strcpy(prog, fptr->program);
    cp = prog;
    while (*cp == ' ')    /* Skip over any spaces proceeding the filter name */
      cp++;
    while (*cp && *cp != ' ')  /* locate the first ' ' after the filter name */
      cp++;
    if (cp != NULL)
      *cp = '\0';                 /* terminate the string at the first space */

    index = 0;
    while (SearchPath[index] != NULL)
    {
      (void) sprintf(buf, "%s/%s", SearchPath[index], prog);
      
      fd = fopen(buf, "r");          /* attempt to access the filter program */
      if (fd != NULL)
      {
        fclose(fd);
        fptr->path = StringDup(SearchPath[index]);  /* save the correct path */
        break;                                            /* cease searching */
      }
      index++;
    }

    /* Check that the filter was located at some location  along the path */
    if (fptr->path == NULL)
    {
      printf("Error: Cannot locate filter '%s' under the name `%s`\n",
             fptr->name, fptr->program );
     Quit(-1);
    }
    fptr = fptr->next;
  }
}

/*****************************************************************************/

/*
  FreeConfigRecords: Completely deallocate a Config Record list structure.
*/
void FreeConfigRecords(FilterRecord *root)
{
  FilterRecord *fptr;
  OptionRecord *oroot, *optr;
  
  while (root != NULL)
  {
    fptr = root;
    root = root->next;

    free(fptr->name);
    free(fptr->program);
    free(fptr->path);

    oroot = fptr->options;
    while (oroot != NULL)
    {
      optr = oroot;
      oroot = oroot->next;
      
      free(optr->name);
      free(optr->command);
      free(optr->value);
    }
  }
}

/*****************************************************************************/

/*
  CreateFilterCommand: Takes FormatFilter and FrontendFilter linked structures
    and constructs a single (piped) command string which can be passed to a
    shell using popen().  Note: written inefficiently but should be robust to
    non ANSI sprintf() implementations.
*/
void CreateFilterCommand(char *cmd, char *fname, FilterRecord *converter,
                         FilterRecord *front_end)
{
  char *cp;
  OptionRecord *optr;

  cp = cmd;
  (void) sprintf(cp, "%s/%s %s", converter->path, converter->program, fname);
  cp = cmd + strlen(cmd);
  
  for (optr = converter->options; optr != NULL; optr = optr->next)
  {
    *cp++ = ' ';     /* ensure options are separated by at least one space */
    (void) sprintf(cp, optr->command, optr->value);
    cp = cmd + strlen(cmd);    
  }
  
  if (front_end != &Frontends)
  {
    (void) sprintf(cp, " | %s/%s", front_end->path, front_end->program);
    cp = cmd + strlen(cmd);
    
    for (optr = front_end->options; optr != NULL; optr = optr->next)
    {
      *cp++ = ' ';     /* ensure options are separated by at least one space */
      (void) sprintf(cp, optr->command, optr->value);
      cp = cmd + strlen(cmd);
    }
  }
}

/*****************************************************************************/

/*
  OpenListFile: read from a list file (which may be compressed) store the
    entries and update the SampleList widget.
*/
void OpenListFile(void)
{
  /*
    Filenames are left truncated if their length exceeds this number of
    characters when they are inserted into the SampleFileList widget.
  */
  const int StringWidth = 28;
  
  int i, rtn, len, start;
  FILE *fd;
  char buf[1024];
  int length;
  int (*fpclose)(FILE *fd) = NULL;         /* pointer to the close function */
  
  /* check for .Z and .gz extensions */
  length = strlen(list_fname);
  if (list_fname[length-2] == '.' && list_fname[length-1] == 'Z')
  {
    (void) sprintf(buf, "%s %s/%s", UNCOMPRESS, list_fpath, list_fname);
    fd = popen(buf, "r");
    fpclose = &pclose;                        /* choose pclose() to close fd */
  }
  else
    if (list_fname[length-3] == '.' && list_fname[length-2] == 'g' &&
        list_fname[length-1] == 'z')
    {
      (void) sprintf(buf, "%s %s/%s", GUNZIP, list_fpath, list_fname);
      fd = popen(buf, "r");
      fpclose = &pclose;                      /* choose pclose() to close fd */
    }
    else
    {
      (void) sprintf(buf, "%s/%s", list_fpath, list_fname);
      fd = fopen(buf, "r");
      fpclose = &fclose;                      /* choose fclose() to close fd */
    }
  assert(fd != NULL);       /* listfile access should have been prev checked */

  /* scan list file adding filename entries to list_names[] */
  list_entries = 0;
  while (1)
  {
    rtn = fscanf(fd, "%[^\n]", buf);   /* read until no more lines available */
    if (rtn == EOF || rtn == 0)
      break;

    list_names[list_entries] = StringDup(buf);
    if (++list_entries == MAX_LIST_LENGTH)
    {
      printf("open_list_file: list file too large, %d entries read\n",
             MAX_LIST_LENGTH);
      fclose(fd);
      Quit(-1);
    }
    (void) fgetc(fd);       /* Skip the \n character at the end of each line */
  }

  list_names[list_entries] = NULL;                         /* terminate list */
  assert(fpclose != NULL);
  fpclose(fd);                       /* call the appropriate closing routine */

  /* Hide the sample list before adding the entries to reduce screen flicker */
  xv_set(SampleFileList, XV_SHOW, FALSE, NULL);

  /* Add entries from list_names[] to the SampleList scrolling widget */
  for (i = 0; i < list_entries; i++)
  {
    strcpy(buf, list_names[i]);

    /*
      if filename is longer than StringWidth then just display last
      StringWidth characters
    */
    len = strlen(buf);
    start = (len > StringWidth) ? len - StringWidth : 0;

    xv_set(SampleFileList,
           PANEL_LIST_INSERT, i,
           PANEL_LIST_STRING, i, &buf[start],
           NULL);
  }

  /* Select first entry by default and redisplay the list */
  xv_set(SampleFileList,
         XV_SHOW, TRUE,
         NULL);

  /* Display list name and number of samples read */
  xv_set(SampleListnameText,
         PANEL_VALUE, list_fname,
         NULL);

  /* Display characteristics of the list file */
  (void) sprintf(buf, "%d", list_entries);
  xv_set(SampleNumSampsText, PANEL_VALUE, buf, NULL);  
}

/*****************************************************************************/

/*
  CreateSpeechRecord: allocate fields within a SpeechRecord depending on
    the 'npoints' and 'dimension' values set.  Check all malloc() calls
    to ensure sucess.  Terminate on failure.
*/

void CreateSpeechRecord(SpeechRecord *sd, int npoints)
{
  int i;
  int fail;
  
  sd->npoints = npoints;
  sd->order = (int *) malloc(npoints * sizeof(int));
  sd->dist = (real *) malloc(npoints * sizeof(real));
  if (sd->order == NULL || sd->dist == NULL)
  {
    printf("Error: during allocation of SpeechRecord (order/dist)\n");
    Quit(-1);
  }

  sd->project1 = (real *) malloc(npoints * sizeof(real));
  sd->project2 = (real *) malloc(npoints * sizeof(real));
  sd->project3 = (real *) malloc(npoints * sizeof(real));
  if (sd->project1 == NULL || sd->project2 == NULL || sd->project3 == NULL)
  {
    printf("Error: during allocation of SpeechRecord (project1/2/3)\n");
    Quit(-1);
  }

  sd->data  = (real **) malloc(npoints * sizeof(real *));
  fail = (sd->data == NULL);
  for (i = 0; i < npoints; i++)
  {
    sd->data[i] = (real *) malloc(dimension * sizeof(real));
    fail = fail || (sd->data == NULL);
  }
  if (fail)
  {
    printf("Error: during allocation of SpeechRecord (data)\n");
    Quit(-1);
  }

  sd->point = (real **) malloc(dimension * sizeof(real *));
  fail = (sd->point == NULL);
  for (i = 0; i < dimension; i++)
  {
    sd->point[i] = (real *) malloc(npoints * sizeof(real));
    fail = fail || (sd->point == NULL);
  }
  if (fail)
  {
    printf("Error: during allocation of SpeechRecord (point)\n");
    Quit(-1);
  }
  sd->ptset = global_ptset_mode;                 /* global point set option */
}

/*****************************************************************************/

/*
  FreeSpeechRecord: deallocate malloced fields within a SpeechRecord.
    Note: all static fields are left intact.
*/

void FreeSpeechRecord(SpeechRecord *sd)
{
  int i;
  
  for (i = 0; i < dimension; i++)
    free(sd->point[i]);
  free(sd->point);
  sd->point = NULL;

  for (i = 0; i < sd->npoints; i++)
    free(sd->data[i]);
  free(sd->data);
  sd->data = NULL;

  free(sd->file_name);
  sd->file_name = NULL;
  free(sd->project3);
  free(sd->project2);
  free(sd->project1);
  sd->project1 = sd->project2 = sd->project3 = NULL;
  free(sd->dist);
  sd->dist = NULL;
  free(sd->order);
  sd->order = NULL;
}

/*****************************************************************************/

/*
  CheckMagicNumber: Check that an eight byte magic number is correct.
*/
int CheckMagicNumber(char *magic, char *reference)
{
  int i;

  /* Checks eight bytes of magic number */
  for (i = 0; i < 8; i++)
    if (magic[i] != reference[i])
      return 1;
  return 0;
}

/*****************************************************************************/

/*
  OpenDataFile: Read the specified file into the specifies SpeechRecord
    structure arranging dynamic allocation of fields using
    CreateSpeechRecord().  On success returns 0.  Any non-zero return value
    indicates a load error.
*/
int OpenDataFile(char* filename, SpeechRecord *sd, int verbose)
{
  int i, j;
  FILE *fd;
  real r;
  FviewFeatureHeader ffh;                           /* Feature header record */
  char fname[1024], buf[1024];
  char *extra;            /* buffer in which to hold addition textual fields */
  double *buffer;       /* input buffer must be double to ensure file compat */
  int unset;       /* indicates that min/max_datum have not been initialized */

  /* process initial '~' character */
  if (filename[0] == '~')
  {
    strcpy(fname, UserHomeDir);      /* substitute the user's home directory */
    strcat(fname, filename + 1);                   /* skip the '~' character */
  }
  else
    strcpy(fname, filename);

  /*
    Open data file directly for reading or create a format converion/
    front-end pipe to generate features.
  */
  if (FormatFilter != &Formats)
  {
    CreateFilterCommand(buf, fname, FormatFilter, FrontendFilter);

    fd = popen(buf, "r");
    if (fd == NULL)
    {
      printf("Load Error: Cannot create input filter for %s\n", filename);
      return 1;
    }
  }
  else
  {
    fd = fopen(fname, "r");
    if (fd == NULL)
    {
      printf("Load Error: Cannot open feature datafile %s\n", filename);
      return 1;
    }
  }

  /* read header record and check the magic number etc. */
  if (fread(&ffh, sizeof(FviewFeatureHeader), 1, fd) != 1)
  {
    printf("Load Error: failed to read header record for %s\n", filename);
    if (FormatFilter != &Formats)
      pclose(fd);
    else
      fclose(fd);
    return 1;
  }
  if (CheckMagicNumber(ffh.magic, FVIEW_FEATURE_MAGIC))
  {
    printf("Load Error: File format in %s is incorrect (magic number wrong)\n",
           filename);
    if (FormatFilter != &Formats)
      pclose(fd);
    else
      fclose(fd);
    return 1;
  }

  /* Extract the feature dimension for the file if not already set */
  if (dimension == 0)
  {
    if (ffh.vector_dimension >= MIN_DIMENSION &&
        ffh.vector_dimension <= MAX_DIMENSION)
    {
      dimension = ffh.vector_dimension;
      (void) sprintf(buf, "%d", dimension);
      xv_set(SampleDimensionText, PANEL_VALUE, buf, NULL);
    }
    else
    {
      printf("Load Error: Cannot cope with feature dimension = %d\n",
             ffh.vector_dimension);
      if (FormatFilter != &Formats)
        pclose(fd);
      else
        fclose(fd);
      return 1;
    }
  }
  else
  {
    if (ffh.vector_dimension != dimension)
    {
      printf("Load Error: Features are wrong dimension (%d instead of %d)\n",
             ffh.vector_dimension, dimension);
      if (FormatFilter != &Formats)
        pclose(fd);
      else
        fclose(fd);
      return 1;
    }
  }

  /* copy `logged' flag across into SpeechRecord */
  if (ffh.logged == FVIEW_FEAT_LINEAR || ffh.logged == FVIEW_FEAT_UNKNOWN ||
      ffh.logged == FVIEW_FEAT_LOGGED)
  {
    sd->logged = ffh.logged;
  }
  else
  {
    printf("Load Error: Features have illegitimate `logged' status (= %d)\n",
           ffh.logged);
    if (FormatFilter != &Formats)
      pclose(fd);
    else
      fclose(fd);
    return 1;
  }

  /* read additional textual information */
  if (ffh.info_size > 0)
  {
    extra = (char *) malloc(ffh.info_size);
    if (fread(extra, sizeof(char), ffh.info_size, fd) != ffh.info_size)
    {
      printf("Load Error: whilst loading textual header from %s\n", filename);
      if (FormatFilter != &Formats)
        pclose(fd);
      else
        fclose(fd);
      return 1;
    }
  }
  else
    extra = NULL;
  
  /* allocate space within record sd */
  CreateSpeechRecord(sd, ffh.number_observations);
  
  /* read in the feature vectors converting to internal format */
  unset = 1;
  buffer = (double*) malloc(dimension * sizeof(double));
  for (i = 0; i < sd->npoints; i++)
  {
    if (fread(buffer, sizeof(double), dimension, fd) != dimension)
    {
      printf("Load Error: data read failed at observation %d in %s\n",
             i, filename);
      FreeSpeechRecord(sd);
      if (FormatFilter != &Formats)
        pclose(fd);
      else
        fclose(fd);
      return 1;
    }
    else
    {
      /* convert from doubles to internal format */
      for (j = 0; j < dimension; j++)
      {
        r = (real) buffer[j];
        sd->data[i][j] = r;
        if (unset)
        {
          sd->min_datum = sd->max_datum = r;
          unset = 0;
        }
        else
        {
          if (r > sd->max_datum)
            sd->max_datum = r;                         /* find maximum datum */
          if (r < sd->min_datum)
            sd->min_datum = r;                         /* find minimum datum */
        }
      }
    }
  }
  free(buffer);

  /* check that logged data has the correct numerical range */
  if (sd->logged == FVIEW_FEAT_LOGGED && sd->min_datum <= 0.0)
  {
    printf("Load Error: logged data contains negative coefficients\n");
    FreeSpeechRecord(sd);
    return 1;
  }

  /* save the file name in speech record */
  sd->file_name = StringDup(filename);
  sd->file_name_length = strlen(filename);

  /*
    precompute the width and height (pixels) needed when writing the text
    string out onto a canvas in the default font.
  */
  sd->label_box.width = (unsigned short)
                     XTextWidth(FontInfo, sd->file_name, sd->file_name_length);
  sd->label_box.height = (unsigned short) LABEL_BOX_HEIGHT;

  sd->color_key = ffh.color_preference;         /* save color preference key */
  sd->nverts = 0;       /* signals that the utterance has not been predicted */
 
  if (FormatFilter != &Formats)
    pclose(fd);
  else
    fclose(fd);

  /* Display textual information if the "info" check flag is active */
  if (verbose)
    printf("Header Info (%d bytes):\n%s\n", ffh.info_size, extra);  
  if (extra != NULL)
    free(extra);
  
  return 0;                        /* signal that utterance loaded correctly */
}

/*****************************************************************************/

/*
  CompressionRange: finds the maximum of the maxima associated with each of
  the loaded utterances.  Stores the result in the global CompressionMaxValue.
*/
void CompressionRange(int samps)
{
  int s;

  assert(samps >= 1);
  CompressionMinValue = spdata[0]->min_datum;
  CompressionMaxValue = spdata[0]->max_datum;
  for (s = 1; s < samps; s++)
  {
    if (spdata[s]->min_datum < CompressionMinValue)
      CompressionMinValue = spdata[s]->min_datum;
    if (spdata[s]->max_datum > CompressionMaxValue)
      CompressionMaxValue = spdata[s]->max_datum;
  }
}

/*****************************************************************************/

/*
  Compression: pre-processes the data prior to projection and display scaling.
    Three modes of operation are supported.  1.  compress_raw_mode displays
    the data samples as loaded without pre-processing.  2.  alpha allows a
    variable degree of compression to be applied to the axes (using a power
    law).  Whilst 3. compress_scale_mode divides all the feature vectors
    (after compression) by the greatest magnitude.
*/
void Compression(SpeechRecord *sd, real alpha)
{
  const real max_exp_arg = 10.0;         /* maximum argument passed to exp() */

  int t, i;
  int log_status, unknown;
  real s, magnitude, max;
  real scaling, r;

  if (compress_raw_mode)
  {
    for (t = 0; t < sd->npoints; t++)
      for (i = 0; i < dimension; i++)        
        sd->point[i][t] = sd->data[t][i];
  }
  else
  {
    s = alpha * alpha * alpha;

    /* If logged status is unknown use global assumption */
    log_status = sd->logged;
    if (log_status == FVIEW_FEAT_UNKNOWN)
    {
      log_status = (sample_assume_logged) ?FVIEW_FEAT_LOGGED:FVIEW_FEAT_LINEAR;
      unknown = 1;
    }
    else
      unknown = 0;

    /* allow for features that are already in log format */
    if (log_status == FVIEW_FEAT_LOGGED)
    {
      scaling = (unknown) ? (max_exp_arg / CompressionMaxValue) : 1.0;
      for (t = 0; t < sd->npoints; t++)
        for (i = 0; i < dimension; i++)
          sd->point[i][t] = (real) exp((double)(s * sd->data[t][i] * scaling));
    }
    else
    {
      /* log_status == FVIEW_FEAT_LINEAR */
      printf("alpha = %f\n", alpha);  /**/
      for (t = 0; t < sd->npoints; t++)
        for (i = 0; i < dimension; i++)
        {
          r = sd->data[t][i];
          /* allow for linear data to be negative */
          if (r >= 0.0)
            sd->point[i][t] = (real) exp((double)(s * log((double)r)));
          else
            sd->point[i][t] = (real) -exp((double)(s * log((double)-r)));
        }
    }
  }

  if (compress_scale_mode)
  {
    /* find the data vector with greatest power */
    max = 0.0;
    for (t = 0; t < sd->npoints; t++)
    {
      magnitude = 0.0;
      for (i = 0; i < dimension; i++)
      {
        s = sd->point[i][t];
        magnitude += (s * s);
      }
      if (magnitude > max)
        max = magnitude;
    }

    /* normalise all data vectors by the greatest power */
    max = sqrt(max);
    for (t = 0; t < sd->npoints; t++)
      for (i = 0; i < dimension; i++)
        sd->point[i][t] /= max;
  }
}

/*****************************************************************************/

/*
  Randomize: generates a set of new (and orthogonal) random basis functions.
*/
void Randomize(void)
{
  real product1, product2;
  int i, j, n1, n2;
  real *u, d;

  u = (real *) malloc(dimension * sizeof(real));
  
  /* Choose elements of the first normal vector */
  product1 = 0.0;
  for (i = 0; i < dimension; i++)
  {
    d = 0.0;
    for (j = 0; j < 12; j++)
      d += drandom();                 /* generate a pseudo-gaussian variable */
    d -= 6.0;
    w1[i] = d;
    product1 += d * d;
  }
  d = sqrt(product1);
  for (i = 0; i < dimension; i++)
    w1[i] /= d;                                    /* ensure unity magnitude */
  
  /* Choose a second vector - not neccessarily normalised or orthogonal */
  n1 = random() % dimension;
  for (i = 0; i < dimension; i++)  
    w2[i] = (i == n1) ? 1.0 : 0.0;

  /* Make second vector normalised and orthogonal to first vector */
  product1 = w1[n1] * w2[n1];                      /* simplified dot product */
  for (i = 0; i < dimension; i++)
    u[i] = w2[i] - (product1 * w1[i]);
  product1 = 0.0;
  for (i = 0; i < dimension; i++) 
    product1 += u[i] * u[i];
  d = sqrt(product1);
  for (i = 0; i < dimension; i++)
    w2[i] = u[i] / d;

  /* Choose a third vector - not neccessarily normalised or orthogonal */
  do {
    n2 = random() % dimension;
  } while (n2 == n1);  
  for (i = 0; i < dimension; i++)  
    w3[i] = (i == n2) ? 1.0 : 0.0;

  /* Make second vector normalised and orthogonal to first and second */
  product1 = w1[n2] * w3[n2];                      /* simplified dot product */
  product2 = w2[n2] * w3[n2];                                       /* ditto */
  for (i = 0; i < dimension; i++)
    u[i] = w3[i] - (product1 * w1[i]) - (product2 * w2[i]);
  product1 = 0.0;
  for (i = 0; i < dimension; i++) 
    product1 += u[i] * u[i];
  d = sqrt(product1);
  for (i = 0; i < dimension; i++)
    w3[i] = u[i] / d;

  free(u);
}

/*****************************************************************************/

/*
  DiagonalBasis: Creates a basis vector diagonally orientated within the
    specified sub-space.
*/
void DiagonalBasis(Xv_opaque SpecifySelector, real *w)
{
  int i;
  unsigned int bits;
  real product = 0.0;
  
  bits = (unsigned int) xv_get(SpecifySelector, PANEL_VALUE);

  for (i = 0; i < dimension; i++) {
    if (bits & 0x0001)
    {
      w[i] = 1.0; 
      product += 1.0;
    }
    else
      w[i] = 0.0;
    bits >>= 1;
  }
  for (i = 0; i < dimension; i++) {
    w[i] /= sqrt(product);
  }
}

/*****************************************************************************/

/*
  RandomBasis: Creates a basis vector randomly orientated within the
    specified sub-space.
*/
void RandomBasis(Xv_opaque SpecifySelector, real *w)
{
  int i;
  unsigned int bits;
  real product = 0.0;

  bits = (unsigned int) xv_get(SpecifySelector, PANEL_VALUE);

  for (i = 0; i < dimension; i++ ) {
    w[i] = (bits & 0x0001) ? drandom() : 0.0;
    product += w[i] * w[i];
    bits >>= 1;
  }
  for (i = 0; i < dimension; i++)  {
    w[i] /= sqrt(product);
  }
}

/*****************************************************************************/

/*
  OrtogonalizeBases: Establish orthogonality between basis vectors w[123].
  Generate a pop-up window error should two of the basis vectors be
  parrallel.  Note: all vectors are assumed to be of unity magnitude as set
  by the Equalise and Randomise routines.  The first vector is left unchanged.
*/
void OrthogonalizeBases(void)
{
  const real DotLimit = 0.9999;
  
  int i, v1, v2;
  int fail = 0;
  real magnitude, product1, product2;
  real *u;
  char buf[1024];

  u = (real *) malloc(dimension * sizeof(real));
  
  /* Make second vector unity magnitude and orthogonal to first vector */
  product1 = 0.0;
  for (i = 0; i < dimension; i++) 
    product1 += w1[i] * w2[i];

  if (product1 > DotLimit)
  {
    v1 = 1;                /* signals which basis vectors are not orthogonal */
    v2 = 2;
    fail = 1;
  }
  else
  {
    /* do the orthogonalisation */
    for (i = 0; i < dimension; i++)
      u[i] = w2[i] - (product1 * w1[i]);
    
    magnitude = 0.0;
    for (i = 0; i < dimension; i++) 
      magnitude += u[i] * u[i];
    
    for (i = 0; i < dimension; i++)
      w2[i] = u[i] / sqrt(magnitude);
  }

  /* Make third vector unity magnitude and orthogonal to first and second */
  product1 = 0.0;
  product2 = 0.0;
  for (i = 0; i < dimension; i++)
  {
    product1 += w1[i] * w3[i];
    product2 += w2[i] * w3[i];
  }

  for (i = 0; i < dimension; i++)
    u[i] = w3[i];

  if (product1 > DotLimit)
  {
    v1 = 1;
    v2 = 3;
    fail = 1;
  }
  else
    for (i = 0; i < dimension; i++)
      u[i] -= (product1 * w1[i]);

  if (product2 > DotLimit)
  {
    v1 = 2;
    v2 = 3;
    fail = 1;
  }
  else
    for (i = 0; i < dimension; i++)
      u[i] -= (product2 * w2[i]);

  if (product1 <= DotLimit || product2 <= DotLimit)
  {
    magnitude = 0.0;
    for (i = 0; i < dimension; i++) 
      magnitude += u[i] * u[i];
    for (i = 0; i < dimension; i++)
      w3[i] = u[i] / sqrt(magnitude);
  }

  free(u);

  if (fail)
  {
    (void) sprintf(buf, "Basis vectors %d and %d", v1, v2);
    (void) notice_prompt(SpecifyPanel, NULL,
                         NOTICE_MESSAGE_STRINGS,
                           "Warning:",
                           buf,
                           "are not currently orthogonal!",
                           NULL,
                         NOTICE_BUTTON_YES, "Continue",
                         NULL);
  }
}

/*****************************************************************************/

/* Creation and projection of points onto 2/3D space */
void Projection(void)
{
  int i, j, s;
  SpeechRecord *sd;

  for (s = 0; s < samples; s++)
  {
    sd = spdata[s];
    
    /* Project points down on initially defined plane */
    for (i = 0; i < sd->npoints; i++) {
      sd->project1[i] = 0.0;
      sd->project2[i] = 0.0;
      sd->project3[i] = 0.0;
      for (j = 0; j < dimension; j++) {
        sd->project1[i] += (sd->point[j][i] * w1[j]);
        sd->project2[i] += (sd->point[j][i] * w2[j]);
        sd->project3[i] += (sd->point[j][i] * w3[j]);
      }
    }
  }
} /* end of projection */

/*****************************************************************************/

/*
  GetEnvironmentVars: Get values from environment variables else establish
    default values.
*/
void GetEnvironmentVars(void)
{
  int index;
  char *cp;
  char *path;

  /* Get home directory of user */
  UserHomeDir = getenv("HOME");
  
  /* Get SearchPath from an environment variable else use default */
  cp = getenv(ENVIRONMENT_PATH_NAME);
  if (cp)
    path = StringDup(cp);
  else
    path = StringDup(DEFAULT_PATH_NAME);

  /* Split path into separate strings by detecting the ':' delimiter */
  for (cp = path, index = 0; *cp != '\0'; cp++)
  {
    if (*cp == ':')
    {
      *cp++ = '\0';            /* terminate the (index-1)th directory string */
      while (*cp == ':')
      {
        cp++;
        if(*cp == '\0')
          break;
      }
      SearchPath[index++] = cp;                      /* set the next pointer */
    }
    else
      if (cp == path)
        SearchPath[index++] = cp;                     /* set initial pointer */

    assert(index < 1024);    /* check that the index doesn't go out of range */
  }
  SearchPath[index] = NULL;

  /* Get 3D color information from an environment variable if set */
  cp = getenv(ENVIRONMENT_COLOR_NAME);
  if (cp)
    ColorString = StringDup(cp);
  else
    ColorString = NULL;
} 

/*****************************************************************************/

/*
  Setup: called once prior to xv_main_loop to initialize variables.
*/
void Setup(int argc, char *argv[])
{
  int i;
  char buf[1024];
  FILE *fd;
  
  /* Setup and verify filter program lists */
  CreateDefaultConverters(&Formats, &Frontends);
  Formats.next = OpenConfigFile(FORMAT_FILE_NAME, 4);      /* 4 fixed params */
  Frontends.next = OpenConfigFile(FRONTEND_FILE_NAME, 3);  /* 3 fixed params */
  CheckConverterList(Formats.next);   /* Check existance of cited converters */
  CheckConverterList(Frontends.next);
  PrepareConverterLists();    /* Setup lists of conversion/front-end filters */

  /* Set initial values for the list file name and its path */
  for (i = 1; i < argc; i++)
  {
    if (strcmp(argv[i], "-list") == 0)
      if (++i < argc)
      {
        list_fname = StringDup(argv[i]);
        break;
      }
  }
  if (i == argc)
    list_fname = StringDup(DEFAULT_LIST_NAME);

  /* Check the existance of list_fname and set the list_fpath */
  for (i = 0; SearchPath[i] != NULL; i++)
  {
    (void) sprintf(buf, "%s/%s", SearchPath[i], list_fname);
    if ((fd = fopen(buf, "r")) != NULL)
      break;
  }
  if (fd != NULL)
  {
    fclose(fd);
    list_fpath = StringDup(SearchPath[i]);
  }
  else
  {
    printf("Error: failed to open list file `%s`\n", list_fname);
    Quit(-1);
  }
}
/*****************************************************************************/

/*
  Initialize: called with each new list file to reset variables etc.
*/
void Initialize(char *list_name, char *list_path)
{
  int i, flags;
  int old_dimension;
  SpeechRecord *sd;

  SetCursors(BusyCursor);

  if (list_name != NULL)
  {
    if (list_fname != NULL)
      free(list_fname);
    list_fname = StringDup(list_name);
  }
  if (list_path != NULL)
  {
    if (list_fpath != NULL)
      free(list_fpath);
    list_fpath = StringDup(list_path);
  }

  old_dimension = dimension;                 /* preserve old dimension value */
  if (dimension != 0)            /* check if a list file is currently loaded */
  {
    FreeGlobalVectors();      /* dispose of vectors before dimension changes */
    dimension = 0;
  }
    

  /* open new list file and ensure that some records were sucessfully read */
  OpenListFile();
  if (list_entries > 0)
  {
    /* OpenListFile selects the first utterance within its list */
    if ((sd = (SpeechRecord *) malloc(sizeof(SpeechRecord))) == NULL)
    {
      printf("Error: malloc of SpeechRecord failed\n");
      Quit(-1);
    }
    
    if (OpenDataFile(list_names[0], sd, 0))
    {
      free(sd);
      samples = 0;
    }
    else
    {
      samples = 1;
      
      /* mark first entry as selected */
      xv_set(SampleFileList, PANEL_LIST_SELECT, 0, TRUE, NULL);
    }
  }
  else
    samples = 0;


  if (dimension > 0)
  {
    CreateGlobalVectors(); /* allocate global vectors depending on dimension */
    ChangeBandWidgets();            /* modify widgets depending on dimension */
  }
  if (samples > 0)
  {
    spdata[0] = sd;
    CompressionRange(samples);
    Compression(spdata[0], compress_alpha);

    /* Set additional fields in the SpeechRecord */
    spdata[0]->color = AllocateColorEntry(spdata[0]->color_key);
    spdata[0]->row = 0;
  }

  /* Set activity of `Recall' button */
  if (dimension == 0 || dimension != old_dimension)
  {
    RecallAllowed = FALSE;
    xv_set(DisplayRecallButton, PANEL_INACTIVE, TRUE, NULL);
  }

  /* If in 3D mode then return to the first utterance */
  if (display_mode == three_dims)
    utterance_index = 0;

  SetCursors(BasicCursor);
  if (samples > 0)
    DisplayRandomizeCallback();
  else
    RefreshGraphics();
}

/*****************************************************************************/

/*
  RefreshGraphics:  Screen Refresh Callback activated whenever a Canvas
    expose or resize is occurs.  If (samples == 0) then hide the display
    window and do not attempt to redraw.
*/
void RefreshGraphics(void)
{
  int status;

  /* Expose/Hide Drawing Window when necessary */
  status = (int) xv_get(DisplayFrame, XV_SHOW);

  if (status == FALSE && samples > 0)
    xv_set(DisplayFrame, XV_SHOW, TRUE, NULL);

  /*
    If (samples == 0) then an error must be displayed and no drawing must
    take place.
  */
  if (samples == 0)
  {
    /* If display is exposed then hide it (and the colour selector window) */
    if (status == TRUE)
    {
      ColorHideWindow();
      xv_set(DisplayFrame,
             FRAME_CMD_PIN_STATE, FRAME_CMD_PIN_OUT,
             XV_SHOW, FALSE,
             NULL);
      XFlush(display);
    }

    (void)
      notice_prompt(MainPanel, NULL,
                    NOTICE_MESSAGE_STRINGS,
                    "Error:",
                    "Nothing to display",
                    "(reselect files/front-ends)",
                    NULL,
                    NOTICE_BUTTON_YES, "Continue",
                    NULL);

    /* Do not attempt to redraw */
    return;
  }
  
  /* obtain Canvas dimensions */
  width = (int) xv_get(DisplayCanvas, XV_WIDTH);
  height = (int) xv_get(DisplayCanvas, XV_HEIGHT);

  /* choose correct display driver function */
  switch (display_mode)
  {
  case two_dims:
    draw_2d_graphics(width, height, 1);
    break;
  case three_dims:
    draw_3d_graphics(width, height, 1);
    break;
  case spectrogram:
    draw_spectrogram(width, height, 1);
    break;
  }
}

/*****************************************************************************/

/*
  Quit: Clear up the Xwindow system and return from xv_main_loop in main().
    Note: code == 0 signifies that a totally legitimate exit has occured.
*/
void Quit(int code)
{
  /* Shutdown X system */
  XUnloadFont(display, FontInfo->fid);
  XFreeGC(display, gc);
  XCloseDisplay(display);

  exit(code);
}

/*****************************************************************************/

/*
  DisableWidgets: disable those parts of the interface that might interfere
    with the operation of the work procedures.
*/
void DisableWidgets(void)
{
  SetCursors(BusyCursor);

  /*
    Reset the list_fname since no change is required and set read only
    during the course of updates.  Also disable the FileList and UpdateButton
    widgets and others that could be dangerous during the update procedure.
  */
  xv_set(SampleListnameText,
         PANEL_VALUE, list_fname,
         PANEL_INACTIVE, TRUE,
         NULL);
  xv_set(SampleUpdateButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(DisplayRandomizeButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ControlAnimateButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ControlSweepButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ControlHighlightButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(SampleFileList, PANEL_INACTIVE, TRUE, NULL);
  xv_set(SampleFormatList, PANEL_INACTIVE, TRUE, NULL);
  xv_set(SampleFrontendList, PANEL_INACTIVE, TRUE, NULL);
  xv_set(SpecifyAnalyseButton, PANEL_INACTIVE, TRUE, NULL);
  XFlush(display);
}

/*****************************************************************************/

/*
  EnableWidgets: enable the full user interface upon completion of a work
    procedure.
*/
void EnableWidgets(void)
{
  /* Reinstate the widgets deactivated during updating */
  xv_set(SampleListnameText, PANEL_INACTIVE, FALSE, NULL);
  xv_set(DisplayRandomizeButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(ControlAnimateButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(ControlSweepButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(ControlHighlightButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(SampleUpdateButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(SampleFileList, PANEL_INACTIVE, FALSE, NULL);
  xv_set(SampleFormatList, PANEL_INACTIVE, FALSE, NULL);
  xv_set(SampleFrontendList, PANEL_INACTIVE, FALSE, NULL);
  xv_set(SpecifyAnalyseButton, PANEL_INACTIVE, FALSE, NULL);
  SetCursors(BasicCursor);
}

/*****************************************************************************/

/*
  main: activate xv widget set and lose control to xv event handler.  All
    other processing is activated by callbacks.
*/
int main(int argc, char** argv)
{
  progname = argv[0];
  srandom((int) getpid());

  signal(SIGINT, FviewSignalHandler);

  GetEnvironmentVars();
  CreateControlWindow(argc, argv);          /* Create XView widget hierarchy */

  Setup(argc, argv);                       /* setup variables once initially */
  Initialize(NULL, NULL);                /* setup variables prior to display */
  
  /*
    This call to xv_main_loop() will return when notify_stop() is used within
    a callback routine at which point it checks to see if a WorkProc has
    been registered and executes it.  Upon completion control returns to
    xv_main_loop().
  */
  while (1)
  {
    xv_main_loop(MainFrame);

    /* Executes the work procedure pointed to by ExecWorkProc */
    if (ExecWorkProc != NULL)
    {
      DisableWidgets();
      (*ExecWorkProc)();
      EnableWidgets();
      ExecWorkProc = NULL;
    }
  }
} /* end main */ 

/*****************************************************************************/

/* end of fview.c */

