/*
  title:    timit.c
  purpose:  Converters TIMIT format sample files into a format suitable for
    use with the fview package.

  author:   Gareth Lee
  date:     02-11-93
  modified: 

  changes:
  02-11-93: Program (loosely) based on sfconverter.c
*/
#define VERSION "1.0"

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "fviewio.h"

/*****************************************************************************/

/* TIMIT standard header - stored in VAX record format - byte swapping */
/* therefore necessary for Sun machines */

typedef struct {
  short header_size;                           /* in VAX word (2-byte) units */
  short version;                                           /* version number */
  short channels;           /* number of channels on which sampling occurred */
  short rate;                                 /* sample rate (units unknown) */
  int   samples;                            /* number of samples in the file */
} TIMIThdr;

/* Byte order options */
#define UNSET_ENDIAN  (-1)
#define BIG_ENDIAN    (0)
#define LITTLE_ENDIAN (1)

#define MAX_HDR_SIZE (10240)

#define VERSION_UNSET (-9999)

/*****************************************************************************/

/* Big <-> Little Endian swappers */
#define REV2(X)	((X&0xFF00)>>8)+((X&0xFF)<<8)
#define B1	0xFF000000
#define B2	0xFF0000
#define REV4(X)	((X&B1)>>24)+((X&B2)>>8)+((X&0xFF00)<<8)+((X&0xFF)<<24)

/*****************************************************************************/

/* Globals */
char               filename[1024];
int                endian = UNSET_ENDIAN;

/* Header records */
TIMIThdr thdr;
FviewSampleHeader fsh;
char header[MAX_HDR_SIZE];

/* for use with getopt */
extern char *optarg;
extern int optind, opterr;

/*****************************************************************************/

void main(int argc, char *argv[])
{
  int index, opt;
  FILE *fd;
  short sample, *short_buffer;
  int version = VERSION_UNSET;
  TIMIThdr swapped;

  assert(sizeof(short) == 2);    /* ensure types have the correct dimensions */
  assert(sizeof(int) == 4);
  
  /* Parse command line */
  while ((opt = getopt(argc, argv, "e:f:v:")) != -1)
    switch(opt) {

    case 'e':
      if (strcmp("big", optarg) == 0 || strcmp("BIG", optarg) == 0)
        endian = BIG_ENDIAN;
      else
        if (strcmp("little", optarg) == 0 || strcmp("LITTLE", optarg) == 0)
          endian = LITTLE_ENDIAN;
      break;

    case 'f':
      strcpy(filename, optarg);
      break;

    case 'v':
      version = atoi(optarg);
      break;
    }
  
  if ((fd = fopen(filename, "rb")) == NULL) {
    fprintf(stderr, "timit: error opening file %s for reading\n", filename);
    exit(-1);
  }

  /* read in header information and reverse the byte order */
  if (fread(&swapped, sizeof(TIMIThdr), 1, fd) != 1)
  {
    fprintf(stderr, "timit: error reading TIMIT header\n");
    exit(-1);
  }
  thdr.header_size = REV2(swapped.header_size);
  thdr.version = REV2(swapped.version);
  thdr.channels = REV2(swapped.channels);
  thdr.rate = REV2(swapped.rate);
  thdr.samples = REV4(swapped.samples);

  /* Check the TIMIT Header record (in 2-byte units) */
  if (thdr.header_size != 6)
  {
    fprintf(stderr, "timit: header size incorrect\n");
    exit(-1);
  }
  if (version != VERSION_UNSET && thdr.version != version)
  {
    fprintf(stderr, "timit: version field (%d) differs from that specified\n",
            thdr.version);
    exit(-1);
  }
  
  /* Create an Fview sample header record */
  memcpy(fsh.magic, FVIEW_SAMPLE_MAGIC, 8);
  fsh.number_samples = (unsigned) thdr.samples;
  fsh.sample_freq = thdr.rate;
  
  /* Create the textual header to go with the file */
  sprintf(header, "TIMIT\nversion=%d\nchannels=%d\nrate=%d\nsamples=%d\n",
          thdr.version, thdr.channels, thdr.rate, thdr.samples);

  /* Write out Fview headers to stdout */
  fsh.info_size = strlen(header) + 1;
  
  if (fwrite(&fsh, sizeof(FviewSampleHeader), 1, stdout) != 1)
  {
    fprintf(stderr, "timit: failed to write FviewSampleHeader\n");
    exit(-1);
  }
  fwrite(header, sizeof(char), fsh.info_size, stdout);
  
  /* copy 16 bit twos compliment samples across and byte-swap samples */
  short_buffer = (short *) malloc(thdr.samples * sizeof(short));
  if (fread(short_buffer, sizeof(short), thdr.samples, fd) != thdr.samples)
  {
    fprintf(stderr, "timit: failed to read sample data\n");
    exit(-1);
  }
  for (index = 0; index < thdr.samples; index++)
  {
    sample = short_buffer[index];
    short_buffer[index] = REV2(sample);
  }
  if (fwrite(short_buffer, sizeof(short), thdr.samples, stdout) !=
      thdr.samples)
  {
    fprintf(stderr, "timit: failed to write sample data\n");
    exit(-1);
  }
  free(short_buffer);
  
  fclose(fd);
  
  /* print out some usefile information */
#ifdef V12
  fprintf(stderr, "timit (%s):file %s, frames %d\n",
          VERSION, filename, thdr.samples);
#endif
  
  fflush(stderr);
}

