/*
  title:    interface.c
  purpose:  xview widget bindings for the fview program.

  author:   Gareth Lee and Hugh Sanderson.
  date:     15-07-93
  modified: 27-10-93

  changes:
  11-08-93: CreateDisplayWindow() mod'd to ensure that the window background is
    always set to white.  This ensures that the trajectories drawn are always
    visible no matter what default screen background colour the user has set.
*/

/* Standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <string.h>
#include <unistd.h>

/* X lib headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>

/* XView headers */
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/cursor.h>
#include <xview/cms.h>

/* Header files associated with other compilation units */
#include "projectors.h"
#include "callbacks.h"
#include "fview.h"
#include "interface.h"

/* Icon Bitmaps */
#include "bitmaps.xbm"

/* Widget handles for objects */
Xv_opaque MainFrame; 
Xv_opaque MainPanel; 
Xv_opaque SampleFrame; 
Xv_opaque SamplePanel;
Xv_opaque SpecifyFrame; 
Xv_opaque SpecifyPanel; 
Xv_opaque RotationSelector1;
Xv_opaque RotationSelector2;
Xv_opaque DisplayFrame;
Xv_opaque DisplayPanel;
Xv_opaque DisplayCanvas;
Xv_opaque DisplaySlider;
Xv_opaque DisplayRandomizeButton;
Xv_opaque DisplayStoreButton;
Xv_opaque DisplayRecallButton;
Xv_opaque DisplayCheckBox;
Xv_opaque DisplayDecrButton;
Xv_opaque DisplayIncrButton;
Xv_opaque ControlAutoScaleButton;
Xv_opaque UpDownSlider;
Xv_opaque LeftRightSlider;
Xv_opaque EnlargeShrinkSlider;
Xv_opaque ControlAnimateButton;
Xv_opaque AngleSlider;
Xv_opaque ControlSweepButton;
Xv_opaque SweepRateSlider;
Xv_opaque IterationSlider;
Xv_opaque ControlSpecifyButton;
Xv_opaque SpecifySelector1;
Xv_opaque SpecifyRandEq1Choice;
Xv_opaque SpecifySelector2;
Xv_opaque SpecifyRandEq2Choice;
Xv_opaque SpecifySelector3;
Xv_opaque SpecifyRandEq3Choice;
Xv_opaque SpecifyProjectButton;
Xv_opaque ControlSamplesButton;
Xv_opaque SampleListnameText;
Xv_opaque SampleDimensionText;
Xv_opaque SampleNumSampsText;
Xv_opaque SampleFileList;
Xv_opaque SampleFormatList;
Xv_opaque SampleFrontendList;
Xv_opaque SampleUpdateButton;
Xv_opaque SampleParamsButton;
Xv_opaque SampleCheckBox;
Xv_opaque ParamsFrame;
Xv_opaque ParamsPanel;
Xv_opaque ParamsFormatNameText;
Xv_opaque ParamsFormatFilterText;
Xv_opaque ParamsFrontendNameText;
Xv_opaque ParamsFrontendFilterText;
Xv_opaque ParamsFormatList;
Xv_opaque ParamsFrontendList;
Xv_opaque ParamsFormatValueText;
Xv_opaque ParamsFrontendValueText;
Xv_opaque ControlHighlightButton;
Xv_opaque HighlightFrame;
Xv_opaque HighlightPanel;
Xv_opaque HighlightSlider;
Xv_opaque HighlightSelector;
Xv_opaque HighlightCheckBox;

Xv_Cursor BasicCursor;
Xv_Cursor BusyCursor;
Server_image ciips_image;
Server_image project_image;

Icon ciips_icon;             /* CIIPS icon for use within the control window */
Icon project_icon;   /* Project icon for association with each of the frames */

/* Display and context variables */
int screen_num = 0;                                /* works on most machines */
unsigned int screen_width, screen_height;        /* size of display (pixels) */
Colormap DefColMap;
char *CName[40];         /* Must Leave this set to 40, cf. set_color_names() */
XColor traj_color[MAX_COLORS];       /* Colours for use drawing trajectories */
XColor spec_color[SPECTRUM_SIZE];    /* Colours for use drawing spectrograms */
XColor dim3_color[3];             /* Colours for use drawing 3D trajectories */
XFontStruct *font_info;
int tty_fd;                      /* file descriptor for the StdLogTTY window */

/* Color set to use for drawing trajectories */
const char* TrajectoryColors[MAX_COLORS] =
{
  "black", "blue", "red", "green", "maroon",
  "dark slate grey", "khaki", "magenta", "NavyBlue", "goldenrod",
  "pink", "purple", "thistle", "sienna", "aquamarine",
  "orchid", "violet", "tomato", "OliveDrab", "orange"
};

/*****************************************************************************/

/*
  SetCursors: Sets the Cursor type as specified in all frames and canvases.
*/
void SetCursors(Xv_Cursor cursor)
{
  Xv_Window win;

  win = (Xv_Window) xv_get(MainFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);
  win = (Xv_Window) xv_get(SampleFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);
  win = (Xv_Window) xv_get(DisplayFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);
  xv_set(canvas_paint_window(DisplayCanvas), WIN_CURSOR, cursor, NULL);
  win = (Xv_Window) xv_get(ParamsFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);
  win = (Xv_Window) xv_get(HighlightFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);
  win = (Xv_Window) xv_get(SpecifyFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);

  XFlush(display);
}

/*****************************************************************************/

/* Colors to use for drawing spectra */
const char* SpectrumColors[SPECTRUM_SIZE] =
{
  "black", "midnight blue", "dark slate blue", "blue violet",
  "medium slate blue", "LightSlateBlue", "turquoise3", "turquoise",
  "aquamarine", "green yellow", "yellow2",
  "gold",
  "orange",
  "orange red",
  "red",
  "white"
};

/*****************************************************************************/

void getGC(Window win, GC *gc, XFontStruct *font_info)
{
  static char dash_list[] = {12, 24};

  unsigned long valuemask = 0; /* ignore XGCvalues and use defaults */
  XGCValues values;
  unsigned int line_width = 1;
  int line_style = LineSolid;
  int cap_style = CapRound;
  int join_style = JoinRound;
  int dash_offset = 0;
  int list_length = 2;
 
  /* Create default Graphics Context */
  *gc = XCreateGC(display, win, valuemask, &values);
 
  /* specify font */
  XSetFont(display, *gc, font_info->fid);

  /* specify black foreground since default window background is
   * white and default foreground is undefined. */
  XSetForeground(display, *gc, BlackPixel(display, screen_num));

  /* set line attributes */
  XSetLineAttributes(display, *gc, line_width, line_style,
                     cap_style, join_style);

  /* set dashes */
  XSetDashes(display, *gc, dash_offset, dash_list, list_length);
} /* end of getGC */
 
/*****************************************************************************/

void load_font(XFontStruct **font_info)
{
  char *fontname = DEFAULT_FONT_NAME;

  /* Load font and get font information structure. */
  if ((*font_info = XLoadQueryFont(display, fontname)) == NULL)
  {
    printf("Error: Cannot open the font '%s'\n", fontname);
    Quit(-1);
  }
} /* end of load_font */

/*****************************************************************************/

void TooSmall(Window win, GC gc, XFontStruct *font_info)
{
  char *string1 = "Too Small";
  int y_offset, x_offset;
 
  y_offset = font_info->ascent + 2;
  x_offset = 2;

  /* output text, centered on each line */
  XDrawString(display, win, gc, x_offset, y_offset, string1,
              strlen(string1));
} /* end of TooSmall */

/*****************************************************************************/

/*
  get_colors: Allocate the colours that will be needed from the (shared) colour
    map.
*/
int get_colors(char **name)
{
  int i, j;
  int def, default_depth;
  Colormap default_cmap;
  unsigned long colors[3];
  unsigned long plane_masks[1];
  XVisualInfo visual_info;
  XColor exact;
  unsigned r1, g1, b1, r2, g2, b2, r3, g3, b3;            /* 3D Color values */

  default_depth = DefaultDepth(display, screen_num);
  default_cmap   = DefaultColormap(display, screen_num);
  if (default_depth == 1) {
    /* must be StaticGray, use black and white */
    return(0);
  }
 
  if (!XMatchVisualInfo(display, screen_num, default_depth, PseudoColor,
                                                               &visual_info)) {
    if (!XMatchVisualInfo(display, screen_num, default_depth, DirectColor,
                                                               &visual_info)) {
      /* No PseudoColor visual available at default_depth.
       * Some applications might try for a GrayScale visual
       * here if they can use gray to advantage, before
       * giving up and using black and white.
       */
      return(0);
    }  
  }   

  /* Setup trajectory colours */
  for (i = 0; i < MAX_COLORS; i++)
    if (!XAllocNamedColor(display, default_cmap, TrajectoryColors[i],
                          &traj_color[i], &exact))
    {
      printf("Error: could not allocate Trajectory color %s (i = %d)\n",
             TrajectoryColors[i], i);
      Quit(-1);
    }

  /* Setup spectrogram colours */
  for (i = 0; i < SPECTRUM_SIZE; i++)
    if (!XAllocNamedColor(display, default_cmap, SpectrumColors[i],
                          &spec_color[i], &exact))
    {
      printf("Error: could not allocate Spectrum color %s (i = %d)\n",
             TrajectoryColors[i], i);
      Quit(-1);
    }


  /* Setup the three 3D color cells */
  if (!XAllocColorCells(display,default_cmap,False,plane_masks,0,colors,3))
  {
    printf("Error: Couldn't allocate %d 3D r/w colors\n", 3);
    Quit(-1);
  }

  /*
    Manually set RGB values for three dimensional colors.  Note: These colors
    have been manually optimised (on a 24bit Sun monitor) for the RED/GREEN
    glasses used.
    */
  def = 1;
  if (ColorString != NULL)
  {
    if (sscanf(ColorString, "%u:%u:%u:%u:%u:%u:%u:%u:%u",
               &r1, &g1, &b1, &r2, &g2, &b2, &r3, &g3, &b3) == 9)
    {
      if (r1 <= 256 && g1 <= 256 && b1 <= 256 &&
          r2 <= 256 && g2 <= 256 && b2 <= 256 &&
          r3 <= 256 && g3 <= 256 && b3 <= 256)
      {
        def = 0;
        
        dim3_color[0].red   = r1 * 256;              /* Color0 == Background */
        dim3_color[0].green = g1 * 256;
        dim3_color[0].blue  = b1 * 256;
        dim3_color[1].red   = r2 * 256;                     /* Color1 == Red */
        dim3_color[1].green = g2 * 256;
        dim3_color[1].blue  = b2 * 256;
        dim3_color[2].red   = r3 * 256;                   /* Color2 == Green */
        dim3_color[2].green = g3 * 256;
        dim3_color[2].blue  = b3 * 256;
      }
      else
      {
        printf("Error: RGB intensity exceeds 256\n");
        Quit(-1);
      }
    }
    else
    {
      printf("Error: format error in RGB string\n");
      Quit(-1);
    }
  }

  /* Setup default RGB colors if no others are specified */
  if (def)
  {
    dim3_color[0].red = 0xdc00;
    dim3_color[0].green = 0x8700;
    dim3_color[0].blue = 0xff00;
    dim3_color[1].red = 0xfa00;
    dim3_color[1].green = 0x0100;
    dim3_color[1].blue = 0x0000;
    dim3_color[2].red = 0x0000;
    dim3_color[2].green = 0xa600;
    dim3_color[2].blue = 0x6600;
  }

  dim3_color[0].flags = DoRed | DoGreen | DoBlue;
  dim3_color[0].pixel = colors[0];  
  dim3_color[1].flags = DoRed | DoGreen | DoBlue;
  dim3_color[1].pixel = colors[1];  
  dim3_color[2].flags = DoRed | DoGreen | DoBlue;
  dim3_color[2].pixel = colors[2];
  XStoreColors(display, default_cmap, dim3_color, 3);

  SetupColorEntries();                         /* Set color allocation table */
  
  return 1;
}  /* End of get_colors */

/*****************************************************************************/

/*
  ChangeBandWidgets: customise any widgets that depend upon the number of
    dimensions set.
*/
void ChangeBandWidgets(void)
{
  int i, width;
  char buf[1024];

  /* reset the two RotationSelector widgets */
  xv_set(RotationSelector1,
         XV_SHOW, FALSE,
         PANEL_CHOICE_STRINGS,
           "Band 1",
           NULL,
         NULL);
  xv_set(RotationSelector2,
         XV_SHOW, FALSE,
         PANEL_CHOICE_STRINGS,
           "Band 1",
           NULL,
         NULL);
  for (i = 1; i < dimension; i++)
  {
    sprintf(buf, "Band %d", i + 1);

    xv_set(RotationSelector1, PANEL_CHOICE_STRING, i, buf, NULL);
    xv_set(RotationSelector2, PANEL_CHOICE_STRING, i, buf, NULL);
  }
  xv_set(RotationSelector1, XV_SHOW, TRUE, NULL);
  xv_set(RotationSelector2, XV_SHOW, TRUE, NULL);

  rotation_dim1 = ROTATION_DIM1;
  xv_set(RotationSelector1, PANEL_VALUE, ROTATION_DIM1, NULL);
  rotation_dim2 = ROTATION_DIM2;
  xv_set(RotationSelector2, PANEL_VALUE, ROTATION_DIM2, NULL);

  /*
    Resize the specify window and reposition/reset the widgets contained
      therein.
  */
  width = 140 + (dimension * 38);
  xv_set(SpecifyPanel, XV_SHOW, FALSE, NULL);
  xv_set(SpecifyFrame, XV_WIDTH, width, NULL);
  xv_set(SpecifyProjectButton, XV_X, width - 70, NULL);  

  i = (width / 2) - 120;
  xv_set(SpecifyRandEq1Choice, XV_X, i, NULL);
  xv_set(SpecifyRandEq2Choice, XV_X, i, NULL);
  xv_set(SpecifyRandEq3Choice, XV_X, i, NULL);

  /* Relabel all the selectors */
  xv_set(SpecifySelector1, PANEL_CHOICE_STRINGS, "1", NULL, NULL);
  xv_set(SpecifySelector2, PANEL_CHOICE_STRINGS, "1", NULL, NULL);
  xv_set(SpecifySelector3, PANEL_CHOICE_STRINGS, "1", NULL, NULL);
  for (i = 1; i < dimension; i++)
  {
    sprintf(buf, "%d", i + 1);

    xv_set(SpecifySelector1, PANEL_CHOICE_STRING, i, buf, NULL);
    xv_set(SpecifySelector2, PANEL_CHOICE_STRING, i, buf, NULL);
    xv_set(SpecifySelector3, PANEL_CHOICE_STRING, i, buf, NULL);
  }

  /* Reset all selector and choice widgets back to their default values */
  xv_set(SpecifySelector1, PANEL_VALUE, 0x0001, NULL);
  xv_set(SpecifyRandEq1Choice, PANEL_VALUE, 0, NULL);
  xv_set(SpecifySelector2, PANEL_VALUE, 0x0002, NULL);
  xv_set(SpecifyRandEq2Choice, PANEL_VALUE, 0, NULL);
  xv_set(SpecifySelector3, PANEL_VALUE, 0x0004, NULL);
  xv_set(SpecifyRandEq3Choice, PANEL_VALUE, 0, NULL);

  /* only expose the panel once all changes have been made */
  xv_set(SpecifyPanel, XV_SHOW, TRUE, NULL);
}

/*****************************************************************************/

/*
  PrepareConverterLists: examine records in the formats and front_ends linked
    lists and prepare their lists accordingly.
*/
void PrepareConverterLists(void)
{
  int index;
  FilterRecord *fptr;
  char buf[1024];
  
  /* Create Formats (conversion filters) list */
  index = 0;
  for (fptr = &Formats; fptr != NULL; fptr = fptr->next)
  {
    sprintf(buf, "%s [->%c]", fptr->name, fptr->format);
    xv_set(SampleFormatList,
           PANEL_LIST_INSERT, index,
           PANEL_LIST_STRING, index, buf,
           PANEL_LIST_CLIENT_DATA, index, fptr,
           NULL);
    
    fptr->index = index;
    index++;
  }
  xv_set(SampleFormatList,
         PANEL_LIST_SELECT, 0, TRUE,
         XV_SHOW, TRUE,
         NULL);
  
  /* Create Front-End filters list */ 
  index = 0;
  for (fptr = &Frontends; fptr != NULL; fptr = fptr->next)
  {
    sprintf(buf, "[%c->] %s", fptr->format, fptr->name);
    xv_set(SampleFrontendList,
           PANEL_LIST_INSERT, index,
           PANEL_LIST_STRING, index, buf,
           PANEL_LIST_CLIENT_DATA, index, fptr,
           NULL);
    
    fptr->index = index;
    index++;
  }
  xv_set(SampleFrontendList,
         PANEL_LIST_SELECT, 0, TRUE,
         XV_SHOW, TRUE,
         NULL);
}

/*****************************************************************************/

/*
  CreateSampleWindow: A pop-up window providing a scrollable list of the
    samples (utterances) within the list file.  Samples can then be
    selected and deselected from the projection display at run time.
    NB: Window geometry is modified between Contracted and Expanded states
    by SampleOptionsCallback().
*/
void CreateSampleWindow(Xv_opaque owner)
{
  unsigned int sample_width = 260;  /* contracted window width, expanded=500 */
  unsigned int sample_height = 300;                         /* window height */
  
  SampleFrame =
    xv_create(owner, FRAME_CMD,
              XV_X, (screen_width - sample_width - 10),
              XV_Y, 10,
              XV_WIDTH,  sample_width,
              XV_HEIGHT, sample_height,
              XV_LABEL, "List File",
              XV_SHOW, FALSE,              
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              FRAME_DONE_PROC, SampleDismissCallback,
              FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
              NULL);

  SamplePanel = xv_get(SampleFrame, FRAME_CMD_PANEL);

  SampleListnameText =
    xv_create(SamplePanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 10,
              PANEL_VALUE_DISPLAY_WIDTH, 150,
              PANEL_LABEL_STRING, "File Name:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_NOTIFY_PROC, SampleListnameCallback,
              NULL);
  SampleDimensionText =
    xv_create(SamplePanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 30,
              PANEL_VALUE_DISPLAY_WIDTH, 84,
              PANEL_LABEL_STRING, "Feature Dimensions:",
              PANEL_VALUE, "0",
              PANEL_READ_ONLY, TRUE,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              NULL);
  SampleNumSampsText =
    xv_create(SamplePanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 50,
              PANEL_VALUE_DISPLAY_WIDTH, 117,
              PANEL_LABEL_STRING, "Samples in List:",
              PANEL_VALUE, "0",
              PANEL_READ_ONLY, TRUE,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              NULL);
  SampleFileList =
    xv_create(SamplePanel, PANEL_LIST,
              XV_X, 10,
              XV_Y, 75,
              PANEL_LIST_WIDTH, 225,
              PANEL_LIST_DISPLAY_ROWS, 8,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_NOTIFY_PROC, SampleListCallback,
              PANEL_READ_ONLY, FALSE,
              PANEL_CHOOSE_ONE, FALSE,
              NULL);
  SampleCheckBox =
    xv_create(SamplePanel, PANEL_CHECK_BOX,
              XV_X, 10,
              XV_Y, 250,
              PANEL_VALUE, 0,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_VERTICAL,
              PANEL_LABEL_STRING, "Options:",
              PANEL_CHOICE_STRINGS,
                "Information",
                "Features",
                NULL,
              PANEL_NOTIFY_PROC, SampleOptionsCallback,
              NULL);

  SampleFormatList =
    xv_create(SamplePanel, PANEL_LIST,
              XV_X, 270,
              XV_Y, 10,
              PANEL_LIST_TITLE, "File Formats:",
              PANEL_LIST_WIDTH, 205,
              PANEL_LIST_DISPLAY_ROWS, 4,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_ONE, TRUE,
              PANEL_NOTIFY_PROC, SampleFormatCallback,
              XV_SHOW, FALSE,
              NULL);

  SampleFrontendList =
    xv_create(SamplePanel, PANEL_LIST,
              XV_X, 270,
              XV_Y, 140,
              PANEL_LIST_TITLE, "Front Ends:",
              PANEL_LIST_WIDTH, 205,
              PANEL_LIST_DISPLAY_ROWS, 4,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_ONE, TRUE,
              PANEL_NOTIFY_PROC, SampleFrontendCallback,
              XV_SHOW, FALSE,
              NULL);

  SampleUpdateButton =
    xv_create(SamplePanel, PANEL_BUTTON,
              XV_X, 290,
              XV_Y, 270,
              PANEL_LABEL_STRING, "Update",
              PANEL_NOTIFY_PROC, SampleUpdateCallback,
              XV_SHOW, FALSE,
              NULL);

  SampleParamsButton =
    xv_create(SamplePanel, PANEL_BUTTON,
              XV_X, 400,
              XV_Y, 270,
              PANEL_LABEL_STRING, "Parameters",
              PANEL_NOTIFY_PROC, SampleParamsCallback,
              XV_SHOW, FALSE,
              NULL);
}

/*****************************************************************************/

/*
  SampleFillScrollList: Fills up a scrolling list widget by traversing down
    through a linked list of OptionRecords.
*/ 
void SampleFillScrollList(Xv_opaque list, OptionRecord *option)
{
  int index;
  OptionRecord *optr;
  char buf[1024];

  index = 0;
  for (optr = option; optr != NULL; optr = optr->next)
  {
    sprintf(buf, "%s = %s", optr->name, optr->value);
    xv_set(list,
           PANEL_LIST_INSERT, index,
           PANEL_LIST_STRING, index, buf,
           PANEL_LIST_CLIENT_DATA, index, optr,
           NULL);
    optr->index = index;
    index++;        
  }
}

/*****************************************************************************/

/*
  SampleEmptyScrollList: Empties a scrolling list widget previously filled 
    by SampleFillScrollList().
*/ 
void SampleEmptyScrollList(Xv_opaque list)
{
  xv_set(list, PANEL_LIST_DELETE_ROWS, 0, 99, NULL);
}

/*****************************************************************************/

void CreateParamsWindow(Xv_opaque owner)
{
  const int params_width = 500;
  const int params_height = 190;
  
  ParamsFrame =
    xv_create(owner, FRAME_CMD,
              XV_X, 100,
              XV_Y, 100,
              XV_WIDTH,  params_width,
              XV_HEIGHT, params_height,
              XV_LABEL, "Filter Parameters",
              XV_SHOW, FALSE,              
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              FRAME_DONE_PROC, ParamsDismissCallback,
              FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
              NULL);

  ParamsPanel = /* xv_get(ParamsFrame, FRAME_CMD_PANEL); */
    xv_create(ParamsFrame, PANEL,
              XV_X, 0,
              XV_Y, 0,
              XV_WIDTH , WIN_EXTEND_TO_EDGE,
              XV_HEIGHT, WIN_EXTEND_TO_EDGE,
              WIN_BORDER, FALSE,
              NULL);

  ParamsFormatNameText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 10,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Format:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_READ_ONLY, TRUE,
              PANEL_VALUE, DEF_FORMAT_FILTER,
              NULL);

  ParamsFormatFilterText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 35,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Filter:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_READ_ONLY, TRUE,
              NULL);

  ParamsFormatList =
    xv_create(ParamsPanel, PANEL_LIST,
              XV_X, 10,
              XV_Y, 60,
              PANEL_LIST_WIDTH, 205,
              PANEL_LIST_DISPLAY_ROWS, 4,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_ONE, TRUE,
              PANEL_NOTIFY_PROC, ParamsFormatCallback,
              NULL);

    ParamsFormatValueText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 160,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Value:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_NOTIFY_PROC, ParamsFormatTextCallback,
              PANEL_READ_ONLY, TRUE,
              PANEL_CLIENT_DATA, NULL,
              NULL);

  ParamsFrontendNameText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 260,
              XV_Y, 10,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Front End:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_READ_ONLY, TRUE,
              PANEL_VALUE, DEF_FRONTEND_FILTER,
              NULL);

  ParamsFrontendFilterText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 260,
              XV_Y, 35,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Filter:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_READ_ONLY, TRUE,
              NULL);

  ParamsFrontendList =
    xv_create(ParamsPanel, PANEL_LIST,
              XV_X, 260,
              XV_Y, 60,
              PANEL_LIST_WIDTH, 205,
              PANEL_LIST_DISPLAY_ROWS, 4,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_ONE, TRUE,
              PANEL_NOTIFY_PROC, ParamsFrontendCallback,
              NULL);

  ParamsFrontendValueText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 260,
              XV_Y, 160,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Value:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_NOTIFY_PROC, ParamsFrontendTextCallback,
              PANEL_READ_ONLY, TRUE,
              PANEL_CLIENT_DATA, NULL,
              NULL);
}

/*****************************************************************************/

/*
  CreateSpecifyWindow: pop-up window with controls for selecting new basis
    functions for projection.  Note: Panel sizes and widget positions are
    customized by CreateBandWidgets() depending on the current value of
    'dimension'.
*/
void CreateSpecifyWindow(Xv_opaque owner)
{
  SpecifyFrame =
    xv_create(owner, FRAME_CMD,
              XV_X, 100,
              XV_Y, 100,
              XV_HEIGHT, 205,
              XV_LABEL, "Projection Basis Vectors",
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              XV_SHOW, FALSE,
              FRAME_DONE_PROC, SpecifyDismissCallback,
              FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
              NULL);

  SpecifyPanel = xv_get(SpecifyFrame, FRAME_CMD_PANEL);

  /* Create Rotation axis selector radio-buttons */
  SpecifySelector1 =
    xv_create(SpecifyPanel, PANEL_TOGGLE,
              XV_X, 5,
              XV_Y, 10,
              PANEL_VALUE, 1,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Subspace 1:",
              PANEL_NOTIFY_PROC, SpecifyToggleCallback,
              PANEL_CLIENT_DATA, 1,
              NULL);

  SpecifyRandEq1Choice =
    xv_create(SpecifyPanel, PANEL_CHOICE,
              XV_Y, 40,
              PANEL_LABEL_STRING, "Basis 1:",
              PANEL_CHOICE_STRINGS,
                "Diagonal", "Random", NULL,
              NULL);

  SpecifySelector2 =
    xv_create(SpecifyPanel, PANEL_TOGGLE,
              XV_X, 5,
              XV_Y, 75,
              PANEL_VALUE, 2,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Subspace 2:",
              PANEL_NOTIFY_PROC, SpecifyToggleCallback,
              PANEL_CLIENT_DATA, 2,
              NULL);

  SpecifyRandEq2Choice =
    xv_create(SpecifyPanel, PANEL_CHOICE,
              XV_Y, 105,
              PANEL_LABEL_STRING, "Basis 2:",
              PANEL_CHOICE_STRINGS,
                "Diagonal", "Random", NULL,
              NULL);

  SpecifySelector3 =
    xv_create(SpecifyPanel, PANEL_TOGGLE,
              XV_X, 5,
              XV_Y, 140,
              PANEL_VALUE, 4,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Subspace 3:",
              PANEL_NOTIFY_PROC, SpecifyToggleCallback,
              PANEL_CLIENT_DATA, 3,
              NULL);

  SpecifyRandEq3Choice =
    xv_create(SpecifyPanel, PANEL_CHOICE,
              XV_Y, 170,
              PANEL_LABEL_STRING, "Basis 3:",
              PANEL_CHOICE_STRINGS,
                "Diagonal", "Random", NULL,
              NULL);

  SpecifyProjectButton =
    xv_create(SpecifyPanel, PANEL_BUTTON,
              XV_Y, 180,
              PANEL_LABEL_STRING, "Project",
              PANEL_NOTIFY_PROC, SpecifyProjectCallback,
              NULL);
}

/*****************************************************************************/

/*
  HighlightWindow: Creates a window to allow control of vertex Prediction
    and display of highlighted vertices.
*/
void CreateHighlightWindow(Xv_opaque owner)
{
  const int highlight_width = 300;
  const int highlight_height = 140;
  
  HighlightFrame =
    xv_create(owner, FRAME_CMD,
              XV_X, 100,
              XV_Y, 140,
              XV_WIDTH, highlight_width,
              XV_HEIGHT, highlight_height,
              XV_LABEL, "Highlight Vertices",
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              XV_SHOW, FALSE,
              FRAME_DONE_PROC, HighlightDismissCallback,
              FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
              NULL);

  HighlightPanel = xv_get(HighlightFrame, FRAME_CMD_PANEL);

  /* Create Rotation axis selector radio-buttons */
  HighlightSelector =
    xv_create(HighlightPanel, PANEL_CHOICE,
              XV_X, 10,
              XV_Y, 10,
              PANEL_VALUE, 0,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_NONE, FALSE,
              PANEL_LABEL_STRING, "Predictor Order:",
              PANEL_CHOICE_STRINGS, "Zeroth", "First", "Second", NULL,
              PANEL_NOTIFY_PROC,HighlightSelectorCallback,
              NULL);

  HighlightCheckBox =
    xv_create(HighlightPanel, PANEL_CHECK_BOX,
              XV_X, 10,
              XV_Y, 40,
              PANEL_VALUE, 0,
              PANEL_CHOICE_NROWS, 2,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Options:",
              PANEL_CHOICE_STRINGS, "Direction", "Velocity", "Verbose", NULL,
              NULL);
  
  HighlightSlider =
    xv_create(HighlightPanel, PANEL_SLIDER,
              XV_X, 10,
              XV_Y, 105,
              PANEL_LABEL_STRING, "Vertices:",
              PANEL_SLIDER_WIDTH, 100,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, TRUE,
              PANEL_SHOW_VALUE, TRUE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 10,
              PANEL_VALUE, 0,
              PANEL_INACTIVE, TRUE,
              PANEL_NOTIFY_PROC, HighlightSliderCallback,
              NULL);
}

/*****************************************************************************/

CreateDisplayWindow(Xv_opaque owner)
{
  int pm,a;
  int Ncols;
  XColor xc;
  XSetWindowAttributes attributes;
  Server_image UpArrowGlyph, DownArrowGlyph;
  
  DisplayFrame =
    xv_create(owner, FRAME,
              XV_X, 0,
              XV_Y, 0,
              XV_WIDTH, 750,
              XV_HEIGHT, 700,
              XV_LABEL, "Trajectory Display",
              XV_SHOW, FALSE,
              FRAME_SHOW_RESIZE_CORNER, TRUE,
              XV_VISUAL_CLASS, PseudoColor,
              NULL);
  DisplayPanel =
    xv_create(DisplayFrame, PANEL,
              XV_X, 0,
              XV_Y, 0,
              XV_WIDTH, WIN_EXTEND_TO_EDGE,
              XV_HEIGHT, 60,
              WIN_BORDER, TRUE,
              NULL);
  DisplayCanvas =
    xv_create(DisplayFrame, CANVAS,
              XV_X, 0,
              XV_Y, 60,
              XV_WIDTH, WIN_EXTEND_TO_EDGE,
              XV_HEIGHT, WIN_EXTEND_TO_EDGE,
              CANVAS_X_PAINT_WINDOW, FALSE,
              CANVAS_RETAINED, TRUE,
              CANVAS_FIXED_IMAGE, TRUE,
              CANVAS_REPAINT_PROC, RefreshGraphics,
              CANVAS_RESIZE_PROC, RefreshGraphics,
              NULL);
  xv_create(DisplayPanel, PANEL_CHOICE_STACK,
              XV_X, 10,
              XV_Y, 6,
              PANEL_LABEL_STRING, "Format:",
              PANEL_CHOICE_STRINGS,
                "2 Dimensions", "3 Dimensions", "Spectrogram",
                NULL,
              PANEL_NOTIFY_PROC, DisplayFormatCallback,
              NULL);

  DisplayRandomizeButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 220,
              XV_Y, 8,
              PANEL_LABEL_STRING, "Randomize",
              PANEL_NOTIFY_PROC, DisplayRandomizeCallback,
              NULL);

  /*
    NB: These two buttons share a common callback routine. The rotuine tells
      between them using the PANEL_CLIENT_DATA value.  The "Recall" button is
      made active on the first press of the "Store" button.
  */
  DisplayStoreButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 325,
              XV_Y, 8,
              PANEL_LABEL_STRING, "Store",
              PANEL_NOTIFY_PROC, DisplayStoreRecallCallback,
              PANEL_CLIENT_DATA, 0,
              NULL);
  DisplayRecallButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 390,
              XV_Y, 8,
              PANEL_LABEL_STRING, "Recall",
              PANEL_NOTIFY_PROC, DisplayStoreRecallCallback,
              PANEL_CLIENT_DATA, 1,
              NULL);
  
  DisplayCheckBox =
    xv_create(DisplayPanel, PANEL_CHECK_BOX,
              XV_X, 10,
              XV_Y, 30,
              PANEL_VALUE, 0,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Options:",
              PANEL_CHOICE_STRINGS, "Raw", "Scaled", "Labels", "Origin", NULL,
              PANEL_NOTIFY_PROC, DisplayCheckBoxCallback,
              NULL);
  DisplaySlider =
    xv_create(DisplayPanel, PANEL_SLIDER,
              XV_X, 600,
              XV_Y, 4,
              PANEL_VALUE, 0,
              PANEL_SLIDER_WIDTH, 100,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 99,
              PANEL_MIN_TICK_STRING, "Linear",
              PANEL_MAX_TICK_STRING, "Compressed",
              PANEL_NOTIFY_PROC, DisplaySliderCallback,
              NULL);

  /* Create icon for display within DisplayDecrButton */
  UpArrowGlyph = (Server_image)
    xv_create(NULL, SERVER_IMAGE,
              XV_WIDTH, up_arrow_width,
              XV_HEIGHT, up_arrow_height,
              SERVER_IMAGE_X_BITS, up_arrow_bits,
              NULL);

  DisplayDecrButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 470,
              XV_Y, 8,
              PANEL_LABEL_IMAGE, UpArrowGlyph,
              PANEL_NOTIFY_PROC, DisplayArrowCallback,
              PANEL_INACTIVE, TRUE,
              PANEL_CLIENT_DATA, 0,
              NULL);

  /* Create icon for display within DisplayIncrButton */
  DownArrowGlyph = (Server_image)
    xv_create(NULL, SERVER_IMAGE,
              XV_WIDTH, down_arrow_width,
              XV_HEIGHT, down_arrow_height,
              SERVER_IMAGE_X_BITS, down_arrow_bits,
              NULL);

  DisplayIncrButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 530,
              XV_Y, 8,
              PANEL_LABEL_IMAGE, DownArrowGlyph,
              PANEL_NOTIFY_PROC, DisplayArrowCallback,
              PANEL_INACTIVE, TRUE,
              PANEL_CLIENT_DATA, 1,
              NULL);

  /* Register Event Handler for Keyboard Events within the window */
/*
  xv_set(canvas_paint_window(DisplayCanvas),
         WIN_EVENT_PROC, DisplayEventHandler,
         WIN_CONSUME_X_EVENT_MASK, KeyPressMask,
         WIN_IGNORE_X_EVENT_MASK, KeyReleaseMask,
         NULL);
*/
  
  /* Obtain handles into window */
  display = (Display *) xv_get(DisplayCanvas,XV_DISPLAY);
  win = (Window) xv_get(canvas_paint_window(DisplayCanvas),XV_XID);

  /* set background color then draw left and right eye point sets */
  attributes.background_pixel = WhitePixel(display, screen_num);
  XChangeWindowAttributes(display, win, CWBackPixel, &attributes);
  XClearWindow(display, win);

  /* Prepare Canvas for drawing */
  (void) get_colors(CName);                           /* Setup color palette */
  load_font(&font_info);                                /* Load Display Font */
  getGC(win, &gc, font_info);              /* create GC for text and drawing */
}

/*****************************************************************************/

/*
  SetWindowManager: Commands controlling window manager and specifying the
    icon to use etc. .
*/
void SetWindowManager(int argc, char* argv[])
{
  Server_image hour_glass_image, hour_glass_mask;
  Xv_singlecolor black, white;
  
  project_image = (Server_image)
    xv_create(NULL, SERVER_IMAGE,
              XV_WIDTH, project_width,
              XV_HEIGHT, project_height,
              SERVER_IMAGE_X_BITS, project_bits,
              NULL);
  project_icon = (Icon)
    xv_create(NULL, ICON,
              XV_WIDTH, project_width,
              XV_HEIGHT, project_height,
              ICON_IMAGE, project_image,
              ICON_TRANSPARENT, TRUE,
              NULL);

  /* Create the set of cursors */
  BasicCursor = (Xv_Cursor)
    xv_create(XV_NULL, CURSOR,
              CURSOR_SRC_CHAR,  OLC_BASIC_PTR,
              CURSOR_MASK_CHAR, OLC_BASIC_MASK_PTR,
              NULL);
  BusyCursor = (Xv_Cursor)
    xv_create(XV_NULL, CURSOR,
              CURSOR_SRC_CHAR,  OLC_BUSY_PTR,
              CURSOR_MASK_CHAR, OLC_BUSY_MASK_PTR,
              NULL);
}

/*****************************************************************************/

void CreateControlWindow(int argc, char* argv[])
{
  unsigned int control_width  = 460;                /* window width (pixels) */
  unsigned int control_height = 330;               /* window height (pixels) */

  /* setup xv graphics system */
  xv_init(XV_INIT_ARGS, argc, argv, NULL);

  /* Create Control window frame and panel */
  MainFrame =
    xv_create(NULL, FRAME,
              XV_WIDTH,  control_width,
              XV_HEIGHT, control_height,
              XV_LABEL, "fview v1.0",
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              XV_SHOW, FALSE,
              NULL);

  /* Creates window containing Canvas for graphics output */
  CreateDisplayWindow(MainFrame);

  /* Create Icon associated with the windows */
  SetWindowManager(argc, argv);
  
  /*
    Get total screen and adjust position of MainFrame accordingly.
    Also set frame icon.
  */
  screen_width  = DisplayWidth(display,  screen_num);
  screen_height = DisplayHeight(display, screen_num);
  xv_set(MainFrame,
         XV_X, (screen_width - control_width - 30),
         XV_Y, (screen_height - control_height - 30),
         FRAME_ICON, project_icon,
         NULL);

  MainPanel =
    xv_create(MainFrame, PANEL,
              XV_X, 0,
              XV_Y, 0,
              XV_WIDTH , WIN_EXTEND_TO_EDGE,
              XV_HEIGHT, WIN_EXTEND_TO_EDGE,
              WIN_BORDER, FALSE,
              NULL);

  /* Create CIIPS icon for display in auto_scale mode */
  ciips_image = (Server_image)
    xv_create(NULL, SERVER_IMAGE,
              XV_WIDTH, ciips_width,
              XV_HEIGHT, ciips_height,
              SERVER_IMAGE_X_BITS, ciips_bits,
              NULL);

  ciips_icon = (Icon)
    xv_create(MainPanel, ICON,
              XV_X, 240,
              XV_Y, 70,
              XV_WIDTH, ciips_width,
              XV_HEIGHT, ciips_height,
              ICON_IMAGE, ciips_image,
              XV_SHOW, FALSE,
              NULL);

  /* Create Rotation axis selector radio-buttons */
  RotationSelector1 =
    xv_create(MainPanel, PANEL_CHOICE_STACK,
              XV_X, 10,
              XV_Y, 10,
              PANEL_VALUE, rotation_dim1,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_NONE, FALSE,
              PANEL_LABEL_STRING, "Rotation Axis 1:",
              PANEL_NOTIFY_PROC, Rotation1Callback,
              NULL);

  RotationSelector2 =
    xv_create(MainPanel, PANEL_CHOICE_STACK,
              XV_X, 240,
              XV_Y, 10,
              PANEL_VALUE, rotation_dim2,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_NONE, FALSE,
              PANEL_LABEL_STRING, "Rotation Axis 2:",
              PANEL_NOTIFY_PROC, Rotation2Callback,
              NULL);

  /* Create Sliders for Position and Scale of Projection */
  UpDownSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 280,
              XV_Y, 50,
              PANEL_SLIDER_WIDTH, 100,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_VERTICAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, -1000,
              PANEL_MAX_VALUE, 1000,
              PANEL_MIN_TICK_STRING, "Down",
              PANEL_MAX_TICK_STRING, "Up",
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, ViewSliderCallback,
              NULL);
  
  LeftRightSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 300,
              XV_Y, 190,
              PANEL_SLIDER_WIDTH, 100,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, -1000,
              PANEL_MAX_VALUE, 1000,
              PANEL_MIN_TICK_STRING, "Left",
              PANEL_MAX_TICK_STRING, "Right",
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, ViewSliderCallback,
              NULL);

  EnlargeShrinkSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 360,
              XV_Y, 50,
              PANEL_SLIDER_WIDTH, 100,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_VERTICAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, -100,
              PANEL_MAX_VALUE, 100,
              PANEL_MIN_TICK_STRING, "Reduce",
              PANEL_MAX_TICK_STRING, "Enlarge",
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, ViewSliderCallback,
              NULL);

  ControlAutoScaleButton =
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 320,
              XV_Y, 250,
              PANEL_LABEL_STRING, "AutoScale",
              PANEL_NOTIFY_PROC, AutoCallback,
              NULL);

  /* Hide buttons from outset if initially in autoscale mode */
  if (auto_scale)
  {
    xv_set(UpDownSlider, XV_SHOW, FALSE, NULL);
    xv_set(LeftRightSlider, XV_SHOW, FALSE, NULL);
    xv_set(EnlargeShrinkSlider, XV_SHOW, FALSE, NULL);
    xv_set(ciips_icon, XV_SHOW, TRUE, NULL);
  }

  /* Controls for Animation System */
  AngleSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 10,
              XV_Y, 50,
              PANEL_LABEL_STRING, "Angle:",
              PANEL_SLIDER_WIDTH, 140,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 1000,
              PANEL_MIN_TICK_STRING, "Small",
              PANEL_MAX_TICK_STRING, "Large",
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, AngleSliderCallback,
              NULL);

  IterationSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 10,
              XV_Y, 110,
              PANEL_LABEL_STRING, "Range:",
              PANEL_SLIDER_WIDTH, 140,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, iterations,
              PANEL_MAX_VALUE, 1000,
              PANEL_MIN_TICK_STRING, "Small",
              PANEL_MAX_TICK_STRING, "Large",
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, IterationSliderCallback,
              NULL);

  ControlAnimateButton = 
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 100,
              XV_Y, 155,
              PANEL_LABEL_STRING, "Animate",
              PANEL_NOTIFY_PROC, AnimateCallback,
              NULL);

  /* Display options */
  SweepRateSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 10,
              XV_Y, 210,
              PANEL_LABEL_STRING, "Sweep:",
              PANEL_SLIDER_WIDTH, 140,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 200,
              PANEL_MIN_TICK_STRING, "Fast",
              PANEL_MAX_TICK_STRING, "Slow",
              PANEL_VALUE, 0,
              NULL);

  ControlSweepButton =
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 100,
              XV_Y, 255,
              PANEL_LABEL_STRING, "Sweep",
              PANEL_NOTIFY_PROC, SweepGraphicsCallback,
              NULL);

  /* Create Remaining buttons along base of panel */
  ControlSamplesButton = 
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 30,
              XV_Y, 300,
              PANEL_LABEL_STRING, "Samples",
              PANEL_NOTIFY_PROC, SamplesCallback,
              NULL);

  ControlSpecifyButton = 
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 150,
              XV_Y, 300,
              PANEL_LABEL_STRING, "Specify",
              PANEL_NOTIFY_PROC, SpecifyCallback,
              NULL);

  ControlHighlightButton =
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 260,
              XV_Y, 300,
              PANEL_LABEL_STRING, "Highlight",
              PANEL_NOTIFY_PROC, HighlightCallback,
              NULL);

    xv_create(MainPanel, PANEL_BUTTON,
            XV_X, 380,
            XV_Y, 300,
            PANEL_LABEL_STRING, "Quit",
            PANEL_NOTIFY_PROC, QuitCallback,
            NULL);

  /*
    Create (hidden) pop-up windows for selecting utterances to view, for
    selecting both format and front-end filters and for controlling the
    parameters of the filters selected.
  */
  CreateSampleWindow(MainFrame);
  CreateParamsWindow(MainFrame);
  
  /* Creates (hidden) pop-up window for setting projection axes */
  CreateSpecifyWindow(MainFrame);

  /* Creates (hidden) pop-up window for controllong prediction/highlighting */
  CreateHighlightWindow(MainFrame);
}

/*****************************************************************************/

/* end of interface.c */
