/*
  title: fview
  purpose: To display two and three dimensional projections of speech
    trajectories through a high dimensional filter-band space.  With
    bindings to an Xview interface defined in interface.c.

  authors:  Gareth Lee & Nicole Duball.
  date:     16-08-93
  modified: 13-11-93

  changes:
  05-09-93: fview.c based on sp_proj3.c.  But with the addition of variable
    dimension feature prediction.  The dimension is read from the first
    entry in the list file all other entries must have the same dimension.
    Band selection widgets automatically customize themselves to suit the
    dimension that is currently selected.
  07-09-93: The SpecifyWindow has been rationalized with the removal of
    various controlling variables (vector_one_on etc).
  13-11-93: Support for a SearchPath added and used by all driver/config/list
    file loads.  The path is read from an environment variable.  The default
    list filename can also be overrided using the -list command line option.
  13-11-93:
*/

/* Standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <signal.h>

/* X lib headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

/* XView headers */
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/notice.h>
#include <xview/cursor.h>

/* Header files associated with other compilation units */
#include "projectors.h"
#include "fviewio.h"
#include "fview.h"
#include "interface.h"
#include "callbacks.h"
#include "entropy.h"

/* Function prototypes */
extern void fclose(FILE *fd);
extern void pclose(FILE *fd);

/* Variable declarations */
int dimension = 0;                /* number of dimensions in feature vectros */
Display *display;
int screen_num;
Window win;
unsigned long valuemask; 
GC gc;
unsigned int width, height;                         /* window size in pixels */
XFontStruct *font_info;
char *progname;                          /* name this program was invoked by */
int iterations = ITERATIONS;
int samples = 0;
double angle = ANGLE;
double *w1, *w2, *w3;                            /* projection basis vectors */
double *stw1, *stw2, *stw3;               /* stored projection basis vectors */
SpeechRecord *spdata[MAX_SAMPLES];
int size = 0;
int up_down = 0;
int left_right = 0;

/* two axes around which to rotate during animation sequences */
int rotation_dim1 = ROTATION_DIM1;
int rotation_dim2 = ROTATION_DIM2;

DisplayMode display_mode;
WorkProc    ExecWorkProc = NULL;
int utterance_index = 0;
int highlighted = 0;                      /* no points highlighted initially */
int max_vertices = 0;  /* maximum number of vertices that can be highlighted */
int auto_scale = 1;            /* automatic trajectory scaling to fit window */
char *list_fname;                                  /* current list file name */
char *list_fpath;                                  /* current list path name */
int list_entries = 0;                     /* number of entries in list_names */
char* list_names[MAX_LIST_LENGTH];             /* names taken from list file */
int compress_raw_mode = 1;    /* by default display samples as they are read */
int compress_scale_mode = 0;  /* scales all traj by their greatest magnitude */
double compress_alpha = 1.0;                    /* compression control value */
int display_label_mode = 1;          /* controls whether traces are labelled */
int display_origin_mode = 0;   /* display the origin within the display area */
int spectrogram_stretch_mode = 0; /* display all spectrograms to diff widths */
FilterRecord Formats;      /* Record for def format converter + root of list */
FilterRecord Frontends;           /* Record for def front-end program + root */
FilterRecord *FormatFilter = &Formats;        /* Chosen Format filter option */
FilterRecord *FrontendFilter = &Frontends;  /* Chosen Frontend filter option */
char *SearchPath[1024];          /* Path to search for driver and list files */
char *ColorString = NULL;       /* ASCII string containing RGB colors for 3D */

/*****************************************************************************/

void SignalHandler(void)
{
  /* Shutdown X system */
  XUnloadFont(display, font_info->fid);
  XFreeGC(display, gc);
  XCloseDisplay(display);

  exit(-1);
}

/*****************************************************************************/

/*
  CreateGlobalVectors: allocate global vectors which depend on feature
    vector dimension.
*/
void CreateGlobalVectors(void)
{
  w1 = (double *) malloc(dimension * sizeof(double));
  w2 = (double *) malloc(dimension * sizeof(double));
  w3 = (double *) malloc(dimension * sizeof(double));
  stw1 = (double *) malloc(dimension * sizeof(double));
  stw2 = (double *) malloc(dimension * sizeof(double));
  stw3 = (double *) malloc(dimension * sizeof(double));

  /* Check malloc() calls suceeded */
  if (w1   == NULL || w2   == NULL || w3   == NULL ||
      stw1 == NULL || stw2 == NULL || stw3 == NULL)
  {
    printf("Error: during allocation of global basis vectors\n");
    Quit(-1);
  }
}

/*****************************************************************************/

/*
  FreeGlobalVectors: free global vectors which depend on feature
    vector dimension.  Set all pointers to NULL as an error check.
*/
void FreeGlobalVectors(void)
{
  free(w1);
  free(w2);
  free(w3);
  w1 = w2 = w3 = NULL;

  free(stw1);
  free(stw2);
  free(stw3);
  stw1 = stw2 = stw3 = NULL;
}

/*****************************************************************************/

/*
  StringDup: Duplicate the source string into a buffer which is malloced to
    exactly the correct size.
  */
char *StringDup(char *source)
{
  char *cp;
  cp = (char *) malloc(strlen(source) + 1);
  if (cp == NULL)
  {
    printf("Error: during StringDup of '%s'\n", source);
    Quit(-1);
  }
  strcpy(cp, source);
  return cp;
}

/*****************************************************************************/

/*
  CreateDefaultConverters: Initialize fields in the default converters
    records.
*/
void CreateDefaultConverters(FilterRecord *dfmt, FilterRecord *dfrt)
{
  /* Setup the default format converter */
  dfmt->name = StringDup(DEF_FORMAT_FILTER);
  dfmt->program = "(Internal)";
  dfmt->path = NULL;
  dfmt->format = 'F';
  dfmt->def_list = StringDup(DEFAULT_LIST_NAME);
  dfmt->index = 0;
  dfmt->options = NULL;
  dfmt->next = NULL;

  /*
    Setup the NULL frontend.  Note: this should be the *only* frontend record
    accpeting 'F' format features
  */
  dfrt->name = StringDup(DEF_FRONTEND_FILTER);
  dfrt->program = "(Internal)";
  dfrt->format = 'F';    
  dfrt->def_list = NULL;
  dfrt->index = 0;
  dfrt->options = NULL;
  dfrt->next = NULL;
}

/*****************************************************************************/

/*
  OpenConfigFile: read in a filter program list from a configuration file.
*/
FilterRecord* OpenConfigFile(char *fname, int fixed)
{
  int i, fields, rtn, length;
  char *field[1024], fmt;
  FILE *fd;
  char buf[1024];
  FilterRecord *root = NULL;
  FilterRecord *fptr;
  OptionRecord *optr;

  assert(fixed == 3 || fixed == 4);   /* check valid fixed field number used */
  
  /* Try all directories along the search path when opening a config file */
  for (i = 0; SearchPath[i] != NULL; i++)
  {
    (void) sprintf(buf, "%s/%s", SearchPath[i], fname);
    if ((fd = fopen(buf, "r")) != NULL)
      break;
  }

  /* Check that the open suceeded somewhere along the path */
  if (fd == NULL)
  {
    printf("Error: Cannot open config file %s\n", buf);
    Quit(-1);
  }

  while (1)
  {
    rtn = fscanf(fd, "%[^\n]", buf);
    if (rtn == EOF || rtn == 0)
      break;

    /* records starting with the '#' character are comments, so ignore them */
    if (buf[0] == '#')
      continue;
    
    fields = 0;
    length = strlen(buf);

    /* parse record and split into ASCIIZ fields */
    field[0] = buf;
    for (i = 0; i < length; i++)
      if (buf[i] == ':')
      {
        buf[i] = '\0';
        fields++;
        field[fields] = &buf[i+1];
      }
    assert(((fields - fixed) % 3) == 0);

    /* Allocate a FilterRecord */
    if (root == NULL)
    {
      fptr = (FilterRecord *) malloc(sizeof(FilterRecord));
      assert(fptr != NULL);
      root = fptr;
    }
    else
    {
      fptr->next = (FilterRecord *) malloc(sizeof(FilterRecord));
      assert(fptr->next != NULL);
      fptr = fptr->next;
    }
    
    /* set up compulsary fields */
    fptr->name = StringDup(field[0]);
    fptr->program = StringDup(field[1]);
    fptr->path = NULL;

    /* store format field converting to upper case if necessary */
    fmt = field[2][0];
    if (fmt >= 'A' && fmt <= 'Z')
      fptr->format = fmt;
    else
      if (fmt >= 'a' && fmt <= 'z')
        fptr->format = fmt - 'a' + 'A';
      else
      {
        printf("Error: format field '%c' incorrect\n", fmt);
        Quit(-1);
      }
    assert(fptr->format >= 'A' && fptr->format <= 'Z');

    fptr->def_list = (fixed == 4) ? StringDup(field[3]) : NULL;
    fptr->options = NULL;
    fptr->next = NULL;

    for  (i = fixed; i < fields; i += 3)
    {
      /* Allocate an OptionRecord */
      if (fptr->options == NULL)
      {
        fptr->options = (OptionRecord *) malloc(sizeof(OptionRecord));
        assert(fptr->options != NULL);
        optr = fptr->options;
      }
      else
      {
        optr->next = (OptionRecord *) malloc(sizeof(OptionRecord));
        assert(optr->next != NULL);
        optr = optr->next;
      }

      /* initialise the name, command and value fields from the record */
      optr->name = StringDup(field[i]);
      optr->command = StringDup(field[i+1]);
      optr->value = StringDup(field[i+2]);
    }
    (void) fgetc(fd);  /* skip over the \n character terminating each record */
  }  
  fclose(fd);

  return root;
}

/*****************************************************************************/

/*
  CheckConverterList: Checks that all converter programs cited in a
    FilterRecord linked-list can be accessed in the directory system by
    searching directories along the SearchPath.  Initialises the path field
    withi the FilterRecord when the filter is located otherwise produces a
    fatal error if any driver cannot be located.
*/
void CheckConverterList(FilterRecord *fptr)
{
  int index;
  FILE *fd;
  char *cp;
  char prog[1024];
  char buf[1024];
  
  while (fptr != NULL)
  {
    /* Separate off trailing parameters before checking */
    strcpy(prog, fptr->program);
    cp = prog;
    while (*cp == ' ')    /* Skip over any spaces proceeding the filter name */
      cp++;
    cp = strchr(cp, ' ');    /* locate the first space after the filter name */
    if (cp != NULL)
      *cp = '\0';                 /* terminate the string at the first space */

    index = 0;
    while (SearchPath[index] != NULL)
    {
      (void) sprintf(buf, "%s/%s", SearchPath[index], prog);
      
      fd = fopen(buf, "r");          /* attempt to access the filter program */
      if (fd != NULL)
      {
        fclose(fd);
        fptr->path = StringDup(SearchPath[index]);  /* save the correct path */
        break;                                            /* cease searching */
      }
      index++;
    }

    /* Check that the filter was located at some location  along the path */
    if (fptr->path == NULL)
    {
      printf("Error: Cannot locate filter '%s' under the name `%s`\n",
             fptr->name, fptr->program );
     Quit(-1);
    }
    fptr = fptr->next;
  }
}

/*****************************************************************************/

/*
  FreeConfigRecords: Completely deallocate a Config Record list structure.
*/
void FreeConfigRecords(FilterRecord *root)
{
  FilterRecord *fptr;
  OptionRecord *oroot, *optr;
  
  while (root != NULL)
  {
    fptr = root;
    root = root->next;

    free(fptr->name);
    free(fptr->program);
    free(fptr->path);

    oroot = fptr->options;
    while (oroot != NULL)
    {
      optr = oroot;
      oroot = oroot->next;
      
      free(optr->name);
      free(optr->command);
      free(optr->value);
    }
  }
}

/*****************************************************************************/

/*
  CreateFilterCommand: Takes FormatFilter and FrontendFilter linked structures
    and constructs a single (piped) command string which can be passed to a
    shell using popen().  Note: written inefficiently but should be robust to
    non ANSI sprintf() implementations.
*/
void CreateFilterCommand(char *cmd, char *fname, FilterRecord *converter,
                         FilterRecord *front_end)
{
  char *cp;
  OptionRecord *optr;

  cp = cmd;
  (void) sprintf(cp, "%s/%s %s", converter->path, converter->program, fname);
  cp = cmd + strlen(cmd);
  
  for (optr = converter->options; optr != NULL; optr = optr->next)
  {
    *cp++ = ' ';     /* ensure options are separated by at least one space */
    (void) sprintf(cp, optr->command, optr->value);
    cp = cmd + strlen(cmd);    
  }
  
  if (front_end != &Frontends)
  {
    (void) sprintf(cp, " | %s/%s", front_end->path, front_end->program);
    cp = cmd + strlen(cmd);
    
    for (optr = front_end->options; optr != NULL; optr = optr->next)
    {
      *cp++ = ' ';     /* ensure options are separated by at least one space */
      (void) sprintf(cp, optr->command, optr->value);
      cp = cmd + strlen(cmd);
    }
  }
}

/*****************************************************************************/

/*
  OpenListFile: read from a list file (which may be compressed) store the
    entries and update the SampleList widget.
*/
void OpenListFile(void)
{
  /*
    Filenames are left truncated if their length exceeds this number of
    characters when they are inserted into the SampleFileList widget.
  */
  const int StringWidth = 28;
  
  int i, rtn, len, start;
  FILE *fd;
  char buf[1024];
  int length;
  void (*fpclose)(FILE *fd) = NULL;         /* pointer to the close function */
  
  /* check for .Z extension */
  length = strlen(list_fname);
  if (list_fname[length-2] == '.' && list_fname[length-1] == 'Z')
  {
    (void) sprintf(buf, "/usr/ucb/zcat %s/%s", list_fpath, list_fname);
    fd = popen(buf, "r");
    fpclose = pclose;                         /* choose pclose() to close fd */
  }
  else
  {
    (void) sprintf(buf, "%s/%s", list_fpath, list_fname);
    fd = fopen(buf, "r");
    fpclose = fclose;                         /* choose fclose() to close fd */
  }
  assert(fd != NULL);       /* listfile access should have been prev checked */

  /* scan list file adding filename entries to list_names[] */
  list_entries = 0;
  while (1)
  {
    rtn = fscanf(fd, "%[^\n]", buf);   /* read until no more lines available */
    if (rtn == EOF || rtn == 0)
      break;

    list_names[list_entries] = StringDup(buf);

    if (++list_entries == MAX_LIST_LENGTH)
    {
      printf("open_list_file: list file too large, %d entries read\n",
             MAX_LIST_LENGTH);
      fclose(fd);
      Quit(-1);
    }
    (void) fgetc(fd);       /* Skip the \n character at the end of each line */
  }

  list_names[list_entries] = NULL;                         /* terminate list */
  assert(fpclose != NULL);
  fpclose(fd);                       /* call the appropriate closing routine */

  /* Hide the sample list before adding the entries to reduce screen flicker */
  xv_set(SampleFileList, XV_SHOW, FALSE, NULL);

  /* Add entries from list_names[] to the SampleList scrolling widget */
  for (i = 0; i < list_entries; i++)
  {
    strcpy(buf, list_names[i]);

    /*
      if filename is longer than StringWidth then just display last
      StringWidth characters
    */
    len = strlen(buf);
    start = (len > StringWidth) ? len - StringWidth : 0;

    xv_set(SampleFileList,
           PANEL_LIST_INSERT, i,
           PANEL_LIST_STRING, i, &buf[start],
           NULL);
  }

  /* Select first entry by default and redisplay the list */
  xv_set(SampleFileList,
         XV_SHOW, TRUE,
         NULL);

  /* Display list name and number of samples read */
  xv_set(SampleListnameText,
         PANEL_VALUE, list_fname,
         NULL);

  /* Display characteristics of the list file */
  (void) sprintf(buf, "%d", list_entries);
  xv_set(SampleNumSampsText, PANEL_VALUE, buf, NULL);  
}

/*****************************************************************************/

/*
  CreateSpeechRecord: allocate fields within a SpeechRecord depending on
    the 'npoints' and 'dimension' values set.  Check all malloc() calls
    to ensure sucess.  Terminate on failure.
*/

void CreateSpeechRecord(SpeechRecord *sd, int npoints)
{
  int i;
  int fail;
  
  sd->npoints = npoints;
  sd->order = (int *) malloc(npoints * sizeof(int));
  sd->dist = (double *) malloc(npoints * sizeof(double));
  if (sd->order == NULL || sd->dist == NULL)
  {
    printf("Error: during allocation of SpeechRecord (order/dist)\n");
    Quit(-1);
  }

  sd->project1 = (double *) malloc(npoints * sizeof(double));
  sd->project2 = (double *) malloc(npoints * sizeof(double));
  sd->project3 = (double *) malloc(npoints * sizeof(double));
  if (sd->project1 == NULL || sd->project2 == NULL || sd->project3 == NULL)
  {
    printf("Error: during allocation of SpeechRecord (project1/2/3)\n");
    Quit(-1);
  }

  sd->data  = (double **) malloc(npoints * sizeof(double *));
  fail = (sd->data == NULL);
  for (i = 0; i < npoints; i++)
  {
    sd->data[i] = (double *) malloc(dimension * sizeof(double));
    fail = fail || (sd->data == NULL);
  }
  if (fail)
  {
    printf("Error: during allocation of SpeechRecord (data)\n");
    Quit(-1);
  }

  sd->point = (double **) malloc(dimension * sizeof(double *));
  fail = (sd->point == NULL);
  for (i = 0; i < dimension; i++)
  {
    sd->point[i] = (double *) malloc(npoints * sizeof(double));
    fail = fail || (sd->point == NULL);
  }
  if (fail)
  {
    printf("Error: during allocation of SpeechRecord (point)\n");
    Quit(-1);
  }
}

/*****************************************************************************/

/*
  FreeSpeechRecord: deallocate malloced fields within a SpeechRecord.
    Note: all static fields are left intact.
*/

void FreeSpeechRecord(SpeechRecord *sd)
{
  int i;
  
  for (i = 0; i < dimension; i++)
    free(sd->point[i]);
  free(sd->point);
  sd->point = NULL;

  for (i = 0; i < sd->npoints; i++)
    free(sd->data[i]);
  free(sd->data);
  sd->data = NULL;

  free(sd->file_name);
  sd->file_name = NULL;
  free(sd->project3);
  free(sd->project2);
  free(sd->project1);
  sd->project1 = sd->project2 = sd->project3 = NULL;
  free(sd->dist);
  sd->dist = NULL;
  free(sd->order);
  sd->order = NULL;
}

/*****************************************************************************/

/*
  CheckMagicNumber: Check that an eight byte magic number is correct.
*/
int CheckMagicNumber(char *magic, char *reference)
{
  int i;

  /* Checks eight bytes of magic number */
  for (i = 0; i < 8; i++)
    if (magic[i] != reference[i])
      return 1;
  return 0;
}

/*****************************************************************************/

/*
  OpenDataFile: Read the specified file into the specifies SpeechRecord
    structure arranging dynamic allocation of fields using
    CreateSpeechRecord().  On success returns 0.  Any non-zero return value
    indicates a load error.
*/
int OpenDataFile(char* filename, SpeechRecord *sd, int verbose)
{
  int i;
  FILE *fd;
  FviewFeatureHeader ffh;                           /* Feature header record */
  char buf[1024];
  char *extra;            /* buffer in which to hold addition textual fields */

  /*
    Open data file directly for reading or create a format converion/
    front-end pipe to generate features.
  */
  if (FormatFilter != &Formats)
  {
    CreateFilterCommand(buf, filename, FormatFilter, FrontendFilter);

    fd = popen(buf, "r");
    if (fd == NULL)
    {
      printf("Load Error: Cannot create input filter for %s\n", filename);
      return 1;
    }
  }
  else
  {
    fd = fopen(filename, "r");
    if (fd == NULL)
    {
      printf("Load Error: Cannot open feature datafile %s\n", filename);
      return 1;
    }
  }

  /* read header record and check the magic number etc. */
  if (fread(&ffh, sizeof(FviewFeatureHeader), 1, fd) != 1)
  {
    printf("Error: failed to read header record for %s\n", filename);
    if (FormatFilter != &Formats)
      pclose(fd);
    else
      fclose(fd);
    return 1;
  }
  if (CheckMagicNumber(ffh.magic, FVIEW_FEATURE_MAGIC))
  {
    printf("Load Error: File format in %s is incorrect (magic number wrong)\n",
           filename);
    if (FormatFilter != &Formats)
      pclose(fd);
    else
      fclose(fd);
    return 1;
  }

  /* Extract the feature dimension for the file if not already set */
  if (dimension == 0)
  {
    if (ffh.vector_dimension >= MIN_DIMENSION &&
        ffh.vector_dimension <= MAX_DIMENSION)
    {
      dimension = ffh.vector_dimension;
      (void) sprintf(buf, "%d", dimension);
      xv_set(SampleDimensionText, PANEL_VALUE, buf, NULL);
    }
    else
    {
      printf("Load Error: Cannot cope with feature dimension = %d\n",
             ffh.vector_dimension);
      if (FormatFilter != &Formats)
        pclose(fd);
      else
        fclose(fd);
      return 1;
    }
  }
  else
  {
    if (ffh.vector_dimension != dimension) {
      printf("Load Error: Features are wrong dimension (%d instead of %d)\n",
             ffh.vector_dimension, dimension);
      if (FormatFilter != &Formats)
        pclose(fd);
      else
        fclose(fd);
      return 1;
    }
  }

  /* read additional textual information */
  if (ffh.info_size > 0)
  {
    extra = (char *) malloc(ffh.info_size);
    if (fread(extra, sizeof(char), ffh.info_size, fd) != ffh.info_size)
    {
      printf("Load Error: whilst loading textual header from %s\n", filename);
      if (FormatFilter != &Formats)
        pclose(fd);
      else
        fclose(fd);
      return 1;
    }
  }
  else
    extra = NULL;
  
  /* allocate space within record sd */
  CreateSpeechRecord(sd, ffh.number_observations);
  
  /* read in the feature vectors */
  for (i = 0; i < sd->npoints; i++)
  {
    if (fread(sd->data[i], sizeof(double), dimension, fd) != dimension)
    {
      printf("Load Error: data read failed at observation %d in %s\n",
             i, filename);
      FreeSpeechRecord(sd);
      if (FormatFilter != &Formats)
        pclose(fd);
      else
        fclose(fd);
      return 1;
    }
  }

  /* save the file name in speech record */
  sd->file_name = StringDup(filename);
  sd->file_name_length = strlen(filename);

  sd->nverts = 0;       /* signals that the utterance has not been predicted */
 
  if (FormatFilter != &Formats)
    pclose(fd);
  else
    fclose(fd);

  /* Display textual information if the "info" check flag is active */
  if (verbose)
    printf("Header Info (%d bytes):\n%s\n", ffh.info_size, extra);  
  if (extra != NULL)
    free(extra);
  
  return 0;                        /* signal that utterance loaded correctly */
}

/*****************************************************************************/

/*
  Compression: pre-processes the data prior to projection and display scaling.
    Three modes of operation are supported.  1.  compress_raw_mode displays
    the data samples as loaded without pre-processing.  2.  alpha allows a
    variable degree of compression to be applied to the axes (using a power
    law).  Whilst 3. compress_scale_mode divides all the feature vectors
    (after compression) by the greatest magnitude.
*/
void Compression(SpeechRecord *sd, double alpha)
{
  int t, i;
  double s, magnitude, max;
  
  if (compress_raw_mode)
  {
    for (t = 0; t < sd->npoints; t++)
      for (i = 0; i < dimension; i++)
        sd->point[i][t] = sd->data[t][i];
  }
  else
  {
    for (t = 0; t < sd->npoints; t++)
      for (i = 0; i < dimension; i++)
        sd->point[i][t] = exp(sd->data[t][i] * alpha);
  }

  if (compress_scale_mode)
  {
    /* find the data vector with greatest power */
    max = 0.0;
    for (t = 0; t < sd->npoints; t++)
    {
      magnitude = 0.0;
      for (i = 0; i < dimension; i++)
      {
        s = sd->point[i][t];
        magnitude += (s * s);
      }
      if (magnitude > max)
        max = magnitude;
    }

    /* normalise all data vectors by the greatest power */
    max = sqrt(max);
    for (t = 0; t < sd->npoints; t++)
      for (i = 0; i < dimension; i++)
        sd->point[i][t] /= max;
  }
}

/*****************************************************************************/

/*
  Randomize: generates a set of new (and ortogonal) random basis functions.
*/
void Randomize(void)
{
  double product1, product2;
  int i;
  double *u;

  u = (double *) malloc(dimension * sizeof(double));
  
  /* Choose first normal vector */
  product1 = 0.0;
  for (i = 0; i < dimension; i++) {
    w1[i] = (2.0 * drandom()) - 1.0;
    product1 += w1[i] * w1[i];
  }
  for (i = 0; i < dimension; i++)
    w1[i] /= sqrt(product1);

  /* Choose a second vector - not neccessarily normalised or orthogonal */
  for (i = 0; i < dimension; i++)  
    w2[i] = (2.0 * drandom()) - 1.0;

  /* Make second vector normalised and orthogonal to first vector */
  product1 = 0.0;
  for (i = 0; i < dimension; i++) 
    product1 += w1[i] * w2[i];
  for (i = 0; i < dimension; i++)
    u[i] = w2[i] - (product1 * w1[i]);
  product1 = 0.0;
  for (i = 0; i < dimension; i++) 
    product1 += u[i] * u[i];
  for (i = 0; i < dimension; i++)
    w2[i] = u[i] / sqrt(product1);      

  /* Choose a third vector - not neccessarily normalised or orthogonal */
  for (i = 0; i < dimension; i++)  
    w3[i] = (2.0 * drandom()) - 1.0;

  /* Make second vector normalised and orthogonal to first and second */
  product1 = product2 = 0.0;
  for (i = 0; i < dimension; i++)
  {
    product1 += w1[i] * w3[i];
    product2 += w2[i] * w3[i];
  }
  for (i = 0; i < dimension; i++)
    u[i] = w3[i] - (product1 * w1[i]) - (product2 * w2[i]);
  product1 = 0.0;
  for (i = 0; i < dimension; i++) 
    product1 += u[i] * u[i];
  for (i = 0; i < dimension; i++)
    w3[i] = u[i] / sqrt(product1);

  free(u);
}

/*****************************************************************************/

/*
  DiagonalBasis: Creates a basis vector diagonally orientated within the
    specified sub-space.
*/
void DiagonalBasis(Xv_opaque SpecifySelector, double *w)
{
  int i;
  unsigned int bits;
  double product = 0.0;
  
  bits = (unsigned int) xv_get(SpecifySelector, PANEL_VALUE);

  for (i = 0; i < dimension; i++) {
    if (bits & 0x0001)
    {
      w[i] = 1.0; 
      product += 1.0;
    }
    else
      w[i] = 0.0;
    bits >>= 1;
  }
  for (i = 0; i < dimension; i++) {
    w[i] /= sqrt(product);
  }
}

/*****************************************************************************/

/*
  RandomBasis: Creates a basis vector randomly orientated within the
    specified sub-space.
*/
void RandomBasis(Xv_opaque SpecifySelector, double *w)
{
  int i;
  unsigned int bits;
  double product = 0.0;

  bits = (unsigned int) xv_get(SpecifySelector, PANEL_VALUE);

  for (i = 0; i < dimension; i++ ) {
    w[i] = (bits & 0x0001) ? drandom() : 0.0;
    product += w[i] * w[i];
    bits >>= 1;
  }
  for (i = 0; i < dimension; i++)  {
    w[i] /= sqrt(product);
  }
}

/*****************************************************************************/

/*
  OrtogonalizeBases: Establish orthogonality between basis vectors w[123].
  Generate a pop-up window error should two of the basis vectors be
  parrallel.  Note: all vectors are assumed to be of unity magnitude as set
  by the Equalise and Randomise routines.  The first vector is left unchanged.
*/
void OrthogonalizeBases(void)
{
  const double DotLimit = 0.9999;
  
  int i, v1, v2;
  int fail = 0;
  double magnitude, product1, product2;
  double *u;
  char buf[1024];

  u = (double *) malloc(dimension * sizeof(double));
  
  /* Make second vector unity magnitude and orthogonal to first vector */
  product1 = 0.0;
  for (i = 0; i < dimension; i++) 
    product1 += w1[i] * w2[i];

  if (product1 > DotLimit)
  {
    v1 = 1;                /* signals which basis vectors are not orthogonal */
    v2 = 2;
    fail = 1;
  }
  else
  {
    /* do the orthogonalisation */
    for (i = 0; i < dimension; i++)
      u[i] = w2[i] - (product1 * w1[i]);
    
    magnitude = 0.0;
    for (i = 0; i < dimension; i++) 
      magnitude += u[i] * u[i];
    
    for (i = 0; i < dimension; i++)
      w2[i] = u[i] / sqrt(magnitude);
  }

  /* Make third vector unity magnitude and orthogonal to first and second */
  product1 = 0.0;
  product2 = 0.0;
  for (i = 0; i < dimension; i++)
  {
    product1 += w1[i] * w3[i];
    product2 += w2[i] * w3[i];
  }

  for (i = 0; i < dimension; i++)
    u[i] = w3[i];

  if (product1 > DotLimit)
  {
    v1 = 1;
    v2 = 3;
    fail = 1;
  }
  else
    for (i = 0; i < dimension; i++)
      u[i] -= (product1 * w1[i]);

  if (product2 > DotLimit)
  {
    v1 = 2;
    v2 = 3;
    fail = 1;
  }
  else
    for (i = 0; i < dimension; i++)
      u[i] -= (product2 * w2[i]);

  if (product1 <= DotLimit || product2 <= DotLimit)
  {
    magnitude = 0.0;
    for (i = 0; i < dimension; i++) 
      magnitude += u[i] * u[i];
    for (i = 0; i < dimension; i++)
      w3[i] = u[i] / sqrt(magnitude);
  }

  Projection();
  RefreshGraphics();
  free(u);

  if (fail)
  {
    (void) sprintf(buf, "Basis vectors %d and %d", v1, v2);
    (void) notice_prompt(SpecifyPanel, NULL,
                         NOTICE_MESSAGE_STRINGS,
                           "Warning:",
                           buf,
                           "are not currently orthogonal!",
                           NULL,
                         NOTICE_BUTTON_YES, "Continue",
                         NULL);
  }
}

/*****************************************************************************/

/* Creation and projection of points onto 2/3D space */
void Projection(void)
{
  int i, j, s;
  SpeechRecord *sd;

  for (s = 0; s < samples; s++)
  {
    sd = spdata[s];
    
    /* Project points down on initially defined plane */
    for (i = 0; i < sd->npoints; i++) {
      sd->project1[i] = 0.0;
      sd->project2[i] = 0.0;
      sd->project3[i] = 0.0;
      for (j = 0; j < dimension; j++) {
        sd->project1[i] += (sd->point[j][i] * w1[j]);
        sd->project2[i] += (sd->point[j][i] * w2[j]);
        sd->project3[i] += (sd->point[j][i] * w3[j]);
      }
    }
  }
} /* end of projection */

/*****************************************************************************/

/*
  GetEnvironmentVars: Get values from environment variables else establish
    default values.
*/
void GetEnvironmentVars(void)
{
  int index;
  char *cp;
  char *path;

  /* Get SearchPath from an environment variable else use default */
  cp = getenv(ENVIRONMENT_PATH_NAME);
  if (cp)
    path = StringDup(cp);
  else
    path = StringDup(DEFAULT_PATH_NAME);

  /* Split path into separate strings by detecting the ':' delimiter */
  for (cp = path, index = 0; *cp != '\0'; cp++)
  {
    if (*cp == ':')
    {
      *cp++ = '\0';            /* terminate the (index-1)th directory string */
      while (*cp == ':')
      {
        cp++;
        if(*cp == '\0')
          break;
      }
      SearchPath[index++] = cp;                      /* set the next pointer */
    }
    else
      if (cp == path)
        SearchPath[index++] = cp;                     /* set initial pointer */

    assert(index < 1024);    /* check that the index doesn't go out of range */
  }
  SearchPath[index] = NULL;

  /* Get 3D color information from an environment variable if set */
  cp = getenv(ENVIRONMENT_COLOR_NAME);
  if (cp)
    ColorString = StringDup(cp);
  else
    ColorString = NULL;
} 

/*****************************************************************************/

/*
  Setup: called once prior to xv_main_loop to initialize variables.
*/
void Setup(int argc, char *argv[])
{
  int i;
  char buf[1024];
  FILE *fd;
  
  /* Setup and verify filter program lists */
  CreateDefaultConverters(&Formats, &Frontends);
  Formats.next = OpenConfigFile(FORMAT_FILE_NAME, 4);      /* 4 fixed params */
  Frontends.next = OpenConfigFile(FRONTEND_FILE_NAME, 3);  /* 3 fixed params */
  CheckConverterList(Formats.next);   /* Check existance of cited converters */
  CheckConverterList(Frontends.next);
  PrepareConverterLists();    /* Setup lists of conversion/front-end filters */

  /* Set initial values for the list file name and its path */
  for (i = 1; i < argc; i++)
  {
    if (strcmp(argv[i], "-list") == 0)
      if (++i < argc)
      {
        list_fname = StringDup(argv[i]);
        break;
      }
  }
  if (i == argc)
    list_fname = StringDup(DEFAULT_LIST_NAME);

  /* Check the existance of list_fname and set the list_fpath */
  for (i = 0; SearchPath[i] != NULL; i++)
  {
    (void) sprintf(buf, "%s/%s", SearchPath[i], list_fname);
    if ((fd = fopen(buf, "r")) != NULL)
      break;
  }
  if (fd != NULL)
  {
    fclose(fd);
    list_fpath = StringDup(SearchPath[i]);
  }
  else
  {
    printf("Error: failed to open list file `%s`\n", list_fname);
    Quit(-1);
  }
}
/*****************************************************************************/

/*
  Initialize: called with each new list file to reset variables etc.
*/
void Initialize(char *list_name, char *list_path)
{
  int i, flags;
  SpeechRecord *sd;

  SetCursors(BusyCursor);

  if (list_name != NULL)
  {
    if (list_fname != NULL)
      free(list_fname);
    list_fname = StringDup(list_name);
  }
  if (list_path != NULL)
  {
    if (list_fpath != NULL)
      free(list_fpath);
    list_fpath = StringDup(list_path);
  }
  OpenListFile();

  /* open_list_file selects the first utterance within its list */
  if ((sd = (SpeechRecord *) malloc(sizeof(SpeechRecord))) == NULL)
  {
    printf("Error: malloc of SpeechRecord failed\n");
    Quit(-1);
  }

  if (dimension != 0)            /* check if a list file is currently loaded */
  {
    FreeGlobalVectors();      /* dispose of vectors before dimension changes */
    dimension = 0;
  }

  if (OpenDataFile(list_names[0], sd, 0))
  {
    free(sd);
    samples = 0;
  }
  else
  {
    samples = 1;
    /* mark first entry as selected */
    xv_set(SampleFileList, PANEL_LIST_SELECT, 0, TRUE, NULL);
  }

  if (dimension > 0)
  {
    CreateGlobalVectors(); /* allocate global vectors depending on dimension */
    ChangeBandWidgets();            /* modify widgets depending on dimension */
  }
  if (samples > 0)
  {
    spdata[0] = sd;
    Compression(spdata[0], compress_alpha);

    /* Set additional fields in the SpeechRecord */
    spdata[0]->color = AllocateColorEntry();
    spdata[0]->row = 0;
  }

  /* Setup DisplayCheckBox using global mode flags */
  flags = 0x0000;
  if (compress_raw_mode)
    flags |= 0x0001;
  if (compress_scale_mode)
    flags |= 0x0002;
  if (display_label_mode)
    flags |= 0x0004;
  if (display_origin_mode)
    flags |= 0x0008;
  xv_set(DisplayCheckBox, PANEL_VALUE, flags, NULL);

  /* If in 3D mode then return to the first utterance */
  if (display_mode == three_dims)
    utterance_index = 0;

  SetCursors(BasicCursor);
  if (samples > 0)
    DisplayRandomizeCallback();
  else
    RefreshGraphics();
}

/*****************************************************************************/

/*
  RefreshGraphics:  Screen Refresh Callback activated whenever a Canvas
    expose or resize is occurs.  If (samples == 0) then hide the display
    window and do not attempt to redraw.
*/
void RefreshGraphics(void)
{
  int status;

  /* Expose/Hide Drawing Window when necessary */
  status = (int) xv_get(DisplayFrame, XV_SHOW);
  if (status == FALSE && samples > 0)
    xv_set(DisplayFrame, XV_SHOW, TRUE, NULL);
  if (status == TRUE && samples == 0)
  {
    xv_set(DisplayFrame, XV_SHOW, FALSE, NULL);
    XFlush(display);

    (void)
      notice_prompt(MainPanel, NULL,
                    NOTICE_MESSAGE_STRINGS,
                    "Error:",
                    "Nothing to display",
                    "(reselect files/front-ends)",
                    NULL,
                    NOTICE_BUTTON_YES, "Continue",
                    NULL);

    /* Do not attempt to redraw */
    return;
  }
  
  /* obtain Canvas dimensions */
  width = (int) xv_get(DisplayCanvas, XV_WIDTH);
  height = (int) xv_get(DisplayCanvas, XV_HEIGHT);

  /* choose correct display driver function */
  switch (display_mode)
  {
  case two_dims:
    draw_2d_graphics(width, height, 1);
    break;
  case three_dims:
    draw_3d_graphics(width, height, 1);
    break;
  case spectrogram:
    draw_spectrogram(width, height, 1);
    break;
  }
}

/*****************************************************************************/

/*
  Quit: Clear up the Xwindow system and return from xv_main_loop in main().
    Note: code == 0 signifies that a totally legitimate exit has occured.
*/
void Quit(int code)
{
  /* Shutdown X system */
  XUnloadFont(display, font_info->fid);
  XFreeGC(display, gc);
  XCloseDisplay(display);

  exit(code);
}

/*****************************************************************************/

/*
  main: activate xv widget set and lose control to xv event handler.  All
    other processing is activated by callbacks.
*/
int main(int argc, char** argv)
{
  progname = argv[0];
  srandom((int) getpid());

  signal(SIGINT, SignalHandler);

  GetEnvironmentVars();
  CreateControlWindow(argc, argv);          /* Create XView widget hierarchy */

  Setup(argc, argv);                       /* setup variables once initially */
  Initialize(NULL, NULL);                /* setup variables prior to display */
  
  /*
    This call to xv_main_loop() will return when notify_stop() is used within
    a callback routine at which point it checks to see if a WorkProc has
    been registered and executes it.  Upon completion control returns to
    xv_main_loop().
  */
  while (1)
  {
    xv_main_loop(MainFrame);

    /* Executes the work procedure pointed to by ExecWorkProc */
    if (ExecWorkProc != NULL)
    {
      SetCursors(BusyCursor);
      (*ExecWorkProc)();
      SetCursors(BasicCursor);
    }
  }
} /* end main */ 

/*****************************************************************************/

/* end of fview.c */

