/*
  title:     entropy.c
  purpose:   Locate observations within sequence associated with high
             predictive entropy.

  author:    Gareth Lee.
  date:      04-07-93
  modified:  11-07-93

  changes:
  11-07-93: consider_direction() added to weight prediction error vectors
    using information about the change of direction occuring at the vertex.
    No change of direction corresponds to a weighting of zero whilst a change
    of 180 degrees results in unity weigthing.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "entropy.h"

int     entropy_verbose = 0;                          /* enables diagnostics */
double* prediction_error;                                 /* distance vector */

/*****************************************************************************/

/*
  linear_predict: generate prediction error vectors for (2 <= t < dims) using
    linear prediction.
*/
void linear_predict(int frames, int dims, double** obs, double** error)
{
  int t, i;

  for (t = 2; t < frames; t++)
    for (i = 0; i < dims; i++)
      error[t][i] = obs[t][i] - (2.0 * obs[t-1][i]) + obs[t-2][i];
}

/*****************************************************************************/

/*
  linear_diff_predict: generate prediction error vectors for (2 <= t < dims)
    using linear prediction of observations and first order differences.
    Note: error needs 2*dims elements in each frame.
*/
void linear_diff_predict(int frames, int dims, double** obs, double** error)
{
  int t, i;

  for (t = 3; t < frames; t++)
    for (i = 0; i < dims; i++) {
      /* prediction error */
      error[t][i] = obs[t][i] - (2.0 * obs[t-1][i]) + obs[t-2][i];

      /* prediction error of first difference */
      error[t][i+dims] = obs[t][i] - (3.0 * obs[t-1][i]) + (3.0 * obs[t-2][i])
                                   - obs[t-3][i];
    }
}

/*****************************************************************************/

/*
  linear_diff2_predict: generate prediction error vectors for (2 <= t < dims)
    using linear prediction of observations and first/second order differences.
    Note: error needs 2*dims elements in each frame.
*/
void linear_diff2_predict(int frames, int dims, double** obs, double** error)
{
  int t, i;
  int dims2;

  dims2 = dims * 2;
  for (t = 4; t < frames; t++)
    for (i = 0; i < dims; i++) {
      /* prediction error */
      error[t][i] = obs[t][i] - (2.0 * obs[t-1][i]) + obs[t-2][i];

      /* prediction error of first difference */
      error[t][i+dims] = obs[t][i] - (3.0 * obs[t-1][i]) + (3.0 * obs[t-2][i])
        - obs[t-3][i];

      /* prediction error of second difference */
      error[t][i+dims2] = obs[t][i] - (4.0 * obs[t-1][i])
                     + (6.0 * obs[t-2][i]) - (4.0 * obs[t-3][i]) + obs[t-4][i];
    }
}

/*****************************************************************************/

/*
  consider_directions: Take changes of direction between succesive vectors.
    If two vectors are in the same direction then weight the error 0.0.
    If they are 360 apart then weight the error as 1.0.
*/
void consider_directions(int frames, int start,
                         int dims, double** obs,
                         int dim2, double** error)
{
  int t, i;
  double dot, mag1, mag2, d1, d2;
  double scale;

  /* Note: start should be 1..frames-2 */
  
  /* weight all errors from t=2..(N-2) */
  for (t = start; t < (frames-1); t++)
  {
    /* find dot product and magnitudes of differential vectors */
    dot = 0.0;
    mag1 = mag2 = 0.0;
    for (i = 0; i < dims; i++)
    {
      d1 = obs[t+1][i] - obs[t][i];              /* next differential vector */
      d2 = obs[t][i] - obs[t-1][i];          /* previous differential vector */
      dot += d1 * d2;
      mag1 += d1 * d1;
      mag2 += d2 * d2;
    }
    dot /= sqrt(mag1) * sqrt(mag2);                      /* dot = cos(theta) */

    /* scale error vector components as a function of the dot value */
    scale = 0.5 * (1.0 - dot);
    for (i = 0; i < dims; i++)
      error[t][i] *= scale;
  }

  /* assume scaling for the final frame is 0.5 */
  scale = 0.5;
  for (i = 0; i < dim2; i++)
    error[frames-1][i] *= scale;
  
}

/*****************************************************************************/

/*
  consider_velcotiy: normalises the error vector by the trajectory veclocity
    measured at the time.
*/
void consider_velocity(int frames, int start,
                       int dims, double** obs,
                       int dim2, double** error)
{
  int t, i;
  double dist, velocity;
  
  /* Note: start should be 1..frames-1 */

  /* weight all errors from t=2..(N-1) */
  for (t = start; t < frames; t++)
  {
    /*
      find the trajectory velocity at time t,
      ie. the Euclidean distance between obs[t] and obs[t-1]
    */
    velocity = 0.0;
    for (i = 0; i < dims; i++)
    {
      dist = obs[t][i] - obs[t-1][i];
      velocity += dist * dist;
    }
    velocity = sqrt(velocity);

    /* normalise the error by the velocity */
    for (i = 0; i < dim2; i++)
      error[t][i] /= velocity;
  }
}

/*****************************************************************************/

/*
  mean_covar: find mean and covariance of error vectors.
*/
void mean_covar(int frames, int dims, double** error,
                double* mean, double** covar)
{
  int t, i, j;
  double err;

  /* zero mean and covariance accumulators */
  for (i = 0; i < dims; i++)
  {
    mean[i] = 0.0;
    for (j = 0; j <= i; j++)
      covar[i][j] = 0.0;
  }

  /* accumulate mean and covariance */
  for (t = 0; t < frames; t++)
    for (i = 0; i < dims; i++)
    {
      err = error[t][i];
      mean[i] += err;
      for (j = 0; j <= i; j++)
        covar[i][j] += err * error[t][j];
    }

  /* normalize mean */
  if (entropy_verbose)
    printf("mean=");
  for (i = 0; i < dims; i++)
  {
    mean[i] /= frames;
    if (entropy_verbose)
      printf("%f ", mean[i]);
  }

  /* subtract mean out of unbiased covariance estimate */
  if (entropy_verbose)
    printf("\ncovar=\n");
  for (i = 0; i < dims; i++)
  {
    for (j = 0; j < i; j++)
    {
      covar[i][j] = covar[j][i] = (covar[i][j] - (frames * mean[i] * mean[j]))
                                                                / (frames - 1);

      if (entropy_verbose)
        printf("%f ", covar[i][j]);
    }
    covar[i][i] = (covar[i][i] - (frames * mean[i] * mean[i])) / (frames - 1);
    if (entropy_verbose)
      printf("%f\n", covar[i][i]);
  }
}

/*****************************************************************************/

/*
  invert: inverts matrix m, leaving result in matrix I.  Note: This routine
    leaves M corrupted.
*/
void invert(int dims, double** M, double** I)
{
  int    a, b, Row;
  double num, denom;
  double* swap;

  for(Row = 0; Row < dims; Row++)
    for(a = 0; a < dims; a++)
      I[Row][a] = (Row == a) ? 1.0 : 0.0;

  for(Row = 0; Row < dims; Row++)
  {
    if(M[Row][Row] == 0.0)
    {
      for(a = Row + 1; a < dims; a++)
        if (M[a][Row] != 0.0) break;

      if (a == dims)
      {
        printf("Matrix non-invertable, fail on row %d left\n", Row);
        exit(1);
      }

      swap = M[Row];         /* swap M[Row] with M[a] by exchanging pointers */
      M[Row] = M[a];
      M[a] = swap;
      swap = I[Row];         /* swap I[Row] with I[a] by exchanging pointers */
      I[Row] = I[a];
      I[a] = swap;
    }
    denom = M[Row][Row];

    for(a = Row + 1; a < dims; a++)
    {
      num = M[a][Row];
      for(b = 0; b < dims; b++)
      {
        M[a][b] = (M[a][b]*denom - M[Row][b]*num) / denom;
        I[a][b] = (I[a][b]*denom - I[Row][b]*num) / denom;
      }
    }
  }

  for(Row = dims - 1; Row >= 0; Row--)
  {
    for(a = Row + 1; a < dims; a++)
    {
      num = M[Row][a];
      denom = M[a][a];
      for(b = 0; b < dims; b++)
      {
        M[Row][b] = (M[Row][b]*denom - M[a][b]*num) / denom;
        I[Row][b] = (I[Row][b]*denom - I[a][b]*num) / denom;
      }
    }
  }

  for(Row = 0; Row < dims; Row++)
  {
    denom = M[Row][Row];
    for (a = 0; a < dims; a++)
      I[Row][a] /= denom;
  }
}

/*****************************************************************************/

/*
  mahalanobis: derives distance of vector from mean using the Mahalanobis
    distance metric specified by the inverse covariance matrix.
*/
double mahalanobis(int dims, double* vector, double* mean, double** icovar)
{
  int i, j;
  double dist, s;
  double* v;

  /* create zero mean vector representation */
  v = (double *) malloc(dims * sizeof(double));
  for (i = 0; i < dims; i++)
    v[i] = vector[i] - mean[i];

  /* find mahalanobis metric */
  dist = 0.0;
  for (i = 0; i < dims; i++)
  {
    s = 0.0;
    for (j = 0; j < dims; j++)
      s += v[j] * icovar[j][i];
    dist += s * v[i];
  }

  free(v);
  return dist;
}

/*****************************************************************************/

/*
  euclidean: returns the euclidean distance between the specified vector
    and mean.
*/
double euclidean(int dims, double* vector, double* mean)
{
  int i;
  double d, dist = 0.0;

  for (i = 0; i < dims; i++) {
    d = vector[i] - mean[i];
    dist += d * d;
  }      
  return sqrt(dist);
}

/*****************************************************************************/

double** make_matrix(int r, int c)
{
  int i;
  double** p;

  p = (double**) malloc(r * sizeof(double*));
  for (i = 0; i < r; i++)
    p[i] = (double*) malloc(c * sizeof(double));
  return p;
}

void free_matrix(int r, double** p)
{
  int i;

  for (i = 0; i < r; i++)
    free(p[i]);
  free(p);
}

/*****************************************************************************/

double* make_vector(int r)
{
  return (double*) malloc(r * sizeof(double));
}

/*****************************************************************************/

/*
  qsort_decision: decision function for qsorting distances into descending
    order.
*/
int qsort_decision(int *a, int *b)
{
  if (prediction_error[*a] < prediction_error[*b])
    return 1;
  else
    return -1;
}

/*****************************************************************************/

/*
  multiply: diagnostic routine for multiplying out matrices and displaying
    their product to stdout.
*/
void multiply(int dims, double **a, double **b)
{
  int i, j, k;
  double s;

  printf("product:\n");
  for (i = 0; i < dims; i++)
  {
    for (j = 0; j < dims; j++)
    {
      s = 0.0;
      for (k = 0; k < dims; k++)
        s += a[i][k] * b[k][j];
      printf("%14f ", s);
      /* note. s = res[i][j] */
    }
    printf("\n");
  }
}

/*****************************************************************************/

/*
  find_segmentation: top level routine takes a sequence of observation vectors
    and saves a sorted list of the significance of each of the observations.
*/
int find_segmentation(int frames, int dims, double** obs, int* order,
                      int direction, int velocity)
{
  int t;
  int i, j;
  
  double** pred_err;                                    /* prediction errors */
  double*  mean;                                              /* mean vector */
  double** covar;                                       /* covariance matrix */
  double** invcov;                              /* inverse covariance matrix */
  
  pred_err = make_matrix(frames, dims);
  mean = make_vector(dims);
  covar = make_matrix(dims, dims);
  invcov = make_matrix(dims, dims);
  prediction_error = make_vector(frames);

  printf("find_segmentation: %d frames: ", frames);
  if (direction)
    printf("with direction.\n");
  else
    printf("without direction.\n");
  
  linear_predict(frames, dims, obs, pred_err);
  if (velocity)
    consider_velocity(frames, 2, dims, obs, dims, pred_err);
  if (direction)
    consider_directions(frames, 2, dims, obs, dims, pred_err);
  mean_covar(frames-2, dims, pred_err+2, mean, covar);

  invert(dims, covar, invcov);

  /* ascertain mahalonobis distances for all predictions */
  for (t = 2; t < frames; t++) {
    /* prediction_error[t] = euclidean(dims, pred_err[t], mean); */
    prediction_error[t] = mahalanobis(dims, pred_err[t], mean, invcov);
  }

  /* sort distances into descending order */
  for (t = 2; t < frames; t++)
    order[t-2] = t;
  qsort(order, frames-2, sizeof(int), qsort_decision);

  if (entropy_verbose)
    for (i = 0; i < frames-2; i++)
      printf("order=%d, M=%f, E=%f\n", order[i],
             mahalanobis(dims, pred_err[order[i]], mean, invcov),
             euclidean(dims,  pred_err[order[i]], mean));
  
  free_vector(prediction_error);
  free_matrix(dims, invcov);
  free_matrix(dims, covar);
  free_vector(mean);
  free_matrix(frames, pred_err);

  return (frames-2);
}

/*****************************************************************************/

/*
  find_diff_segmentation: top level routine takes a sequence of observation
    vectors and saves a sorted list of the significance of each of the
    observations.  This version works using differentail features.
*/
int find_diff_segmentation(int frames, int dims, double** obs, int* order,
                           int direction, int velocity)
{
  int t;
  int i, j;
  int error_dims;
  
  double** pred_err;                                    /* prediction errors */
  double*  mean;                                              /* mean vector */
  double** covar;                                       /* covariance matrix */
  double** invcov;                              /* inverse covariance matrix */

  /*
    error vector consists of obs prediction errors concatenated with
    differential obs prediction errors
  */
  error_dims = dims * 2;

  pred_err = make_matrix(frames, error_dims);
  mean = make_vector(error_dims);
  covar = make_matrix(error_dims, error_dims);
  invcov = make_matrix(error_dims, error_dims);
  prediction_error = make_vector(frames);

  printf("find_diff_segmentation: %d frames\n", frames);
  
  linear_diff_predict(frames, dims, obs, pred_err);
  if (velocity)
    consider_velocity(frames, 3, dims, obs, error_dims, pred_err);
  if (direction)
    consider_directions(frames, 3, dims, obs, error_dims, pred_err);
  mean_covar(frames-3, error_dims, pred_err+3, mean, covar);
  invert(error_dims, covar, invcov);

  /* ascertain mahalonobis distances for all predictions */
  for (t = 3; t < frames; t++) {
    /* prediction_error[t] = euclidean(error_dims, pred_err[t], mean); */
    prediction_error[t] = mahalanobis(error_dims, pred_err[t], mean, invcov);
  }

  /* sort distances into descending order */
  for (t = 3; t < frames; t++)
    order[t-3] = t;
  qsort(order, frames-3, sizeof(int), qsort_decision);

  if (entropy_verbose)
    for (i = 0; i < frames-3; i++)
      printf("order=%d, M=%f, E=%f\n", order[i],
             mahalanobis(error_dims, pred_err[order[i]], mean, invcov),
             euclidean(error_dims,  pred_err[order[i]], mean));
  
  free_vector(prediction_error);
  free_matrix(error_dims, invcov);
  free_matrix(error_dims, covar);
  free_vector(mean);
  free_matrix(frames, pred_err);

  return (frames-3);
}

/*****************************************************************************/

/*
  find_diff2_segmentation: top level routine takes a sequence of observation
    vectors and saves a sorted list of the significance of each of the
    observations.  This version works using second differential features.
*/
int find_diff2_segmentation(int frames, int dims, double** obs, int* order,
                            int direction, int velocity)
{
  int t;
  int i, j;
  int error_dims;
  
  double** pred_err;                                    /* prediction errors */
  double*  mean;                                              /* mean vector */
  double** covar;                                       /* covariance matrix */
  double** invcov;                              /* inverse covariance matrix */

  /*
    error vector consists of obs prediction errors concatenated with
    differential obs prediction errors
  */
  error_dims = dims * 3;

  pred_err = make_matrix(frames, error_dims);
  mean = make_vector(error_dims);
  covar = make_matrix(error_dims, error_dims);
  invcov = make_matrix(error_dims, error_dims);
  prediction_error = make_vector(frames);

  printf("find_diff2_segmentation: %d frames\n", frames);
  
  linear_diff2_predict(frames, dims, obs, pred_err);
  if (velocity)
    consider_velocity(frames, 4, dims, obs, error_dims, pred_err);
  if (direction)
    consider_directions(frames, 4, dims, obs, error_dims, pred_err);
  mean_covar(frames-4, error_dims, pred_err+4, mean, covar);
  invert(error_dims, covar, invcov);

  /* ascertain mahalonobis distances for all predictions */
  for (t = 4; t < frames; t++) {
    /* prediction_error[t] = euclidean(error_dims, pred_err[t], mean); */
    prediction_error[t] = mahalanobis(error_dims, pred_err[t], mean, invcov);
  }

  /* sort distances into descending order */
  for (t = 4; t < frames; t++)
    order[t-4] = t;
  qsort(order, frames-4, sizeof(int), qsort_decision);

  if (entropy_verbose)
    for (i = 0; i < frames-4; i++)
      printf("order=%d, M=%f, E=%f\n", order[i],
             mahalanobis(error_dims, pred_err[order[i]], mean, invcov),
             euclidean(error_dims,  pred_err[order[i]], mean));
  
  free_vector(prediction_error);
  free_matrix(error_dims, invcov);
  free_matrix(error_dims, covar);
  free_vector(mean);
  free_matrix(frames, pred_err);

  return (frames-4);
}

/*****************************************************************************/

/* end of entropy_seg.c */
