/*
          This program is a simulation of a 6 legged robot guided by vision.
          To run it type: insectv maze.dat, where maze.dat is the name of
          a maze file.
          To start the simulation, type "g" in the "maze" window.
          To stop it, type "s" in the same window.
          To quit, type "q".

*/

#include "insectv.h"

#define deltax 0.3
#define deltay 0.3
#define deltaz 0.3

/* Here, 2 degrees. */
#define deltatheta 6 

/* Color. */
short rgbvec[3];

/* Define the viewpoint. */
vertex ILookfrom = {-3.0,4.4,0.0};
vertex ILookat = {0,0,0};
vertex Lookfrom,Lookat;

/* Define the camera viewpoint. */
vertex ICamLookfrom = {1.07,0.5,0.25};
vertex ICamLookat = {2.07,0.5,0.25};
vertex CamLookfrom,CamLookat;
vertex TCamLookfrom,TCamLookat;

double twist = 0;

/* Define the eye(center of eye ball). */
Object eye = 1;
vertex Ieye_center = {1.0,0.5,0};
vertex eye_center,Teye_center;

/* Window ids. */
long W[2];

/* Definition of the robot in a given instant. */
double incxl[NUMBEROFLEGS],incyl[NUMBEROFLEGS];
double incxb,inczb;
double dtheta;

/* Motion indicators. */
int rotating;
int moving;
int legmoving;
int movebody;
int begin_rotation;

/* Key. */
long dev;

/* To add rgb colors. */
unsigned long Add_Color();
unsigned long Scalar_Mult_Color();

main(argc,argv)
int argc;
char *argv[];
{
    short val;
    double dx,dz;
    int i,m = 0;
    int skip = 1;

    /* Set up graphics environment. */
    GSettings();
 
    /* Set up control keys. */
    set_control();

    /* Initialize robot coordinates. */
    Init_Robot();

    /* Set(load) up the maze. */
    Load_Maze(argv[1]);

    /* Initial drawing. */
    Initial_Draw();

    while( (moving == 1) || (rotating == 1) || 
           ((dev = qread(&val))&&(dev != QKEY))) {

       /* Reset. */
       if (dev == CKEY) {
           Init_Robot();
           Initial_Draw();
           continue;
       }

       /* Position the robot. */
       Move_Position_rob();

       m++;
       /* Update the viewing points and draw. */
       Update_View_Draw(&m);

       /* Vision routine that controls rotation. */
       if (rotating == 1)
           Rotation_Control(&skip);

      qreset();

    } /* end while */
    gexit();
}

Rotation_Control(skip)
int *skip;
{
   if (*skip == 0) { 
      if (Aligned_Edges()) {
          rotating = 0;
          moving = 1;
          *skip = 1;
          printf("Stoping rotation.\n");
      }
   }
   else *skip = 0;
}

Update_View_Draw(m)
int *m;
{
  int i;

  for(i = 0; i < 2; i++) {

      winset(W[i]);
      pushmatrix();
      if (!rotating) {

          if (i == 0) {
              Lookfrom.x += incxb/8;
              Lookfrom.z += inczb/8;
              Lookat.x += incxb/8;
              Lookat.z += inczb/8;
              lookat(Lookfrom.x,
                     Lookfrom.y,
                     Lookfrom.z,
                     Lookat.x,
                     Lookat.y,
                     Lookat.z,twist);
          }
          else {
             if (*m == 7) *m = 0;
             lookat(CamLookfrom.x,
                    CamLookfrom.y,
                    CamLookfrom.z,
                    CamLookat.x,
                    CamLookat.y,
                    CamLookat.z,twist);
         }
    }
    else { /* Do the rotation. */
       *m = 0;
        if (i == 0)
           lookat(Lookfrom.x,
                  Lookfrom.y,
                  Lookfrom.z,
                  Lookat.x,
                  Lookat.y,
                  Lookat.z,twist);
        else {
           lookat(CamLookfrom.x,
                  CamLookfrom.y,
                  CamLookfrom.z,
                  CamLookat.x,
                  CamLookat.y,
                  CamLookat.z,twist);
        }
     }
    Draw();
    popmatrix();

  } /* end for */
}

Load_Maze(mazefile)
char *mazefile;
{

   FILE *fp;
   int i = 0;
   char temp[30];
   short R,G,B;

   if ((fp = fopen(mazefile,"r")) == NULL) {
      printf("Couldn't find mazefile: \"%s\".\n",
             mazefile);
      return(1);
   }

   while(!feof(fp)) {
      fscanf(fp,"%s",temp); 
      fscanf(fp,"%s",temp); 
      Maze[i].size = temp[0];
      fscanf(fp,"%s",temp); 
      Maze[i].theta = atof(temp);
      fscanf(fp,"%s",temp); 
      Maze[i].t.x = atof(temp);
      fscanf(fp,"%s",temp); 
      Maze[i].t.y = atof(temp);
      fscanf(fp,"%s",temp); 
      Maze[i].t.z = atof(temp);
      fscanf(fp,"%s",temp);R = (short)atoi(temp);
      fscanf(fp,"%s",temp);G = (short)atoi(temp);
      fscanf(fp,"%s",temp);B = (short)atoi(temp);
      Maze[i].color[0] = R;
      Maze[i].color[1] = G;
      Maze[i].color[2] = B;

      i++;
   }
   NOOFWALLS = i - 1;
}

Draw_Maze()
{
   int i;
   char size;
   double theta;
   double dx,dz;

   for(i = 0; i < NOOFWALLS; i++) {
       size = Maze[i].size;
       theta = Maze[i].theta;
       rgbvec[0] = Maze[i].color[0];
       rgbvec[1] = Maze[i].color[1];
       rgbvec[2] = Maze[i].color[2];
       dx = Maze[i].t.x;
       dz = Maze[i].t.z;
       pushmatrix();
       translate(dx,0,dz);
       rot(theta,'y');
       c3s(rgbvec);
       Draw_Wall(size);
       popmatrix();
   }
}

Draw_Wall(size)
char size;
{
   int i,j;
   vertex w[8];
   double vert[3];

   if (size == 's') {
       for(i = 0; i < 8; i++)
           w[i] = sm[i];
   }
   else if (size == 'm') {
            for(i = 0; i < 8; i++)
                w[i] = med[i];
        }
        else {
            for(i = 0; i < 8; i++)
                w[i] = lar[i];
        }

  for(i = 0; i < 6; i++) {
      /* if (i == 5 || i == 4)
          cpack(RGB_DARKGRAY); */

      bgnpolygon();
      for(j = 0; j < 4;j++) {
          vert[0] = w[face[i][j]].x;
          vert[1] = w[face[i][j]].y;
          vert[2] = w[face[i][j]].z;
          v3d(vert);
      }
      endpolygon();
  }
}

Initial_Draw()
{
  int i;

  for(i = 0; i < 2;i++) {
      winset(W[i]);
      zbuffer(TRUE);
      pushmatrix();
      if (i == 0)
         lookat(Lookfrom.x,Lookfrom.y,Lookfrom.z,
                Lookat.x,Lookat.y,Lookat.z,twist);
      else
         lookat(CamLookfrom.x,
                CamLookfrom.y,
                CamLookfrom.z,
                CamLookat.x,
                CamLookat.y,
                CamLookat.z,twist);
      Draw();
      popmatrix();
  }
}

Move_Position_rob()
{
   double dx = 0,dy = 0,dz = 0;
   int i,j;
   static double theta;
   static int skip = 1;

   switch(dev) {
     case GKEY:
          rotating = 0;
          if (legmoving)
              Move_legs_forward();
          else {
              if (movebody) {
                  Move_body_forward();
                  movebody = 0;
              } 
              else {
                 Device devs[1] = SKEY;
                 short vals[1];

                 Move_middle_forward();
                 legmoving = 1;
                 getdev((long)1,devs,vals);
                 if (vals[0] == 1)
                     moving = 0;
                 else /* Vision routines. */
                    Look_Around();
              }
	  }
          break;

     case LKEY:
     case RKEY:
          if (!rotating || begin_rotation) {
              begin_rotation = 1;
              /* Get the original coordinates. */
              for(i = 0; i < 8; i++)
                  obody[i] = body[i];
              TCamLookfrom = CamLookfrom;
              TCamLookat = CamLookat;
              Teye_center = eye_center;
              theta = 0;

              for(j = 0; j < NUMBEROFLEGS; j++)
                  for(i = 0; i < 8; i++) {
                    oupper_leg[j][i]=upper_leg[j][i];
                    olower_leg[j][i]=lower_leg[j][i];
                  }

              /* Calculate center of rotation. */
              get_center();

              rotating = 1;  
	  }
          if (dev == RKEY) {
             dtheta += deltatheta;
             theta += deltatheta;
          }
          else {
             dtheta -= deltatheta;
             theta -= deltatheta;
	  }

          rotate_robot(theta*M_PI/180.0,xc,zc);
          /* Vision routine that controls rotation. */

          break;
   }
}

get_center()
{
   int i;
   double xmin,zmin,xmax,zmax;

   /* Start the rotation. */
   xmin = INF;zmin = INF;
   xmax = -INF;zmax = -INF;

   for(i = 0; i < 8; i++) {
       xmin = MIN(obody[i].x,xmin);
       zmin = MIN(obody[i].z,zmin);
       xmax = MAX(obody[i].x,xmax);
       zmax = MAX(obody[i].z,zmax);
   }

   xc = xmin + (xmax - xmin)/2.0;
   zc = zmin + (zmax - zmin)/2.0;
}


Write_Image(Image)
unsigned long Image[5000];
{
  static int first = 1;
  static long pwin;

  if (first) {
      first = 0;
      prefsize(200, 250);
      prefposition(350,550,600,850);
      pwin = winopen("Processed Image");
      clear();
      RGBmode();
      gconfig();    
  }
  else winset(pwin);

  lrectwrite(0,0,200,250,Image);
}

Load_Image(Image,dir)
unsigned long Image[];
char dir;
{
  int xmin,xmax,ymin,ymax;

  winset(W[1]);
  readsource(SRC_FRONT);

  if (dir == 'm') {
      xmin = 150;xmax = 350;ymin = 0; ymax = 250;
  }
  else {
         if (dir == 'r') {
             xmin = 300;xmax = 500;ymin = 0; ymax = 250;
         }
         else {
             xmin = 0;xmax = 200;ymin = 0; ymax = 250;
         }
  }

  lrectread(xmin,ymin,xmax,ymax,Image);
}

Check_For_Corners()
{

  int left,right;
  double atx,atz;

  winset(W[1]);
  /* Look right and left to search for corners. */

  /* Look right first. */
  pushmatrix();
  printf("Scaning right of image  ...... \n");
  rotate_point(3.4*deltatheta*M_PI/180.0,
               CamLookfrom.x,CamLookfrom.z,
               CamLookat.x,CamLookat.z,&atx,&atz);
  lookat(CamLookfrom.x,CamLookfrom.y,CamLookfrom.z,
         atx,CamLookat.y,atz,twist);
  Draw();
  if (Check_Corner('r'))
      right = 1;
  else right = 0;
  popmatrix();

  /* Then look left. */
  pushmatrix();
  printf("Scaning left of image ...... \n");
  rotate_point(-3.4*deltatheta*M_PI/180.0,
               CamLookfrom.x,CamLookfrom.z,
               CamLookat.x,CamLookat.z,&atx,&atz);
  lookat(CamLookfrom.x,CamLookfrom.y,CamLookfrom.z,
         atx,CamLookat.y,atz,twist);
  Draw();
  if (Check_Corner('l'))
      left = 1;
  else left = 0;
  popmatrix();


  if ((left == 1) && (right == 0))
     return(-1);
  return(left + right);

}

Check_Corner(c)
char c;
{
   unsigned long Image[150000];
   unsigned long DerivImage[50000];

   /* Load image. */
   Load_Image(Image,c);

   /* Process image. Compute derivative. */
   Compute_Derivative(200,250,DerivImage,Image);
  
   /* For now, just write image. */
   Write_Image(DerivImage);   

   return(Scan_For_Corner(201,250,DerivImage));

}

Scan_For_Corner(Nx,Ny,Image)
int Nx,Ny;
unsigned long Image[50000];
{
   int a[20] = {0,0,1,0,1,0,0,0,0,1,1,1};
   static int pos = 0;
   int index;
   int i,j,si,sj = -1;
   int vertical = 0;

   si = 40;
   for(j = 0; j < Ny - 2; j++) {
       index = si + j*Nx;
       if (Image[index] == 0x00000000) {
           sj = j;
           j = Ny;
       }
   }
   if (sj == -1) {
       printf("Couldn't find a starting point. \n");
       winset(W[1]);
       return(0);
   }

   /* Walk thru the line till find a fork. */
   i = si; j = sj;

   while( (i != Nx - 1) && (j != Ny - 1) ) {
         /* Check for vertical line at (i,j). */
         vertical = Is_Vertical( &Image[i+j*Nx],&Image[i+1+j*Nx],
                                 &Image[i+(j+1)*Nx],&Image[i+1+(j+1)*Nx],
                                 &Image[i+(j+2)*Nx],&Image[i+1+(j+2)*Nx],
                                 &Image[i+(j+3)*Nx],&Image[i+1+(j+3)*Nx],
                                 &Image[i+(j+4)*Nx],&Image[i+1+(j+4)*Nx],
                                 &Image[i+(j+5)*Nx],&Image[i+1+(j+5)*Nx] );

         /* Check to see if found a vertical line. */
         if (vertical) {
             printf("Found a vertical line at (%d,%d).\n",i,j);
             /* Check neighbors of bottom of the line to see if
                form a fork. */
             if ( (Image[i+2+j*Nx] == 0x00000000) ||
                  (Image[i+2+(j+1)*Nx] == 0x00000000) ||
                  (Image[i+2+(j-1)*Nx] == 0x00000000) ||
                  (Image[i+3+(j)*Nx] == 0x00000000) ||
                  (Image[i+3+(j-1)*Nx] == 0x00000000) ||
                  (Image[i+3+(j+1)*Nx] == 0x00000000) ) {

                 /* Color red the rest of the fork. */
                 Color_Rest_Fork(&Image[i+2+j*Nx],
                                 &Image[i+2+(j+1)*Nx],
                                 &Image[i+2+(j-1)*Nx],
                                 &Image[i+3+(j)*Nx],
                                 &Image[i+3+(j-1)*Nx],
                                 &Image[i+3+(j+1)*Nx]);

                 printf("Found fork at (%d,%d).\n",i,j);   
                 Write_Image(Image);
                 sleep(1);
                 /* Set window back. */
                 winset(W[1]);
                 return(1);
	     }
             else {
                 printf("No fork found at pixel (%d,%d).\n",i,j);
                 Write_Image(Image);
                 winset(W[1]);
                 return(0);                
             }
	   }

          /* Mark red current pixels. */
          Image[i + j*Nx] = 0xff0000ff;
          Image[i + (j + 1)*Nx] = 0xff0000ff;
          if (Image[i + (j + 2)*Nx] == 0x00000000)
              Image[i + (j + 2)*Nx] = 0xff0000ff;
             
          Write_Image(Image);

          /* Get next pixel. */
          if (Get_Next_Pixel(&i,&j,Image[i+1+j*Nx],Image[i+1+(j+1)*Nx],
                               Image[i+1+(j+2)*Nx]) == 0) {
                 printf("No next pixel at pixel (%d,%d).\n",i,j);
                 winset(W[1]);
                 return(0);                
           }


   } /* End while. */
   
   printf("Found no corner. \n");
   /* Set window back. */
   winset(W[1]);
   return(0);
}

Color_Rest_Fork(p1,p2,p3,p4,p5,p6)
unsigned long *p1,*p2,*p3,*p4,*p5,*p6;
{
    if (*p1 == 0x00000000) *p1 = 0xff0000ff;
    if (*p2 == 0x00000000) *p2 = 0xff0000ff;
    if (*p3 == 0x00000000) *p3 = 0xff0000ff;
    if (*p4 == 0x00000000) *p4 = 0xff0000ff;
    if (*p5 == 0x00000000) *p5 = 0xff0000ff;
    if (*p6 == 0x00000000) *p6 = 0xff0000ff;
}


Is_Vertical(p1,p1s,p2,p2s,p3,p3s,p4,p4s,p5,p5s,p6,p6s)
unsigned long *p1,*p1s,*p2,*p2s,*p3,*p3s,*p4,*p4s,*p5,*p5s,*p6,*p6s;
{
   if ( (*p1 == 0x00000000) &&
        (*p2 == 0x00000000) &&
        (*p3 == 0x00000000) &&
        (*p4 == 0x00000000) &&
        (*p5 == 0x00000000) &&
        (*p6 == 0x00000000) ) {

      /* Color then red before returning. */
      *p1 = 0xff0000ff;*p2 = 0xff0000ff;*p3 = 0xff0000ff;
      *p4 = 0xff0000ff;*p5 = 0xff0000ff;*p6 = 0xff0000ff;
      *p1s = 0xff0000ff;*p2s = 0xff0000ff;*p3s = 0xff0000ff;
      *p4s = 0xff0000ff;*p5s = 0xff0000ff;*p6s = 0xff0000ff;
      return(1);
    }
   else
      return(0);
}


Check_up_down(p1,p2,p3,p4)
unsigned long *p1,*p2,*p3,*p4;
{
printf("p1 = %x, p2 = %x, p3 = %x,p4 = %x\n",*p1,*p2,*p3,*p4);

   if ( ( (*p2 == 0x00000000) && (*p3 != 0x00000000) ) ||
        ( (*p1 == 0x00000000) && (*p2 != 0x00000000) ) ) {

      /* It is up. */
      return(1);
    }
   else
     if ( ( (*p3 == 0x00000000) && (*p2 != 0x00000000) ) ||
          ( (*p4 == 0x00000000) && (*p3 != 0x00000000) ) ) {

      /* Then it is down. */
      return(0);
     }

  /* Otherwise it should report an error. */
  printf("This should be an ERROR !!!! \n");
}


Get_Next_Pixel(i,j,p1,p2,p3)
int *i,*j;
unsigned long p1,p2,p3;
{
    if (p1 == 0x00000000) {
       *i = *i + 1;
    }
    else {
       if (p2 == 0x00000000) {
         *i = *i + 1; *j = *j + 1;
       }
       else {
          if (p3 == 0x00000000) {
             *i = *i + 1; *j = *j + 2;
          }
          else {
              return(0);
          }
       }
    }
    return(1);
}

Gen_Random_Turn()
{
   static int turn = 0;
   int t[3]={1,0};

   turn++;
   return(t[turn - 1]);
}

Aligned_Edges()
{
  unsigned long Image[150000];
  unsigned long DerivImage[50000];
  int i,j;
  
  /* Load camera image. */
  Load_Image(Image,'m');

  /* Process image. Compute derivative. */
  Compute_Derivative(200,250,DerivImage,Image);
  
  /* For now, just print the result. */
  Write_Image(DerivImage);
  
  return(Scan_For_Alignment(201,500,DerivImage));
}

Scan_For_Alignment(Nx,Ny,Image)
int Nx,Ny;
unsigned long Image[50000];
{
   int index;
   int i,j,si,sj = -1;
   int vertical = 0,right = 0;

   si = 100;
   for(j = 0; j < Ny - 2; j++) {
       index = si + j*Nx;
       if (Image[index] == 0x00000000) {
           sj = j;
           j = Ny;
       }
       Image[index] = 0xff0000ff;
       Write_Image(Image);
   }
   if (sj == -1) {
       printf("Couldn't find a starting point. \n");
       winset(W[1]);
       return(0);
   }

   /* Draw red line. */

   /* Walk thru the horizontal line until it is aligned. */
   i = si; j = sj;
   /* Check right. */
   for(i = si+1; i < Nx - 4;i++) {
       if ( (Image[i + sj*Nx] == 0x00000000) &&
            (Image[i + (sj+1)*Nx] == 0x00000000) ) {
           right = 1;
           /* Check for vertical line at (i,j). */
           vertical = Is_Vertical( &Image[i+sj*Nx],&Image[i+1+sj*Nx],
                           &Image[i+(sj+1)*Nx],&Image[i+1+(sj+1)*Nx],
                           &Image[i+(sj+2)*Nx],&Image[i+1+(sj+2)*Nx],
                           &Image[i+(sj+3)*Nx],&Image[i+1+(sj+3)*Nx],
                           &Image[i+(sj+4)*Nx],&Image[i+1+(sj+4)*Nx],
                           &Image[i+(sj+5)*Nx],&Image[i+1+(sj+5)*Nx] );

           Image[i + sj*Nx] = 0xff0000ff;
           Image[i + (sj+1)*Nx] = 0xff0000ff;
           Write_Image(Image);

           if (vertical)
               i = Nx; /* exit loop. */    
       }
       else {
         right = 0;
         i = Nx; /* exit loop. */    
       }
   }

   Write_Image(Image);

   if (right) { /* If right is okay, check left. */
       /* Check left. */
       for(i = si - 1; i > 4;i--) {
           if ( (Image[i + sj*Nx] == 0x00000000) &&
                (Image[i + (sj+1)*Nx] == 0x00000000) ) {
               right = 1;
               /* Check for vertical line at (i,j). */
               vertical = Is_Vertical( &Image[i+sj*Nx],&Image[i+1+sj*Nx],
                             &Image[i+(sj+1)*Nx],&Image[i+1+(sj+1)*Nx],
                             &Image[i+(sj+2)*Nx],&Image[i+1+(sj+2)*Nx],
                             &Image[i+(sj+3)*Nx],&Image[i+1+(sj+3)*Nx],
                             &Image[i+(sj+4)*Nx],&Image[i+1+(sj+4)*Nx],
                             &Image[i+(sj+5)*Nx],&Image[i+1+(sj+5)*Nx] );
              Image[i + sj*Nx] = 0xff0000ff;
              Image[i + (sj+1)*Nx] = 0xff0000ff;
              Write_Image(Image);
              if (vertical)
                 i = 1; /* exit loop. */    
           }
           else {
              right = 0;
              i = 1; /* exit loop. */    
	   }
       } /* end for. */
   }

   Write_Image(Image);
   sleep(1);

   if (right) { /* Stop rotation and go. */
       moving = 1; rotating = 0;
       dev = GKEY;
       return(1);
    }

   /* Didn't find horizontal line. */
   moving = 0;
   return(0);
}

Look_Around()
{
  int corner_pos;

  if (Too_Close()) {
      printf("\n\nToo close to the wall.\n");
      moving = 0; begin_rotation = 1; rotating = 1;

      /* Check for corners to the left or to the right.*/
      if ((corner_pos = Check_For_Corners()) == 0) {

         /* No corners were seen. */
         if (Gen_Random_Turn() == 1) /* Turn right. */
            dev = RKEY;
         else dev = LKEY;

      }
      else {
         if (corner_pos == 1) /* Corner is to the right.
                                 Generate a left turn.*/
            dev = LKEY;
         else {
            if (corner_pos == -1)/* Corner to left.
                                   Generate right turn.*/
               dev = RKEY;
            else {
              if (corner_pos != 2)
                 printf("Couldn't identify the image.n");
              /* If not error, there are 2 corners.
                 Dead end. Stop. */
              moving = 0; rotating = 0;begin_rotation=0;
            }
	 }
      }
  }
}

Draw()
{
  rgbvec[0] = 0;rgbvec[1] = 0;rgbvec[2] = 0;
  c3s(rgbvec);  /* RGB_BLACK */
  clear();
  zclear();
  draw_floor();
  Draw_Maze();
  draw_rob();
  swapbuffers();
}

GSettings()
{
   long xsize, ysize;
   float aspect;
   int i;

   if (getgdesc(GD_BITS_NORM_SNG_RED) == 0) {
    fprintf(stderr, "single buffered RGB not available on this machine\n");
	return 1;
    }

    prefsize(900, 400);
    prefposition(500,1400,0,400);
    W[0] = winopen("Maze");
    prefsize(500, 300);
    prefposition(800,1300,600,900);
    W[1] = winopen("Camera view");

    for(i = 0; i < 2; i++) {
        winset(W[i]);
        mmode(MVIEWING);
        getsize(&xsize, &ysize);
        aspect = (float)xsize / (float)ysize;
        perspective(900, aspect, 0.1,20);
        RGBmode();
        doublebuffer();
        /*dither(DT_OFF);*/
        gconfig();
        /* backface(TRUE); */
    }
}

Init_View_Points()
{
   Lookfrom = ILookfrom;
   Lookat = ILookat;

   CamLookfrom = ICamLookfrom;
   CamLookat = ICamLookat;

   eye_center = Ieye_center;
}

Init_Robot()
{
   int i,j;
   int angle;

   rotating = 0;moving = 0;legmoving = 1;
   begin_rotation = 0;

   Init_View_Points();

   /* First, initialize increments. */
   incxb = 0; inczb = 0;dtheta = 0;
   for(i = 0; i < NUMBEROFLEGS; i++) {
      incxl[i] = 0.0; incyl[i] = 0.0;      
   }

   /* Set body. */
   for(i = 0; i < 8; i++)
      body[i] = b[i];

   /* Set eye. */
   eye = 1;
   makeobj(eye);
   /* RGB_RED */
   rgbvec[0] = 255;rgbvec[1] = 0;rgbvec[2] = 0;
   c3s(rgbvec);
   for(angle = 0; angle < 3600; angle += 300) {
       rotate(300,'y');
       circf(0.0, 0.0,0.05);
   }
   closeobj();

   /* Here, the legs are numbered 0 thru 7, where 0,1,2 are the
      front,middle and back left legs and 3,4 and 5 are the front,
      middle and back right legs.
   */

   /* Set upper and lower legs. */
   for(j = 0; j < NUMBEROFLEGS; j++)
       for(i = 0; i < 8; i++) {
           if (j < 3) {
               upper_leg[j][i] = ul[i];
               lower_leg[j][i] = ll[i];
           }
           else {
                upper_leg[j][i] = ur[i];
                lower_leg[j][i] = lr[i];
           }
   }

   /* Adjust legs to obtain the middle and back legs. */
   for(j = 0; j < NUMBEROFLEGS; j++)
       for(i = 0; i < 8; i++) {
           switch(j) {
             case 1:
	     case 4:
               upper_leg[j][i].x += -0.6;
               lower_leg[j][i].x += -0.6;
               break;
	     case 2:
             case 5:
               upper_leg[j][i].x += -1.2;
               lower_leg[j][i].x += -1.2;
               break;
           }               
       }
}

/* *******************  Draw routines. *********************/
void draw_rob()
{
 pushmatrix();
 translate(eye_center.x,eye_center.y,eye_center.z);
 callobj(eye);
 popmatrix();

 draw_body();
 draw_legs();
}

draw_legs()
{
   int i;

   for(i = 0; i < NUMBEROFLEGS; i++)
       draw_leg(i);
}

draw_leg(l)
int l;
{
  int i,j;
  double vert[3];

  /* Draw upper leg. */
  /* RGB_MYBLUE */
  rgbvec[0] = 100;rgbvec[1] = 100;rgbvec[2] = 130;
  c3s(rgbvec);
  for(i = 0; i < 6; i++) {
      if (i == 5 || i == 4)
          c3s(rgbvec); /* RGB_MYBLUE */

      bgnpolygon();
      for(j = 0; j < 4;j++) {
          vert[0] = upper_leg[l][face[i][j]].x;
          vert[1] = upper_leg[l][face[i][j]].y;
          vert[2] = upper_leg[l][face[i][j]].z;
          v3d(vert);
      }
      endpolygon();
  }
  /* Draw lower leg. */
  /* RGB_MYBLUE */
  rgbvec[0] = 100;rgbvec[1] = 100;rgbvec[2] = 130;
  c3s(rgbvec);
  for(i = 0; i < 6; i++) {
      if (i == 5 || i == 4)
          c3s(rgbvec);

      bgnpolygon();
      for(j = 0; j < 4;j++) {
          vert[0] = lower_leg[l][face[i][j]].x;
          vert[1] = lower_leg[l][face[i][j]].y;
          vert[2] = lower_leg[l][face[i][j]].z;
          v3d(vert);
      }
      endpolygon();
  }
}

draw_body()
{
  int i,j;
  double vert[3];

  /* RGB_GRAY */
  rgbvec[0] = 70;rgbvec[1] = 70;rgbvec[2] = 100;
  c3s(rgbvec);
  for(i = 0; i < 6; i++) {
      if (i == 5 || i == 4) {
         /* RGB_DARKGRAY */
         rgbvec[0] = 40;rgbvec[1] = 40;rgbvec[2] = 70;
         c3s(rgbvec);
      }

      bgnpolygon();
      for(j = 0; j < 4;j++) {
          vert[0] = body[face[i][j]].x;
          vert[1] = body[face[i][j]].y;
          vert[2] = body[face[i][j]].z;
          v3d(vert);
      }
      endpolygon();
  }
}

void draw_floor()
{
   double x,z;
   double p[3];
   int sw = 0;

   for (z = -30; z < 30; z+= 3.0) {
       for (x = -30; x < 27; x+= 3.0) {
           bgnpolygon();
               sw = (sw == 0 ? 1: 0);
               if (sw) {
                   /* RGB_DARKBLUE */
                   rgbvec[0] = 120;rgbvec[1] = 120;
                   rgbvec[2] = 120;
                   c3s(rgbvec);
               }
               else {
                   /*cpack(RGB_LIGHTBLUE);*/
                   rgbvec[0] = 120;rgbvec[1] = 120;
                   rgbvec[2] = 120;
                   c3s(rgbvec);
	       }
               p[0]=x;p[1]=0;p[2]=z;v3d(p);
               p[0]=x+3;p[2]=z;v3d(p);
               p[0]=x+3;p[2]=z-3;v3d(p);
               p[0]=x;p[2]=z-3;v3d(p);
               p[0]=x;p[2]=z;v3d(p);
            endpolygon();
       }
   }
}


Move_legs_forward()
{
   int i;
   double dx,dz;
   double theta;
   static int turn = 0;
   int seq[4] = {0,3,2,5};

   theta = (M_PI/180.0)*dtheta;
   /* Move all legs forward in the 'x' direction. */
   dx = deltax*cos(theta);
   dz = deltaz*sin(theta);
   incxl[0] += deltax;
   for(i = 0; i < 8; i++) {
       if ( (i == 0) || (i == 1) || 
            (i == 4) || (i == 5) ) {
            /*printf("dx = %f, dz = %f, theta = %f\n",
                      dx,dz,dtheta);*/
            upper_leg[seq[turn]][i].x += dx;
            upper_leg[seq[turn]][i].z += dz;
       }
       lower_leg[seq[turn]][i].x += dx;
       lower_leg[seq[turn]][i].z += dz;
   }

   /* Update turn(leg of the turn). */
   turn++;
   if (turn > 3) {
       /* Move_body_forward();*/
       legmoving = 0;movebody = 1;
       turn = 0;
   }

   /* Start moving legs. If the last leg was moved, stop
      and start moving body. */
   moving = 1;
}

Move_middle_forward()
{
   int i,l;
   double dx,dz;
   double theta;

   theta = (M_PI/180.0)*dtheta;
   dx = deltax*cos(theta);
   dz = deltaz*sin(theta);

   /* Also move the middle leg. */
   for(l = 0; l < 2; l++) {
       for(i = 0; i < 8; i++) {
           if ( (i == 0) || (i == 1) || 
                (i == 4) || (i == 5) ) {
                /* printf("leg %d dx = %f, dz = %f, 
                theta = %f\n",3*l + 1,dx,dz,dtheta); */
                upper_leg[3*l + 1][i].x += dx;
                upper_leg[3*l + 1][i].z += dz;
           }
           lower_leg[3*l + 1][i].x += dx;
           lower_leg[3*l + 1][i].z += dz;
      }
   }
}

Move_body_forward()
{
   int i,l;
   double dx,dz;
   double theta;

   theta = (M_PI/180.0)*dtheta;

   /* Move body and upper leg junctions. */
   dx = deltax*cos(theta);
   dz = deltaz*sin(theta);
   incxb = dx;inczb = dz;
   for(i = 0; i < 8; i++) {
       body[i].x += dx;
       body[i].z += dz;
   }
   for(l = 0; l < NUMBEROFLEGS; l++) {          
       for(i = 0; i < 8; i++) {
           if ( (i == 3) || (i == 2) ||
                (i == 6) || (i == 7) ) {
              upper_leg[l][i].x += dx;
              upper_leg[l][i].z += dz;
           }
       }
   }
   /* Update camera. */
   CamLookfrom.x += dx; CamLookfrom.z += dz;
   CamLookat.x += dx; CamLookat.z += dz;
   eye_center.x += dx; eye_center.z += dz;

}

rotate_point(theta,xc,zc,ox,oz,newx,newz)
double theta,xc,zc,ox,oz,*newx,*newz;
{

  *newx = (ox - xc)*cos(theta) -
          (oz - zc)*sin(theta) + xc;
  *newz = (ox - xc)*sin(theta) + 
          (oz - zc)*cos(theta) + zc;
}

rotate_robot(theta,xc,zc)
double theta;
double xc,zc;
{
   int i,j;

   /* Rotate body. */
   for(i = 0; i < 8; i++)
       rotate_point(theta,xc,zc,obody[i].x,obody[i].z,
                    &body[i].x,&body[i].z);

   /* Rotate camera and lookat. */
  rotate_point(theta,xc,zc,TCamLookfrom.x,TCamLookfrom.z,
               &CamLookfrom.x,&CamLookfrom.z);
  rotate_point(theta,xc,zc,TCamLookat.x,TCamLookat.z,
               &CamLookat.x,&CamLookat.z);

  /* Rotate the eyes. */
  rotate_point(theta,xc,zc,Teye_center.x,Teye_center.z,
               &eye_center.x,&eye_center.z);


   
   /* Rotate legs. */
   for(j = 0; j < NUMBEROFLEGS; j++)
       for(i = 0; i < 8; i++) {
           upper_leg[j][i].x = 
             (oupper_leg[j][i].x - xc)*cos(theta) -
             (oupper_leg[j][i].z - zc)*sin(theta);
           upper_leg[j][i].z = 
             (oupper_leg[j][i].x - xc)*sin(theta) + 
             (oupper_leg[j][i].z - zc)*cos(theta);

           lower_leg[j][i].x = 
             (olower_leg[j][i].x - xc)*cos(theta) -
             (olower_leg[j][i].z - zc)*sin(theta);
           lower_leg[j][i].z = 
             (olower_leg[j][i].x - xc)*sin(theta) + 
             (olower_leg[j][i].z - zc)*cos(theta);
       }
   /* Translate back. */
   for(j = 0; j < NUMBEROFLEGS; j++)
       for(i = 0; i < 8; i++) {
           upper_leg[j][i].x += xc;
           upper_leg[j][i].z += zc;
           lower_leg[j][i].x += xc;
           lower_leg[j][i].z += zc;
       }
}

set_control()
{
   qdevice(CKEY);
   qdevice(GKEY);
   qdevice(QKEY);
/*
   qdevice(LKEY);
   qdevice(RKEY);
   qdevice(UPARROWKEY);
*/
}


Too_Close()
{
  unsigned long Image[150000];
  unsigned long DerivImage[50000];
  int i,j,limit;

  /* Load camera image. */
  Load_Image(Image,'m');

  /* Process image. Compute derivative. */
  Compute_Derivative(200,250,DerivImage,Image);
  
  /* For now, just print the result. */
  Write_Image(DerivImage);
  
  /* Check to see if it needs course correction. */
  if (Needs_Course_Correction(201,250,DerivImage,&limit)) {
      moving = 0; rotating = 1; begin_rotation = 1;
      return(0);
  }

  /* Check for threshold. */
  return(limit);
}

Check_Threshold(Nx,Ny,Image)
int Nx,Ny;
unsigned long Image[15000];
{
   int i,j,th = 100; /* 95 */

}

Needs_Course_Correction(Nx,Ny,Image,limit)
int Nx,Ny;
unsigned long Image[5000];
int *limit;
{
   int index;
   int i,j,si,sj = -1,th = 100; /* was 95. */
   int vertical = 0,up = 0;

   /* Check distance limit. */
   i = 100; /* Take the midpoint. */
   for(j = 0; j < Ny; j++) {
       if (Image[i + j*Nx] == 0x00000000) {
          if (j < th)
             *limit = 1;
          else
            *limit = 0;
          j = Ny; /* exit loop. */
       }
   }

   si = 100;
   for(j = 0; j < Ny - 2; j++) {
       index = si + j*Nx;
       if (Image[index] == 0x00000000) {
           sj = j;
           j = Ny;
       }
       Image[index] = 0xff0000ff;
       Write_Image(Image);
   }
   if (sj == -1) {
       printf("Couldn't find a starting point. \n");
       winset(W[1]);
       return(0);
   }

   /* Walk thru the horizontal line until it is aligned. */
   i = si; j = sj;
   /* Check right. */
   for(i = si+1; i < Nx - 4;i++) {
       if ( (Image[i + sj*Nx] == 0x00000000) &&
            (Image[i + (sj+1)*Nx] == 0x00000000) ) {
           /* Check for vertical line at (i,j). */
           vertical = Is_Vertical( &Image[i+sj*Nx],&Image[i+1+sj*Nx],
                           &Image[i+(sj+1)*Nx],&Image[i+1+(sj+1)*Nx],
                           &Image[i+(sj+2)*Nx],&Image[i+1+(sj+2)*Nx],
                           &Image[i+(sj+3)*Nx],&Image[i+1+(sj+3)*Nx],
                           &Image[i+(sj+4)*Nx],&Image[i+1+(sj+4)*Nx],
                           &Image[i+(sj+5)*Nx],&Image[i+1+(sj+5)*Nx] );

           Image[i + sj*Nx] = 0xff0000ff;
           Image[i + (sj+1)*Nx] = 0xff0000ff;
           Write_Image(Image);

           if (vertical)
              i = Nx; /* exit loop, to check left side. */    
       }
       else 
           if (Image[i+1+(sj+1)*Nx] == 0x00000000)
               continue;
          else
          {
printf("A\n");
         up = Check_up_down(&Image[i+(sj+2)*Nx],&Image[i+(sj+1)*Nx],
                            &Image[i+(sj)*Nx],&Image[i+(sj-1)*Nx]);
         if (up) dev = LKEY;
         else dev = RKEY;
         printf("\n\n %s right discontinuity.  Correcting course to %s.\n",
                (up == 1 ? "up" : "down"),(dev == RKEY ? "right":"left"));
         return(1);
       }
   }

   Write_Image(Image);

   /* If right is okay, check left. */
   for(i = si - 1; i > 4;i--) {
       if ( (Image[i + sj*Nx] == 0x00000000) &&
            (Image[i + (sj+1)*Nx] == 0x00000000) ) {
            /* Check for vertical line at (i,j). */
            vertical = Is_Vertical( &Image[i+sj*Nx],&Image[i+1+sj*Nx],
                            &Image[i+(sj+1)*Nx],&Image[i+1+(sj+1)*Nx],
                            &Image[i+(sj+2)*Nx],&Image[i+1+(sj+2)*Nx],
                            &Image[i+(sj+3)*Nx],&Image[i+1+(sj+3)*Nx],
                            &Image[i+(sj+4)*Nx],&Image[i+1+(sj+4)*Nx],
                            &Image[i+(sj+5)*Nx],&Image[i+1+(sj+5)*Nx] );
            Image[i + sj*Nx] = 0xff0000ff;
            Image[i + (sj+1)*Nx] = 0xff0000ff;
            Write_Image(Image);
            if (vertical)
               return(0); /* No correction is needed. */
         }
         else {
printf("B\n");
printf("i+sj = %x, i +(sj+1) = %x\n",Image[i+sj*Nx],Image[i+(sj+1)*Nx]);
            up = Check_up_down(&Image[i+(sj+2)*Nx],&Image[i+(sj+1)*Nx],
                               &Image[i+(sj)*Nx],&Image[i+(sj-1)*Nx]);
            if (up) dev = RKEY;
            else dev = LKEY;
            printf("\n\n %s left discontinuity. Correcting course to %s.\n",
                  (up == 1 ? "up" : "down"),(dev == RKEY ? "right":"left"));
            Write_Image(Image);
            return(1);
	 }
  } /* end for. */
/*printf("Shouldn't arrive here.\n");*/
}

Compute_Derivative(Nx,Ny,DerivImage,Image)
int Nx,Ny;
unsigned long DerivImage[50000],Image[150000];
{
   unsigned long UpDerivImage[50000];
   unsigned long RightDerivImage[50000];

   /* First, compute up derivative of the image. */
   Compute_UpDeriv(Nx,Ny,UpDerivImage,Image);

   /* Then, compute the right derivative of the image. */
   Compute_RightDeriv(Nx,Ny,RightDerivImage,Image);

   /* Combine up and right derivative. */
   Combine_Derivatives(Nx,Ny,
          DerivImage,UpDerivImage,RightDerivImage);

}

Combine_Derivatives(Nx,Ny,Image,Up,Right)
int Nx,Ny;
unsigned long Image[50000],Up[50000],Right[50000];
{
   int i,j;
   
   for(j = 0; j < Ny; j++)
       for(i = 0; i < Nx; i++)
           if ( (Up[i + j*Nx] == 0) ||
                (Right[i + j*Nx] == 0) )
                Image[i + j*Nx] = 0;
           else Image[i + j*Nx] = 0xffffffff;
}

Compute_UpDeriv(Nx,Ny,DerivImage,Image)
int Nx,Ny;
unsigned long DerivImage[50000],Image[150000];
{
   int i,j,lowest = 0;

   
   for(j = 0; j < Ny; j++)
       for(i = 0; i < Nx; i++) {
           if ( (j == 0) || (j == (Ny - 1)) ) {
                 DerivImage[i + j*Nx] = 0xffffffff;
                 continue;
           }
           DerivImage[i + j*Nx] = Add_Color('-',
                       Image[i + (j + 1)*Nx],
                       Image[i + (j - 1)*Nx]);
           lowest = MIN(DerivImage[i + j*Nx],lowest);
    }
}


Compute_RightDeriv(Nx,Ny,DerivImage,Image)
int Nx,Ny;
unsigned long DerivImage[50000],Image[150000];
{
   int i,j,lowest = 0;
   
   for(j = 0; j < Ny; j++)
       for(i = 0; i < Nx; i++) {
           if ( (i == 0) || (i == (Nx - 1)) ) {
                 DerivImage[i + j*Nx] = 0xffffffff;
                 continue;
           }
           DerivImage[i + j*Nx] = Add_Color('-',
                       Image[i + 1 + j*Nx],
                       Image[i - 1 + j*Nx]);

           lowest = MIN(DerivImage[i + j*Nx],lowest);
    }
}

unsigned long Add_Color(s,t1,t2)
char s;
unsigned long t1,t2;
{
  unsigned char r,g,b,r1,g1,b1,r2,g2,b2;
  unsigned long t;

  t = 0;

  r1 = (t1>>0)&0xff; r2 = (t2>>0)&0xff;
  g1 = (t1>>8)&0xff; g2 = (t2>>8)&0xff;
  b1 = (t1>>16)&0xff; b2 = (t2>>16)&0xff;

  r = (s == '+' ? r1 + r2: r1 - r2);
  g = (s == '+' ? g1 + g2: g1 - g2);
  b = (s == '+' ? b1 + b2: b1 - b2);

  t = t | (unsigned long)r;
  t = t | ((unsigned long)g)<<8;
  t = t | ((unsigned long)b)<<16;
  t = t | ((unsigned long)0xff)<<24;

  /* Change color picture into black&white. */
  if (t != 0xff000000)
      t = 0x00000000;
  else t = 0xffffffff;

  /* printf("rgb %d %d %d,rgb1 %d %d %d,rgb2 %d %d %d\n",
            r,g,b,r1,g1,b1,r2,g2,b2);*/
  return(t);
}

























