;;; -*- Mode: LISP; Package: USER; Base: 10; Syntax: Common-lisp -*-

;;; LaHaShem HaAretz U'Mloah


;;; This file contains a program profiling utility routine.
;;; It is used to determine where the processing time is spent in user
;;; programs. It is similar in intent to the PC metering tools provided
;;; by Symbolics. The Symbolics tools however, only gather info on the
;;; lowest level function which is running when a meter sample is taken.
;;; As most of the samples indicate system functions such as CAR, this
;;; gives little information about the running characteristics of the users
;;; code. These tools instead trace up the stack giving a metering "point"
;;; to every nested function at each sample tick. The tick rate is given
;;; by the variable *profile-sample-rate* which is how many seconds between
;;; samples. Various reporting mechanisms then return the amount of time
;;; spent in each function. Clearly each percentages lies between 0% and 100%
;;; though their sum will total more than 100%. For example if function A
;;; calls B and C, B calls D and E, and C calles F and G then the following
;;; might be a reasonable profiling result:
;;; 100% A
;;;  60% B
;;;  40% C
;;;  20% D
;;;  40% E
;;;  30% F
;;;   5% G
;;; Note that the sum of F and G is less than E as some time was spent
;;; directly in E and not its decendents. If a function's decedents each
;;; have only one caller than the sum of the decendent percentages should
;;; be less than or equal to the parent percentage. This may not be the
;;; case when a function has more than one caller. Also a recursive call
;;; to a function is only counted once per sample tick.
;;;

;;; Sampling is accomplished by the macro profile. To sample the execution
;;; of some lisp form <form> issue:
;;; (profile <form>)
;;; or
;;; :Profile <form>
;;; which runs <form> and returns the results of <form> (just like eval)
;;; but which gathers profiling statistics as a side effect.
;;; Profiling terminates either when <form> returns, or when <form> is aborted.
;;; As the profiler runs in a background process which accesses global
;;; variables, only one process may be profiled at a time. In order to
;;; achieve the greatest accuracy, the process to be profiled should run
;;; at priority 0 and no other process of priority greater than or equal to 0
;;; should be doing significant work.
;;;
;;; Gathering of sample ticks is cummulative. It is possible to do several
;;; calls to profile and print the cummulative results together. Clearing
;;; of the sample data base is accomplished by issuing:
;;; (initialize-profiler)
;;; or
;;; :Initialize Profiler
;;;
;;; By issuing the function:
;;; (profiler-graph)
;;; or
;;; :Show Profiler Graph
;;; a graph window is displayed which represents the calling structure
;;; of user functions. Root functions appear at the left margin while the
;;; caller-callee relationship progresses to the right. Recursive calls
;;; are pruned. A root function is defined to be any user function which
;;; is not called by any other user function. Functions which have been
;;; sampled are annotated with with their sampling percentages. Scrolling
;;; can be accomplished either with the scroll bars or by the following
;;; key-strokes: Scroll     - down
;;;              m-Scroll   - up
;;;              c-Scroll   - right
;;;              c-m-Scroll - left
;;; The profiler graph window is running in a separate process so it is
;;; possible to go back and forth between it and other windows including
;;; the window which issued the call to profiler-graph. It is possible to
;;; select the profiler graph window by issuing a <Select> !.
;;; Typing <Abort> to the profiler graph window will kill both the window
;;; and the process. Normally the profiler-graph function will only display
;;; user functions which have been sampled. It is possible to display all
;;; user functions by issuing the command:
;;; (profiler-graph t)
;;; or
;;; :Show Program Graph
;;; This way sampled functions will be annotated with their percentages
;;; while unsampled functions will be displayed without annotation.
;;;
;;; Bugs:
;;; 1) Should handle :internal
;;; 4) Should get rid of cursor blinker in profiler graph window
;;; 5) Should also report percentage time spent in user function as
;;;    lowest level user function on the stack.
;;; 6) Should also keep track of percentage of time A is called by B
;;;    versus A being called by C.
;;; 7) Someday, I will get arround to doing storage allocation metering as
;;;    well.
;;;
;;; Acknowledgements:
;;;    Thanks are due to:
;;;       David Vinayak Wallace (Gumby@MCC)
;;;       Michael Greenwald (Greenwald@STONY-BROOK.SCRC.Symbolics)
;;;       Larry Dennenberg (ldenenbe@ALEXANDER.BBN)
;;;       David Moon (Moon@STONY-BROOK.SCRC.Symbolics)
;;; for providing the pointers to system code needed to produce this program.
;;; This program uses many Symbolics internal system functions and may not
;;; work in releases other than Genera 7.1 and may not work in all
;;; circumstances even then. I do not claim to understand the internals of
;;; the 3600 series. Rather I did a lot of programming by analogy.
;;;                                      Qobi
;;;                                      Palo Alto, 8-July-1987
;;;
;;; (c) Copyright 1987 by Jeffrey Mark Siskind.  Massively hacked by Johan de Kleer.
;;; You are free to use and distribute this software so long as you comply
;;; with the following provisions:
;;; 1) Any copy of this software must retain this notice.
;;; 2) It may not be used as part of a commercial product or for direct
;;;    monetary gain.
;;; 3) Anyone who either distributes or receives a copy of this software must
;;;    notify me of the recepient at one of the addresses listed below.
;;; 4) Anyone who modifies this software must provide me with a copy of the
;;;    modifications which retains this notice.
;;; 5) Any use of this software must include appropriate references to its
;;;    creators.
;;; 6) This software is provided on an AS IS basis. Correct operation and
;;;    suitablity for any purpose is not guaranteed.
;;;
;;; Enjoy!
;;;
;;; Jeffrey Mark Siskind
;;; Theory of Computation Group
;;; MIT Laboratory for Computer Science
;;; 545 Technology Square NE43-340
;;; Cambridge MA 02139
;;; 617/253-5293
;;; Qobi@XX.LCS.MIT.EDU
;;;       or
;;; Intelligent Systems Laboratory #1601
;;; Xerox PARC
;;; 3333 Coyote Hill Road
;;; Palo Alto CA 94304
;;; 415/494-4319
;;; Siskind.PA@Xerox.COM

(export '*profile-sample-rate*)

(defvar *profile-sample-rate* 0.1 "How many seconds between profiling samples")

(defvar *profile?* nil)

(defvar *profile-count* 0)

(defvar *profile-count-table* (make-hash-table :test #'equal))

(defvar *profile-definitions-in-buffer* nil)

(defvar *profile-decendent-functions-table* nil)

(defvar *profile-root-functions* nil)

(defflavor profiler-graph-window
        ()
        (dw:dynamic-window)
  (:default-init-plist
   :margin-components '((dw:margin-borders :thickness 1)
                        (dw:margin-white-borders :thickness 3)
                        (dw:margin-label
                         :margin :bottom
                         :style (:sans-serif :italic :normal)
                         :string "Profiler Graph")
                        (dw:margin-whitespace
                         :margin :bottom
                         :thickness 3)
                        (dw:margin-scroll-bar :margin :left)
                        (dw:margin-scroll-bar :margin :bottom)
                        (dw:margin-white-borders :thickness 5)
                        (dw:margin-ragged-borders :thickness 1))))

(defvar *roots* nil)

(defun make-profiler-tables ()
  (clrhash *profile-count-table*)
  (make-profiler-table (cdddr *roots*)))

;;; The *profile-count-table* will contain the number of non-recursive ticks assigned
;;; to this function.
(defun make-profiler-table (slots)
  (dolist (slot slots)
    (incf (gethash (first slot) *profile-count-table* 0)
	  (- (second slot) (third slot)))
    (make-profiler-table (cdddr slot))))


(defun profiler (process)
 (loop with touched?-table = (make-hash-table :test #'equal)
       while *profile?*
       do (incf *profile-count*)
          (clrhash touched?-table)
          (si:process-enable-arrest-reason process :profile)
          (si:inhibit-gc-flips
	    (update-calling-tree (sys:sg-frame-pointer (process-stack-group process)) nil))
          (si:process-disable-arrest-reason process :profile)
          (sleep *profile-sample-rate* :sleep-reason "Profile Sleep")
       finally
         (setq *profile?* t)))

;;; This gathers the data one really wants.  This could be considerably optimized.
;;; **** this does not handle evals!?.  Need to do those by hand.
;;; Optimizations: (1) collect this data, process this all later, (2) keep actually
;;; buffer as top parts of stack hardly ever change.

;;; Notice that at every recursion, the stack gets walked up one function.  Thus,
;;; the exact reverse of normal.
;;; Every level in the tree looks like (function total-count recursive-count . tree)

(defun update-calling-tree (frame function-stack &aux function slot tree)
  (setq function (dbg:frame-real-function frame))
  (cond ((or (null (locativep frame))
	     (eq function #'profiler)
	     (eq function #'com-profile))
	 (incf (second *roots*))
	 (if (member function function-stack) (incf (third *roots*)) :test #'eq)
	 *roots*)
	;; Forget about anything inside of error for now --- this only messes things up.
	((eq function #'error) nil)
	((null (with-stack-list* (stack function function-stack)
		 (setq tree
		       (update-calling-tree (dbg:frame-previous-active-frame frame) stack))))
	 nil)
	;; Don't include useless things.
	((eq function #'si:maybe-preempt-current-process) nil)
	((setq slot (assoc function (cdddr tree) :test #'eq))
	 (incf (second slot))
	 (if (member function function-stack) (incf (third slot)))
	 (and (neq function #'si:print)
	      (neq function #'format)
	      slot))
	(t (setq slot (cons function (cons 1 (cons 0 nil))))
	   (rplacd (cddr tree) (cons slot (cdddr tree)))
	   (and (neq function #'si:print)
		(neq function #'format)
		slot))))

;;; Given a FRAME, return the function it corresponds to.
;;; Bug is that frame-self-value will sometimes return <DTP-EVEN-PC ...>
DBG:
(DEFUN FRAME-REAL-FUNCTION (FRAME)
  (DECLARE (VALUES REAL-FUNCTION FIRST-INTERESTING-LOCAL))
  (LET ((FUNCTION (FRAME-FUNCTION FRAME))
	(OBJECT (FRAME-SELF-VALUE FRAME T)))
    (IF (AND (instancep OBJECT)
	     (LET* ((OPERATION (FRAME-ARG-VALUE FRAME #-3600 0 #+3600 2))
		    (HANDLER (UNLESS (FUNCTIONP OPERATION)
			       (GET-HANDLER-FOR OBJECT OPERATION))))
	       (IF HANDLER
		   (EQ FUNCTION (IF (SYMBOLP HANDLER) (FSYMEVAL HANDLER) HANDLER))
		   (OR (EQ FUNCTION #'FLAVOR::HANDLE-REAL-UNCLAIMED-MESSAGE)
		       (EQ FUNCTION #'FLAVOR::HANDLE-UNCLAIMED-MESSAGE)))))
	(VALUES OBJECT #-3600 0 #+3600 2)
	(VALUES FUNCTION 0))))

(export 'initialize-profiler)

(defun initialize-profiler ()
  (setq *roots* (cons 'TOP (cons 0 (cons 0 nil)))
        *profile?* nil
	*profile-count* 0)
 (clrhash *profile-count-table*)
 t)

;;; Make sure profiler is initialized when first loaded.
(unless *roots* (initialize-profiler))

(cp:define-command (com-initialize-profiler
                    :command-table "Global"
                    :provide-output-destination-keyword nil)
    ()
   (initialize-profiler))

(export 'profile)

(defmacro profile (form)
  `(unwind-protect
       (progn (setq *profile?* t)
              (process-run-function "Profiler" #'profiler si:current-process)
              ,form)
     (setq *profile?* nil)))

(cp:define-command (com-profile :command-table "Global" :values t)
    ((form 'sys:expression
           :documentation "A form to be executed while profiling is enabled"
           :prompt "A form to profile"))
   (profile (eval form)))

(export 'profiler-graph)

(defun profiler-graph (&rest ignore)
 (process-run-function
  "Profiler Graph"
   (lambda ()
     (let ((window (tv:make-window 'profiler-graph-window)))
       (make-profiler-tables)
       (unwind-protect
           (progn
            (send window :select)
            (format-graph-from-root
	      (cdddr *roots*)
	      (lambda (slot stream &aux total-count)
		(if (= (setq total-count (gethash (first slot) *profile-count-table*))
		       (second slot))
		    ;; *** fix from format documentation.
		    (format stream "~S ~D%"
			    (case (zl:typep (first slot))
			      (:SYMBOL (first slot))
			      (:COMPILED-FUNCTION (si:compiled-function-name (first slot)))
			      (T (first slot)))
			    (round
			      (* (/ (second slot) *profile-count*) 100.0)))
		    (format stream "~S ~D%(~D%)"
			    (case (zl:typep (first slot))
			      (:SYMBOL (first slot))
			      (:COMPILED-FUNCTION (si:compiled-function-name (first slot)))
			      (T (first slot)))
			    (round
			      (* (/ (second slot) *profile-count*) 100.0))
			    (round (* (/ total-count *profile-count*) 100.0)))))
	      ;; Ignore .5% or less functions.
	      (lambda (slot &aux callees)
		(dolist (slot (cdddr slot))
		  (when (> (/ (second slot) *profile-count*) 0.005)
		    (push slot callees)))
		(sort callees #'(lambda (a b) (> (second a) (second b)))))
             :stream window
             :dont-draw-duplicates nil
             :key #'identity
             :test #'equal
             :root-is-sequence t
             :orientation :horizontal
             :direction :after
             :default-drawing-mode :arrow
             :cutoff-depth nil
             :balance-evenly nil
             :border '(:shape :rectangle :thickness 2)
             :row-spacing 40
             :within-row-spacing 20
             :column-spacing 20
             :within-column-spacing 10)
            (loop for char = (send window :tyi)
                  with *terminal-io* = window
                  do (case char
                       (#\scroll (cp::com-scroll-window :screen 1 :y))
                       (#\m-scroll (cp::com-scroll-window :screen -1 :y))
                       (#\c-scroll (cp::com-scroll-window :screen 1 :x))
                       (#\c-m-scroll (cp::com-scroll-window :screen -1 :x))
                       (otherwise (send window :beep)))))
         (send window :kill)))))
 t)

(cp:define-command (com-show-profiler-graph
                    :command-table "Global"
                    :provide-output-destination-keyword nil)
    ()
   (profiler-graph nil))

(cp:define-command (com-show-program-graph
                    :command-table "Global"
                    :provide-output-destination-keyword nil)
    ()
   (profiler-graph t))

(tv:add-select-key #\! 'profiler-graph-window "Profiler Graph" nil)

;;; Tam V'Nishlam Shevah L'El Borei Olam

