" (c) 1993 Copyright (c) University of Washington
  Written by Tony Barrett.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to
  bug-ucpop@cs.washington.edu; the same address should be used for problems."

(in-package "ZENO")

(defvar *sc* nil)
(defvar *nodes-visited* 0)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 2. Handy interface functions.

(defun ISC-CONTROL (prob scs)
  "Use search controller to perform an iterative deepening 
   best first search"
  (sc-control* prob scs #'sc:isrch))

(defun SC-CONTROL (prob scs)
  "Use search controller to perform a best first search"
  (sc-control* prob scs #'sc:srch))

(defun sc-show (prob scs &optional (display "mizar:0"))
  "Use search controller to perform a best first search,
   followed by a graphical exploration of search space"
  (enable-vcr)
  (rule-net:collect-firings)
  (sc-control* prob scs #'sc:srch)
  (rule-net:ignore-firings)
  (play "sc-show" display)
  (disable-vcr))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. Formulating a search for Zeno

(defun SC-CONTROL* (prob scs srch)
  (when (symbolp prob) 
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (init-zeno)
  (let* ((sc (funcall scs prob))	; Setting up the controller
	 (start (make-goal-plan (problem-start prob) (problem-end prob)
				(problem-inits prob) (problem-goal prob)))
	 (init-time (get-internal-run-time))
	 (plan (funcall srch start sc *search-limit*))
	 (total-time (- (get-internal-run-time) init-time)))
    (record-stat .cpu-time. total-time)
    (record-stat .success. (plan-test plan))
    (when plan (display-plan plan))
    (display-stats)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6. Special purpose search controllers

;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Mimicing the old best first search
;;;
;;; (sc-control 'sussman-anomaly #'bf-mimic)

(defun bf-mimic (prob)
  (declare (ignore prob))
  (let ((sc (zeno-sc 'bf-mimic)))
    (sc:rule sc 'select-ranked
	     :when '((:node $p) (rank-plan $p $n))
	     :effect '(:rank :node $n $p))
    (sc:rule sc 'select-threats
	     :when '((:flaw $g1)
		     (threat nil $g1 $l $t))
	     :effect '(:rank :flaw -1 $g1))
    (sc:def-clause sc (rank-plan p n)
      (bound! 'rank-plan p)
      (when (plan-p p)
	(list (rule-net::match n (rank3 p)))))
    sc))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; A search controller for the prodigy blocks world domain
;;;
;;; (sc-control 'prodigy-sussman #'speed-prodigy-bw)

(defun speed-prodigy-bw (prob)
  (let ((s (bf-mimic prob)))
    (sc:rule s 'rank-goals
	     :when '((:flaw $f) (rank-goals $f $n))
	     :effect '(:rank :flaw $n $f))
    (sc:rule s 'only-puton-for-goals
	     :when '((:current :flaw $g)
		     (reason $p (:step $s $add))
		     (operator $s (stack $b $c) $p)
		     (goal $p $g $t $sn) (neq $sn :goal))
	     :effect '(:reject :node $p))
    (sc:rule s 'prefer-ons
	     :when '((:candidate :flaw 3 $g1)
		     (:candidate :flaw 3 $g2)
		     (goal nil $g1 (on $a $b) :goal)
		     (goal nil $g2 (on $b $c) :goal))
	     :effect '(:prefer :flaw $g2 $g1))
    (sc:def-clause s (rank-goals g n)
      (bound! 'rank-goals g)
      (when (and (openc-p g) 
		 (member (car (openc-condition g)) 
			 '(on on-table holding clear arm-empty)))
	(list (rule-net::match n (cdr (assoc (car (openc-condition g)) 
					     '((on . 3) (on-table . 2)
					       (holding . 1) (clear . 4)
					       (arm-empty . 5))))))))
    s))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; A search controller for the prodigy processp domain
;;;
;;; (sc-control (make-metal-prob "p101") #'speed-machine)

(defun speed-machine (prob)
  (let ((s (bf-mimic prob)))
    (sc:rule s 'rank-goals
	     :when '((:flaw $t) (rank-goals $t $n))
	     :effect '(:rank :flaw $n $t))
    (sc:def-clause s (rank-goals g n)
      (bound! 'rank-goals g)
      (when (and (openc-p g) 
		 (member (car (openc-condition g))
			 '(is-a is-of-type :or)))
	(list (rule-net::match n (cdr (assoc (car (openc-condition g))
					     '((is-a . -2) (is-of-type . -2) 
					       (:or . -1))))))))
    s))

;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; A search controller for the refrigerator domain
;;;
;;; (sc-control 'fixa #'speed-fridge)
;;; (sc-control 'fixb #'speed-fridge)
  
(defun speed-fridge (prob)
  (let ((s (bf-mimic prob)))
    (sc:rule s 'reject-symmetric-holds
	      :when '((reason $p1 (:fact (holds $t $a1 $f)))
		      (reason $p2 (:fact (holds $t $a2 $f)))
		      (prefered-screw $a1 $a2))
	      :effect '(:reject :node $p2))
    (sc:rule s 'no-screw-abstraction
	     :when '((:flaw $s) (goal nil $s $g $sn) (no-screw-abstraction $g))
	     :effect '(:rank :flaw -1 $s))
    (sc:def-clause s (prefered-screw a1 a2)
      (bound! 'prefered-screw a1 a2)
      (when (and (symbolp a1) (symbolp a2)
		 (string< (symbol-name a1) (symbol-name a2)))
	'(nil)))
    (sc:def-clause s (no-screw-abstraction g)
      (bound! 'no-screw-abstraction g)
      (when (not (member (theta-pred g) '(screwed screw)))
	'(nil)))
    s))

;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; A search controller for the ferry domain
;;;
;;; (sc-control 'test-ferry #'speed-ferry)

(defun speed-ferry (prob)
  (declare (ignore prob))
  (let ((s (ucpop-sc 'speed-ferry)))
    (sc:rule s 'select-threats
	     :when '((threat nil $g1 $l $t))
	     :effect '(:select :flaw $g1))
    (sc:rule s 'road-abstraction
	     :when '((goal nil $s $g $sn) (in-road-abstraction $g))
	     :effect '(:select :flaw $s))
    (sc:rule s 'price-plan
	     :when   '((:node $p) (sail-cost $p $n))
	     :effect '(:rank :node $n $p))
    (sc:rule s 'reject-partial-sails
	     :when '((:current :flaw $g)
		     (reason $p (:step $s $a))
		     (operator $s (debark $t1 $t2 $x $y) $p)
		     (goal nil $g $t $sn) (neq $sn :goal))
	     :effect '(:reject :node $p))
    (sc:def-clause s (in-road-abstraction g)
      (bound! 'in-road-abstraction g)
      (when (member (car g) '(at empty-ferry on auto place))
	'(nil)))
    (sc:def-clause s (sail-cost p n)
      (bound! 'sail-cost p)
      (when (plan-p p)
	(let ((cost 0))
	  (dolist (s (plan-steps p))
	    (incf cost (case (car (p-step-action s))
			 ('sail 40)
			 ('board 2)
			 ('debark 2)
			 (t 0))))
	  (list (rule-net::match n cost)))))
    s))

;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; A search controller for the STRIPS world
;;;
;;; (sc-control 'move-boxes #'speed-strips)

(defun speed-strips (prob)
  (let ((s (bf-mimic prob)))
    (sc:rule s 'rank-goals
             :when '((:flaw $f) (rank-goals $f $n))
             :effect '(:rank :flaw $n $f))
    (sc:def-clause s (rank-goals g n)
      (bound! 'rank-goals g)
      (when (openc-p g)
	(let ((x (assoc (car (openc-condition g))
			'((in-room . 1) (is-type . 0) (connects . -1)
			  (loc-in-room . 8) (pushable . 0) (next-to . 5)
			  (statis . 6)))))
	  (when x 
	    (setf x (if (member 'robot (openc-condition g))
			(1+ (cdr x)) (cdr x))))
	  (when x (list (rule-net::match n x))))))
    s))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. Making a search controller for ZENO

(defun flaw-fn (sc p)
  (let ((n (assoc :new (plan-other p))))
    (when n (new-unsafes p)))
  (when (plan-flaws p)
    (do ()
	((or (null (plan-flaws p))
	     (numberp (flaw-rank (car (plan-flaws p)))))
	 (setf (plan-flaws p) 
	   (nconc (mapcar #'(lambda (x)
			      (setf (flaw-rank (cdr x)) (car x))
			      (cdr x))
			  (sc:sort-entries :flaw sc))
		  (plan-flaws p))))
      (sc:assertion sc `(:flaw ,(pop (plan-flaws p)))))
    (let* ((f (car (plan-flaws p))))
      (dolist (a (plan-flaws p))
	(when (< (flaw-rank a) (flaw-rank f)) (setf f a)))
      f)))
      
(defun repair-fn (sc n f)
  (incf *nodes-visited*)
  (preprocess-plan n)
  (dolist (p (postprocess-children n (handle-flaw f n)))
    (sc:assertion sc `(:node ,p))
    (let* ((r1 (bind-variable (cdr (assoc :reason (plan-other p))) 
			      (plan-bindings p))))
;      (print `(reason ,p ,r1)) ;****
      (sc:assertion sc `(reason ,p ,r1))))
  (let ((trace (rule-net:dump-firings)))
    (when trace
      (push (cons :trace trace) (plan-other n)))))

(defun zeno-sc (name)
  (let ((s (sc:make-sc 
	    :name name
	    :flaw-fn #'flaw-fn
	    :repair-fn #'repair-fn
	    )))
    (sc:def-action s (note node slot what value)
      (bound! 'note node slot what value)
      (when (plan-p node)
	(let* ((pad (assoc :note-pad (plan-other node)))
	       (s (assoc slot (cdr pad) :test #'equal)))
	  (unless s
	    (setf s (car (push (cons slot nil) (cdr pad)))))
	  (case what
	    (:set (setf (cdr s) value))
	    (:add (push value (cdr s)))
	    (:del (setf (cdr s) (remove value (cdr s) :test #'equal)))
	    (otherwise
	     (error "action NOTE does not recognize command [~a]" what))))))
    (sc:def-clause s (noted node slot value)
      (bound! 'noted node slot)
      (when (plan-p node)
	(sc::match* value 
		    (cdr (assoc slot
				(cdr (assoc :note-pad (plan-other node))) 
				:test #'equal)))))
    (sc:def-clause s (operator s op p)
      (bound! 'operator s p)
      (when (and (plan-p p) (numberp s))
	(let ((act (bind-variable (get-operator p s) (plan-bindings p))))
	  (sc::match* op (list act)))))
    (sc:def-clause s (goal p g term step)
      (bound! 'goal p g)
      (when (openc-p g)
	(sc::match* (list term step) 
		    (list (list (if (plan-p p)
				    (bind-variable (openc-condition g)
						   (plan-bindings p))
				  (openc-condition g))
				(openc-step g))))))
    (sc:def-clause s (threat p g l s)
      (bound! 'threat p g)
      (when (unsafe-p g)
	(let* ((link (unsafe-link g))
	       (s1 (link-Si link))
	       (c (if (plan-p p) 
		      (bind-variable (link-condition link) (plan-bindings p))
		    (link-condition link)))
	       (s2 (link-Sj link))
	       (step (effect-id (unsafe-clobber-effect g))))
	  (list (rule-net::match (list l s) `((,s1 ,c ,s2) ,step))))))
    (sc:def-clause s (neq p x)
      (bound! 'neq p x)
      (when (not (listp (rule-net::match p x))) '(nil)))
    s))

;;;;;;;;;;;;;;;;
;;; Getting the action name for an ID (for computing reasons)
(defun GET-OPERATOR (plan id)
  (cond ((not (numberp id)) id)
	((= id 0) ':initial)
	(t (dolist (s (plan-steps plan))
	     (when (= (p-step-id s) id) (return (p-step-action s)))))))

(defun bound! (clause &rest args)
  (dolist (arg args)
    (when (rule-net::variable$ arg)
      (error "~%Clause [~a] expects variable [~a] to be bound" clause arg))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;

(defun test-sussman ()
  (sc-control 'sussman-anomaly #'bf-mimic))

(defun test-prodigy-bw ()
  (sc-control 'prodigy-sussman #'speed-prodigy-bw))

(defun test-fixa ()
  (sc-control 'fixa #'speed-fridge))

(defun test-fixb ()
  (sc-control 'fixb #'speed-fridge))

(defun test-ferry ()
  (sc-control 'test-ferry #'speed-ferry))

(defun test-ferry-a ()
  (sc-control 'test-ferry #'alp-test))

(defun alp-test (prob)
  (let ((s (bf-mimic prob)))
    (alpine-sc prob s)
    s))

(defun test-ferry-s ()
  (sc-control 'test-ferry #'stat-test))

(defun stat-test (prob)
  (let ((s (bf-mimic prob)))
    (static-sc prob s)
    s))

(defun alp-stat-test (prob)
  (let ((s (bf-mimic prob)))
    (alpine-sc prob s)
    (static-sc prob s)
    s))
