;;
;;; second file of zeno
;;

(in-package "ZENO")
(use-package "VARIABLE")

;; 
;;; Links to support f(x)=v
;;

(defun rlink-pt->goal (plan step effect open-cond
		       new-step? bindings peeled-goal peeled-ca)
  (let* ((etime (effect-time effect))
	 (gtime (openc-time open-cond))
	 (etheta (effect-post effect))
	 (gtheta (openc-condition open-cond))
	 (v1 (theta-var etheta))
	 (v2 (theta-var gtheta))
	 (newb nil)
	 (equals nil)
	 (plans nil))
    (setf equals `(= ,v1 ,(deref-fnvar v2 (i-start gtime))
		     ,(deref-fnvar v2 (i-end gtime))))
    (setf newb (acons (second equals) (third equals) nil))
    (setf newb (acons (third equals) (fourth equals) newb))
    (setf plans
      (follow-link-instructions
       `((:link (:closed ,etime ,(i-end gtime))
		:time ((<= ,etime ,(i-start gtime)))))
       plan step effect open-cond new-step?
       bindings peeled-goal peeled-ca))
    (dolist (p plans)
      (push etheta (plan-names p))
      (add-metric-constraint p equals)
      (setf (plan-bindings plan)
	(add-bind newb (plan-bindings plan))))
    (values plans)))

(defun rlink-open->goal (plan step effect open-cond
			 new-step? bindings peeled-goal peeled-ca)
  (let* ((etime (effect-time effect))
	 (gtime (openc-time open-cond))
	 (etheta (effect-post effect))
	 (gtheta (openc-condition open-cond))
	 (v2 (theta-var gtheta))
	 (v1 (theta-var etheta))
	 (newb nil)
	 (new-ca nil)
	 (equals nil)
	 (plans nil))
    (setf equals `(= ,(deref-fnvar v2 (i-start gtime))
		     ,(deref-fnvar v2 (i-end gtime))
		     ,(deref-fnvar v1 (i-start gtime)))) ;; connects to new-ca
    (setf newb (acons (second equals) (third equals) nil))
    (setf newb (acons (third equals) (fourth equals) newb))
    (setf new-ca (deref-constraint peeled-ca (i-start gtime)))
    (setf plans
      (follow-link-instructions
       `((:link (:open-start ,(i-start etime) ,(i-end gtime))
		:time ((< ,(i-start etime) ,(i-start gtime))
		       (<= ,(i-end etime) ,(i-end gtime))))
	 (:link ,etime
		:time ((< ,(i-start etime) ,(i-start gtime))
		       (< ,(i-end gtime) ,(i-end etime)))))
       plan step effect open-cond new-step?
       bindings peeled-goal new-ca))
    (dolist (p plans)
      (push etheta (plan-names p))
      (add-metric-constraint p equals)
      (setf (plan-bindings plan)
         (add-bind newb (plan-bindings plan))))
    (values plans)))

(defun rlink-open-start->goal (plan step effect open-cond
				    new-step? bindings peeled-goal peeled-ca)
  (cond ((effect-influence-p effect)
	 (link-influence->goal plan step effect open-cond
			       new-step? bindings peeled-goal peeled-ca))
	(t
	 (*rlink-open-start->goal plan step effect open-cond
				  new-step? bindings peeled-goal peeled-ca))))

(defun *rlink-open-start->goal (plan step effect open-cond
				     new-step? bindings peeled-goal peeled-ca)
  (let* ((etime (effect-time effect))
	 (gtime (openc-time open-cond))
	 (etheta (effect-post effect))
	 (gtheta (openc-condition open-cond))
	 (v2 (theta-var gtheta))
	 (v1 (theta-var etheta))
	 (newb nil)
	 (new-ca nil)
	 (equals nil)
	 (plans nil))
    (setf equals `(= ,(deref-fnvar v2 (i-start gtime))
		     ,(deref-fnvar v2 (i-end gtime))
		     ,(deref-fnvar v1 (i-start gtime))));; connects to new-ca
    (setf newb (acons (second equals) (third equals) nil))
    (setf newb (acons (third equals) (fourth equals) newb))
    (setf new-ca (deref-constraint peeled-ca (i-start gtime)))
    (setf plans
	  (follow-link-instructions
	   `((:link (:open-start ,(i-start etime) ,(i-end gtime))
		    :time ((< ,(i-start etime) ,(i-start gtime))
			   (<= ,(i-end etime) ,(i-end gtime))))
	     (:link ,etime
		    :time ((< ,(i-start etime) ,(i-start gtime))
			   (< ,(i-end gtime) ,(i-end etime)))))
	   plan step effect open-cond new-step?
	   bindings peeled-goal new-ca))
    (dolist (p plans)
					;      (push etheta (plan-names p))  ;... may be useless here
	    (add-metric-constraint p equals)
	    (setf (plan-bindings plan)
		  (add-bind newb (plan-bindings plan))))
    (values plans)))


(defun link-influence->goal (plan step effect open-cond
                             new-step? bindings peeled-goal peeled-ca)
  (nconc
    (persistent-influence-support plan step effect open-cond
           new-step? bindings peeled-goal peeled-ca)
    (active-influence-support plan step effect open-cond
            new-step? bindings peeled-goal peeled-ca)
   ))

(defun set-theta-time (theta time)
   ;; Theta is a funarg, and we want to set its time to TIME.
   `(== (,(theta-pred theta) ,time ,@(theta-args theta))
        ,(deref-fnvar (theta-var theta) time)))

(defun persistent-influence-support
  (plan step effect open-cond new-step? bindings peeled-goal peeled-ca)
  (let* ((etime (effect-time effect))
 	 (gtime (openc-time open-cond))
	 (newb nil)
	 (etheta (effect-post effect))
	 (gtheta (openc-condition open-cond))
	 (dt (effect-influence-p effect));; stored here
	 (v2 (theta-var gtheta)) (v1 (theta-var etheta))
	 (t0 (i-start etime)) (t1 (i-end etime))
	 (t2 (i-start gtime)) (t3 (i-end gtime))
	 (equals nil)
	 (plans nil))
    (if peeled-goal
        (setf peeled-goal (flatten-and-tree 
			   `(:and ,(set-theta-time etheta t0)
				  ,peeled-goal)))
      (setf peeled-goal (set-theta-time etheta t0)))
    (setf equals `(= ,(deref-fnvar v2 t2)
		     ,(deref-fnvar v2 t3)
		     ,(deref-fnvar v1 t1)))
    (setf newb (acons (second equals) (third equals) nil))
    (setf newb (acons (third equals) (fourth equals) newb))
    (setf peeled-ca
	  (list
	   `(= ,(deref-fnvar v1 t1)
	       (+ ,(deref-fnvar v1 t0) (* ,dt (- ,t1 ,t0))))
	   peeled-ca))
    (setq plans
	  (follow-link-instructions
	   `((:link (:closed ,t0 ,t3) :time ((<= ,t1 ,t2))))
	   plan step effect open-cond new-step?
	   bindings peeled-goal peeled-ca))
    (dolist (p plans)
	    (push (set-theta-time etheta t0) (plan-names p))
	    (push (set-theta-time etheta t1) (plan-names p))
	    (push (set-theta-time gtheta t2) (plan-names p))
	    (push (set-theta-time gtheta t3) (plan-names p))
	    (add-metric-constraint p equals)
	    (setf (plan-bindings plan)
		  (add-bind newb (plan-bindings plan))))
    (values plans)))

(defun active-influence-support
  (plan step effect open-cond new-step? bindings peeled-goal peeled-ca)
  (let* ((etime (effect-time effect))
 	 (gtime (openc-time open-cond))
	 (etheta (effect-post effect))
	 (gtheta (openc-condition open-cond))
	 (dt (effect-influence-p effect));; stored here
	 (v2 (theta-var gtheta))
	 (v1 (theta-var etheta))
	 (t0 (i-start etime)) (t1 (i-end etime))
	 (t2 (i-start gtime)) (t3 (i-end gtime))
	 (plans nil))
    (if peeled-goal
        (setf peeled-goal (flatten-and-tree 
			   `(:and ,(set-theta-time etheta t0) ,peeled-goal)))
      (setf peeled-goal (set-theta-time etheta t0)))
    (setf peeled-ca
	  (list				;*
					;      `(= ,(deref-fnvar v1 t1)
					;	     (+ ,(deref-fnvar v2 t3) (* ,dt (- ,t1 ,t3))))
	   `(= ,(deref-fnvar v2 t3)
	       (+ ,(deref-fnvar v2 t2) (* ,dt (- ,t3 ,t2))))
	   `(= ,(deref-fnvar v2 t2)
	       (+ ,(deref-fnvar v1 t0) (* ,dt (- ,t2 ,t0))))
	   peeled-ca))
    (setq plans
	  (follow-link-instructions
	   `((:link (:closed ,t0 ,t1)
		    :time ((<= ,t0 ,t2) (<= ,t3 ,t1))))
	   plan step effect open-cond new-step?
	   bindings peeled-goal peeled-ca))
    (dolist (p plans)
	    (push (set-theta-time etheta t0) (plan-names p))
					;      (push (set-theta-time etheta t1) (plan-names p))
	    (push (set-theta-time gtheta t2) (plan-names p))
	    (push (set-theta-time gtheta t3) (plan-names p))
	    )
    (values plans)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4.3. Adding link to initial conditions for closed world model
;;;
;;; Not implemented yet...!

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a link from a negated goal to the initial conditions
;;;  If a link can be added, a new plan is created with that link
;;;  A link will be added (from :not P to initial conditions) if
;;;
;;;   - for every initial condition Q, P does not unify with Q or
;;;   - for every initial condition Q with which P may unify, there
;;;       exists a binding constraint such that P and Q do not unify
;;;
;;;  If one of the above conditions holds, then a new plan is returned
;;;  with the link added, and any appropriate binding constraints added.

(defun check-goal-time (spec pred schema)
  (unless (eq-member (i-type spec) '(:point :closed :open-end))
    (error "Interval preconditions must have a well-defined starting point.~%~
            A goal for ~s from action ~s references~%~
            interval ~(~s~)."
	   pred schema spec)))

(defun CW-ASSUME (open-cond plan)
  (let* ((theta (openc-condition open-cond))
	 (id (openc-step open-cond))
	 (init  (get-step-with-id plan 0))
	 (effects (cdr (assoc (theta-pred theta) (p-step-add init))))
         (bind-goals nil)
	 (etime (p-step-start init))
	 (gtime (openc-time open-cond))
	 (codesg (plan-bindings plan))
	 (b nil))
    (check-goal-time
     gtime (theta-pred theta)
     (car (p-step-action (get-step-with-id plan id))))
    (dolist (e effects)
      (when (setf b (mgu (effect-post e) (cadr theta) codesg))
	(setf b (car b))
	(unless b (return-from CW-ASSUME nil))
	(push (if (null (cdr b))
		  `(:neq ,(caar b) ,(cdar b))
		`(:or ,@(mapcar #'(lambda (x) 
				    `(:neq ,(car x) ,(cdr x)))
				(car b))))
	      bind-goals)))
    (setf b 
      (tweak-plan 
       plan
       :reason `(:cw-assumption)
       :flaws (remove-1 open-cond (plan-flaws plan))
       :add-goal (when bind-goals
		   (make-openc
		    :condition `(:and ,@bind-goals)
		    :step id))
       :links (cons (make-link
		     :Si 0 :Sj id
		     :condition theta
		     :time (ecase (i-type gtime)
			     (:point `(:closed ,etime ,gtime))
			     (:open-end `(:open-end ,etime ,(i-end gtime)))
			     (:closed `(:closed ,etime ,(i-end gtime)))))
		    (plan-links plan))))
    (when b
      ;; record the new link for later threat detection!
      (setf (plan-other b)
	(acons :new (list (car (plan-links b)) nil)
	       (plan-other b)))
      (cons b nil))))
     

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5. Handling unsafe links

(defun STILL-UNSAFE? (unsafe-ln plan &aux binds)
  ;; Return the bindings necessary to make UNSAFE-LN still unsafe, or
  ;; NIL if it has been resolved.
  (let* ((effect (unsafe-clobber-effect unsafe-ln))
	 (etheta (unsafe-clobber-condition unsafe-ln))
	 (link (unsafe-link unsafe-ln))
	 (ltheta (link-condition link))
	 (ltime (link-time link))
	 (etime (effect-time effect))
	 (ztm (plan-ordering plan)))
    (setf binds (affects effect etheta ltheta (plan-bindings plan)))
    (if (and binds
	     (times-possibly-overlap-p ztm etime ltime))
         binds)))

(defun HANDLE-UNSAFE (unsafe-ln plan &aux binds)
  (setf binds (still-unsafe? unsafe-ln plan))
  (if binds
      (nconc (disable unsafe-ln (car binds) plan)
	     (demote unsafe-ln plan)
	     (promote unsafe-ln plan)
	     )
    (list (tweak-plan plan
		      :reason `(:bogus ,unsafe-ln)
		      :flaws (remove-1 unsafe-ln (plan-flaws plan))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5.1. resolving an unsafe link
;;;
;;;  disable
;;;  demote
;;;  promote

(defvar *minimal-interference-p* t
 "Set to T to insist on minimal interference.")

(defun disable (unsafe-ln binds plan)
  (let* ((effect (unsafe-clobber-effect unsafe-ln))
	 (subgoal (negate&simplify-precond effect binds)))
    (when subgoal
      (let ((etime (effect-time effect))
	    (effect (unsafe-clobber-effect unsafe-ln))
	    (ltime (link-time (unsafe-link unsafe-ln)))
	    (ztm (copy-ztm (plan-ordering plan)))
	    (flaws (remove-1 unsafe-ln (plan-flaws plan)))
	    (p nil)
	    )
	(if *debug* (format t "~&Disabling with ~s~%" subgoal))
	(setf p
	  (tweak-plan plan
		      :reason `(:goal ,subgoal ,(effect-id effect))
		      :flaws flaws
		      :ordering ztm
		      :add-goal
		      (if subgoal
			  (make-openc :condition subgoal
				      :step (effect-id effect)
				      :time nil))))
	(when p
	  ;; universal and/or point effect must start before link ends
	  (set-time< p (i-start etime) (i-end ltime))
	  (list p))))))

(defun get-protected-interval (link plan)
  ;; If we want minimal interference, then consider the interval
  ;; from the beginning of the source step, to the end of the
  ;; consuming step, as an 'interval of possible interference.'
  ;;
  ;; Any step whose effects interfere with the link-condition will
  ;; therefore have to be before the source step, or after the
  ;; consuming step, but nowhere in-between.
  ;;
    (cond (*minimal-interference-p*
	   `(:closed
	      ,(p-step-start (get-step-with-id plan (link-Si link)))
	      ,(p-step-end (get-step-with-id plan (link-Sj link)))))
	  (t
	   (link-time link))))

(defun get-effect-interval (effect plan)
  ;; If we want minimal interference, then consider the entire
  ;; source step as an 'interval of possible interference.'  THus,
  ;; steps may overlap only when they have *nothing* in common with
  ;; respect to the goals of the plan.
  ;;
    (cond ((and *minimal-interference-p*
		(> (effect-id effect) 0))
	   `(:closed
	      ,(p-step-start (get-step-with-id plan (effect-id effect)))
	      ,(p-step-end (get-step-with-id plan (effect-id effect)))))
	  (t
	   (effect-time effect))))
	       
(defun demote (unsafe-ln plan)
  (let ((ltime (get-protected-interval (unsafe-link unsafe-ln) plan))
	(etime (get-effect-interval (unsafe-clobber-effect unsafe-ln) plan))
	(ztm (plan-ordering plan)))
    (when (and ;(plan-bindings plan)
	       (order-ok? ztm etime ltime))
      (if *debug*
	  (format t "~&Demoting ~s before ~s~%" etime ltime))
      (setf ztm (copy-ztm ztm))
      (order-times ztm etime ltime)
      (values
       (list
	(tweak-plan plan :reason `(:order ,etime ,ltime)
		    :ordering ztm
		    :flaws (remove-1 unsafe-ln (plan-flaws plan))))))))

(defun promote (unsafe-ln plan)
  (let ((ltime (get-protected-interval (unsafe-link unsafe-ln) plan))
	(etime (get-effect-interval (unsafe-clobber-effect unsafe-ln) plan))
	(ztm (plan-ordering plan)))
    (when (and ;(plan-bindings plan)
	       (order-ok? ztm ltime etime))
      (if *debug*
	  (format t "~&Promoting ~s after ~s~%" etime ltime))
      (setf ztm (copy-ztm ztm))
      (order-times ztm ltime etime)
      (values
       (list
	(tweak-plan plan :reason `(:order ,ltime ,etime)
		    :ordering ztm
		    :flaws (remove-1 unsafe-ln (plan-flaws plan))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5.2. detecting unsafety conditions

(defun test-link (plan link)
  ;; determine whether the given link is threatened by any
  ;; step in the plan.
  (let ((bind (plan-bindings plan))
	(ztm (plan-ordering plan))
        (orig-effect (link-effect link))
	(ltime (link-time link))
	(ltheta (link-condition link))
	(key (theta-pred (link-condition link)))
	(etheta nil)
	(threats nil))
    (dolist (step (plan-steps plan))
     (dolist (effect (cdr (assoc key (p-step-add step))))
      (unless (eq effect orig-effect)
	(setf etheta (effect-post effect))
	(when (and (times-possibly-overlap-p ztm (effect-time effect) ltime)
	           (affects effect etheta ltheta bind))
	   (push (make-unsafe :link link 
		              :clobber-effect effect
			      :clobber-condition etheta)
		  threats)))))
    (values threats)))

(defun test-effects (plan effects)
  ;; determine whether the given effects threaten any
  ;; links in the plan.
  (let ((bind (plan-bindings plan))
	(ztm (plan-ordering plan))
	(threats nil)
	(ltheta nil)
	(ltime nil)
	(key nil)
	(etheta nil)
	(orig-effect nil))
    (dolist (link (plan-links plan))
      (setf orig-effect (link-effect link)
	    ltheta (link-condition link)
	    ltime (link-time link))
      (setf key (theta-pred ltheta))
      (dolist (effect (cdr (assoc key effects)))
	(unless (eq orig-effect effect)
	  (setf etheta (effect-post effect))
	  (when (and (times-possibly-overlap-p
		      ztm (effect-time effect) ltime)
		     (affects effect etheta ltheta bind)
		     )
	    (push (make-unsafe :link link 
			       :clobber-effect effect
			       :clobber-condition etheta)
		  threats)))))
    (values threats)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6. handle partial orderings.

#-aclpc
(defun unionq (seq1 seq2)
  (declare (optimize (speed 3) (safety 1))
	   (type list seq1)
	   (type list seq2))
  (let ((new seq1))
    (dolist (other seq2)
      (unless (eq-member other new)
	(push other new)))
    (values new)))

#-aclpc
(defun intersectionq (seq1 seq2)
  (declare (optimize (speed 3) (safety 1))
	   (type list seq1)
	   (type list seq2))
  (let ((result nil))
    (dolist (elt1 seq1)
      (when (eq-member elt1 seq2)
	(push elt1 result)))
    (values result)))

(defun same-time? (plan t1 t2)
  (or (eq t1 t2)
      (let ((ztm (plan-ordering plan)))
	(let ((z1 (find-ztime! ztm t1))
	      (z2 (find-ztime! ztm t2)))
	  (cond ((and z1 z2)
		 (= (ztime-mask z1)
		    (logand (ztime-mask z1)
			    (ztime-<= z2)
			    (ztime->= z2))))
		(t
		 (eq t1 t2)))))))

(defun set-time<= (plan before after)
  (let ((ztm (plan-ordering plan)))
    (zset<= ztm
	    (find-ztime ztm before)
	    (find-ztime ztm after))))

(defun set-time< (plan before after)
  (let ((ztm (plan-ordering plan)))
    (zset< ztm
	   (find-ztime ztm before)
	   (find-ztime ztm after))))

(defun set-time= (plan time1 time2)
  (let ((ztm (plan-ordering plan)))
    (let ((t1 (find-ztime ztm time1))
	  (t2 (find-ztime ztm time2)))
      (zset<= ztm t1 t2)
      (zset<= ztm t2 t1))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 7. creating plan entries

;;;;;;;;;;;;;;;;
;;
;; if we match a universal ?x versus ?y, don't include (?x . ?y) in
;; the binding list.

(defun peel-binds (binds effect &aux (ret nil))
  (dolist (b binds ret)
    (unless (eq-member (car b) (effect-forall effect)) (push b ret))))

;;;;;;;;;;;;;;;;
;;;  create a modifixed version of plan
(defun tweak-plan (plan &key
			reason
                        (steps :same)
                        (links :same)
                        (flaws :same)
                        (ordering :same)
                        (bindings :same)
                        (high-step :same)
			(add-goal nil)
			(names :same)
			(constraints :same)
			)
  "return a new plan that is a copy of plan with the supplied
   instance variable tweaked."
  (flet ((tweak-it (keyword plan-accessor)
           (if (eq keyword :same)
               (funcall plan-accessor plan)
	     keyword)))
    (when (tweak-it bindings #'plan-bindings)
      (let ((plan1 (make-plan
		    :names (tweak-it names #'plan-names)
		    :constraints (tweak-it constraints #'plan-constraints)
		    :steps (tweak-it steps #'plan-steps)
		    :links (tweak-it links #'plan-links)
		    :flaws (tweak-it flaws #'plan-flaws)
		    :ordering (tweak-it ordering  #'plan-ordering)
		    :bindings (tweak-it bindings #'plan-bindings)
		    :high-step (tweak-it high-step #'plan-high-step)
		    :other (if (eq reason :same) (plan-other plan)
			     (acons :reason reason nil)))))
	(when add-goal
          (if plan 
	      (setf (plan-constraints plan1) 
                (copy-metric-constraints plan)))
	  (if (openc-condition add-goal)
	      (setf plan1
		(handle-and (list :and 
				  (canonical (openc-condition add-goal)))
			    (openc-step add-goal)
			    (openc-time add-goal)
			    plan1
			    (openc-marked? add-goal)))
	    (setf plan1 plan)))
	(if (and plan1
		 (null (plan-bindings plan1)))
	    (setf plan1 nil))
	(when plan1
	  (push (cons :distance 
		      (if plan (1+ (cdr (assoc :distance (plan-other plan))))
			0))
		(plan-other plan1))
 	  (vcr-frame plan reason plan1)
	  )
	plan1))))

;;;;;;;;;;;;;;;;
;;; 

(defun eq->bind (vars not?)
  (let ((raw (cons (car vars) (cadr vars))))
    (if not?
	(cons :not raw)
      raw)))

(defun new-goal? (form plan)
  (cond ((is-a-fn? form)
	 (new-functional-goal? form plan))
	(t
	 (new-logical-goal? form plan))))

(defun new-functional-goal? (form plan)
  (let ((time (theta-time form))
	(var (theta-var form))
	(bind (plan-bindings plan))
	(ltheta nil))
    (dolist (l (plan-links plan))
      (setf ltheta (link-condition l))
      (when (and (is-a-fn? ltheta)
		 (eq (theta-time ltheta) time)
		 (nec-mgu ltheta form bind))
	(add-metric-constraint plan `(= ,var ,(theta-var ltheta)))
;	(format t "~&Aha!  Found redundant goal ~s~%" form)
	(return-from new-functional-goal? nil)))
    (dolist (g (plan-flaws plan))
      (when (openc-p g)
	(setf ltheta (openc-condition g))
	(when (and (is-a-fn? ltheta)
		   (eq (theta-time ltheta) time)
		   (nec-mgu ltheta form bind))
	  (add-metric-constraint plan `(= ,var ,(theta-var ltheta)))
;	(format t "~&Aha!  Found redundant goal ~s~%" form)
	  (return-from new-functional-goal? nil))))
    (values T)))

(defun new-logical-goal? (form plan)
  (let ((time (theta-time form))
	(bind (plan-bindings plan))
	(ltheta nil))
    (dolist (l (plan-links plan))
      (setf ltheta (link-condition l))
      (when (and (not (is-a-fn? ltheta))
		 (eq (theta-time ltheta) time)
		 (nec-mgu ltheta form bind))
	(return-from new-logical-goal? nil)))
    (dolist (g (plan-flaws plan))
      (when (openc-p g)
	(setf ltheta (openc-condition g))
	(when (and (eq (car ltheta) (car form))
		   (eq (theta-time ltheta) time)
		   (nec-mgu ltheta form bind))
	  (return-from new-logical-goal? nil))))
    (values T)))


(defun handle-and (form step-id time plan
		   &optional (marked? nil))
  (let ((bs nil))
    (dolist (g (cdr form))
      (cond ((eq (car g) :eq)
	     (push (eq->bind (cdr g) nil) bs))
	    ((eq (car g) :neq)
	     (push (eq->bind (cdr g) t) bs))
	    ((eq (car g) :and) 
	     (handle-and g step-id time plan marked?))
	    ((eq (car g) :forall)
	     (handle-and (handle-forall g step-id plan)
			 step-id time plan marked?))
	    ((eq (car g) :exists)
	     (handle-and (handle-exists g step-id plan)
			 step-id time plan marked?))
	    ((r-formula-p g)
	     (add-metric-constraint-over-time plan g time))
	    ((eq g :false)
	     (add-metric-constraint plan `(= 1 0)))
	    ((eq g :true) nil)
	    ((new-goal? g plan)
	     (push (make-openc :condition g
			       :marked? marked?
			       :step step-id
			       :time (compute-goal-time g time))
		   (plan-flaws plan)))
	    (t nil)))
    (cond (bs
	   (setf bs (add-bind bs (plan-bindings plan)))
           (setf (plan-bindings plan) bs)
	   (if bs plan nil))
	  (t plan))))

(defun compute-goal-time (form time)
  (if time
      time
    (if (not (eq-member (car form) '(:and :or :forall :exists)))
	(theta-time form)
      time)))
	
(defun add-metric-constraint-over-time (plan con time)
  (cond ((null time)
	 ;; simple constraint w/out temporal functions
	 (add-metric-constraint plan con))
	((not (eq :closed (car time)))
	 (error "Illegal time interval for metric constraints!~%~s~%~s"
		con time))
	(t
	 (add-metric-constraint
	  plan
	  (deref-constraint con (i-start time)))
	 (add-metric-constraint
	  plan
	  (deref-constraint con (i-end time))))))
	 
;;;;;;;;;;;;;;;;
;;;  see if one goal is a strict negation of the other.
(defun negates (n1 n2)
  (let ((p1 (if (eq (car n1) :not) n1 n2))
	(p2 (if (eq (car n1) :not) n2 n1)))
    (and (eq (car p1) :not)
	 (equal p2 (cadr p1)))))

;;;;;;;;;;;;;;;;
;;; 
(defun temporal-quantifier? (args)
  (dolist (a args nil)
    (if (eq (car a) 'time) (return t))))

(defun handle-forall (goal step-id plan)
  (cond ((temporal-quantifier? (cadr goal))
	 (handle-forall-with-time goal step-id plan))
	(t
	 (handle-normal-forall goal plan))))

(defun extract-universal-times (goal)
  (let ((times nil))
    (dolist (var (cadr goal))
      (when (eq (car var) 'time)
	(push var times)))
    (values times)))

(defun handle-forall-with-time (goal step-id plan)
  ;; goal is of the form (:forall ((time ?t) ()*) ...).  extract
  ;; out the timepoint ?t, post goals for the start and end of step-id,
  ;; add the constraint to the list of universal constraints, then extract
  ;; all time from the goal.
  (let ((utimes (extract-universal-times goal))
	(unies (cadr goal)))
    (cond ((> (list-length utimes) 1)
	   (error "this zeno implementation can't handle multiple ~
                   universal times:~%~s for step ~s."
		  goal (get-step-with-id plan step-id)))
	  (t
	   ;; assuming goal looks like (:forall ((time t)) <c>)
	   ;; First, transform existentials into functions.
	   (setf goal (car (fixup-interval-time-refs goal)))
	   ;; extract out all temporal refs, leaving others.
	   (setf unies (set-difference unies utimes))
	   ;; utimes as ((time ?t code from to)); cddar gives (code from to)
	   (setf utimes (cddar utimes))
	   (if (null utimes)
	       (error "Missing temporal specifier for :FORALL time:~%~s" goal))
	   (if (null unies)
	       (setf goal (third goal))
	       (setf goal `(:forall ,unies ,(third goal))))
	   (push
	    (make-openc :condition goal :time utimes
			:step step-id :marked? nil)
	    (plan-flaws plan))
	   (values nil)))))
	
(defun handle-normal-forall (goal plan &aux temp)
  (declare (ignore plan))
  (let ((entries (cdr (assoc (theta-pred (third goal)) *forever*)))
	(opens (third goal)))
    (labels
	((handle* (vars alst)
	   (if (null vars) (push (v-sublis alst opens) temp)
	     (dolist (e entries)
	       ;; entries will be (type :forever instance)
	       (when (eq (car e) (caar vars))
		 (handle* (cdr vars) (acons (cadar vars) (third e) alst)))))))
      (handle* (cadr goal) nil))
    (cons :and temp)))

;;;;;;;;;;;;;;;;
;;;
(defun V-SUBLIS (alst e)
  (cond ((consp e) (cons (v-sublis alst (car e)) (v-sublis alst (cdr e))))
	((variable? e)
	 (let ((a (assoc e alst)))
	   (if a (cdr a) e)))
	(t e)))
	 
;;;;;;;;;;;;;;;;
;;; 
(defun HANDLE-EXISTS (goal step-id plan)
  (cond ((temporal-quantifier? (cadr goal))
	 (handle-exists-with-time goal step-id plan))
	(t
	 (handle-normal-exists goal))))

(defun HANDLE-NORMAL-EXISTS (goal)
  (let ((alst (mapcar #'(lambda (x) 
                         (if (atom x) (uniquify-var x)
                             (cons (cadr x) (uniquify-var (cadr x)))))
		      (cadr goal)))
	(subgoals nil))
    (dolist (item (cadr goal))
      (if (consp item)
	  (push `(,(car item) :forever ,@(cdr item)) subgoals)))
    (v-sublis alst `(:and ,(caddr goal) ,@subgoals))))

(defun HANDLE-EXISTS-WITH-TIME (goal step-id plan)
   ;; I assume that we have only 1 quantifer!!
   ;;
   ;; goal is (:exists ((time ?t :type from to)) ...)
   ;;
  (let ((step (get-step-with-id plan step-id))
        (time (cdr (first (cadr goal))))
        (ztm (plan-ordering plan)))
    (if (second (cadr goal))
	(error "I'm sorry -- temporal quantifiers must appear alone~%~
                in :exists clauses (see step ~a)."
	       (car (p-step-action step))))
    (let ((t1 (find-ztime ztm (i-start (cdr time))))
          (t2 (find-ztime ztm (i-end (cdr time))))
          (new (find-ztime ztm (uniquify-var (car time)))))
       (ecase (i-type (cdr time))
          (:point (zset= ztm new t1))
          (:open-end (zset<= ztm t1 new) (zset< ztm new t2))
          (:open-start (zset< ztm t1 new) (zset<= ztm new t2))
          (:open (zset< ztm t1 new) (zset< ztm new t2))
          (:closed (zset<= ztm t1 new) (zset<= ztm new t2)))
       `(:and
              ,(sublis (acons (car time) (ztime-id new) nil) 
                        (third goal))))))

;;;;;;;;;;;;;;;;
;;; 
(defun HANDLE-NOT (eqn)
  ;; Negate EQN
  (let ((head (car eqn))
	(tail (cdr eqn)))
    (cond ((eq :eq head)  (negate-eq tail))
	  ((eq :neq head) (cons :eq tail))
	  ((eq :not head) (canonical (car tail)))
	  ((eq :or head)  (cons :and (mapcar #'handle-not tail)))
	  ((eq :and head) (cons :or (mapcar #'handle-not tail)))
	  ((eq :forall head) 
	   `(:exists ,(car tail) ,(handle-not (cadr tail))))
	  ((eq :exists head) 
	   `(:forall ,(car tail) ,(handle-not (cadr tail))))
	  ((r-formula-p eqn)
	   (handle-negated-constraint eqn))
	  ((and (consp eqn) (is-a-fn? eqn))
	   :false)
	  ((eq eqn :false) :true)
	  ((eq eqn :true) :false)
	  (t (list :not eqn)))))

(defun HANDLE-NEGATED-CONSTRAINT (c)
  (ecase (car c)
    (=  (list :or
	      (cons '< (cdr c))
	      (cons '> (cdr c))))
    (<  (cons '>= (cdr c)))
    (>  (cons '<= (cdr c)))
    (>= (cons '< (cdr c)))
    (<= (cons '> (cdr c)))))

(defun NEGATE-eq (c)
  (cond ((eq :true (car c))
	 `(:eq :false ,@(cdr c)))
	((eq :false (car c))
	 `(:eq :true ,@(cdr c)))
	((eq :true (cadr c))
	 `(:eq :false ,(car c)))
	((eq :false (cadr c))
	 `(:eq :true ,(car c)))
	(t
	 `(:neq ,@c))))

;;;;;;;;;;;;;;;;
;;; 
(defun CANONICAL (eqn)
;  (if (equal (second eqn) '(:and)) (error "yoq"))
  (cond ((atom eqn) eqn)
	((eq :not (car eqn)) (handle-not (cadr eqn)))
	((or (eq (car eqn) :or) (eq (car eqn) :and))
	 (cons (car eqn) (mapcar #'canonical (cdr eqn))))
	((eq :forall (car eqn)) 
	 `(:forall ,(cadr eqn) ,(canonical (caddr eqn))))
	((eq :exists (car eqn)) 
	 `(:exists ,(cadr eqn) ,(canonical (caddr eqn))))
	(t eqn)))

;;;;;;;;;;;;;;;;
;;;  Make a plan and keep track of the total number of plans created.
(defun MAKE-PLAN (&rest args)
  (count-stat .created.)
  (apply #'make-plan* args))
